--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Ada.Containers.Hashed_Sets;
with Ada.Exceptions; use Ada.Exceptions;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;

with System;
with System.Memory;

with Langkit_Support.Generic_API; use Langkit_Support.Generic_API;
with Langkit_Support.Hashes;
with Langkit_Support.Slocs; use Langkit_Support.Slocs;
with Langkit_Support.Token_Data_Handlers;
use Langkit_Support.Token_Data_Handlers;

with Libadalang.Common;       use Libadalang.Common;
with Libadalang.Generic_API;  use Libadalang.Generic_API;
with Libadalang.Generic_API.Introspection;
use Libadalang.Generic_API.Introspection;
with Libadalang.Implementation;
with Libadalang.Lexer_Implementation;
use Libadalang.Lexer_Implementation;

with Libadalang.Unparsing_Implementation;
use Libadalang.Unparsing_Implementation;

package body Libadalang.Rewriting_Implementation is

   use type System.Memory.size_t;

   function Convert is new Ada.Unchecked_Conversion
     (Rewriting_Handle, Rewriting_Handle_Pointer);
   function Convert is new Ada.Unchecked_Conversion
     (Rewriting_Handle_Pointer, Rewriting_Handle);

   function Hash is new Langkit_Support.Hashes.Hash_Access
     (Node_Rewriting_Handle_Type, Node_Rewriting_Handle);
   package NRH_Sets is new Ada.Containers.Hashed_Sets
     (Element_Type        => Node_Rewriting_Handle,
      Equivalent_Elements => "=",
      Hash                => Hash);

   procedure Pre_Check (Value : Boolean; Msg : String);
   --  Raise a Precondition_Failure exception with the given message
   --  if the Value is False.

   function Parent_Is_List (Node : Node_Rewriting_Handle) return Boolean
   is (Node.Parent /= null and then Node.Parent.Children.Kind = Expanded_List);
   --  Return whether Node's parent is a list node.
   --
   --  Since a tied and non-root node rewriting handle can exist only when its
   --  parent is expanded, it is safe checking the parent's Children field.

   function Index_For
     (Handle : Node_Rewriting_Handle;
      Member : Struct_Member_Ref) return Positive;
   --  Return the 1-based index of the ``Member`` parse field in the node
   --  referenced ``Handle``. Raise a ``Precondition_Failure`` if there is no
   --  such member.

   function Index_In_Parent_List
     (Handle : Node_Rewriting_Handle) return Positive;
   --  Assuming ``Handle`` is a node whose parent is a list node, return its
   --  1-based index in that list node.

   procedure Set_Child
     (Handle : Node_Rewriting_Handle;
      Index  : Positive;
      Child  : Node_Rewriting_Handle);
   --  Assign ``Child`` to the child slot at the given ``Index`` in ``Handle``

   ---------------
   -- Pre_Check --
   ---------------

   procedure Pre_Check (Value : Boolean; Msg : String) is
   begin
      if not Value then
         raise Precondition_Failure with Msg;
      end if;
   end Pre_Check;

   ---------------
   -- Index_For --
   ---------------

   function Index_For
     (Handle : Node_Rewriting_Handle;
      Member : Struct_Member_Ref) return Positive
   is
      Result : Positive := 1;
   begin
      if Language (Owner (Member)) /= Self_Id then
         raise Precondition_Failure with "invalid member language";
      elsif Is_Property (Member) then
         raise Precondition_Failure with "got property, parse field expected";
      end if;

      declare
         T            : constant Type_Ref := Kind_To_Type (Handle.Kind);
         Node_Members : constant Struct_Member_Ref_Array := Members (T);
      begin
         if Is_Null_For (Member, T) then
            raise Precondition_Failure
              with "invalid reference to the node rewriting handle of a null"
                   & " field";
         end if;
         for M of Node_Members loop
            if M = Member then
               return Result;
            elsif not Is_Property (M) and not Is_Null_For (M, T) then
               Result := Result + 1;
            end if;
         end loop;
      end;

      return (raise Precondition_Failure with "no such member on this node");
   end Index_For;

   --------------------------
   -- Index_In_Parent_List --
   --------------------------

   function Index_In_Parent_List
     (Handle : Node_Rewriting_Handle) return Positive
   is
      Parent : constant Node_Rewriting_Handle := Handle.Parent;
   begin
      for I in 1 .. Natural (Parent.Children.Vector.Length) loop
         if Parent.Children.Vector.Element (I) = Handle then
            return I;
         end if;
      end loop;

      --  We should not be able to reach this point, as ``Handle`` should
      --  always be present in the list of children of ``Handle.Parent``
      --  (otherwise the tree of node rewriting handles is corrupted).

      return (raise Program_Error);
   end Index_In_Parent_List;

   

   

   

   

   

   

   

   

   

   

   

   

   

   

   function Handle (Context : Internal_Context) return Rewriting_Handle is
     (Convert (Get_Rewriting_Handle (Context)));

   function Context (Handle : Rewriting_Handle) return Internal_Context is
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Context;
   end Context;

   function Allocate
     (Kind          : Ada_Node_Kind_Type;
      Context       : Rewriting_Handle;
      Unit_Handle   : Unit_Rewriting_Handle;
      Parent_Handle : Node_Rewriting_Handle)
      return Node_Rewriting_Handle
      with Pre =>
         Context /= No_Rewriting_Handle
         and then (Unit_Handle = No_Unit_Rewriting_Handle
                   or else Unit_Handle.Context_Handle = Context)
         and then (Parent_Handle = No_Node_Rewriting_Handle
                   or else Parent_Handle.Context_Handle = Context);

   function Allocate
     (Node          : Bare_Ada_Node;
      Context       : Rewriting_Handle;
      Unit_Handle   : Unit_Rewriting_Handle;
      Parent_Handle : Node_Rewriting_Handle)
      return Node_Rewriting_Handle
      with Pre =>
         Context /= No_Rewriting_Handle
         and then (Unit_Handle = No_Unit_Rewriting_Handle
                   or else Unit_Handle.Context_Handle = Context)
         and then (Parent_Handle = No_Node_Rewriting_Handle
                   or else Parent_Handle.Context_Handle = Context);
   --  Allocate a handle for Node and register it in Unit_Handle's map

   function Allocate_Stub
     (Context : Rewriting_Handle) return Node_Rewriting_Handle;
   --  Allocate a stub rewriting node in ``Context``, to be used as a temporary
   --  node in ``Rotate``.

   procedure Expand_Children (Node : Node_Rewriting_Handle)
      with Pre => Node /= No_Node_Rewriting_Handle;
   --  If Node.Children.Kind is Unexpanded, populate Node's list of Children to
   --  mimic the related bare AST node. Otherwise, do nothing.

   procedure Free_Handles (Handle : in out Rewriting_Handle);
   --  Free all resources tied to Handle. This also releases the rewriting
   --  handle singleton in Handle's Context.

   procedure Tie
     (Handle, Parent : Node_Rewriting_Handle;
      Unit           : Unit_Rewriting_Handle);
   --  Tie the node represented by handle so that either:
   --
   --    * it is the root of Unit (Parent is null);
   --    * it is a child of Parent (Unit is null).
   --
   --  Do nothing if Handle is null.

   procedure Untie (Handle : Node_Rewriting_Handle);
   --  Untie the node represented by Handle. Do nothing if Handle is null.

   -------------------------
   -- C_Context_To_Handle --
   -------------------------

   function C_Context_To_Handle
     (Context : Internal_Context) return Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Handle (Context);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Context_To_Handle;

   -------------------------
   -- C_Handle_To_Context --
   -------------------------

   function C_Handle_To_Context
     (Handle : Rewriting_Handle) return Internal_Context is
   begin
      Clear_Last_Exception;
      return Context (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Handle_To_Context;

   ---------------------
   -- Start_Rewriting --
   ---------------------

   function Start_Rewriting
     (Context : Internal_Context) return Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle (Context) = No_Rewriting_Handle,
         "Handle (Context) must be null");
   

      declare
         Result : constant Rewriting_Handle := new Rewriting_Handle_Type'
           (Context   => Context,
            Units     => <>,
            Pool      => Create,
            New_Nodes => <>,
            Stubs     => <>);
      begin
         Result.New_Nodes := Nodes_Pools.Create (Result.Pool);
         Result.Stubs := Nodes_Pools.Create (Result.Pool);
         Set_Rewriting_Handle (Context, Convert (Result));
         return Result;
      end;
   end Start_Rewriting;

   -----------------------
   -- C_Start_Rewriting --
   -----------------------

   function C_Start_Rewriting
     (Context : Internal_Context) return Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Start_Rewriting (Context);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Start_Rewriting;

   ---------------------
   -- Abort_Rewriting --
   ---------------------

   procedure Abort_Rewriting (Handle : in out Rewriting_Handle) is
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   
      Free_Handles (Handle);
   end Abort_Rewriting;

   -----------------------
   -- C_Abort_Rewriting --
   -----------------------

   procedure C_Abort_Rewriting (Handle : Rewriting_Handle) is
      H : Rewriting_Handle := Handle;
   begin
      Clear_Last_Exception;
      Abort_Rewriting (H);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Abort_Rewriting;

   -----------
   -- Apply --
   -----------

   function Apply (Handle : in out Rewriting_Handle) return Apply_Result is

      type Processed_Unit_Record is record
         Unit     : Internal_Unit;
         New_Data : Reparsed_Unit;
      end record;
      type Processed_Unit is access Processed_Unit_Record;
      procedure Free is new Ada.Unchecked_Deallocation
        (Processed_Unit_Record, Processed_Unit);

      package Processed_Unit_Vectors is new Ada.Containers.Vectors
        (Positive, Processed_Unit);

      Units  : Processed_Unit_Vectors.Vector;
      Result : Apply_Result := (Success => True);

   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

      --  Try to reparse all units that were potentially modified
      for Unit_Handle of Handle.Units loop
         declare
            PU    : constant Processed_Unit := new Processed_Unit_Record'
              (Unit     => Unit_Handle.Unit,
               New_Data => <>);
            Input : Internal_Lexer_Input :=
              (Kind        => Bytes_Buffer,
               Charset     => <>,
               Read_BOM    => False,
               Bytes       => System.Null_Address,
               Bytes_Count => 0);
            Bytes : String_Access;

            function Error_Result return Apply_Result
            is ((Success => False, Unit => PU.Unit, Diagnostics => <>));
         begin
            Units.Append (PU);

            --  Reparse (i.e. unparse and then parse) this rewritten unit
            begin
               Bytes := Unparse
                 (Create_Abstract_Node (Unit_Handle.Root),
                  PU.Unit,
                  Preserve_Formatting => True,
                  As_Unit             => True);
            exception
               when Exc : Malformed_Tree_Error =>
                  Result := Error_Result;
                  Append
                    (Result.Diagnostics,
                     No_Source_Location_Range,
                     To_Text (Exception_Message (Exc)));
                  exit;
            end;
            Input.Charset := Unit_Handle.Unit.Charset;
            Input.Bytes := Bytes.all'Address;
            Input.Bytes_Count := Bytes.all'Length;
            Do_Parsing (PU.Unit, Input, PU.New_Data);
            Free (Bytes);

            --  If there is a parsing error, abort the rewriting process
            if not PU.New_Data.Diagnostics.Is_Empty then
               Result := Error_Result;
               Result.Diagnostics.Move (PU.New_Data.Diagnostics);
               Destroy (PU.New_Data);
               exit;
            end if;
         end;
      end loop;

      --  If all reparsing went fine, actually replace the AST nodes all over
      --  the context and free all resources associated to Handle.
      if Result.Success then
         for PU of Units loop
            Update_After_Reparse (PU.Unit, PU.New_Data);
         end loop;
         Free_Handles (Handle);
      end if;

      --  Clean-up our local resources and return
      for PU of Units loop
         Free (PU);
      end loop;
      return Result;
   end Apply;

   -------------
   -- C_Apply --
   -------------

   procedure C_Apply
     (Handle : Rewriting_Handle;
      Result : access C_Apply_Result)
   is
      H          : Rewriting_Handle := Handle;
      Ada_Result : Apply_Result;
   begin
      Clear_Last_Exception;

      Ada_Result := Apply (H);
      Result.Success := (if Ada_Result.Success then 1 else 0);
      if not Ada_Result.Success then
         Result.Unit := Ada_Result.Unit;
         if Ada_Result.Diagnostics.Is_Empty then
            Result.Diagnostics_Count := 0;
            Result.Diagnostics := null;
         else
            Result.Diagnostics_Count :=
              Interfaces.C.int (Ada_Result.Diagnostics.Length);
            declare
               type Array_Type is
                 array (1 .. Natural (Result.Diagnostics_Count))
                 of ada_diagnostic;
               Diagnostics_Address : constant System.Address :=
                 System.Memory.Alloc (Array_Type'Size / 8);
               Diagnostics         : Array_Type
                 with Import, Address => Diagnostics_Address;
            begin
               Result.Diagnostics :=
                 C_Diagnostic_Array.To_Pointer (Diagnostics_Address);
               for I in Diagnostics'Range loop
                  declare
                     D : Diagnostic renames Ada_Result.Diagnostics (I);
                  begin
                     Diagnostics (I).Sloc_Range := Wrap (D.Sloc_Range);
                     Diagnostics (I).Message :=
                       Wrap_Alloc (To_Text (D.Message));
                  end;
               end loop;
            end;
         end if;
      end if;

   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Apply;

   -----------------------
   -- Free_Apply_Result --
   -----------------------

   procedure Free_Apply_Result (Result : access C_Apply_Result) is
      use type Interfaces.C.int;
   begin
      Clear_Last_Exception;

      if Result.Success = 0 then
         if Result.Diagnostics_Count /= 0 then
            declare
               type Array_Type is
                 array (1 .. Natural (Result.Diagnostics_Count))
                 of ada_diagnostic;
               Diagnostics_Address : constant System.Address :=
                 (C_Diagnostic_Array.To_Address (Result.Diagnostics));
               Diagnostics         : Array_Type
                 with Import, Address => Diagnostics_Address;
            begin
               for D of Diagnostics loop
                  ada_destroy_text
                    (D.Message'Unrestricted_Access);
               end loop;
            end;
            System.Memory.Free
              (C_Diagnostic_Array.To_Address (Result.Diagnostics));
         end if;
      end if;

   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end Free_Apply_Result;

   ------------------
   -- Unit_Handles --
   ------------------

   function Unit_Handles
     (Handle : Rewriting_Handle) return Unit_Rewriting_Handle_Array is
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

      declare
         Count  : constant Natural := Natural (Handle.Units.Length);
         Result : Unit_Rewriting_Handle_Array (1 .. Count);
         I      : Positive := 1;
      begin
         for Unit of Handle.Units loop
            Result (I) := Unit;
            I := I + 1;
         end loop;
         return Result;
      end;
   end Unit_Handles;

   --------------------
   -- C_Unit_Handles --
   --------------------

   function C_Unit_Handles
     (Handle : Rewriting_Handle) return C_Unit_Array.Object_Pointer is
   begin
      Clear_Last_Exception;

      declare
         Units : constant Unit_Rewriting_Handle_Array := Unit_Handles (Handle);
         type Array_Type is
           array (Units'First .. Units'Last + 1)
           of Unit_Rewriting_Handle;
         Result_Address : constant System.Address :=
           System.Memory.Alloc (Array_Type'Size / 8);
         Result         : Array_Type
           with Import, Address => Result_Address;
      begin
         for I in Units'Range loop
            Result (I) := Units (I);
         end loop;
         Result (Result'Last) := null;
         return C_Unit_Array.To_Pointer (Result_Address);
      end;

   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Unit_Handles;

   ------------
   -- Handle --
   ------------

   function Handle (Unit : Internal_Unit) return Unit_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle (Context (Unit)) /= No_Rewriting_Handle,
         "Handle (Context (Unit)) should not be null");
   
      
      Pre_Check
        (not Has_Diagnostics (Unit),
         "Unit must not have diagnostics");
   

      declare
         use Unit_Maps;

         Context        : constant Internal_Context := Unit.Context;
         Context_Handle : constant Rewriting_Handle := Handle (Context);
         Filename       : constant Unbounded_String :=
            To_Unbounded_String (Get_Filename (Unit));

         Cur : constant Cursor := Context_Handle.Units.Find (Filename);
      begin
         if Cur /= No_Element then
            return Element (Cur);
         end if;

         declare
            Result : constant Unit_Rewriting_Handle :=
               new Unit_Rewriting_Handle_Type'(Context_Handle => Context_Handle,
                                               Unit           => Unit,
                                               Root           => <>,
                                               Nodes          => <>);
         begin
            Context_Handle.Units.Insert (Filename, Result);
            Result.Root := Handle (Root (Unit));
            return Result;
         end;
      end;
   end Handle;

   ----------
   -- Unit --
   ----------

   function Unit (Handle : Unit_Rewriting_Handle) return Internal_Unit
   is
   begin
      
      Pre_Check
        (Handle /= No_Unit_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Unit;
   end Unit;

   ----------------------
   -- C_Unit_To_Handle --
   ----------------------

   function C_Unit_To_Handle
     (Unit : Internal_Unit) return Unit_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Handle (Unit);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Unit_To_Handle;

   ----------------------
   -- C_Handle_To_Unit --
   ----------------------

   function C_Handle_To_Unit
     (Handle : Unit_Rewriting_Handle) return Internal_Unit is
   begin
      Clear_Last_Exception;
      return Unit (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Handle_To_Unit;

   ----------
   -- Root --
   ----------

   function Root (Handle : Unit_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      
      Pre_Check
        (Handle /= No_Unit_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Root;
   end Root;

   ------------
   -- C_Root --
   ------------

   function C_Root
     (Handle : Unit_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Root (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Root;

   --------------
   -- Set_Root --
   --------------

   procedure Set_Root
     (Handle : Unit_Rewriting_Handle;
      Root   : Node_Rewriting_Handle) is
   begin
      
      Pre_Check
        (Handle /= No_Unit_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Root = No_Node_Rewriting_Handle or else not Tied (Root),
         "Root must not be tied to another rewriting context.");
   

      Untie (Handle.Root);
      Handle.Root := Root;
      Tie (Root, No_Node_Rewriting_Handle, Handle);
   end Set_Root;

   ----------------
   -- C_Set_Root --
   ----------------

   procedure C_Set_Root
     (Handle : Unit_Rewriting_Handle;
      Root   : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Set_Root (Handle, Root);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Set_Root;

   -------------
   -- Unparse --
   -------------

   function Unparse
     (Handle : Unit_Rewriting_Handle) return Unbounded_Text_Type is
   begin
      
      Pre_Check
        (Handle /= No_Unit_Rewriting_Handle,
         "Handle should not be null");
   
      return Unparsing_Implementation.Unparse
        (Node                => Create_Abstract_Node (Handle.Root),
         Unit                => Handle.Unit,
         Preserve_Formatting => True,
         As_Unit             => True);
   end Unparse;

   ---------------
   -- C_Unparse --
   ---------------

   procedure C_Unparse
     (Handle : Unit_Rewriting_Handle; Result : access ada_text)
   is
      Text : Unbounded_Text_Type;
   begin
      Clear_Last_Exception;
      Text := Unparse (Handle);
      Result.all := Wrap_Alloc (Text);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Unparse;

   ------------
   -- Handle --
   ------------

   function Handle (Node : Bare_Ada_Node) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle (Context (Node.Unit)) /= No_Rewriting_Handle,
         "Handle (Context (Node.Unit)) should not be null");
   
      
      Pre_Check
        (not Has_Diagnostics (Node.Unit),
         "Node.Unit must not have diagnostics");
   

      if Node = null then
         return No_Node_Rewriting_Handle;
      end if;

      declare
         use Node_Maps;

         Unit_Handle : constant Unit_Rewriting_Handle :=
            Handle (Node.Unit);
         Cur         : constant Cursor := Unit_Handle.Nodes.Find (Node);
      begin
         --  If we have already built a handle for this node, just return it
         if Cur /= No_Element then
            return Element (Cur);

         --  Otherwise, if this node has a parent, make sure this parent has
         --  its own handle, then expand its children. This last must create
         --  the handle we are supposed to return.
         elsif Node.Parent /= null then
            Expand_Children (Handle (Node.Parent));
            return Element (Unit_Handle.Nodes.Find (Node));
         end if;

         --  Otherwise, we are dealing with the root node: just create its
         --  rewriting handle.
         return Allocate (Node, Unit_Handle.Context_Handle, Unit_Handle,
                          No_Node_Rewriting_Handle);
      end;
   end Handle;

   ----------------------
   -- C_Node_To_Handle --
   ----------------------

   function C_Node_To_Handle (Node : ada_base_node) return Node_Rewriting_Handle
   is
      N : Bare_Ada_Node;
   begin
      Clear_Last_Exception;
      N := Unwrap (Node);
      return Handle (N);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Node_To_Handle;

   ----------
   -- Node --
   ----------

   function Node
     (Handle : Node_Rewriting_Handle) return Bare_Ada_Node is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Node;
   end Node;

   ----------------------
   -- C_Handle_To_Node --
   ----------------------

   function C_Handle_To_Node
     (Handle : Node_Rewriting_Handle) return ada_base_node
   is
      N : Bare_Ada_Node;
   begin
      Clear_Last_Exception;
      N := Node (Handle);
      return Wrap (N);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return Wrap (null);
   end C_Handle_To_Node;

   -------------
   -- Context --
   -------------

   function Context (Handle : Node_Rewriting_Handle) return Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Context_Handle;
   end Context;

   -----------------------
   -- C_Node_To_Context --
   -----------------------

   function C_Node_To_Context
     (Node : Node_Rewriting_Handle) return Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Context (Node);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Node_To_Context;

   -------------
   -- Unparse --
   -------------

   function Unparse (Handle : Node_Rewriting_Handle) return Text_Type is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return To_Wide_Wide_String
        (Unparsing_Implementation.Unparse
           (Create_Abstract_Node (Handle),
            Unit                => null,
            Preserve_Formatting => True,
            As_Unit             => False));
   end Unparse;

   ---------------
   -- C_Unparse --
   ---------------

   procedure C_Unparse
     (Handle : Node_Rewriting_Handle; Result : access ada_text) is
   begin
      Clear_Last_Exception;
      declare
         Text : constant Text_Type := Unparse (Handle);
      begin
         Result.all := Wrap_Alloc (Text);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Unparse;

   --------------
   -- Allocate --
   --------------

   function Allocate
     (Kind          : Ada_Node_Kind_Type;
      Context       : Rewriting_Handle;
      Unit_Handle   : Unit_Rewriting_Handle;
      Parent_Handle : Node_Rewriting_Handle)
      return Node_Rewriting_Handle
   is
      Tied : constant Boolean := Unit_Handle /= No_Unit_Rewriting_Handle;
   begin
      return new Node_Rewriting_Handle_Type'
        (Context_Handle => Context,
         Node           => null,
         Parent         => Parent_Handle,
         Previous       => No_Node_Rewriting_Handle,
         Next           => No_Node_Rewriting_Handle,
         Kind           => Kind,
         Tied           => Tied,
         Root_Of        =>
           (if Tied and then Parent_Handle = No_Node_Rewriting_Handle
            then Unit_Handle
            else No_Unit_Rewriting_Handle),
         Children       => Unexpanded_Children);
   end Allocate;

   --------------
   -- Allocate --
   --------------

   function Allocate
     (Node          : Bare_Ada_Node;
      Context       : Rewriting_Handle;
      Unit_Handle   : Unit_Rewriting_Handle;
      Parent_Handle : Node_Rewriting_Handle)
      return Node_Rewriting_Handle
   is
      Result : constant Node_Rewriting_Handle := Allocate
        (Node.Kind, Context, Unit_Handle, Parent_Handle);
   begin
      Result.Node := Node;
      if Result.Tied then
         Unit_Handle.Nodes.Insert (Node, Result);
      end if;
      return Result;
   end Allocate;

   -------------------
   -- Allocate_Stub --
   -------------------

   function Allocate_Stub
     (Context : Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      return Allocate
        (Kind          => Ada_Node_Kind_Type'First,
         Context       => Context,
         Unit_Handle   => No_Unit_Rewriting_Handle,
         Parent_Handle => No_Node_Rewriting_Handle);
   end Allocate_Stub;

   ---------------------
   -- Expand_Children --
   ---------------------

   procedure Expand_Children (Node : Node_Rewriting_Handle) is
      Children : Node_Children renames Node.Children;
   begin
      --  If this handle has already be expanded, there is nothing to do

      if Children.Kind /= Unexpanded then
         return;
      end if;

      --  Otherwise, expand to the appropriate children form

      declare
         N           : constant Bare_Ada_Node := Node.Node;
         Unit_Handle : constant Unit_Rewriting_Handle :=
            Handle (N.Unit);

         function Allocate_Child
           (Child : Bare_Ada_Node) return Node_Rewriting_Handle
         is (if Child = null
             then No_Node_Rewriting_Handle
             else Allocate
                   (Child, Unit_Handle.Context_Handle, Unit_Handle, Node));
      begin
         if Is_Token_Node (N) then

            --  N is a token node: its expanded form contains only its text

            Children := (Kind => Expanded_Token_Node,
                         Text => To_Unbounded_Wide_Wide_String (Text (N)));

         elsif Is_List_Node (N.Kind) then

            --  N is a list node: its expanded form contains a doubly linked
            --  list for its children.

            declare
               Count                : constant Natural := Children_Count (N);
               First, Last, Current : Node_Rewriting_Handle :=
                 No_Node_Rewriting_Handle;
            begin
               for I in 1 .. Count loop
                  Current := Allocate_Child (Implementation.Child (N, I));
                  if First = No_Node_Rewriting_Handle then
                     First := Current;
                     Last := Current;
                  else
                     Last.Next := Current;
                     Current.Previous := Last;
                     Last := Current;
                  end if;
               end loop;
               Children :=
                 (Kind  => Expanded_List,
                  First => First,
                  Last  => Last,
                  Count => Count);
            end;

         else
            --  N is a regular node: its expanded form contains a vector for
            --  all non-null syntax fields.

            Children := (Kind => Expanded_Regular, Vector => <>);
            declare
               Count : constant Natural := Children_Count (N);
            begin
               Children.Vector.Reserve_Capacity
                 (Ada.Containers.Count_Type (Count));
               for I in 1 .. Count loop
                  Children.Vector.Append
                    (Allocate_Child (Implementation.Child (N, I)));
               end loop;
            end;
         end if;
      end;
   end Expand_Children;

   ------------------
   -- Free_Handles --
   ------------------

   procedure Free_Handles (Handle : in out Rewriting_Handle) is

      procedure Free is new Ada.Unchecked_Deallocation
        (Rewriting_Handle_Type, Rewriting_Handle);
      procedure Free is new Ada.Unchecked_Deallocation
        (Unit_Rewriting_Handle_Type, Unit_Rewriting_Handle);
      procedure Free is new Ada.Unchecked_Deallocation
        (Node_Rewriting_Handle_Type, Node_Rewriting_Handle);

      Ctx : constant Internal_Context := Context (Handle);
   begin
      --  Free all resources tied to Handle
      for Unit of Handle.Units loop
         for Node of Unit.Nodes loop
            Free (Node);
         end loop;
         Free (Unit);
      end loop;
      for Node of Handle.New_Nodes loop
         declare
            N : Node_Rewriting_Handle := Node;
         begin
            Free (N);
         end;
      end loop;
      for Node of Handle.Stubs loop
         declare
            N : Node_Rewriting_Handle := Node;
         begin
            Free (N);
         end;
      end loop;
      Free (Handle.Pool);
      Free (Handle);

      --  Release the rewriting handle singleton for its context
      Set_Rewriting_Handle (Ctx, Convert (Handle));
   end Free_Handles;

   ---------
   -- Tie --
   ---------

   procedure Tie
     (Handle, Parent : Node_Rewriting_Handle;
      Unit           : Unit_Rewriting_Handle) is
   begin
      if Handle /= No_Node_Rewriting_Handle then
         Handle.Parent := Parent;
         Handle.Tied := True;
         if Parent = No_Node_Rewriting_Handle then
            Handle.Root_Of := Unit;
         end if;
      end if;
   end Tie;

   -----------
   -- Untie --
   -----------

   procedure Untie (Handle : Node_Rewriting_Handle) is
   begin
      if Handle /= No_Node_Rewriting_Handle then
         Handle.Parent := No_Node_Rewriting_Handle;
         Handle.Previous := No_Node_Rewriting_Handle;
         Handle.Next := No_Node_Rewriting_Handle;
         Handle.Tied := False;
         Handle.Root_Of := No_Unit_Rewriting_Handle;
      end if;
   end Untie;

   ----------
   -- Kind --
   ----------

   function Kind (Handle : Node_Rewriting_Handle) return Ada_Node_Kind_Type is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Kind;
   end Kind;

   ------------
   -- C_Kind --
   ------------

   function C_Kind (Handle : Node_Rewriting_Handle) return ada_node_kind_enum is
   begin
      Clear_Last_Exception;
      declare
         K : constant Ada_Node_Kind_Type := Kind (Handle);
      begin
         return ada_node_kind_enum (K'Enum_Rep);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return 0;
   end C_Kind;

   -----------
   -- Image --
   -----------

   function Image (Handle : Node_Rewriting_Handle) return String is
   begin
      if Handle = No_Node_Rewriting_Handle then
         return "None";
      end if;

      declare
         Tied_Suffix : constant String :=
           (if Tied (Handle) then " (tied)" else "");
      begin
         if Node (Handle) = null then
            declare
               K          : constant Ada_Node_Kind_Type := Kind (Handle);
               Tok_Suffix : constant String :=
                 (if Is_Token_Node (K)
                  then " " & Image (Text (Handle), With_Quotes => True)
                  else "");
            begin
               return "<" & K'Image & Tok_Suffix & Tied_Suffix & ">";
            end;
         else
            declare
               Ent : constant Internal_Entity :=
                 (Node => Node (Handle),
                  Info => No_Entity_Info);
               Img : constant String := Image (Ent);
            begin
               return Img (Img'First .. Img'Last - 1) & Tied_Suffix & ">";
            end;
         end if;
      end;
   end Image;

   -------------
   -- C_Image --
   -------------

   procedure C_Image
     (Handle : Node_Rewriting_Handle; Result : access ada_text) is
   begin
      Clear_Last_Exception;
      declare
         Img : constant Text_Type := To_Text (Image (Handle));
      begin
         Result.all := Wrap_Alloc (Img);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Image;

   ----------
   -- Tied --
   ----------

   function Tied (Handle : Node_Rewriting_Handle) return Boolean is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Tied;
   end Tied;

   ------------
   -- C_Tied --
   ------------

   function C_Tied (Handle : Node_Rewriting_Handle) return Interfaces.C.int is
   begin
      Clear_Last_Exception;
      return (if Tied (Handle) then 1 else 0);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return 0;
   end C_Tied;

   ------------
   -- Parent --
   ------------

   function Parent
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Handle.Parent;
   end Parent;

   --------------
   -- C_Parent --
   --------------

   function C_Parent
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Parent (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Parent;

   --------------------
   -- Children_Count --
   --------------------

   function Children_Count (Handle : Node_Rewriting_Handle) return Natural is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return
        (case Handle.Children.Kind is
         when Unexpanded          => Children_Count (Handle.Node),
         when Expanded_Regular    => Natural (Handle.Children.Vector.Length),
         when Expanded_List       => Handle.Children.Count,
         when Expanded_Token_Node => 0);
   end Children_Count;

   ----------------------
   -- C_Children_Count --
   ----------------------

   function C_Children_Count
     (Handle : Node_Rewriting_Handle) return Interfaces.C.int is
   begin
      Clear_Last_Exception;
      return Interfaces.C.int (Children_Count (Handle));
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return 0;
   end C_Children_Count;

   -----------
   -- Child --
   -----------

   function Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref) return Node_Rewriting_Handle
   is
      Index : Positive;
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   

      Index := Index_For (Handle, Field);

      --  If this handle represents an already existing node, make sure it is
      --  expanded so we have a handle to return.
      Expand_Children (Handle);
      return Handle.Children.Vector.Element (Index);
   end Child;

   -------------
   -- C_Child --
   -------------

   function C_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Interfaces.C.int) return Node_Rewriting_Handle
   is
      F : Struct_Member_Ref;
   begin
      Clear_Last_Exception;
      F := From_Index (Self_Id, Struct_Member_Index (Field));
      return Child (Handle, F);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Child;

   --------------
   -- Children --
   --------------

   function Children
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle_Array
   is
      I : Positive := 1;
      N : Node_Rewriting_Handle;
   begin
      return Result : Node_Rewriting_Handle_Array
                        (1 .. Children_Count (Handle))
      do
         if Is_List_Node (Handle) then
            N := First_Child (Handle);
            while N /= No_Node_Rewriting_Handle loop
               Result (I) := N;
               I := I + 1;
               N := Next_Child (N);
            end loop;
         else
            declare
               T            : constant Type_Ref :=
                 Kind_To_Type (Kind (Handle));
               Node_Members : constant Struct_Member_Ref_Array := Members (T);
            begin
               for M of Node_Members loop
                  if Is_Field (M) and then not Is_Null_For (M, T) then
                     Result (I) := Child (Handle, M);
                     I := I + 1;
                  end if;
               end loop;
            end;
         end if;
      end return;
   end Children;

   ----------------
   -- C_Children --
   ----------------

   procedure C_Children
     (Handle   : Node_Rewriting_Handle;
      Children : access C_Node_Array.Object_Pointer;
      Count    : access Interfaces.C.int) is
   begin
      Clear_Last_Exception;

      declare
         Result : constant Node_Rewriting_Handle_Array :=
           Rewriting_Implementation.Children (Handle);

         subtype Array_Type is Node_Rewriting_Handle_Array (Result'Range);
         Children_Address : constant System.Address :=
           System.Memory.Alloc (Array_Type'Size / 8);
         Ada_Children     : Array_Type
           with Import, Address => Children_Address;
      begin
         Ada_Children := Result;
         Children.all := C_Node_Array.To_Pointer (Children_Address);
         Count.all := Result'Length;
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Children;

   ---------------
   -- Set_Child --
   ---------------

   procedure Set_Child
     (Handle : Node_Rewriting_Handle;
      Index  : Positive;
      Child  : Node_Rewriting_Handle) is
   begin
      --  If this handle represents an already existing node, make sure it is
      --  expanded so that its children vector can be modified.
      Expand_Children (Handle);

      declare
         Child_Slot : Node_Rewriting_Handle renames
            Handle.Children.Vector.Reference (Index);
      begin
         --  Untie the child to be replaced if it exists
         Untie (Child_Slot);

         --  Tie the new child if it exists
         Tie (Child, Handle, No_Unit_Rewriting_Handle);

         Child_Slot := Child;
      end;
   end Set_Child;

   ---------------
   -- Set_Child --
   ---------------

   procedure Set_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref;
      Child  : Node_Rewriting_Handle) is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Child = No_Node_Rewriting_Handle or else not Tied (Child),
         "Child must not be tied to another rewriting context.");
   

      Set_Child (Handle, Index_For (Handle, Field), Child);
   end Set_Child;

   -----------------
   -- C_Set_Child --
   -----------------

   procedure C_Set_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Interfaces.C.int;
      Child  : Node_Rewriting_Handle)
   is
      F : Struct_Member_Ref;
   begin
      Clear_Last_Exception;
      F := From_Index (Self_Id, Struct_Member_Index (Field));
      Set_Child (Handle, F, Child);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Set_Child;

   ----------
   -- Text --
   ----------

   function Text (Handle : Node_Rewriting_Handle) return Text_Type is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_Token_Node (Kind (Handle)),
         "Expected a token node. Got " & Kind (Handle)'Image);
   

      case Handle.Children.Kind is
         when Unexpanded =>
            if Is_Token_Node (Handle.Kind) then
               return Text (Handle.Node);
            else
               raise Program_Error;
            end if;
         when Expanded_Regular | Expanded_List =>
            return (raise Program_Error);
         when Expanded_Token_Node =>
            return To_Wide_Wide_String (Handle.Children.Text);
      end case;
   end Text;

   ------------
   -- C_Text --
   ------------

   procedure C_Text
     (Handle : Node_Rewriting_Handle; Result : access ada_text) is
   begin
      Clear_Last_Exception;
      declare
         T : constant Text_Type := Text (Handle);
      begin
         Result.all := Wrap_Alloc (T);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Text;

   --------------
   -- Set_Text --
   --------------

   procedure Set_Text (Handle : Node_Rewriting_Handle; Text : Text_Type) is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_Token_Node (Kind (Handle)),
         "Expected a token node. Got " & Kind (Handle)'Image);
   

      --  Make sure Handle is expanded so we have a Text field to override
      Expand_Children (Handle);

      Handle.Children.Text := To_Unbounded_Wide_Wide_String (Text);
   end Set_Text;

   ----------------
   -- C_Set_Text --
   ----------------

   procedure C_Set_Text
     (Handle : Node_Rewriting_Handle; Text : access ada_text)
   is
      T : constant Text_Type (1 .. Natural (Text.Length))
      with Import, Address => Text.Chars;
   begin
      Clear_Last_Exception;
      Set_Text (Handle, T);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Set_Text;

   -------------
   -- Replace --
   -------------

   procedure Replace (Handle, New_Node : Node_Rewriting_Handle) is
      Parent : Node_Rewriting_Handle;
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Tied (Handle),
         "Handle must be tied to an analysis unit.");
   
      
      Pre_Check
        (New_Node = No_Node_Rewriting_Handle or else not Tied (New_Node),
         "New_Node must not be tied to another rewriting context.");
   

      if Handle = New_Node then
         return;
      end if;

      if Handle.Root_Of = No_Unit_Rewriting_Handle then

         --  If Handle is not the root node of its owning unit, go replace it
         --  in its parent's children list.

         if Parent_Is_List (Handle) then
            if New_Node = No_Node_Rewriting_Handle then
               Remove_Child (Handle);
               return;
            end if;

            Parent := Handle.Parent;
            if Handle.Previous = No_Node_Rewriting_Handle then
               Parent.Children.First := New_Node;
            else
               Handle.Previous.Next := New_Node;
            end if;
            if Handle.Next = No_Node_Rewriting_Handle then
               Parent.Children.Last := New_Node;
            else
               Handle.Next.Previous := New_Node;
            end if;
            Tie (New_Node, Parent, No_Unit_Rewriting_Handle);
            New_Node.Previous := Handle.Previous;
            New_Node.Next := Handle.Next;
            Untie (Handle);
         else
            Set_Child (Handle.Parent, Index_In_Parent_List (Handle), New_Node);
         end if;

      else
         --  Otherwise, replace it as a root node
         Set_Root (Handle.Root_Of, New_Node);
      end if;
   end Replace;

   ---------------
   -- C_Replace --
   ---------------

   procedure C_Replace (Handle, New_Node : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Replace (Handle, New_Node);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Replace;

   ------------
   -- Rotate --
   ------------

   procedure Rotate (Handles : Node_Rewriting_Handle_Array) is

      function Non_Null_Tied (Handle : Node_Rewriting_Handle) return Boolean
      is (Handle /= No_Node_Rewriting_Handle and then Tied (Handle));

      RH : Rewriting_Handle := No_Rewriting_Handle;
   begin
      --  Rotate is a no-op if there are less than two handles or none is tied

      if Handles'Length < 2 then
         return;
      end if;

      for H of Handles loop
         if Non_Null_Tied (H) then
            RH := H.Context_Handle;
            exit;
         end if;
      end loop;
      if RH = No_Rewriting_Handle then
         return;
      end if;

      --  Check that each non-null handle is present at most once in the input
      --  list.

      declare
         Handle_Set : NRH_Sets.Set;
      begin
         for H of Handles loop
            if H /= No_Node_Rewriting_Handle then
               begin
                  Handle_Set.Insert (H);
               exception
                  when Constraint_Error =>
                     raise Precondition_Failure with
                       "non-null handles can be present at most once";
               end;
            end if;
         end loop;
      end;

      --  Now that inputs are validated, we can start the rotation. As a
      --  reminder: replace H1 by H2, H2 by H3, ... Or in other words: put H2
      --  in the location of H1, put H3 in the location of H1.

      declare
         Stubs       : array (Handles'Range) of Node_Rewriting_Handle;
         Stub_Cursor : Nodes_Pools.Cursor := Nodes_Pools.First (RH.Stubs);
      begin
         --  First create stubs to replace the nodes that will be replaced with
         --  other nodes.

         for I in Handles'Range loop
            declare
               Node : constant Node_Rewriting_Handle := Handles (I);
               Repl : constant Node_Rewriting_Handle :=
                 Handles (if I = Handles'Last then Handles'First else I + 1);
               Stub : Node_Rewriting_Handle renames Stubs (I);
            begin
               --  Get a stub if needed. If an already allocated stub is
               --  available, use it, otherwise allocate one.

               if Non_Null_Tied (Node) and then Non_Null_Tied (Repl) then
                  if Nodes_Pools.Has_Element (RH.Stubs, Stub_Cursor) then
                     Stub := Nodes_Pools.Get (RH.Stubs, Stub_Cursor);
                     Stub_Cursor := Nodes_Pools.Next (RH.Stubs, Stub_Cursor);
                  else
                     Stub := Allocate_Stub (RH);
                     Nodes_Pools.Append (RH.Stubs, Stub);
                  end if;
               else
                  Stub := No_Node_Rewriting_Handle;
               end if;
            end;
         end loop;

         --  Now replace nodes with their stubs. We have to do this as separate
         --  pass because the replacement affects the "Tied" predicate that is
         --  used in the first pass.

         for I in Handles'Range loop
            if Handles (I) /= No_Node_Rewriting_Handle then
               Replace (Handles (I), Stubs (I));
            end if;
         end loop;

         --  Now that all the nodes to rotate are untied, replace stubs with
         --  the definitive nodes.

         for I in Handles'Range loop
            declare
               Stub : constant Node_Rewriting_Handle := Stubs (I);
               Repl : Node_Rewriting_Handle;
            begin
               if Stub /= No_Node_Rewriting_Handle then
                  Repl := Handles
                    (if I = Handles'Last then Handles'First else I + 1);
                  Replace (Stub, Repl);
               end if;
            end;
         end loop;
      end;
   end Rotate;

   --------------
   -- C_Rotate --
   --------------

   procedure C_Rotate
     (Handles : C_Node_Array.Object_Pointer;
      Count   : Interfaces.C.int) is
   begin
      Clear_Last_Exception;

      declare
         subtype Array_Type is
           Node_Rewriting_Handle_Array (1 .. Natural (Count));
         Handles_Address : constant System.Address :=
           C_Node_Array.To_Address (Handles);
         Ada_Handles     : constant Array_Type
           with Import, Address => Handles_Address;
      begin
         Rotate (Ada_Handles);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Rotate;

   ------------------
   -- Is_List_Node --
   ------------------

   function Is_List_Node (Handle : Node_Rewriting_Handle) return Boolean is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      return Is_List_Node (Kind (Handle));
   end Is_List_Node;

   -----------------
   -- First_Child --
   -----------------

   function First_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle)),
         "Expected a list node. Got " & Kind (Handle)'Image);
   

      Expand_Children (Handle);
      return Handle.Children.First;
   end First_Child;

   -------------------
   -- C_First_Child --
   -------------------

   function C_First_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return First_Child (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_First_Child;

   ----------------
   -- Last_Child --
   ----------------

   function Last_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle)),
         "Expected a list node. Got " & Kind (Handle)'Image);
   

      Expand_Children (Handle);
      return Handle.Children.Last;
   end Last_Child;

   ------------------
   -- C_Last_Child --
   ------------------

   function C_Last_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Last_Child (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Last_Child;

   ----------------
   -- Next_Child --
   ----------------

   function Next_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Handle.Parent /= No_Node_Rewriting_Handle,
         "Handle.Parent should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle.Parent)),
         "Expected a list node. Got " & Kind (Handle.Parent)'Image);
   

      return Handle.Next;
   end Next_Child;

   ------------------
   -- C_Next_Child --
   ------------------

   function C_Next_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      Clear_Last_Exception;
      return Next_Child (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Next_Child;

   --------------------
   -- Previous_Child --
   --------------------

   function Previous_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Handle.Parent /= No_Node_Rewriting_Handle,
         "Handle.Parent should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle.Parent)),
         "Expected a list node. Got " & Kind (Handle.Parent)'Image);
   

      return Handle.Previous;
   end Previous_Child;

   ----------------------
   -- C_Previous_Child --
   ----------------------

   function C_Previous_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      Clear_Last_Exception;
      return Previous_Child (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Previous_Child;

   -------------------
   -- Insert_Before --
   -------------------

   procedure Insert_Before (Handle, New_Sibling : Node_Rewriting_Handle) is
      Old_Previous, Parent : Node_Rewriting_Handle;
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      Parent := Handle.Parent;
      
      Pre_Check
        (Parent /= No_Node_Rewriting_Handle,
         "Parent should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Parent)),
         "Expected a list node. Got " & Kind (Parent)'Image);
   
      
      Pre_Check
        (New_Sibling = No_Node_Rewriting_Handle or else not Tied (New_Sibling),
         "New_Sibling must not be tied to another rewriting context.");
   

      Old_Previous := Handle.Previous;
      if Old_Previous = No_Node_Rewriting_Handle then
         Handle.Parent.Children.First := New_Sibling;
      else
         Old_Previous.Next := New_Sibling;
      end if;
      New_Sibling.Previous := Old_Previous;
      New_Sibling.Next := Handle;
      Handle.Previous := New_Sibling;
      Tie (New_Sibling, Parent, No_Unit_Rewriting_Handle);
      Parent.Children.Count := Parent.Children.Count + 1;
   end Insert_Before;

   ---------------------
   -- C_Insert_Before --
   ---------------------

   procedure C_Insert_Before (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Insert_Before (Handle, New_Sibling);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Insert_Before;

   ------------------
   -- Insert_After --
   ------------------

   procedure Insert_After (Handle, New_Sibling : Node_Rewriting_Handle) is
      Old_Next, Parent : Node_Rewriting_Handle;
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      Parent := Handle.Parent;
      
      Pre_Check
        (Parent /= No_Node_Rewriting_Handle,
         "Parent should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Parent)),
         "Expected a list node. Got " & Kind (Parent)'Image);
   
      
      Pre_Check
        (New_Sibling = No_Node_Rewriting_Handle or else not Tied (New_Sibling),
         "New_Sibling must not be tied to another rewriting context.");
   

      Old_Next := Handle.Next;
      if Old_Next = No_Node_Rewriting_Handle then
         Handle.Parent.Children.Last := New_Sibling;
      else
         Old_Next.Previous := New_Sibling;
      end if;
      New_Sibling.Next := Old_Next;
      New_Sibling.Previous := Handle;
      Handle.Next := New_Sibling;
      Tie (New_Sibling, Parent, No_Unit_Rewriting_Handle);
      Parent.Children.Count := Parent.Children.Count + 1;
   end Insert_After;

   --------------------
   -- C_Insert_After --
   --------------------

   procedure C_Insert_After (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Insert_After (Handle, New_Sibling);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Insert_After;

   ------------------
   -- Insert_First --
   ------------------

   procedure Insert_First (Handle, New_Child : Node_Rewriting_Handle) is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle)),
         "Expected a list node. Got " & Kind (Handle)'Image);
   
      
      Pre_Check
        (New_Child = No_Node_Rewriting_Handle or else not Tied (New_Child),
         "New_Child must not be tied to another rewriting context.");
   

      Expand_Children (Handle);
      if Handle.Children.First /= No_Node_Rewriting_Handle then
         Handle.Children.First.Previous := New_Child;
         New_Child.Next := Handle.Children.First;
      end if;
      Handle.Children.First := New_Child;
      if Handle.Children.Last = No_Node_Rewriting_Handle then
         Handle.Children.Last := New_Child;
      end if;
      Tie (New_Child, Handle, No_Unit_Rewriting_Handle);
      Handle.Children.Count := Handle.Children.Count + 1;
   end Insert_First;

   --------------------
   -- C_Insert_First --
   --------------------

   procedure C_Insert_First (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Insert_First (Handle, New_Sibling);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Insert_First;

   -----------------
   -- Insert_Last --
   -----------------

   procedure Insert_Last (Handle, New_Child : Node_Rewriting_Handle) is
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle)),
         "Expected a list node. Got " & Kind (Handle)'Image);
   
      
      Pre_Check
        (New_Child = No_Node_Rewriting_Handle or else not Tied (New_Child),
         "New_Child must not be tied to another rewriting context.");
   

      Expand_Children (Handle);
      if Handle.Children.Last /= No_Node_Rewriting_Handle then
         Handle.Children.Last.Next := New_Child;
         New_Child.Previous := Handle.Children.Last;
      end if;
      Handle.Children.Last := New_Child;
      if Handle.Children.First = No_Node_Rewriting_Handle then
         Handle.Children.First := New_Child;
      end if;
      Tie (New_Child, Handle, No_Unit_Rewriting_Handle);
      Handle.Children.Count := Handle.Children.Count + 1;
   end Insert_Last;

   -------------------
   -- C_Insert_Last --
   -------------------

   procedure C_Insert_Last (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Insert_Last (Handle, New_Sibling);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Insert_Last;

   ------------------
   -- Remove_Child --
   ------------------

   procedure Remove_Child (Handle : Node_Rewriting_Handle) is
      Parent : Node_Rewriting_Handle;
   begin
      
      Pre_Check
        (Handle /= No_Node_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Handle.Parent /= No_Node_Rewriting_Handle,
         "Handle.Parent should not be null");
   
      
      Pre_Check
        (Is_List_Node (Kind (Handle.Parent)),
         "Expected a list node. Got " & Kind (Handle.Parent)'Image);
   

      Expand_Children (Handle);
      Parent := Handle.Parent;
      if Parent.Children.First = Handle then
         Parent.Children.First := Handle.Next;
         if Handle.Next = No_Node_Rewriting_Handle then
            Parent.Children.Last := No_Node_Rewriting_Handle;
         else
            Handle.Next.Previous := No_Node_Rewriting_Handle;
         end if;
      elsif Parent.Children.Last = Handle then
         Parent.Children.Last := Handle.Previous;
         Parent.Children.Last.Next := No_Node_Rewriting_Handle;
      else
         Handle.Previous.Next := Handle.Next;
         Handle.Next.Previous := Handle.Previous;
      end if;
      Untie (Handle);
      Parent.Children.Count := Parent.Children.Count - 1;
   end Remove_Child;

   --------------------
   -- C_Remove_Child --
   --------------------

   procedure C_Remove_Child (Handle : Node_Rewriting_Handle) is
   begin
      Clear_Last_Exception;
      Remove_Child (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
   end C_Remove_Child;

   -----------
   -- Clone --
   -----------

   function Clone
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
   is
      Result : Node_Rewriting_Handle;
   begin
      if Handle = No_Node_Rewriting_Handle then
         return Handle;
      end if;

      --  Make sure the original handle is expanded so we can iterate on it
      Expand_Children (Handle);

      --  If the input handle is associated to a node, so should be the cloned
      --  handle, so that its formatting is copied as well.
      Result :=
        (if Handle.Node = null
         then Allocate (Handle.Kind, Handle.Context_Handle,
                        No_Unit_Rewriting_Handle, No_Node_Rewriting_Handle)
         else Allocate (Handle.Node, Handle.Context_Handle,
                        No_Unit_Rewriting_Handle, No_Node_Rewriting_Handle));
      Nodes_Pools.Append (Handle.Context_Handle.New_Nodes, Result);

      --  Recursively clone children
      case Handle.Children.Kind is
         when Unexpanded =>
            raise Program_Error;

         when Expanded_Token_Node =>
            Result.Children := (Kind => Expanded_Token_Node,
                                Text => Handle.Children.Text);

         when Expanded_List =>
            declare
               First, Last, Cloned : Node_Rewriting_Handle :=
                 No_Node_Rewriting_Handle;
               Current             : Node_Rewriting_Handle :=
                 Handle.Children.First;
            begin
               while Current /= No_Node_Rewriting_Handle loop
                  Cloned := Clone (Current);
                  Tie (Cloned, Result, No_Unit_Rewriting_Handle);
                  if First = No_Node_Rewriting_Handle then
                     First := Cloned;
                     Last := Cloned;
                  else
                     Last.Next := Cloned;
                     Cloned.Previous := Last;
                     Last := Cloned;
                  end if;
                  Current := Next_Child (Current);
               end loop;
               Result.Children :=
                 (Kind  => Expanded_List,
                  First => First,
                  Last  => Last,
                  Count => Handle.Children.Count);
            end;

         when Expanded_Regular =>
            Result.Children := (Kind => Expanded_Regular, Vector => <>);
            Result.Children.Vector.Reserve_Capacity
              (Handle.Children.Vector.Length);
            for I in 1 .. Handle.Children.Vector.Last_Index loop
               declare
                  Child : constant Node_Rewriting_Handle :=
                    Clone (Handle.Children.Vector.Element (I));
               begin
                  Tie (Child, Result, No_Unit_Rewriting_Handle);
                  Result.Children.Vector.Append (Child);
               end;
            end loop;
      end case;

      return Result;
   end Clone;

   -------------
   -- C_Clone --
   -------------

   function C_Clone
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      return Clone (Handle);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Clone;

   -----------------
   -- Create_Node --
   -----------------

   function Create_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (not Is_Error_Node (Kind),
         "Expected a non-error node. Got " & Kind'Image);
   

      if Is_Token_Node (Kind) then
         return Create_Token_Node (Handle, Kind, "");
      else
         declare
            Count    : constant Integer := Kind_To_Node_Children_Count (Kind);
            Children : constant Node_Rewriting_Handle_Array (1 ..  Count) :=
               (others => No_Node_Rewriting_Handle);
         begin
            return Create_Regular_Node (Handle, Kind, Children);
         end;
      end if;
   end Create_Node;

   -------------------
   -- C_Create_Node --
   -------------------

   function C_Create_Node
     (Handle : Rewriting_Handle;
      Kind   : ada_node_kind_enum) return Node_Rewriting_Handle
   is
      K : Ada_Node_Kind_Type;
   begin
      Clear_Last_Exception;
      K := Ada_Node_Kind_Type'Enum_Val (Kind);
      return Create_Node (Handle, K);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Create_Node;

   -----------------------
   -- Create_Token_Node --
   -----------------------

   function Create_Token_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type;
      Text   : Text_Type) return Node_Rewriting_Handle is
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (Is_Token_Node (Kind),
         "Expected a token node. Got " & Kind'Image);
   

      declare
         Result : constant Node_Rewriting_Handle := Allocate
           (Kind, Handle, No_Unit_Rewriting_Handle, No_Node_Rewriting_Handle);
      begin
         Result.Children := (Kind => Expanded_Token_Node,
                             Text => To_Unbounded_Wide_Wide_String (Text));
         Nodes_Pools.Append (Handle.New_Nodes, Result);
         return Result;
      end;
   end Create_Token_Node;

   -------------------------
   -- C_Create_Token_Node --
   -------------------------

   function C_Create_Token_Node
     (Handle : Rewriting_Handle;
      Kind   : ada_node_kind_enum;
      Text   : access ada_text) return Node_Rewriting_Handle
   is
      K : Ada_Node_Kind_Type;
      T : constant Text_Type (1 .. Natural (Text.Length))
      with Import, Address => Text.Chars;
   begin
      Clear_Last_Exception;
      K := Ada_Node_Kind_Type'Enum_Val (Kind);
      return Create_Token_Node (Handle, K, T);
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Create_Token_Node;

   -------------------------
   -- Create_Regular_Node --
   -------------------------

   function Create_Regular_Node
     (Handle   : Rewriting_Handle;
      Kind     : Ada_Node_Kind_Type;
      Children : Node_Rewriting_Handle_Array) return Node_Rewriting_Handle
   is
      List : Boolean;
   begin
      
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   
      
      Pre_Check
        (not Is_Token_Node (Kind),
         "Expected a token node. Got " & Kind'Image);
   
      
      Pre_Check
        (not Is_Error_Node (Kind),
         "Expected a non-error node. Got " & Kind'Image);
   
      List := Is_List_Node (Kind);
      for One_Child of Children loop
         if List then
            
      Pre_Check
        (One_Child /= No_Node_Rewriting_Handle and then not Tied (One_Child),
         "One_Child must not be tied to another rewriting context.");
   
         else
            
      Pre_Check
        (One_Child = No_Node_Rewriting_Handle or else not Tied (One_Child),
         "One_Child must not be tied to another rewriting context.");
   
         end if;
      end loop;

      declare
         Result : Node_Rewriting_Handle := Allocate
           (Kind, Handle, No_Unit_Rewriting_Handle, No_Node_Rewriting_Handle);
      begin
         if List then
            declare
               First, Last : Node_Rewriting_Handle := No_Node_Rewriting_Handle;
            begin
               for C of Children loop
                  if First = No_Node_Rewriting_Handle then
                     First := C;
                  else
                     Last.Next := C;
                     C.Previous := Last;
                  end if;
                  Last := C;
                  Tie (C, Result, No_Unit_Rewriting_Handle);
               end loop;
               Result.Children :=
                 (Kind  => Expanded_List,
                  First => First,
                  Last  => Last,
                  Count => Children'Length);
            end;
         else
            Result.Children := (Kind   => Expanded_Regular,
                                Vector => <>);
            Result.Children.Vector.Reserve_Capacity (Children'Length);
            for C of Children loop
               Result.Children.Vector.Append (C);
               if C /= No_Node_Rewriting_Handle then
                  Tie (C, Result, No_Unit_Rewriting_Handle);
               end if;
            end loop;
         end if;
         Nodes_Pools.Append (Handle.New_Nodes, Result);
         return Result;
      end;
   end Create_Regular_Node;

   ---------------------------
   -- C_Create_Regular_Node --
   ---------------------------

   function C_Create_Regular_Node
     (Handle   : Rewriting_Handle;
      Kind     : ada_node_kind_enum;
      Children : C_Node_Array.Object_Pointer;
      Count    : Interfaces.C.int) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      declare
         K : constant Ada_Node_Kind_Type := Ada_Node_Kind_Type'Enum_Val (Kind);

         subtype Array_Type is
           Node_Rewriting_Handle_Array (1 .. Natural (Count));
         Children_Address : constant System.Address :=
           C_Node_Array.To_Address (Children);
         Ada_Children     : constant Array_Type
           with Import, Address => Children_Address;
      begin
         return Create_Regular_Node (Handle, K, Ada_Children);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Create_Regular_Node;

   --------------------------
   -- Create_From_Template --
   --------------------------

   function Create_From_Template
     (Handle    : Rewriting_Handle;
      Template  : Text_Type;
      Arguments : Node_Rewriting_Handle_Array;
      Rule      : Grammar_Rule) return Node_Rewriting_Handle
   is
      type State_Type is (
         Default,
         --  Default state: no meta character being processed

         Open_Brace,
         --  The previous character is a open brace: the current one
         --  determines what it means.

         Close_Brace
         --  The previous character is a closing brace: the current one must be
         --  another closing brace.
      );

      Buffer   : Unbounded_Wide_Wide_String;
      State    : State_Type := Default;
      Next_Arg : Positive := Arguments'First;
   begin
      for One_Argument of Arguments loop
         
      Pre_Check
        (One_Argument = No_Node_Rewriting_Handle
            or else Context (One_Argument) = Handle,
         "One_Argument should be associated to rewriting context Handle.");
   
      end loop;

      --  Interpret the template looping over its characters with a state
      --  machine.
      for C of Template loop
         case State is
         when Default =>
            case C is
            when '{' =>
               State := Open_Brace;
            when '}' =>
               State := Close_Brace;
            when others =>
               Append (Buffer, C);
            end case;

         when Open_Brace =>
            case C is
            when '{' =>
               State := Default;
               Append (Buffer, C);
            when '}' =>
               State := Default;
               if Next_Arg in Arguments'Range then
                  declare
                     Unparsed_Arg : constant Wide_Wide_String :=
                        Unparse (Arguments (Next_Arg));
                  begin
                     Next_Arg := Next_Arg + 1;
                     Append (Buffer, Unparsed_Arg);
                  end;
               else
                  raise Template_Args_Error with
                     "not enough arguments provided";
               end if;
            when others =>
               raise Template_Format_Error with
                  "standalone ""{"" character";
            end case;

         when Close_Brace =>
            case C is
            when '}' =>
               State := Default;
               Append (Buffer, C);
            when others =>
               raise Template_Format_Error with
                  "standalone ""}"" character";
            end case;
         end case;
      end loop;

      --  Make sure that there is no standalone metacharacter at the end of the
      --  template.
      case State is
         when Default => null;
         when Open_Brace =>
            raise Template_Format_Error with "standalone ""{"" character";
         when Close_Brace =>
            raise Template_Format_Error with "standalone ""}"" character";
      end case;

      --  Make sure all given arguments were consumed
      if Next_Arg in Arguments'Range then
         raise Template_Args_Error with "too many arguments provided";
      end if;

      --  Now parse the resulting buffer and create the corresponding tree of
      --  nodes.
      declare
         Context  : constant Internal_Context :=
           Rewriting_Implementation.Context (Handle);
         Unit     : constant Internal_Unit := Templates_Unit (Context);
         Reparsed : Reparsed_Unit;
         Text     : constant Text_Type := To_Wide_Wide_String (Buffer);
         Input    : constant Internal_Lexer_Input :=
           (Kind       => Text_Buffer,
            Text       => Text'Address,
            Text_Count => Text'Length);

         function Transform
           (Node   : Bare_Ada_Node;
            Parent : Node_Rewriting_Handle) return Node_Rewriting_Handle;
         --  Turn a node from the Reparsed unit into a recursively expanded
         --  node rewriting handle.

         ---------------
         -- Transform --
         ---------------

         function Transform
           (Node   : Bare_Ada_Node;
            Parent : Node_Rewriting_Handle) return Node_Rewriting_Handle
         is
            Result : Node_Rewriting_Handle;
         begin
            if Node = null then
               return No_Node_Rewriting_Handle;
            end if;

            --  Allocate the handle for Node, and don't forget to remove the
            --  backlink to Node itself as it exists only temporarily for
            --  template instantiation. Also, track the newly allocated node
            --  so that it is freed correctly upon destruction of the
            --  rewriting context.
            Result := Allocate (Node, Handle, No_Unit_Rewriting_Handle,
                                Parent);
            Result.Node := null;
            Nodes_Pools.Append (Handle.New_Nodes, Result);

            if Is_Token_Node (Node) then
               declare
                  Index : constant Natural := Natural (Node.Token_Start_Index);
                  Data  : constant Stored_Token_Data :=
                     Reparsed.TDH.Tokens.Get (Index);
                  Text  : constant Text_Type := Reparsed.TDH.Source_Buffer
                    (Data.Source_First .. Data.Source_Last);
               begin
                  Result.Children :=
                    (Kind => Expanded_Token_Node,
                     Text => To_Unbounded_Wide_Wide_String (Text));
               end;

            elsif Is_List_Node (Node.Kind) then
               declare
                  Count                : constant Natural :=
                    Children_Count (Node);
                  First, Last, Current : Node_Rewriting_Handle :=
                    No_Node_Rewriting_Handle;
               begin
                  for I in 1 .. Count loop
                     Current :=
                       Transform (Implementation.Child (Node, I), Result);
                     if First = No_Node_Rewriting_Handle then
                        First := Current;
                        Last := Current;
                     else
                        Last.Next := Current;
                        Current.Previous := Last;
                        Last := Current;
                     end if;
                     Tie (Current, Result, No_Unit_Rewriting_Handle);
                  end loop;
                  Result.Children :=
                    (Kind  => Expanded_List,
                     First => First,
                     Last  => Last,
                     Count => Count);
               end;

            else
               declare
                  Count : constant Natural := Children_Count (Node);
               begin
                  Result.Children := (Kind => Expanded_Regular, Vector => <>);
                  Result.Children.Vector.Reserve_Capacity
                    (Ada.Containers.Count_Type (Count));
                  for I in 1 .. Count loop
                     declare
                        C : constant Node_Rewriting_Handle :=
                          Transform (Child (Node, I), Result);
                     begin
                        Tie (C, Result, No_Unit_Rewriting_Handle);
                        Result.Children.Vector.Append (C);
                     end;
                  end loop;
               end;
            end if;
            return Result;
         end Transform;

      begin
         Set_Rule (Unit, Rule);
         Do_Parsing (Unit, Input, Reparsed);
         if not Reparsed.Diagnostics.Is_Empty then
            Destroy (Reparsed);
            raise Template_Instantiation_Error;
         end if;

         declare
            Result : constant Node_Rewriting_Handle :=
               Transform (Reparsed.Ast_Root, No_Node_Rewriting_Handle);
         begin
            Destroy (Reparsed);
            return Result;
         end;
      end;
   end Create_From_Template;

   ----------------------------
   -- C_Create_From_Template --
   ----------------------------

   function C_Create_From_Template
     (Handle    : Rewriting_Handle;
      Template  : access ada_text;
      Arguments : C_Node_Array.Object_Pointer;
      Count     : Interfaces.C.int;
      Rule      : ada_grammar_rule) return Node_Rewriting_Handle is
   begin
      Clear_Last_Exception;
      declare
         Ada_Template : constant Text_Type (1 .. Natural (Template.Length))
         with Import, Address => Template.Chars;

         subtype Array_Type is
           Node_Rewriting_Handle_Array (1 .. Natural (Count));
         Arguments_Address : constant System.Address :=
           C_Node_Array.To_Address (Arguments);
         Ada_Arguments     : constant Array_Type
           with Import, Address => Arguments_Address;
      begin
         return Create_From_Template
           (Handle, Ada_Template, Ada_Arguments, Rule);
      end;
   exception
      when Exc : others =>
         Set_Last_Exception (Exc);
         return null;
   end C_Create_From_Template;


         function Create_Constrained_Array_Indices
           (Handle : Rewriting_Handle
               ; Constrained_Array_Indices_F_List : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Constrained_Array_Indices,
                 (1 => Constrained_Array_Indices_F_List));
         end;


         function Create_Unconstrained_Array_Indices
           (Handle : Rewriting_Handle
               ; Unconstrained_Array_Indices_F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Unconstrained_Array_Indices,
                 (1 => Unconstrained_Array_Indices_F_Types));
         end;


         function Create_Aspect_Assoc
           (Handle : Rewriting_Handle
               ; Aspect_Assoc_F_Id : Node_Rewriting_Handle
               ; Aspect_Assoc_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Aspect_Assoc,
                 (1 => Aspect_Assoc_F_Id,
                  2 => Aspect_Assoc_F_Expr));
         end;


         function Create_At_Clause
           (Handle : Rewriting_Handle
               ; At_Clause_F_Name : Node_Rewriting_Handle
               ; At_Clause_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_At_Clause,
                 (1 => At_Clause_F_Name,
                  2 => At_Clause_F_Expr));
         end;


         function Create_Attribute_Def_Clause
           (Handle : Rewriting_Handle
               ; Attribute_Def_Clause_F_Attribute_Expr : Node_Rewriting_Handle
               ; Attribute_Def_Clause_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Attribute_Def_Clause,
                 (1 => Attribute_Def_Clause_F_Attribute_Expr,
                  2 => Attribute_Def_Clause_F_Expr));
         end;


         function Create_Enum_Rep_Clause
           (Handle : Rewriting_Handle
               ; Enum_Rep_Clause_F_Type_Name : Node_Rewriting_Handle
               ; Enum_Rep_Clause_F_Aggregate : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Enum_Rep_Clause,
                 (1 => Enum_Rep_Clause_F_Type_Name,
                  2 => Enum_Rep_Clause_F_Aggregate));
         end;


         function Create_Record_Rep_Clause
           (Handle : Rewriting_Handle
               ; Record_Rep_Clause_F_Name : Node_Rewriting_Handle
               ; Record_Rep_Clause_F_At_Expr : Node_Rewriting_Handle
               ; Record_Rep_Clause_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Record_Rep_Clause,
                 (1 => Record_Rep_Clause_F_Name,
                  2 => Record_Rep_Clause_F_At_Expr,
                  3 => Record_Rep_Clause_F_Components));
         end;


         function Create_Aspect_Spec
           (Handle : Rewriting_Handle
               ; Aspect_Spec_F_Aspect_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Aspect_Spec,
                 (1 => Aspect_Spec_F_Aspect_Assocs));
         end;


         function Create_Contract_Case_Assoc
           (Handle : Rewriting_Handle
               ; Contract_Case_Assoc_F_Guard : Node_Rewriting_Handle
               ; Contract_Case_Assoc_F_Consequence : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Contract_Case_Assoc,
                 (1 => Contract_Case_Assoc_F_Guard,
                  2 => Contract_Case_Assoc_F_Consequence));
         end;


         function Create_Pragma_Argument_Assoc
           (Handle : Rewriting_Handle
               ; Pragma_Argument_Assoc_F_Name : Node_Rewriting_Handle
               ; Pragma_Argument_Assoc_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Pragma_Argument_Assoc,
                 (1 => Pragma_Argument_Assoc_F_Name,
                  2 => Pragma_Argument_Assoc_F_Expr));
         end;


         function Create_Entry_Spec
           (Handle : Rewriting_Handle
               ; Entry_Spec_F_Entry_Name : Node_Rewriting_Handle
               ; Entry_Spec_F_Family_Type : Node_Rewriting_Handle
               ; Entry_Spec_F_Entry_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Entry_Spec,
                 (1 => Entry_Spec_F_Entry_Name,
                  2 => Entry_Spec_F_Family_Type,
                  3 => Entry_Spec_F_Entry_Params));
         end;


         function Create_Subp_Spec
           (Handle : Rewriting_Handle
               ; Subp_Spec_F_Subp_Kind : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Name : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Params : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Returns : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subp_Spec,
                 (1 => Subp_Spec_F_Subp_Kind,
                  2 => Subp_Spec_F_Subp_Name,
                  3 => Subp_Spec_F_Subp_Params,
                  4 => Subp_Spec_F_Subp_Returns));
         end;


         function Create_Synthetic_Binary_Spec
           (Handle : Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Left_Param : Node_Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Right_Param : Node_Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Binary_Spec,
                 (1 => Synthetic_Binary_Spec_F_Left_Param,
                  2 => Synthetic_Binary_Spec_F_Right_Param,
                  3 => Synthetic_Binary_Spec_F_Return_Type_Expr));
         end;


         function Create_Synthetic_Unary_Spec
           (Handle : Rewriting_Handle
               ; Synthetic_Unary_Spec_F_Right_Param : Node_Rewriting_Handle
               ; Synthetic_Unary_Spec_F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Unary_Spec,
                 (1 => Synthetic_Unary_Spec_F_Right_Param,
                  2 => Synthetic_Unary_Spec_F_Return_Type_Expr));
         end;


         function Create_Component_List
           (Handle : Rewriting_Handle
               ; Component_List_F_Components : Node_Rewriting_Handle
               ; Component_List_F_Variant_Part : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Component_List,
                 (1 => Component_List_F_Components,
                  2 => Component_List_F_Variant_Part));
         end;


         function Create_Known_Discriminant_Part
           (Handle : Rewriting_Handle
               ; Known_Discriminant_Part_F_Discr_Specs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Known_Discriminant_Part,
                 (1 => Known_Discriminant_Part_F_Discr_Specs));
         end;


         function Create_Entry_Completion_Formal_Params
           (Handle : Rewriting_Handle
               ; Entry_Completion_Formal_Params_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Entry_Completion_Formal_Params,
                 (1 => Entry_Completion_Formal_Params_F_Params));
         end;


         function Create_Generic_Formal_Part
           (Handle : Rewriting_Handle
               ; Generic_Formal_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Formal_Part,
                 (1 => Generic_Formal_Part_F_Decls));
         end;


         function Create_Null_Record_Def
           (Handle : Rewriting_Handle
               ; Base_Record_Def_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Null_Record_Def,
                 (1 => Base_Record_Def_F_Components));
         end;


         function Create_Record_Def
           (Handle : Rewriting_Handle
               ; Base_Record_Def_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Record_Def,
                 (1 => Base_Record_Def_F_Components));
         end;


         function Create_Aggregate_Assoc
           (Handle : Rewriting_Handle
               ; Aggregate_Assoc_F_Designators : Node_Rewriting_Handle
               ; Aggregate_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Aggregate_Assoc,
                 (1 => Aggregate_Assoc_F_Designators,
                  2 => Aggregate_Assoc_F_R_Expr));
         end;


         function Create_Multi_Dim_Array_Assoc
           (Handle : Rewriting_Handle
               ; Aggregate_Assoc_F_Designators : Node_Rewriting_Handle
               ; Aggregate_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Multi_Dim_Array_Assoc,
                 (1 => Aggregate_Assoc_F_Designators,
                  2 => Aggregate_Assoc_F_R_Expr));
         end;


         function Create_Composite_Constraint_Assoc
           (Handle : Rewriting_Handle
               ; Composite_Constraint_Assoc_F_Ids : Node_Rewriting_Handle
               ; Composite_Constraint_Assoc_F_Constraint_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Composite_Constraint_Assoc,
                 (1 => Composite_Constraint_Assoc_F_Ids,
                  2 => Composite_Constraint_Assoc_F_Constraint_Expr));
         end;


         function Create_Iterated_Assoc
           (Handle : Rewriting_Handle
               ; Iterated_Assoc_F_Spec : Node_Rewriting_Handle
               ; Iterated_Assoc_F_Key_Expr : Node_Rewriting_Handle
               ; Iterated_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Iterated_Assoc,
                 (1 => Iterated_Assoc_F_Spec,
                  2 => Iterated_Assoc_F_Key_Expr,
                  3 => Iterated_Assoc_F_R_Expr));
         end;


         function Create_Param_Assoc
           (Handle : Rewriting_Handle
               ; Param_Assoc_F_Designator : Node_Rewriting_Handle
               ; Param_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Param_Assoc,
                 (1 => Param_Assoc_F_Designator,
                  2 => Param_Assoc_F_R_Expr));
         end;


         function Create_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Abstract_State_Decl_F_Name : Node_Rewriting_Handle
               ; Abstract_State_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Abstract_State_Decl,
                 (1 => Abstract_State_Decl_F_Name,
                  2 => Abstract_State_Decl_F_Aspects));
         end;


         function Create_Anonymous_Expr_Decl
           (Handle : Rewriting_Handle
               ; Anonymous_Expr_Decl_F_Expr : Node_Rewriting_Handle
               ; Anonymous_Expr_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Anonymous_Expr_Decl,
                 (1 => Anonymous_Expr_Decl_F_Expr,
                  2 => Anonymous_Expr_Decl_F_Aspects));
         end;


         function Create_Component_Decl
           (Handle : Rewriting_Handle
               ; Component_Decl_F_Ids : Node_Rewriting_Handle
               ; Component_Decl_F_Component_Def : Node_Rewriting_Handle
               ; Component_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Component_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Component_Decl,
                 (1 => Component_Decl_F_Ids,
                  2 => Component_Decl_F_Component_Def,
                  3 => Component_Decl_F_Default_Expr,
                  4 => Component_Decl_F_Aspects));
         end;


         function Create_Discriminant_Spec
           (Handle : Rewriting_Handle
               ; Discriminant_Spec_F_Ids : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Type_Expr : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Default_Expr : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Discriminant_Spec,
                 (1 => Discriminant_Spec_F_Ids,
                  2 => Discriminant_Spec_F_Type_Expr,
                  3 => Discriminant_Spec_F_Default_Expr,
                  4 => Discriminant_Spec_F_Aspects));
         end;


         function Create_Generic_Formal_Obj_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Formal_Obj_Decl,
                 (1 => Generic_Formal_F_Decl,
                  2 => Generic_Formal_F_Aspects));
         end;


         function Create_Generic_Formal_Package
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Formal_Package,
                 (1 => Generic_Formal_F_Decl,
                  2 => Generic_Formal_F_Aspects));
         end;


         function Create_Generic_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Formal_Subp_Decl,
                 (1 => Generic_Formal_F_Decl,
                  2 => Generic_Formal_F_Aspects));
         end;


         function Create_Generic_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Formal_Type_Decl,
                 (1 => Generic_Formal_F_Decl,
                  2 => Generic_Formal_F_Aspects));
         end;


         function Create_Param_Spec
           (Handle : Rewriting_Handle
               ; Param_Spec_F_Ids : Node_Rewriting_Handle
               ; Param_Spec_F_Has_Aliased : Node_Rewriting_Handle
               ; Param_Spec_F_Mode : Node_Rewriting_Handle
               ; Param_Spec_F_Type_Expr : Node_Rewriting_Handle
               ; Param_Spec_F_Default_Expr : Node_Rewriting_Handle
               ; Param_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Param_Spec,
                 (1 => Param_Spec_F_Ids,
                  2 => Param_Spec_F_Has_Aliased,
                  3 => Param_Spec_F_Mode,
                  4 => Param_Spec_F_Type_Expr,
                  5 => Param_Spec_F_Default_Expr,
                  6 => Param_Spec_F_Aspects));
         end;


         function Create_Synthetic_Formal_Param_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Formal_Param_Decl_F_Param_Type : Node_Rewriting_Handle
               ; Synthetic_Formal_Param_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Formal_Param_Decl,
                 (1 => Synthetic_Formal_Param_Decl_F_Param_Type,
                  2 => Synthetic_Formal_Param_Decl_F_Aspects));
         end;


         function Create_Generic_Package_Internal
           (Handle : Rewriting_Handle
               ; Base_Package_Decl_F_Package_Name : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Aspects : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Public_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Private_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Package_Internal,
                 (1 => Base_Package_Decl_F_Package_Name,
                  2 => Base_Package_Decl_F_Aspects,
                  3 => Base_Package_Decl_F_Public_Part,
                  4 => Base_Package_Decl_F_Private_Part,
                  5 => Base_Package_Decl_F_End_Name));
         end;


         function Create_Package_Decl
           (Handle : Rewriting_Handle
               ; Base_Package_Decl_F_Package_Name : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Aspects : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Public_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Private_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Package_Decl,
                 (1 => Base_Package_Decl_F_Package_Name,
                  2 => Base_Package_Decl_F_Aspects,
                  3 => Base_Package_Decl_F_Public_Part,
                  4 => Base_Package_Decl_F_Private_Part,
                  5 => Base_Package_Decl_F_End_Name));
         end;


         function Create_Discrete_Base_Subtype_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Discrete_Base_Subtype_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Discrete_Base_Subtype_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Discrete_Base_Subtype_Decl_F_Aspects));
         end;


         function Create_Subtype_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Subtype_Decl_F_Subtype : Node_Rewriting_Handle
               ; Subtype_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subtype_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Subtype_Decl_F_Subtype,
                  3 => Subtype_Decl_F_Aspects));
         end;


         function Create_Classwide_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Classwide_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Classwide_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Classwide_Type_Decl_F_Aspects));
         end;


         function Create_Incomplete_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Incomplete_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Incomplete_Type_Decl_F_Discriminants,
                  3 => Incomplete_Type_Decl_F_Aspects));
         end;


         function Create_Incomplete_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Incomplete_Formal_Type_Decl_F_Is_Tagged : Node_Rewriting_Handle
               ; Incomplete_Formal_Type_Decl_F_Default_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Incomplete_Formal_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Incomplete_Type_Decl_F_Discriminants,
                  3 => Incomplete_Type_Decl_F_Aspects,
                  4 => Incomplete_Formal_Type_Decl_F_Is_Tagged,
                  5 => Incomplete_Formal_Type_Decl_F_Default_Type));
         end;


         function Create_Incomplete_Tagged_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Incomplete_Tagged_Type_Decl_F_Has_Abstract : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Incomplete_Tagged_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Incomplete_Type_Decl_F_Discriminants,
                  3 => Incomplete_Type_Decl_F_Aspects,
                  4 => Incomplete_Tagged_Type_Decl_F_Has_Abstract));
         end;


         function Create_Protected_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Interfaces : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Protected_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Protected_Type_Decl_F_Discriminants,
                  3 => Protected_Type_Decl_F_Aspects,
                  4 => Protected_Type_Decl_F_Interfaces,
                  5 => Protected_Type_Decl_F_Definition));
         end;


         function Create_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Task_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Task_Type_Decl_F_Discriminants,
                  3 => Task_Type_Decl_F_Aspects,
                  4 => Task_Type_Decl_F_Definition));
         end;


         function Create_Single_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Single_Task_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Task_Type_Decl_F_Discriminants,
                  3 => Task_Type_Decl_F_Aspects,
                  4 => Task_Type_Decl_F_Definition));
         end;


         function Create_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Anonymous_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Anonymous_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Type_Decl_F_Discriminants,
                  3 => Type_Decl_F_Type_Def,
                  4 => Anonymous_Type_Decl_F_Aspects));
         end;


         function Create_Synth_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Anonymous_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synth_Anonymous_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Type_Decl_F_Discriminants,
                  3 => Type_Decl_F_Type_Def,
                  4 => Anonymous_Type_Decl_F_Aspects));
         end;


         function Create_Concrete_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Concrete_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Concrete_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Type_Decl_F_Discriminants,
                  3 => Type_Decl_F_Type_Def,
                  4 => Concrete_Type_Decl_F_Aspects));
         end;


         function Create_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Formal_Type_Decl_F_Default_Type : Node_Rewriting_Handle
               ; Formal_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Formal_Type_Decl,
                 (1 => Base_Type_Decl_F_Name,
                  2 => Type_Decl_F_Discriminants,
                  3 => Type_Decl_F_Type_Def,
                  4 => Formal_Type_Decl_F_Default_Type,
                  5 => Formal_Type_Decl_F_Aspects));
         end;


         function Create_Abstract_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Abstract_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Abstract_Subp_Decl,
                 (1 => Classic_Subp_Decl_F_Overriding,
                  2 => Classic_Subp_Decl_F_Subp_Spec,
                  3 => Abstract_Subp_Decl_F_Aspects));
         end;


         function Create_Abstract_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Abstract_Formal_Subp_Decl,
                 (1 => Classic_Subp_Decl_F_Overriding,
                  2 => Classic_Subp_Decl_F_Subp_Spec,
                  3 => Formal_Subp_Decl_F_Default_Expr,
                  4 => Formal_Subp_Decl_F_Aspects));
         end;


         function Create_Concrete_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Concrete_Formal_Subp_Decl,
                 (1 => Classic_Subp_Decl_F_Overriding,
                  2 => Classic_Subp_Decl_F_Subp_Spec,
                  3 => Formal_Subp_Decl_F_Default_Expr,
                  4 => Formal_Subp_Decl_F_Aspects));
         end;


         function Create_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subp_Decl,
                 (1 => Classic_Subp_Decl_F_Overriding,
                  2 => Classic_Subp_Decl_F_Subp_Spec,
                  3 => Subp_Decl_F_Aspects));
         end;


         function Create_Entry_Decl
           (Handle : Rewriting_Handle
               ; Entry_Decl_F_Overriding : Node_Rewriting_Handle
               ; Entry_Decl_F_Spec : Node_Rewriting_Handle
               ; Entry_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Entry_Decl,
                 (1 => Entry_Decl_F_Overriding,
                  2 => Entry_Decl_F_Spec,
                  3 => Entry_Decl_F_Aspects));
         end;


         function Create_Enum_Literal_Decl
           (Handle : Rewriting_Handle
               ; Enum_Literal_Decl_F_Name : Node_Rewriting_Handle
               ; Enum_Literal_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Enum_Literal_Decl,
                 (1 => Enum_Literal_Decl_F_Name,
                  2 => Enum_Literal_Decl_F_Aspects));
         end;


         function Create_Synthetic_Char_Enum_Lit
           (Handle : Rewriting_Handle
               ; Enum_Literal_Decl_F_Name : Node_Rewriting_Handle
               ; Enum_Literal_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Char_Enum_Lit,
                 (1 => Enum_Literal_Decl_F_Name,
                  2 => Enum_Literal_Decl_F_Aspects));
         end;


         function Create_Generic_Subp_Internal
           (Handle : Rewriting_Handle
               ; Generic_Subp_Internal_F_Subp_Spec : Node_Rewriting_Handle
               ; Generic_Subp_Internal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Subp_Internal,
                 (1 => Generic_Subp_Internal_F_Subp_Spec,
                  2 => Generic_Subp_Internal_F_Aspects));
         end;


         function Create_Synthetic_Subp_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Subp_Decl_F_Aspects : Node_Rewriting_Handle
               ; Synthetic_Subp_Decl_F_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Subp_Decl,
                 (1 => Synthetic_Subp_Decl_F_Aspects,
                  2 => Synthetic_Subp_Decl_F_Spec));
         end;


         function Create_Accept_Stmt_Body
           (Handle : Rewriting_Handle
               ; Accept_Stmt_Body_F_Name : Node_Rewriting_Handle
               ; Accept_Stmt_Body_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Accept_Stmt_Body,
                 (1 => Accept_Stmt_Body_F_Name,
                  2 => Accept_Stmt_Body_F_Aspects));
         end;


         function Create_Expr_Function
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Expr_Function_F_Expr : Node_Rewriting_Handle
               ; Expr_Function_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Expr_Function,
                 (1 => Base_Subp_Body_F_Overriding,
                  2 => Base_Subp_Body_F_Subp_Spec,
                  3 => Expr_Function_F_Expr,
                  4 => Expr_Function_F_Aspects));
         end;


         function Create_Null_Subp_Decl
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Null_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Null_Subp_Decl,
                 (1 => Base_Subp_Body_F_Overriding,
                  2 => Base_Subp_Body_F_Subp_Spec,
                  3 => Null_Subp_Decl_F_Aspects));
         end;


         function Create_Subp_Body
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Body_F_Aspects : Node_Rewriting_Handle
               ; Subp_Body_F_Decls : Node_Rewriting_Handle
               ; Subp_Body_F_Stmts : Node_Rewriting_Handle
               ; Subp_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subp_Body,
                 (1 => Base_Subp_Body_F_Overriding,
                  2 => Base_Subp_Body_F_Subp_Spec,
                  3 => Subp_Body_F_Aspects,
                  4 => Subp_Body_F_Decls,
                  5 => Subp_Body_F_Stmts,
                  6 => Subp_Body_F_End_Name));
         end;


         function Create_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Subp_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subp_Renaming_Decl,
                 (1 => Base_Subp_Body_F_Overriding,
                  2 => Base_Subp_Body_F_Subp_Spec,
                  3 => Subp_Renaming_Decl_F_Renames,
                  4 => Subp_Renaming_Decl_F_Aspects));
         end;


         function Create_Package_Body_Stub
           (Handle : Rewriting_Handle
               ; Package_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Package_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Package_Body_Stub,
                 (1 => Package_Body_Stub_F_Name,
                  2 => Package_Body_Stub_F_Aspects));
         end;


         function Create_Protected_Body_Stub
           (Handle : Rewriting_Handle
               ; Protected_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Protected_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Protected_Body_Stub,
                 (1 => Protected_Body_Stub_F_Name,
                  2 => Protected_Body_Stub_F_Aspects));
         end;


         function Create_Subp_Body_Stub
           (Handle : Rewriting_Handle
               ; Subp_Body_Stub_F_Overriding : Node_Rewriting_Handle
               ; Subp_Body_Stub_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subp_Body_Stub,
                 (1 => Subp_Body_Stub_F_Overriding,
                  2 => Subp_Body_Stub_F_Subp_Spec,
                  3 => Subp_Body_Stub_F_Aspects));
         end;


         function Create_Task_Body_Stub
           (Handle : Rewriting_Handle
               ; Task_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Task_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Task_Body_Stub,
                 (1 => Task_Body_Stub_F_Name,
                  2 => Task_Body_Stub_F_Aspects));
         end;


         function Create_Entry_Body
           (Handle : Rewriting_Handle
               ; Entry_Body_F_Entry_Name : Node_Rewriting_Handle
               ; Entry_Body_F_Index_Spec : Node_Rewriting_Handle
               ; Entry_Body_F_Params : Node_Rewriting_Handle
               ; Entry_Body_F_Aspects : Node_Rewriting_Handle
               ; Entry_Body_F_Barrier : Node_Rewriting_Handle
               ; Entry_Body_F_Decls : Node_Rewriting_Handle
               ; Entry_Body_F_Stmts : Node_Rewriting_Handle
               ; Entry_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Entry_Body,
                 (1 => Entry_Body_F_Entry_Name,
                  2 => Entry_Body_F_Index_Spec,
                  3 => Entry_Body_F_Params,
                  4 => Entry_Body_F_Aspects,
                  5 => Entry_Body_F_Barrier,
                  6 => Entry_Body_F_Decls,
                  7 => Entry_Body_F_Stmts,
                  8 => Entry_Body_F_End_Name));
         end;


         function Create_Package_Body
           (Handle : Rewriting_Handle
               ; Package_Body_F_Package_Name : Node_Rewriting_Handle
               ; Package_Body_F_Aspects : Node_Rewriting_Handle
               ; Package_Body_F_Decls : Node_Rewriting_Handle
               ; Package_Body_F_Stmts : Node_Rewriting_Handle
               ; Package_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Package_Body,
                 (1 => Package_Body_F_Package_Name,
                  2 => Package_Body_F_Aspects,
                  3 => Package_Body_F_Decls,
                  4 => Package_Body_F_Stmts,
                  5 => Package_Body_F_End_Name));
         end;


         function Create_Protected_Body
           (Handle : Rewriting_Handle
               ; Protected_Body_F_Name : Node_Rewriting_Handle
               ; Protected_Body_F_Aspects : Node_Rewriting_Handle
               ; Protected_Body_F_Decls : Node_Rewriting_Handle
               ; Protected_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Protected_Body,
                 (1 => Protected_Body_F_Name,
                  2 => Protected_Body_F_Aspects,
                  3 => Protected_Body_F_Decls,
                  4 => Protected_Body_F_End_Name));
         end;


         function Create_Task_Body
           (Handle : Rewriting_Handle
               ; Task_Body_F_Name : Node_Rewriting_Handle
               ; Task_Body_F_Aspects : Node_Rewriting_Handle
               ; Task_Body_F_Decls : Node_Rewriting_Handle
               ; Task_Body_F_Stmts : Node_Rewriting_Handle
               ; Task_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Task_Body,
                 (1 => Task_Body_F_Name,
                  2 => Task_Body_F_Aspects,
                  3 => Task_Body_F_Decls,
                  4 => Task_Body_F_Stmts,
                  5 => Task_Body_F_End_Name));
         end;


         function Create_Entry_Index_Spec
           (Handle : Rewriting_Handle
               ; Entry_Index_Spec_F_Id : Node_Rewriting_Handle
               ; Entry_Index_Spec_F_Subtype : Node_Rewriting_Handle
               ; Entry_Index_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Entry_Index_Spec,
                 (1 => Entry_Index_Spec_F_Id,
                  2 => Entry_Index_Spec_F_Subtype,
                  3 => Entry_Index_Spec_F_Aspects));
         end;


         function Create_Error_Decl
           (Handle : Rewriting_Handle
               ; Error_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Error_Decl,
                 (1 => Error_Decl_F_Aspects));
         end;


         function Create_Exception_Decl
           (Handle : Rewriting_Handle
               ; Exception_Decl_F_Ids : Node_Rewriting_Handle
               ; Exception_Decl_F_Renames : Node_Rewriting_Handle
               ; Exception_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Exception_Decl,
                 (1 => Exception_Decl_F_Ids,
                  2 => Exception_Decl_F_Renames,
                  3 => Exception_Decl_F_Aspects));
         end;


         function Create_Exception_Handler
           (Handle : Rewriting_Handle
               ; Exception_Handler_F_Exception_Name : Node_Rewriting_Handle
               ; Exception_Handler_F_Handled_Exceptions : Node_Rewriting_Handle
               ; Exception_Handler_F_Stmts : Node_Rewriting_Handle
               ; Exception_Handler_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Exception_Handler,
                 (1 => Exception_Handler_F_Exception_Name,
                  2 => Exception_Handler_F_Handled_Exceptions,
                  3 => Exception_Handler_F_Stmts,
                  4 => Exception_Handler_F_Aspects));
         end;


         function Create_For_Loop_Var_Decl
           (Handle : Rewriting_Handle
               ; For_Loop_Var_Decl_F_Id : Node_Rewriting_Handle
               ; For_Loop_Var_Decl_F_Id_Type : Node_Rewriting_Handle
               ; For_Loop_Var_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_For_Loop_Var_Decl,
                 (1 => For_Loop_Var_Decl_F_Id,
                  2 => For_Loop_Var_Decl_F_Id_Type,
                  3 => For_Loop_Var_Decl_F_Aspects));
         end;


         function Create_Generic_Package_Decl
           (Handle : Rewriting_Handle
               ; Generic_Decl_F_Formal_Part : Node_Rewriting_Handle
               ; Generic_Package_Decl_F_Package_Decl : Node_Rewriting_Handle
               ; Generic_Package_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Package_Decl,
                 (1 => Generic_Decl_F_Formal_Part,
                  2 => Generic_Package_Decl_F_Package_Decl,
                  3 => Generic_Package_Decl_F_Aspects));
         end;


         function Create_Generic_Subp_Decl
           (Handle : Rewriting_Handle
               ; Generic_Decl_F_Formal_Part : Node_Rewriting_Handle
               ; Generic_Subp_Decl_F_Subp_Decl : Node_Rewriting_Handle
               ; Generic_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Subp_Decl,
                 (1 => Generic_Decl_F_Formal_Part,
                  2 => Generic_Subp_Decl_F_Subp_Decl,
                  3 => Generic_Subp_Decl_F_Aspects));
         end;


         function Create_Generic_Package_Instantiation
           (Handle : Rewriting_Handle
               ; Generic_Package_Instantiation_F_Name : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Generic_Pkg_Name : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Params : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Package_Instantiation,
                 (1 => Generic_Package_Instantiation_F_Name,
                  2 => Generic_Package_Instantiation_F_Generic_Pkg_Name,
                  3 => Generic_Package_Instantiation_F_Params,
                  4 => Generic_Package_Instantiation_F_Aspects));
         end;


         function Create_Generic_Subp_Instantiation
           (Handle : Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Overriding : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Kind : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Subp_Name : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Generic_Subp_Name : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Params : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Subp_Instantiation,
                 (1 => Generic_Subp_Instantiation_F_Overriding,
                  2 => Generic_Subp_Instantiation_F_Kind,
                  3 => Generic_Subp_Instantiation_F_Subp_Name,
                  4 => Generic_Subp_Instantiation_F_Generic_Subp_Name,
                  5 => Generic_Subp_Instantiation_F_Params,
                  6 => Generic_Subp_Instantiation_F_Aspects));
         end;


         function Create_Generic_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Package_Renaming_Decl,
                 (1 => Generic_Package_Renaming_Decl_F_Name,
                  2 => Generic_Package_Renaming_Decl_F_Renames,
                  3 => Generic_Package_Renaming_Decl_F_Aspects));
         end;


         function Create_Generic_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Kind : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Generic_Subp_Renaming_Decl,
                 (1 => Generic_Subp_Renaming_Decl_F_Kind,
                  2 => Generic_Subp_Renaming_Decl_F_Name,
                  3 => Generic_Subp_Renaming_Decl_F_Renames,
                  4 => Generic_Subp_Renaming_Decl_F_Aspects));
         end;


         function Create_Label_Decl
           (Handle : Rewriting_Handle
               ; Label_Decl_F_Name : Node_Rewriting_Handle
               ; Label_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Label_Decl,
                 (1 => Label_Decl_F_Name,
                  2 => Label_Decl_F_Aspects));
         end;


         function Create_Named_Stmt_Decl
           (Handle : Rewriting_Handle
               ; Named_Stmt_Decl_F_Name : Node_Rewriting_Handle
               ; Named_Stmt_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Named_Stmt_Decl,
                 (1 => Named_Stmt_Decl_F_Name,
                  2 => Named_Stmt_Decl_F_Aspects));
         end;


         function Create_Number_Decl
           (Handle : Rewriting_Handle
               ; Number_Decl_F_Ids : Node_Rewriting_Handle
               ; Number_Decl_F_Expr : Node_Rewriting_Handle
               ; Number_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Number_Decl,
                 (1 => Number_Decl_F_Ids,
                  2 => Number_Decl_F_Expr,
                  3 => Number_Decl_F_Aspects));
         end;


         function Create_Object_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Object_Decl,
                 (1 => Object_Decl_F_Ids,
                  2 => Object_Decl_F_Has_Aliased,
                  3 => Object_Decl_F_Has_Constant,
                  4 => Object_Decl_F_Mode,
                  5 => Object_Decl_F_Type_Expr,
                  6 => Object_Decl_F_Default_Expr,
                  7 => Object_Decl_F_Renaming_Clause,
                  8 => Object_Decl_F_Aspects));
         end;


         function Create_Extended_Return_Stmt_Object_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Extended_Return_Stmt_Object_Decl,
                 (1 => Object_Decl_F_Ids,
                  2 => Object_Decl_F_Has_Aliased,
                  3 => Object_Decl_F_Has_Constant,
                  4 => Object_Decl_F_Mode,
                  5 => Object_Decl_F_Type_Expr,
                  6 => Object_Decl_F_Default_Expr,
                  7 => Object_Decl_F_Renaming_Clause,
                  8 => Object_Decl_F_Aspects));
         end;


         function Create_No_Type_Object_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_No_Type_Object_Renaming_Decl,
                 (1 => Object_Decl_F_Ids,
                  2 => Object_Decl_F_Has_Aliased,
                  3 => Object_Decl_F_Has_Constant,
                  4 => Object_Decl_F_Mode,
                  5 => Object_Decl_F_Type_Expr,
                  6 => Object_Decl_F_Default_Expr,
                  7 => Object_Decl_F_Renaming_Clause,
                  8 => Object_Decl_F_Aspects));
         end;


         function Create_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Package_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Package_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Package_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Package_Renaming_Decl,
                 (1 => Package_Renaming_Decl_F_Name,
                  2 => Package_Renaming_Decl_F_Renames,
                  3 => Package_Renaming_Decl_F_Aspects));
         end;


         function Create_Single_Protected_Decl
           (Handle : Rewriting_Handle
               ; Single_Protected_Decl_F_Name : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Aspects : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Interfaces : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Single_Protected_Decl,
                 (1 => Single_Protected_Decl_F_Name,
                  2 => Single_Protected_Decl_F_Aspects,
                  3 => Single_Protected_Decl_F_Interfaces,
                  4 => Single_Protected_Decl_F_Definition));
         end;


         function Create_Single_Task_Decl
           (Handle : Rewriting_Handle
               ; Single_Task_Decl_F_Task_Type : Node_Rewriting_Handle
               ; Single_Task_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Single_Task_Decl,
                 (1 => Single_Task_Decl_F_Task_Type,
                  2 => Single_Task_Decl_F_Aspects));
         end;


         function Create_Synthetic_Object_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Object_Decl,
                 (1 => Synthetic_Object_Decl_F_Aspects));
         end;


         function Create_Case_Stmt_Alternative
           (Handle : Rewriting_Handle
               ; Case_Stmt_Alternative_F_Choices : Node_Rewriting_Handle
               ; Case_Stmt_Alternative_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Case_Stmt_Alternative,
                 (1 => Case_Stmt_Alternative_F_Choices,
                  2 => Case_Stmt_Alternative_F_Stmts));
         end;


         function Create_Compilation_Unit
           (Handle : Rewriting_Handle
               ; Compilation_Unit_F_Prelude : Node_Rewriting_Handle
               ; Compilation_Unit_F_Body : Node_Rewriting_Handle
               ; Compilation_Unit_F_Pragmas : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Compilation_Unit,
                 (1 => Compilation_Unit_F_Prelude,
                  2 => Compilation_Unit_F_Body,
                  3 => Compilation_Unit_F_Pragmas));
         end;


         function Create_Component_Clause
           (Handle : Rewriting_Handle
               ; Component_Clause_F_Id : Node_Rewriting_Handle
               ; Component_Clause_F_Position : Node_Rewriting_Handle
               ; Component_Clause_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Component_Clause,
                 (1 => Component_Clause_F_Id,
                  2 => Component_Clause_F_Position,
                  3 => Component_Clause_F_Range));
         end;


         function Create_Component_Def
           (Handle : Rewriting_Handle
               ; Component_Def_F_Has_Aliased : Node_Rewriting_Handle
               ; Component_Def_F_Has_Constant : Node_Rewriting_Handle
               ; Component_Def_F_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Component_Def,
                 (1 => Component_Def_F_Has_Aliased,
                  2 => Component_Def_F_Has_Constant,
                  3 => Component_Def_F_Type_Expr));
         end;


         function Create_Composite_Constraint
           (Handle : Rewriting_Handle
               ; Composite_Constraint_F_Constraints : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Composite_Constraint,
                 (1 => Composite_Constraint_F_Constraints));
         end;


         function Create_Delta_Constraint
           (Handle : Rewriting_Handle
               ; Delta_Constraint_F_Delta : Node_Rewriting_Handle
               ; Delta_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Delta_Constraint,
                 (1 => Delta_Constraint_F_Delta,
                  2 => Delta_Constraint_F_Range));
         end;


         function Create_Digits_Constraint
           (Handle : Rewriting_Handle
               ; Digits_Constraint_F_Digits : Node_Rewriting_Handle
               ; Digits_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Digits_Constraint,
                 (1 => Digits_Constraint_F_Digits,
                  2 => Digits_Constraint_F_Range));
         end;


         function Create_Range_Constraint
           (Handle : Rewriting_Handle
               ; Range_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Range_Constraint,
                 (1 => Range_Constraint_F_Range));
         end;


         function Create_Declarative_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Declarative_Part,
                 (1 => Declarative_Part_F_Decls));
         end;


         function Create_Private_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Private_Part,
                 (1 => Declarative_Part_F_Decls));
         end;


         function Create_Public_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Public_Part,
                 (1 => Declarative_Part_F_Decls));
         end;


         function Create_Elsif_Expr_Part
           (Handle : Rewriting_Handle
               ; Elsif_Expr_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Elsif_Expr_Part_F_Then_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Elsif_Expr_Part,
                 (1 => Elsif_Expr_Part_F_Cond_Expr,
                  2 => Elsif_Expr_Part_F_Then_Expr));
         end;


         function Create_Elsif_Stmt_Part
           (Handle : Rewriting_Handle
               ; Elsif_Stmt_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Elsif_Stmt_Part_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Elsif_Stmt_Part,
                 (1 => Elsif_Stmt_Part_F_Cond_Expr,
                  2 => Elsif_Stmt_Part_F_Stmts));
         end;


         function Create_Abstract_State_Decl_Expr
           (Handle : Rewriting_Handle
               ; Abstract_State_Decl_Expr_F_State_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Abstract_State_Decl_Expr,
                 (1 => Abstract_State_Decl_Expr_F_State_Decl));
         end;


         function Create_Allocator
           (Handle : Rewriting_Handle
               ; Allocator_F_Subpool : Node_Rewriting_Handle
               ; Allocator_F_Type_Or_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Allocator,
                 (1 => Allocator_F_Subpool,
                  2 => Allocator_F_Type_Or_Expr));
         end;


         function Create_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Aggregate,
                 (1 => Base_Aggregate_F_Ancestor_Expr,
                  2 => Base_Aggregate_F_Assocs));
         end;


         function Create_Bracket_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Bracket_Aggregate,
                 (1 => Base_Aggregate_F_Ancestor_Expr,
                  2 => Base_Aggregate_F_Assocs));
         end;


         function Create_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Delta_Aggregate,
                 (1 => Base_Aggregate_F_Ancestor_Expr,
                  2 => Base_Aggregate_F_Assocs));
         end;


         function Create_Bracket_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Bracket_Delta_Aggregate,
                 (1 => Base_Aggregate_F_Ancestor_Expr,
                  2 => Base_Aggregate_F_Assocs));
         end;


         function Create_Null_Record_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Null_Record_Aggregate,
                 (1 => Base_Aggregate_F_Ancestor_Expr,
                  2 => Base_Aggregate_F_Assocs));
         end;


         function Create_Bin_Op
           (Handle : Rewriting_Handle
               ; Bin_Op_F_Left : Node_Rewriting_Handle
               ; Bin_Op_F_Op : Node_Rewriting_Handle
               ; Bin_Op_F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Bin_Op,
                 (1 => Bin_Op_F_Left,
                  2 => Bin_Op_F_Op,
                  3 => Bin_Op_F_Right));
         end;


         function Create_Relation_Op
           (Handle : Rewriting_Handle
               ; Bin_Op_F_Left : Node_Rewriting_Handle
               ; Bin_Op_F_Op : Node_Rewriting_Handle
               ; Bin_Op_F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Relation_Op,
                 (1 => Bin_Op_F_Left,
                  2 => Bin_Op_F_Op,
                  3 => Bin_Op_F_Right));
         end;


         function Create_Case_Expr_Alternative
           (Handle : Rewriting_Handle
               ; Case_Expr_Alternative_F_Choices : Node_Rewriting_Handle
               ; Case_Expr_Alternative_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Case_Expr_Alternative,
                 (1 => Case_Expr_Alternative_F_Choices,
                  2 => Case_Expr_Alternative_F_Expr));
         end;


         function Create_Concat_Op
           (Handle : Rewriting_Handle
               ; Concat_Op_F_First_Operand : Node_Rewriting_Handle
               ; Concat_Op_F_Other_Operands : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Concat_Op,
                 (1 => Concat_Op_F_First_Operand,
                  2 => Concat_Op_F_Other_Operands));
         end;


         function Create_Concat_Operand
           (Handle : Rewriting_Handle
               ; Concat_Operand_F_Operator : Node_Rewriting_Handle
               ; Concat_Operand_F_Operand : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Concat_Operand,
                 (1 => Concat_Operand_F_Operator,
                  2 => Concat_Operand_F_Operand));
         end;


         function Create_Case_Expr
           (Handle : Rewriting_Handle
               ; Case_Expr_F_Expr : Node_Rewriting_Handle
               ; Case_Expr_F_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Case_Expr,
                 (1 => Case_Expr_F_Expr,
                  2 => Case_Expr_F_Cases));
         end;


         function Create_If_Expr
           (Handle : Rewriting_Handle
               ; If_Expr_F_Cond_Expr : Node_Rewriting_Handle
               ; If_Expr_F_Then_Expr : Node_Rewriting_Handle
               ; If_Expr_F_Alternatives : Node_Rewriting_Handle
               ; If_Expr_F_Else_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_If_Expr,
                 (1 => If_Expr_F_Cond_Expr,
                  2 => If_Expr_F_Then_Expr,
                  3 => If_Expr_F_Alternatives,
                  4 => If_Expr_F_Else_Expr));
         end;


         function Create_Contract_Cases
           (Handle : Rewriting_Handle
               ; Contract_Cases_F_Contract_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Contract_Cases,
                 (1 => Contract_Cases_F_Contract_Cases));
         end;


         function Create_Decl_Expr
           (Handle : Rewriting_Handle
               ; Decl_Expr_F_Decls : Node_Rewriting_Handle
               ; Decl_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Decl_Expr,
                 (1 => Decl_Expr_F_Decls,
                  2 => Decl_Expr_F_Expr));
         end;


         function Create_Format_String_Literal
           (Handle : Rewriting_Handle
               ; Format_String_Literal_F_Opening_Chunk : Node_Rewriting_Handle
               ; Format_String_Literal_F_Mid_Exprs : Node_Rewriting_Handle
               ; Format_String_Literal_F_Trailing_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Format_String_Literal,
                 (1 => Format_String_Literal_F_Opening_Chunk,
                  2 => Format_String_Literal_F_Mid_Exprs,
                  3 => Format_String_Literal_F_Trailing_Expr));
         end;


         function Create_Membership_Expr
           (Handle : Rewriting_Handle
               ; Membership_Expr_F_Expr : Node_Rewriting_Handle
               ; Membership_Expr_F_Op : Node_Rewriting_Handle
               ; Membership_Expr_F_Membership_Exprs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Membership_Expr,
                 (1 => Membership_Expr_F_Expr,
                  2 => Membership_Expr_F_Op,
                  3 => Membership_Expr_F_Membership_Exprs));
         end;


         function Create_Array_Subcomponent_Choice_Name
           (Handle : Rewriting_Handle
               ; Array_Subcomponent_Choice_Name_F_Name : Node_Rewriting_Handle
               ; Array_Subcomponent_Choice_Name_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Array_Subcomponent_Choice_Name,
                 (1 => Array_Subcomponent_Choice_Name_F_Name,
                  2 => Array_Subcomponent_Choice_Name_F_Suffix));
         end;


         function Create_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Attribute_Ref_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Attribute_Ref,
                 (1 => Attribute_Ref_F_Prefix,
                  2 => Attribute_Ref_F_Attribute,
                  3 => Attribute_Ref_F_Args));
         end;


         function Create_Call_Expr
           (Handle : Rewriting_Handle
               ; Call_Expr_F_Name : Node_Rewriting_Handle
               ; Call_Expr_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Call_Expr,
                 (1 => Call_Expr_F_Name,
                  2 => Call_Expr_F_Suffix));
         end;


         function Create_Defining_Name
           (Handle : Rewriting_Handle
               ; Defining_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Defining_Name,
                 (1 => Defining_Name_F_Name));
         end;


         function Create_Synthetic_Defining_Name
           (Handle : Rewriting_Handle
               ; Defining_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Defining_Name,
                 (1 => Defining_Name_F_Name));
         end;


         function Create_Discrete_Subtype_Name
           (Handle : Rewriting_Handle
               ; Discrete_Subtype_Name_F_Subtype : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Discrete_Subtype_Name,
                 (1 => Discrete_Subtype_Name_F_Subtype));
         end;


         function Create_Dotted_Name
           (Handle : Rewriting_Handle
               ; Dotted_Name_F_Prefix : Node_Rewriting_Handle
               ; Dotted_Name_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Dotted_Name,
                 (1 => Dotted_Name_F_Prefix,
                  2 => Dotted_Name_F_Suffix));
         end;


         function Create_End_Name
           (Handle : Rewriting_Handle
               ; End_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_End_Name,
                 (1 => End_Name_F_Name));
         end;


         function Create_Explicit_Deref
           (Handle : Rewriting_Handle
               ; Explicit_Deref_F_Prefix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Explicit_Deref,
                 (1 => Explicit_Deref_F_Prefix));
         end;


         function Create_Qual_Expr
           (Handle : Rewriting_Handle
               ; Qual_Expr_F_Prefix : Node_Rewriting_Handle
               ; Qual_Expr_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Qual_Expr,
                 (1 => Qual_Expr_F_Prefix,
                  2 => Qual_Expr_F_Suffix));
         end;


         function Create_Reduce_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Reduce_Attribute_Ref,
                 (1 => Reduce_Attribute_Ref_F_Prefix,
                  2 => Reduce_Attribute_Ref_F_Attribute,
                  3 => Reduce_Attribute_Ref_F_Args));
         end;


         function Create_Update_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Update_Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Update_Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Update_Attribute_Ref_F_Values : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Update_Attribute_Ref,
                 (1 => Update_Attribute_Ref_F_Prefix,
                  2 => Update_Attribute_Ref_F_Attribute,
                  3 => Update_Attribute_Ref_F_Values));
         end;


         function Create_Paren_Expr
           (Handle : Rewriting_Handle
               ; Paren_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Paren_Expr,
                 (1 => Paren_Expr_F_Expr));
         end;


         function Create_Quantified_Expr
           (Handle : Rewriting_Handle
               ; Quantified_Expr_F_Quantifier : Node_Rewriting_Handle
               ; Quantified_Expr_F_Loop_Spec : Node_Rewriting_Handle
               ; Quantified_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Quantified_Expr,
                 (1 => Quantified_Expr_F_Quantifier,
                  2 => Quantified_Expr_F_Loop_Spec,
                  3 => Quantified_Expr_F_Expr));
         end;


         function Create_Raise_Expr
           (Handle : Rewriting_Handle
               ; Raise_Expr_F_Exception_Name : Node_Rewriting_Handle
               ; Raise_Expr_F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Raise_Expr,
                 (1 => Raise_Expr_F_Exception_Name,
                  2 => Raise_Expr_F_Error_Message));
         end;


         function Create_Un_Op
           (Handle : Rewriting_Handle
               ; Un_Op_F_Op : Node_Rewriting_Handle
               ; Un_Op_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Un_Op,
                 (1 => Un_Op_F_Op,
                  2 => Un_Op_F_Expr));
         end;


         function Create_For_Loop_Iter_Filter
           (Handle : Rewriting_Handle
               ; For_Loop_Iter_Filter_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_For_Loop_Iter_Filter,
                 (1 => For_Loop_Iter_Filter_F_Expr));
         end;


         function Create_Format_String_Chunk
           (Handle : Rewriting_Handle
               ; Format_String_Chunk_F_Expr : Node_Rewriting_Handle
               ; Format_String_Chunk_F_String_Tok : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Format_String_Chunk,
                 (1 => Format_String_Chunk_F_Expr,
                  2 => Format_String_Chunk_F_String_Tok));
         end;


         function Create_Handled_Stmts
           (Handle : Rewriting_Handle
               ; Handled_Stmts_F_Stmts : Node_Rewriting_Handle
               ; Handled_Stmts_F_Exceptions : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Handled_Stmts,
                 (1 => Handled_Stmts_F_Stmts,
                  2 => Handled_Stmts_F_Exceptions));
         end;


         function Create_Library_Item
           (Handle : Rewriting_Handle
               ; Library_Item_F_Has_Private : Node_Rewriting_Handle
               ; Library_Item_F_Item : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Library_Item,
                 (1 => Library_Item_F_Has_Private,
                  2 => Library_Item_F_Item));
         end;


         function Create_For_Loop_Spec
           (Handle : Rewriting_Handle
               ; For_Loop_Spec_F_Var_Decl : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Loop_Type : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Has_Reverse : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Iter_Expr : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Iter_Filter : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_For_Loop_Spec,
                 (1 => For_Loop_Spec_F_Var_Decl,
                  2 => For_Loop_Spec_F_Loop_Type,
                  3 => For_Loop_Spec_F_Has_Reverse,
                  4 => For_Loop_Spec_F_Iter_Expr,
                  5 => For_Loop_Spec_F_Iter_Filter));
         end;


         function Create_While_Loop_Spec
           (Handle : Rewriting_Handle
               ; While_Loop_Spec_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_While_Loop_Spec,
                 (1 => While_Loop_Spec_F_Expr));
         end;


         function Create_Multi_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Multi_Abstract_State_Decl_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Multi_Abstract_State_Decl,
                 (1 => Multi_Abstract_State_Decl_F_Decls));
         end;


         function Create_Params
           (Handle : Rewriting_Handle
               ; Params_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Params,
                 (1 => Params_F_Params));
         end;


         function Create_Paren_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Paren_Abstract_State_Decl_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Paren_Abstract_State_Decl,
                 (1 => Paren_Abstract_State_Decl_F_Decl));
         end;


         function Create_Pp_Elsif_Directive
           (Handle : Rewriting_Handle
               ; Pp_Elsif_Directive_F_Expr : Node_Rewriting_Handle
               ; Pp_Elsif_Directive_F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Pp_Elsif_Directive,
                 (1 => Pp_Elsif_Directive_F_Expr,
                  2 => Pp_Elsif_Directive_F_Then_Kw));
         end;


         function Create_Pp_If_Directive
           (Handle : Rewriting_Handle
               ; Pp_If_Directive_F_Expr : Node_Rewriting_Handle
               ; Pp_If_Directive_F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Pp_If_Directive,
                 (1 => Pp_If_Directive_F_Expr,
                  2 => Pp_If_Directive_F_Then_Kw));
         end;


         function Create_Pragma_Node
           (Handle : Rewriting_Handle
               ; Pragma_Node_F_Id : Node_Rewriting_Handle
               ; Pragma_Node_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Pragma_Node,
                 (1 => Pragma_Node_F_Id,
                  2 => Pragma_Node_F_Args));
         end;


         function Create_Protected_Def
           (Handle : Rewriting_Handle
               ; Protected_Def_F_Public_Part : Node_Rewriting_Handle
               ; Protected_Def_F_Private_Part : Node_Rewriting_Handle
               ; Protected_Def_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Protected_Def,
                 (1 => Protected_Def_F_Public_Part,
                  2 => Protected_Def_F_Private_Part,
                  3 => Protected_Def_F_End_Name));
         end;


         function Create_Range_Spec
           (Handle : Rewriting_Handle
               ; Range_Spec_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Range_Spec,
                 (1 => Range_Spec_F_Range));
         end;


         function Create_Renaming_Clause
           (Handle : Rewriting_Handle
               ; Renaming_Clause_F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Renaming_Clause,
                 (1 => Renaming_Clause_F_Renamed_Object));
         end;


         function Create_Synthetic_Renaming_Clause
           (Handle : Rewriting_Handle
               ; Renaming_Clause_F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Renaming_Clause,
                 (1 => Renaming_Clause_F_Renamed_Object));
         end;


         function Create_Select_When_Part
           (Handle : Rewriting_Handle
               ; Select_When_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Select_When_Part_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Select_When_Part,
                 (1 => Select_When_Part_F_Cond_Expr,
                  2 => Select_When_Part_F_Stmts));
         end;


         function Create_Accept_Stmt
           (Handle : Rewriting_Handle
               ; Accept_Stmt_F_Body_Decl : Node_Rewriting_Handle
               ; Accept_Stmt_F_Entry_Index_Expr : Node_Rewriting_Handle
               ; Accept_Stmt_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Accept_Stmt,
                 (1 => Accept_Stmt_F_Body_Decl,
                  2 => Accept_Stmt_F_Entry_Index_Expr,
                  3 => Accept_Stmt_F_Params));
         end;


         function Create_Accept_Stmt_With_Stmts
           (Handle : Rewriting_Handle
               ; Accept_Stmt_F_Body_Decl : Node_Rewriting_Handle
               ; Accept_Stmt_F_Entry_Index_Expr : Node_Rewriting_Handle
               ; Accept_Stmt_F_Params : Node_Rewriting_Handle
               ; Accept_Stmt_With_Stmts_F_Stmts : Node_Rewriting_Handle
               ; Accept_Stmt_With_Stmts_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Accept_Stmt_With_Stmts,
                 (1 => Accept_Stmt_F_Body_Decl,
                  2 => Accept_Stmt_F_Entry_Index_Expr,
                  3 => Accept_Stmt_F_Params,
                  4 => Accept_Stmt_With_Stmts_F_Stmts,
                  5 => Accept_Stmt_With_Stmts_F_End_Name));
         end;


         function Create_For_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_For_Loop_Stmt,
                 (1 => Base_Loop_Stmt_F_Spec,
                  2 => Base_Loop_Stmt_F_Stmts,
                  3 => Base_Loop_Stmt_F_End_Name));
         end;


         function Create_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Loop_Stmt,
                 (1 => Base_Loop_Stmt_F_Spec,
                  2 => Base_Loop_Stmt_F_Stmts,
                  3 => Base_Loop_Stmt_F_End_Name));
         end;


         function Create_While_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_While_Loop_Stmt,
                 (1 => Base_Loop_Stmt_F_Spec,
                  2 => Base_Loop_Stmt_F_Stmts,
                  3 => Base_Loop_Stmt_F_End_Name));
         end;


         function Create_Begin_Block
           (Handle : Rewriting_Handle
               ; Begin_Block_F_Stmts : Node_Rewriting_Handle
               ; Begin_Block_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Begin_Block,
                 (1 => Begin_Block_F_Stmts,
                  2 => Begin_Block_F_End_Name));
         end;


         function Create_Decl_Block
           (Handle : Rewriting_Handle
               ; Decl_Block_F_Decls : Node_Rewriting_Handle
               ; Decl_Block_F_Stmts : Node_Rewriting_Handle
               ; Decl_Block_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Decl_Block,
                 (1 => Decl_Block_F_Decls,
                  2 => Decl_Block_F_Stmts,
                  3 => Decl_Block_F_End_Name));
         end;


         function Create_Case_Stmt
           (Handle : Rewriting_Handle
               ; Case_Stmt_F_Expr : Node_Rewriting_Handle
               ; Case_Stmt_F_Pragmas : Node_Rewriting_Handle
               ; Case_Stmt_F_Alternatives : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Case_Stmt,
                 (1 => Case_Stmt_F_Expr,
                  2 => Case_Stmt_F_Pragmas,
                  3 => Case_Stmt_F_Alternatives));
         end;


         function Create_Extended_Return_Stmt
           (Handle : Rewriting_Handle
               ; Extended_Return_Stmt_F_Decl : Node_Rewriting_Handle
               ; Extended_Return_Stmt_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Extended_Return_Stmt,
                 (1 => Extended_Return_Stmt_F_Decl,
                  2 => Extended_Return_Stmt_F_Stmts));
         end;


         function Create_If_Stmt
           (Handle : Rewriting_Handle
               ; If_Stmt_F_Cond_Expr : Node_Rewriting_Handle
               ; If_Stmt_F_Then_Stmts : Node_Rewriting_Handle
               ; If_Stmt_F_Alternatives : Node_Rewriting_Handle
               ; If_Stmt_F_Else_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_If_Stmt,
                 (1 => If_Stmt_F_Cond_Expr,
                  2 => If_Stmt_F_Then_Stmts,
                  3 => If_Stmt_F_Alternatives,
                  4 => If_Stmt_F_Else_Stmts));
         end;


         function Create_Named_Stmt
           (Handle : Rewriting_Handle
               ; Named_Stmt_F_Decl : Node_Rewriting_Handle
               ; Named_Stmt_F_Stmt : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Named_Stmt,
                 (1 => Named_Stmt_F_Decl,
                  2 => Named_Stmt_F_Stmt));
         end;


         function Create_Select_Stmt
           (Handle : Rewriting_Handle
               ; Select_Stmt_F_Guards : Node_Rewriting_Handle
               ; Select_Stmt_F_Else_Stmts : Node_Rewriting_Handle
               ; Select_Stmt_F_Abort_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Select_Stmt,
                 (1 => Select_Stmt_F_Guards,
                  2 => Select_Stmt_F_Else_Stmts,
                  3 => Select_Stmt_F_Abort_Stmts));
         end;


         function Create_Abort_Stmt
           (Handle : Rewriting_Handle
               ; Abort_Stmt_F_Names : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Abort_Stmt,
                 (1 => Abort_Stmt_F_Names));
         end;


         function Create_Assign_Stmt
           (Handle : Rewriting_Handle
               ; Assign_Stmt_F_Dest : Node_Rewriting_Handle
               ; Assign_Stmt_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Assign_Stmt,
                 (1 => Assign_Stmt_F_Dest,
                  2 => Assign_Stmt_F_Expr));
         end;


         function Create_Call_Stmt
           (Handle : Rewriting_Handle
               ; Call_Stmt_F_Call : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Call_Stmt,
                 (1 => Call_Stmt_F_Call));
         end;


         function Create_Delay_Stmt
           (Handle : Rewriting_Handle
               ; Delay_Stmt_F_Has_Until : Node_Rewriting_Handle
               ; Delay_Stmt_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Delay_Stmt,
                 (1 => Delay_Stmt_F_Has_Until,
                  2 => Delay_Stmt_F_Expr));
         end;


         function Create_Exit_Stmt
           (Handle : Rewriting_Handle
               ; Exit_Stmt_F_Loop_Name : Node_Rewriting_Handle
               ; Exit_Stmt_F_Cond_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Exit_Stmt,
                 (1 => Exit_Stmt_F_Loop_Name,
                  2 => Exit_Stmt_F_Cond_Expr));
         end;


         function Create_Goto_Stmt
           (Handle : Rewriting_Handle
               ; Goto_Stmt_F_Label_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Goto_Stmt,
                 (1 => Goto_Stmt_F_Label_Name));
         end;


         function Create_Label
           (Handle : Rewriting_Handle
               ; Label_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Label,
                 (1 => Label_F_Decl));
         end;


         function Create_Raise_Stmt
           (Handle : Rewriting_Handle
               ; Raise_Stmt_F_Exception_Name : Node_Rewriting_Handle
               ; Raise_Stmt_F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Raise_Stmt,
                 (1 => Raise_Stmt_F_Exception_Name,
                  2 => Raise_Stmt_F_Error_Message));
         end;


         function Create_Requeue_Stmt
           (Handle : Rewriting_Handle
               ; Requeue_Stmt_F_Call_Name : Node_Rewriting_Handle
               ; Requeue_Stmt_F_Has_Abort : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Requeue_Stmt,
                 (1 => Requeue_Stmt_F_Call_Name,
                  2 => Requeue_Stmt_F_Has_Abort));
         end;


         function Create_Return_Stmt
           (Handle : Rewriting_Handle
               ; Return_Stmt_F_Return_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Return_Stmt,
                 (1 => Return_Stmt_F_Return_Expr));
         end;


         function Create_Simple_Decl_Stmt
           (Handle : Rewriting_Handle
               ; Simple_Decl_Stmt_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Simple_Decl_Stmt,
                 (1 => Simple_Decl_Stmt_F_Decl));
         end;


         function Create_Subunit
           (Handle : Rewriting_Handle
               ; Subunit_F_Name : Node_Rewriting_Handle
               ; Subunit_F_Body : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subunit,
                 (1 => Subunit_F_Name,
                  2 => Subunit_F_Body));
         end;


         function Create_Task_Def
           (Handle : Rewriting_Handle
               ; Task_Def_F_Interfaces : Node_Rewriting_Handle
               ; Task_Def_F_Public_Part : Node_Rewriting_Handle
               ; Task_Def_F_Private_Part : Node_Rewriting_Handle
               ; Task_Def_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Task_Def,
                 (1 => Task_Def_F_Interfaces,
                  2 => Task_Def_F_Public_Part,
                  3 => Task_Def_F_Private_Part,
                  4 => Task_Def_F_End_Name));
         end;


         function Create_Access_To_Subp_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Access_To_Subp_Def_F_Has_Protected : Node_Rewriting_Handle
               ; Access_To_Subp_Def_F_Subp_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Access_To_Subp_Def,
                 (1 => Access_Def_F_Has_Not_Null,
                  2 => Access_To_Subp_Def_F_Has_Protected,
                  3 => Access_To_Subp_Def_F_Subp_Spec));
         end;


         function Create_Anonymous_Type_Access_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Anonymous_Type_Access_Def_F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Anonymous_Type_Access_Def,
                 (1 => Access_Def_F_Has_Not_Null,
                  2 => Anonymous_Type_Access_Def_F_Type_Decl));
         end;


         function Create_Type_Access_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Type_Access_Def_F_Has_All : Node_Rewriting_Handle
               ; Type_Access_Def_F_Has_Constant : Node_Rewriting_Handle
               ; Type_Access_Def_F_Subtype_Indication : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Type_Access_Def,
                 (1 => Access_Def_F_Has_Not_Null,
                  2 => Type_Access_Def_F_Has_All,
                  3 => Type_Access_Def_F_Has_Constant,
                  4 => Type_Access_Def_F_Subtype_Indication));
         end;


         function Create_Array_Type_Def
           (Handle : Rewriting_Handle
               ; Array_Type_Def_F_Indices : Node_Rewriting_Handle
               ; Array_Type_Def_F_Component_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Array_Type_Def,
                 (1 => Array_Type_Def_F_Indices,
                  2 => Array_Type_Def_F_Component_Type));
         end;


         function Create_Derived_Type_Def
           (Handle : Rewriting_Handle
               ; Derived_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_Limited : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_Synchronized : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Subtype_Indication : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Interfaces : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Record_Extension : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_With_Private : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Derived_Type_Def,
                 (1 => Derived_Type_Def_F_Has_Abstract,
                  2 => Derived_Type_Def_F_Has_Limited,
                  3 => Derived_Type_Def_F_Has_Synchronized,
                  4 => Derived_Type_Def_F_Subtype_Indication,
                  5 => Derived_Type_Def_F_Interfaces,
                  6 => Derived_Type_Def_F_Record_Extension,
                  7 => Derived_Type_Def_F_Has_With_Private));
         end;


         function Create_Enum_Type_Def
           (Handle : Rewriting_Handle
               ; Enum_Type_Def_F_Enum_Literals : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Enum_Type_Def,
                 (1 => Enum_Type_Def_F_Enum_Literals));
         end;


         function Create_Interface_Type_Def
           (Handle : Rewriting_Handle
               ; Interface_Type_Def_F_Interface_Kind : Node_Rewriting_Handle
               ; Interface_Type_Def_F_Interfaces : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Interface_Type_Def,
                 (1 => Interface_Type_Def_F_Interface_Kind,
                  2 => Interface_Type_Def_F_Interfaces));
         end;


         function Create_Mod_Int_Type_Def
           (Handle : Rewriting_Handle
               ; Mod_Int_Type_Def_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Mod_Int_Type_Def,
                 (1 => Mod_Int_Type_Def_F_Expr));
         end;


         function Create_Private_Type_Def
           (Handle : Rewriting_Handle
               ; Private_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Private_Type_Def_F_Has_Tagged : Node_Rewriting_Handle
               ; Private_Type_Def_F_Has_Limited : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Private_Type_Def,
                 (1 => Private_Type_Def_F_Has_Abstract,
                  2 => Private_Type_Def_F_Has_Tagged,
                  3 => Private_Type_Def_F_Has_Limited));
         end;


         function Create_Decimal_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Delta : Node_Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Digits : Node_Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Decimal_Fixed_Point_Def,
                 (1 => Decimal_Fixed_Point_Def_F_Delta,
                  2 => Decimal_Fixed_Point_Def_F_Digits,
                  3 => Decimal_Fixed_Point_Def_F_Range));
         end;


         function Create_Floating_Point_Def
           (Handle : Rewriting_Handle
               ; Floating_Point_Def_F_Num_Digits : Node_Rewriting_Handle
               ; Floating_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Floating_Point_Def,
                 (1 => Floating_Point_Def_F_Num_Digits,
                  2 => Floating_Point_Def_F_Range));
         end;


         function Create_Ordinary_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; Ordinary_Fixed_Point_Def_F_Delta : Node_Rewriting_Handle
               ; Ordinary_Fixed_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Ordinary_Fixed_Point_Def,
                 (1 => Ordinary_Fixed_Point_Def_F_Delta,
                  2 => Ordinary_Fixed_Point_Def_F_Range));
         end;


         function Create_Record_Type_Def
           (Handle : Rewriting_Handle
               ; Record_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Record_Type_Def_F_Has_Tagged : Node_Rewriting_Handle
               ; Record_Type_Def_F_Has_Limited : Node_Rewriting_Handle
               ; Record_Type_Def_F_Record_Def : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Record_Type_Def,
                 (1 => Record_Type_Def_F_Has_Abstract,
                  2 => Record_Type_Def_F_Has_Tagged,
                  3 => Record_Type_Def_F_Has_Limited,
                  4 => Record_Type_Def_F_Record_Def));
         end;


         function Create_Signed_Int_Type_Def
           (Handle : Rewriting_Handle
               ; Signed_Int_Type_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Signed_Int_Type_Def,
                 (1 => Signed_Int_Type_Def_F_Range));
         end;


         function Create_Anonymous_Type
           (Handle : Rewriting_Handle
               ; Anonymous_Type_F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Anonymous_Type,
                 (1 => Anonymous_Type_F_Type_Decl));
         end;


         function Create_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Subtype_Indication,
                 (1 => Subtype_Indication_F_Has_Not_Null,
                  2 => Subtype_Indication_F_Name,
                  3 => Subtype_Indication_F_Constraint));
         end;


         function Create_Constrained_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Constrained_Subtype_Indication,
                 (1 => Subtype_Indication_F_Has_Not_Null,
                  2 => Subtype_Indication_F_Name,
                  3 => Subtype_Indication_F_Constraint));
         end;


         function Create_Discrete_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Discrete_Subtype_Indication,
                 (1 => Subtype_Indication_F_Has_Not_Null,
                  2 => Subtype_Indication_F_Name,
                  3 => Subtype_Indication_F_Constraint));
         end;


         function Create_Synthetic_Type_Expr
           (Handle : Rewriting_Handle
               ; Synthetic_Type_Expr_F_Target_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Synthetic_Type_Expr,
                 (1 => Synthetic_Type_Expr_F_Target_Type));
         end;


         function Create_Unconstrained_Array_Index
           (Handle : Rewriting_Handle
               ; Unconstrained_Array_Index_F_Subtype_Name : Node_Rewriting_Handle
               ; Unconstrained_Array_Index_F_Lower_Bound : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Unconstrained_Array_Index,
                 (1 => Unconstrained_Array_Index_F_Subtype_Name,
                  2 => Unconstrained_Array_Index_F_Lower_Bound));
         end;


         function Create_Use_Package_Clause
           (Handle : Rewriting_Handle
               ; Use_Package_Clause_F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Use_Package_Clause,
                 (1 => Use_Package_Clause_F_Packages));
         end;


         function Create_Use_Type_Clause
           (Handle : Rewriting_Handle
               ; Use_Type_Clause_F_Has_All : Node_Rewriting_Handle
               ; Use_Type_Clause_F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Use_Type_Clause,
                 (1 => Use_Type_Clause_F_Has_All,
                  2 => Use_Type_Clause_F_Types));
         end;


         function Create_Value_Sequence
           (Handle : Rewriting_Handle
               ; Value_Sequence_F_Iter_Assoc : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Value_Sequence,
                 (1 => Value_Sequence_F_Iter_Assoc));
         end;


         function Create_Variant
           (Handle : Rewriting_Handle
               ; Variant_F_Choices : Node_Rewriting_Handle
               ; Variant_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Variant,
                 (1 => Variant_F_Choices,
                  2 => Variant_F_Components));
         end;


         function Create_Variant_Part
           (Handle : Rewriting_Handle
               ; Variant_Part_F_Discr_Name : Node_Rewriting_Handle
               ; Variant_Part_F_Variant : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_Variant_Part,
                 (1 => Variant_Part_F_Discr_Name,
                  2 => Variant_Part_F_Variant));
         end;


         function Create_With_Clause
           (Handle : Rewriting_Handle
               ; With_Clause_F_Has_Limited : Node_Rewriting_Handle
               ; With_Clause_F_Has_Private : Node_Rewriting_Handle
               ; With_Clause_F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            
      Pre_Check
        (Handle /= No_Rewriting_Handle,
         "Handle should not be null");
   

            return Create_Regular_Node
              (Handle, Ada_With_Clause,
                 (1 => With_Clause_F_Has_Limited,
                  2 => With_Clause_F_Has_Private,
                  3 => With_Clause_F_Packages));
         end;

end Libadalang.Rewriting_Implementation;
