/*
 *
 * This file is part of Tulip (https://tulip.labri.fr)
 *
 * Authors: David Auber and the Tulip development Team
 * from LaBRI, University of Bordeaux
 *
 * Tulip is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *
 * Tulip is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 */

#ifndef TULIP_SELECTION_H
#define TULIP_SELECTION_H

#include <tulip/PropertyTypes.h>
#include <tulip/AbstractProperty.h>
#include <tulip/PropertyAlgorithm.h>

namespace tlp {

class PropertyContext;
template <class Tnode, class Tedge, class Tprop>
class AbstractProperty;
template <typename vectType, typename eltType, typename propType>
class AbstractVectorProperty;

/**
 * @ingroup Graph
 * @brief A graph property that maps a Boolean value to graph elements.
 */
class TLP_SCOPE BooleanProperty : public AbstractProperty<tlp::BooleanType, tlp::BooleanType> {
public:
  BooleanProperty(Graph *g, const std::string &n = "")
      : AbstractProperty<BooleanType, BooleanType>(g, n) {}
  // PropertyInterface inherited methods
  PropertyInterface *clonePrototype(Graph *, const std::string &) const override;
  static const std::string propertyTypename;
  const std::string &getTypename() const override {
    return propertyTypename;
  }
  DEFINE_GET_CPP_CLASS_NAME;

  /**
   * Reverses all values associated to graph elements,
   * i.e true => false, false => true.
   * If sg is nullptr, the graph given when creating the property is considered.
   */
  void reverse(const Graph *sg = nullptr);

  /**
   * Reverses all the direction of edges of the visible graph
   * which are true in this BooleanProperty.
   * * If sg is nullptr, the graph given when creating the property is considered.
   */
  void reverseEdgeDirection(Graph *sg = nullptr);
};

/**
 * @ingroup Graph
 * @brief A graph property that maps a std::vector<bool> value to graph elements.
 */
class TLP_SCOPE BooleanVectorProperty
    : public AbstractVectorProperty<tlp::BooleanVectorType, tlp::BooleanType> {
public:
  BooleanVectorProperty(Graph *g, const std::string &n = "")
      : AbstractVectorProperty<BooleanVectorType, tlp::BooleanType>(g, n) {}
  // PropertyInterface inherited methods
  PropertyInterface *clonePrototype(Graph *, const std::string &) const override;
  static const std::string propertyTypename;
  const std::string &getTypename() const override {
    return propertyTypename;
  }
  DEFINE_GET_CPP_CLASS_NAME;
};
} // namespace tlp

#endif
