%feature("docstring") OT::Normal
"Normal distribution.

Available constructors:
    Normal(*d=1*)

    Normal(*mu, sigma*)

    Normal(*mu, sigma, R*)

    Normal(*mu, Sigma*)

Parameters
----------
d : int, :math:`d \\geq 1`
    Dimension.

mu : float :math:`\\mu`, sequence of float, :math:`\\vect{\\mu} \\in \\Rset^d`
    Mean.

    Default is :math:`\\vect{\\mu} = (0, \\dots, 0) \\in \\Rset^d`.

sigma : float :math:`\\sigma`, sequence of float, :math:`\\vect{\\sigma} \\in \\Rset^{*d}_+`
    Scale parameter.

    Default is :math:`\\vect{\\sigma} = (1, \\dots, 1) \\in \\Rset^{*d}_+`.

R : :class:`~openturns.CorrelationMatrix` :math:`\\mat{R} \\in  \\cM_{d \\times d}(\\Rset)`
    Correlation matrix.

    By default the identity matrix.

Sigma : :class:`~openturns.CovarianceMatrix` :math:`\\mat{\\Sigma} \\in  \\cM_{d \\times d}(\\Rset)`
    Covariance matrix.

Notes
-----
Its probability density function is defined as:

.. math::

    f_{\\vect{X}}(\\vect{x}) = \\frac{1}
             {\\displaystyle (2\\pi)^{\\frac{n}{2}}
                            (\\mathrm{det}\\mat{\\Sigma})^{\\frac{1}{2}}}
             \\displaystyle exp \\left(-\\frac{1}{2}\\Tr{(\\vect{x} - \\vect{\\mu})}
                                     \\mat{\\Sigma}^{-1}(\\vect{x} - \\vect{\\mu})\\right)

with :math:`\\mat{\\Sigma} = \\diag(\\vect{\\sigma}) \\mat{R} \\diag(\\vect{\\sigma})`.
Both :math:`\\mat{R}` and :math:`\\mat{\\Sigma}` are symmetric positive definite.

The first moments of the distribution are:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        \\Expect{\\vect{X}} & = & \\vect{\\mu} \\\\
        \\Var{\\vect{X}} & = & \\mat{\\Sigma}
    \\end{eqnarray*}

Examples
--------
Create a distribution:

>>> import openturns as ot
>>> distribution = ot.Normal(0.0, 1.0)

Draw a sample:

>>> sample = distribution.getSample(5)

Create a multivariate normal distribution with constant :math:`\\sigma` and identity correlation matrix:

>>> dimension = 5
>>> mu = ot.Point(dimension)
>>> sigma = ot.Point(dimension, 0.1)
>>> R = ot.CorrelationMatrix(dimension)
>>> distribution = ot.Normal(mu, sigma, R)
>>> print('distribution=', distribution)
distribution= Normal(mu = [0,0,0,0,0], sigma = [0.1,0.1,0.1,0.1,0.1], R = 5x5
[[ 1 0 0 0 0 ]
 [ 0 1 0 0 0 ]
 [ 0 0 1 0 0 ]
 [ 0 0 0 1 0 ]
 [ 0 0 0 0 1 ]])

Create a multivariate normal distribution from a covariance matrix:

>>> Sigma = ot.CovarianceMatrix(dimension)
>>> Sigma[1, 1] = 3.0
>>> Sigma[0, 2] = -0.5
>>> Sigma[3, 4] = -0.2
>>> distribution = ot.Normal(mu, Sigma)
>>> print('distribution=', distribution)
distribution= Normal(mu = [0,0,0,0,0], sigma = [1,1.73205,1,1,1], R = 5x5
[[  1    0   -0.5  0    0   ]
 [  0    1    0    0    0   ]
 [ -0.5  0    1    0    0   ]
 [  0    0    0    1   -0.2 ]
 [  0    0    0   -0.2  1   ]])"
