%feature("docstring") OT::NLopt
"Interface to NLopt.

This class exposes the solvers from the non-linear optimization library [nlopt2009]_.

More details about available algorithms are available `here <http://ab-initio.mit.edu/wiki/index.php/NLopt_Algorithms>`_.

Parameters
----------
problem : :class:`~openturns.OptimizationProblem`
    Optimization problem to solve.
algoName : str
    The NLopt identifier of the algorithm.
    Use :func:`GetAlgorithmNames()` to list available names.


Notes
-----
Here are some properties of the different algorithms:

+----------------------------+-------------------+-----------------------------+
| Algorithm                  | Derivative info   | Constraint support          |
+============================+===================+=============================+
| AUGLAG                     | no derivative     | all                         |
+----------------------------+-------------------+-----------------------------+
| AUGLAG_EQ                  | no derivative     | all                         |
+----------------------------+-------------------+-----------------------------+
| GD_MLSL                    | first derivative  | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GD_MLSL_LDS                | first derivative  | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GD_STOGO (optional)        | first derivative  | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GD_STOGO_RAND (optional)   | first derivative  | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_AGS (optional)          | no derivative     | bounds required, inequality |
+----------------------------+-------------------+-----------------------------+
| GN_CRS2_LM                 | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_DIRECT                  | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_DIRECT_L                | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_DIRECT_L_NOSCAL         | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_DIRECT_L_RAND           | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_DIRECT_L_RAND_NOSCAL    | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_ESCH                    | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_ISRES                   | no derivative     | bounds required, all        |
+----------------------------+-------------------+-----------------------------+
| GN_MLSL                    | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_MLSL_LDS                | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| GN_ORIG_DIRECT             | no derivative     | bounds required, inequality |
+----------------------------+-------------------+-----------------------------+
| GN_ORIG_DIRECT_L           | no derivative     | bounds required, inequality |
+----------------------------+-------------------+-----------------------------+
| G_MLSL                     | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| G_MLSL_LDS                 | no derivative     | bounds required             |
+----------------------------+-------------------+-----------------------------+
| LD_AUGLAG                  | first derivative  | all                         |
+----------------------------+-------------------+-----------------------------+
| LD_AUGLAG_EQ               | first derivative  | all                         |
+----------------------------+-------------------+-----------------------------+
| LD_CCSAQ                   | first derivative  | bounds, inequality          |
+----------------------------+-------------------+-----------------------------+
| LD_LBFGS                   | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_MMA                     | first derivative  | bounds, inequality          |
+----------------------------+-------------------+-----------------------------+
| LD_SLSQP                   | first derivative  | all                         |
+----------------------------+-------------------+-----------------------------+
| LD_TNEWTON                 | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_TNEWTON_PRECOND         | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_TNEWTON_PRECOND_RESTART | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_TNEWTON_RESTART         | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_VAR1                    | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LD_VAR2                    | first derivative  | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LN_AUGLAG                  | no derivative     | all                         |
+----------------------------+-------------------+-----------------------------+
| LN_AUGLAG_EQ               | no derivative     | all                         |
+----------------------------+-------------------+-----------------------------+
| LN_BOBYQA                  | no derivative     | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LN_COBYLA                  | no derivative     | all                         |
+----------------------------+-------------------+-----------------------------+
| LN_NELDERMEAD              | no derivative     | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LN_NEWUOA                  | no derivative     | none                        |
+----------------------------+-------------------+-----------------------------+
| LN_NEWUOA_BOUND            | no derivative     | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LN_PRAXIS                  | no derivative     | bounds                      |
+----------------------------+-------------------+-----------------------------+
| LN_SBPLX                   | no derivative     | bounds                      |
+----------------------------+-------------------+-----------------------------+

Availability of algorithms marked as optional may vary depending on the NLopt version
or compilation options used.

See also
--------
AbdoRackwitz, Cobyla, SQP, TNC

Examples
--------
>>> import openturns as ot
>>> dim = 4
>>> bounds = ot.Interval([-3.0] * dim, [5.0] * dim)
>>> linear = ot.SymbolicFunction(['x1', 'x2', 'x3', 'x4'], ['x1+2*x2-3*x3+4*x4'])
>>> problem = ot.OptimizationProblem(linear, ot.Function(), ot.Function(), bounds)
>>> print(ot.NLopt.GetAlgorithmNames())  # doctest: +SKIP
[AUGLAG,AUGLAG_EQ,GD_MLSL,GD_MLSL_LDS,...
>>> algo = ot.NLopt(problem, 'LD_MMA')  # doctest: +SKIP
>>> algo.setStartingPoint([0.0] * 4)  # doctest: +SKIP
>>> algo.run()  # doctest: +SKIP
>>> result = algo.getResult()  # doctest: +SKIP
>>> x_star = result.getOptimalPoint()  # doctest: +SKIP
>>> y_star = result.getOptimalValue()  # doctest: +SKIP"

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::setAlgorithmName
"Accessor to the algorithm name.

Parameters
----------
algoName : str
    The NLopt identifier of the algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::getAlgorithmName
"Accessor to the algorithm name.

Returns
-------
algoName : str
    The NLopt identifier of the algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::setInitialStep
"Initial local derivative-free algorithms step accessor.

Parameters
----------
dx : sequence of float
    The initial step."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::getInitialStep
"Initial local derivative-free algorithms step accessor.

Returns
-------
dx : :class:`~openturns.Point`
    The initial step."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::setLocalSolver
"Local solver accessor.

Parameters
----------
solver : :class:`~openturns.NLopt`
    The local solver."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::getLocalSolver
"Local solver accessor.

Returns
-------
solver : :class:`~openturns.NLopt`
    The local solver."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::GetAlgorithmNames
"Accessor to the list of algorithms provided by NLopt, by names.

Returns
-------
names : :class:`~openturns.Description`
    List of algorithm names provided by NLopt, according to its naming convention.

Examples
--------
>>> import openturns as ot
>>> print(ot.NLopt.GetAlgorithmNames())  # doctest: +SKIP
[AUGLAG,AUGLAG_EQ,GD_MLSL,..."

// ---------------------------------------------------------------------

%feature("docstring") OT::NLopt::SetSeed
"Initialize the random generator seed.

Parameters
----------
seed : int
    The RNG seed."
