%feature("docstring") OT::LinearTaylor
"First order polynomial response surface by Taylor expansion.

Available constructors:
    LinearTaylor(*center, function*)

Parameters
----------
center : sequence of float
    Point :math:`\\vect{x}_0` where the Taylor expansion of the function
    :math:`h` is performed.
function : :class:`~openturns.Function`
    Function :math:`h` to be approximated.

Notes
-----
The approximation of the model response :math:`\\vect{y} = h(\\vect{x})` around a
specific set :math:`\\vect{x}_0 = (x_{0,1},\\dots,x_{0,n_{X}})` of input
parameters may be of interest. One may then substitute :math:`h` for its Taylor
expansion at point :math:`\\vect{x}_0`. Hence :math:`h` is replaced with a first
or second-order polynomial :math:`\\widehat{h}` whose evaluation is inexpensive,
allowing the analyst to apply the uncertainty anaysis methods.

We consider here the first order Taylor expansion around :math:`\\ux=\\vect{x}_0`.

.. math::

    \\vect{y} \\, \\approx \\, \\widehat{h}(\\vect{x}) \\,
      = \\, h(\\vect{x}_0) \\, +
        \\, \\sum_{i=1}^{n_{X}} \\; \\frac{\\partial h}{\\partial x_i}(\\vect{x}_0).\\left(x_i - x_{0,i} \\right)

Introducing a vector notation, the previous equation rewrites:

.. math::

    \\vect{y} \\, \\approx \\, \\vect{y}_0 \\, + \\, \\vect{\\vect{L}} \\: \\left(\\vect{x}-\\vect{x}_0\\right)

where:

- :math:`\\vect{y_0} = (y_{0,1} , \\dots, y_{0,n_Y})^{\\textsf{T}} = h(\\vect{x}_0)`
  is the vector model response evaluated at :math:`\\vect{x}_0`;
- :math:`\\vect{x}` is the current set of input parameters;
- :math:`\\vect{\\vect{L}} = \\left( \\frac{\\partial y_{0,j}}{\\partial x_i} \\,,\\, i=1,\\ldots, n_X \\,,\\, j=1,\\ldots, n_Y \\right)`
  is the transposed Jacobian matrix evaluated at :math:`\\vect{x}_0`.

See also
--------
QuadraticTaylor, LinearLeastSquares, QuadraticLeastSquares

Examples
--------
>>> import openturns as ot
>>> formulas = ['x1 * sin(x2)', 'cos(x1 + x2)', '(x2 + 1) * exp(x1 - 2 * x2)']
>>> myFunc = ot.SymbolicFunction(['x1', 'x2'], formulas)
>>> myTaylor = ot.LinearTaylor([1, 2], myFunc)
>>> myTaylor.run()
>>> responseSurface = myTaylor.getMetaModel()
>>> print(responseSurface([1.2,1.9]))
[1.13277,-1.0041,0.204127]"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::getCenter
"Get the center.

Returns
-------
center : :class:`~openturns.Point`
    Point :math:`\\vect{x}_0` where the Taylor expansion of the function is
    performed."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::getConstant
"Get the constant vector of the approximation.

Returns
-------
constantVector : :class:`~openturns.Point`
    Constant vector of the approximation, equal to :math:`h(x_0)`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::getLinear
"Get the gradient of the function at :math:`\\vect{x}_0`.

Returns
-------
gradient : :class:`~openturns.Matrix`
    Gradient of the function :math:`h` at the point :math:`\\vect{x}_0` (the
    transposition of the jacobian matrix)."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::getInputFunction
"Get the function.

Returns
-------
function : :class:`~openturns.Function`
    Function :math:`h` to be approximated."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::getMetaModel
"Get an approximation of the function.

Returns
-------
approximation : :class:`~openturns.Function`
    An approximation of the function :math:`h` by a Linear Taylor expansion at
    the point :math:`\\vect{x}_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearTaylor::run
"Perform the Linear Taylor expansion around :math:`\\vect{x}_0`."
