Texture2D inRTT;
Texture2D inTex0;
Texture2D inTex1;
Texture2D inTex2;

SamplerState g_samLinear
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Wrap;
    AddressV = Wrap;
};

SamplerState g_samClamp
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Clamp;
    AddressV = Clamp;
};

struct v2p
{
    float4 position : SV_POSITION;
    float2 texCoord : TEXCOORD0;
};

// vignetting effect (makes corners of image darker)
float vignette(float2 pos, float inner, float outer)
{
  float r = length(pos);
  r = 1.0 - smoothstep(inner, outer, r);
  return r;
}

float4 toneMap(float4 inColour, float4 lum, float4 key)
{  
  //Reinhard's Photographic Tone Reproduction Operator
  float4 scaledlum = inColour * (key/(lum + 0.0001));
  inColour.rgb = (scaledlum*(1+scaledlum/(lum + 0.0001*lum + 0.0001))/(0.5+scaledlum)).rgb;

	return clamp(inColour, 0.0, 1.0);
}

float4 radial(Texture2D inTex,
              float2 texcoord,
              int samples,
              float startScale,
              float scaleMul,
              float4 lum,
              float4 key)
{
    float4 c = 0;
    float scale = startScale;
    for(int i=0; i<samples; i++) {
        float2 uv = ((texcoord-0.5)*scale)+0.5;
        float4 s = toneMap(inTex.Sample(g_samLinear, uv), lum, key);
        c += s;
        scale *= scaleMul;
    }
    c /= samples;
    return c;
}

static const float3 LUMINANCE_VECTOR = float3(0.2125, 0.7154f, 0.0721);

float4 initLuminence(v2p input,
	uniform float2 texelSize // depends on size of source texture
  ) : SV_Target
{
  float accum = 0.0;
  
	float2 texOffset[16] = {
		-1.5, -1.5,
		-0.5, -1.5,
		 0.5, -1.5,
		 1.5, -1.5,
		 
		-1.5, -0.5,
		-0.5, -0.5,
		 0.5, -0.5,
		 1.5, -0.5,
		 
		-1.5, 0.5,
		-0.5, 0.5,
		 0.5, 0.5,
		 1.5, 0.5,
		 
		-1.5, 1.5,
		-0.5, 1.5,
		 0.5, 1.5,
		 1.5, 1.5
	};

	for( int i = 0; i < 16; i++ )
  {
      // Get colour from source
      //original code log() this value
      accum += dot(inRTT.Sample(g_samLinear, input.texCoord + texelSize * texOffset[i]).rgb, LUMINANCE_VECTOR) +0.0001;
  }
    
	// take average of 16 samples
	accum *= 0.0625;
  
  return float4(accum, accum, accum, 1.0);
}

float4 finalLuminance(v2p input,
	uniform float2 texelSize // depends on size of source texture
    ) : SV_Target
{
  float3 accum = float3(0.0, 0.0, 0.0);

	float2 texOffset[16] = {
		-1.5, -1.5,
		-0.5, -1.5,
		 0.5, -1.5,
		 1.5, -1.5,
		 
		-1.5, -0.5,
		-0.5, -0.5,
		 0.5, -0.5,
		 1.5, -0.5,
		 
		-1.5, 0.5,
		-0.5, 0.5,
		 0.5, 0.5,
		 1.5, 0.5,
		 
		-1.5, 1.5,
		-0.5, 1.5,
		 0.5, 1.5,
		 1.5, 1.5
	};

	for( int i = 0; i < 16; i++ )
  {
    // Get colour from source
    accum += inRTT.Sample(g_samLinear, input.texCoord + texelSize * texOffset[i]).rgb;
  }
  
	// take average of 16 samples
	accum *= 0.0625;
  
  // Divide the sum to complete the average, and perform an exp() to complete
  // the average luminance calculation
  // min / max white
  return float4(clamp(accum, 0.4, 0.7), 1.0);
}

float4 brightpass(v2p input,
	uniform float2 texelSize
    ) : SV_Target
{
	float4 avgLum = inTex0.Sample(g_samLinear, float2(0.5, 0.5));
	float4 Key = inTex1.Sample(g_samLinear, float2(0.5,0.5));
	
	float4 accum = float4(0.0, 0.0, 0.0, 0.0);

	float2 texOffset[9] = {
		-1.0, -1.0,
		 0.0, -1.0,
		 1.0, -1.0,
		-1.0,  0.0,
		 0.0,  0.0,
		 1.0,  0.0,
		-1.0,  1.0,
		 0.0,  1.0,
		 1.0,  1.0
	};

	for( int i = 0; i < 9; i++ )
  {
    // Get colour from source
    accum += inRTT.Sample(g_samLinear, input.texCoord + texelSize * texOffset[i]);
  }
    
	// take average of 9 samples
	accum *= 0.1111111111111111;

	return toneMap(accum, avgLum, Key);
}

/* Gaussian bloom, requires offsets and weights to be provided externally
*/
float4 bloom(v2p input,
  uniform float2 sampleOffsets[15],
  uniform float4 sampleWeights[15]
  ) : SV_Target
{
  float4 accum = float4(0.0, 0.0, 0.0, 1.0);
  float2 sampleUV;
  
  for( int i = 0; i < 15; i++ )
  {
      // Sample from adjacent points, 7 each side and central
      sampleUV = input.texCoord + sampleOffsets[i];
      accum += sampleWeights[i] * inRTT.Sample(g_samLinear, sampleUV);
  }
  
  return accum;
}

/* Final scene composition, with tone mapping
*/
float4 finalToneMapping(v2p input,
	uniform float effectAmount,
	uniform float blurAmount,
	uniform float gamma
    ) : SV_Target
{
	// Get main scene colour
  float4 sceneCol = inRTT.Sample(g_samClamp, input.texCoord);

	// Get luminence value
	float4 lum = inTex1.Sample(g_samClamp, float2(0.5, 0.5));

	// Get key
	float4 key = inTex2.Sample(g_samClamp, float2(0.5, 0.5));

	// tone map this
	float4 toneMappedSceneCol = toneMap(sceneCol, lum, key);
	
	// Get bloom colour
  float4 toneMappedBloom = toneMap(inTex0.Sample(g_samClamp, input.texCoord), lum, key);
  float4 effect = radial(inTex0, input.texCoord, 30, 1.0, 0.95, lum, key);

  float4 c = lerp(toneMappedSceneCol, toneMappedBloom, blurAmount);
	c += c * (1.0 - effectAmount);
	c += effect * effectAmount;
  
  // vignette effect
  //c *= vignette(input.texCoord*2-1, 0.7, 1.5);

  // gamma correction
  c.rgb = pow(c.rgb, gamma);

  return c;
}


static const float TauCone = 0.01;
static const float TauRod = 0.04;

float4 adaptLuminance(v2p input,
	uniform float dTime,
	uniform float AdaptationScale
    ) : SV_Target
{
  float3 Lum = inRTT.Sample(g_samLinear, float2(0.5, 0.5)).rgb;
  float3 oldLum = inTex0.Sample(g_samLinear, float2(0.5, 0.5)).rgb;

  //determin if rods or cones are active
  //Perceptual Effects in Real-time Tone Mapping: Equ(7)    
  float3 sigma = saturate(0.4/(0.04+Lum));

  //interpolate tau from taurod and taucone depending on lum
  //Perceptual Effects in Real-time Tone Mapping: Equ(12)
  float3 Tau = lerp(TauCone, TauRod, sigma) * AdaptationScale;

  //calculate adaption
  //Perceptual Effects in Real-time Tone Mapping: Equ(5)
  float3 tm = (1 - exp(-(dTime)/Tau));
  float3 nlum = oldLum + (Lum - oldLum) * tm;
  return float4(nlum, 1.0);     	
}


float4 autokey(v2p input
  ) : SV_Target
{
  float3 avgLum = inRTT.Sample(g_samLinear, float2(0.5, 0.5)).rgb;

  //Perceptual Effects in Real-time Tone Mapping: Equ(11)
  return float4(saturate(1.0 - 1.0/(avgLum*0.1+1.0)) + 0.1, 1.0);
}


float4 colorTex(v2p input
  ) : SV_Target
{
  return inRTT.Sample(g_samLinear, input.texCoord);
}

    
