%feature("docstring") OT::EmpiricalBernsteinCopula
"Empirical Bernstein copula.

Parameters
----------
sample : :class:`~openturns.Sample`
    A sample of size :math:`\\sampleSize>0` and dimension :math:`d`.
    
    Default value is [[0.0], [0.0]].
m : int, :math:`1 \\leq m \\leq \\sampleSize`
    The number of sub-intervals in which all the edges of the unit cube
    :math:`[0, 1]^d` are regularly partitioned.
    
    Default value is 1.
isRankSample : bool
    Flag to tell if the given sample is already a normalized rank sample.
    
    Default value is False.

Notes
-----
Let :math:`(\\inputReal_1, \\dots, \\inputReal_\\sampleSize)` a sample. Let :math:`m` be the bin number.
Note that when :math:`m=1`, the empirical Bernstein copula is the independent copula. When :math:`m=\\sampleSize`, the empirical Bernstein
copula is also called the *Beta copula* (see [segers2016]_).

**Case 1:** :math:`isRankSample = False`.

In that case, if :math:`m` does not divide :math:`\\sampleSize`, then a part of the sample is
removed in order to get a new size :math:`\\tilde{\\sampleSize}` such that :math:`m` divides :math:`\\tilde{\\sampleSize}`. At most, :math:`m-1` points are removed from the initial sample. This condition guarantees that the :class:`~openturns.EmpiricalBernsteinCopula`
built is a copula.

We still denote by  :math:`\\sampleSize` the size of the sample used to create the empirical Bernstein copula.

Let :math:`(\\vect{v}_1, \\dots, \\vect{v}_\\sampleSize)` be the associated normalized rank sample defined  by:

.. math::

     \\vect{v}_k = \\dfrac{1}{\\sampleSize} rank(\\inputReal_k)

for all :math:`1 \\leq i \\leq \\sampleSize` where the rank of a vector is the vector of the component ranks.

Let :math:`r_i^j` and :math:`s_i^j` defined by:

.. math::

     r_i^j & = \\lfloor m v_i^j \\rfloor \\\\
     s_i^j & = m-r_i^j+1

for all :math:`1 \\leq i \\leq \\sampleSize` and :math:`1 \\leq j \\leq d` and where :math:`\\lfloor x \\rfloor` is
the largest integer less than or equal to :math:`x`.

Let :math:`B(r_i^j , s_i^j, 0, 1)` be a
:class:`~openturns.Beta` distribution defined on :math:`[0,1]` parametrized by the shape parameters :math:`r_i^j` and
:math:`s_i^j`. We denote by :math:`F_{B(r_i^j, s_i^j, 0, 1)}` its cumulated distribution function.

Then the empirical Bernstein copula :math:`C^B_\\sampleSize` is defined by:

.. math::
    :label: defEmpCopBernstein

    C^B_\\sampleSize(\\vect{u}) = \\dfrac{1}{\\sampleSize}\\sum_{i=1}^\\sampleSize\\prod_{j=1}^d F_{B(r_i^j, s_i^j, 0, 1)}(u^j)

for :math:`\\vect{u}\\in[0,1]^d`.

**Case 2:** :math:`isRankSample = True`.

In that case, the given sample is assumed to be already a normalized rank sample.
Even if the bin number :math:`m` does not divide :math:`\\sampleSize`, the whole sample is used. The
empirical Bernstein copula is still defined by :eq:`defEmpCopBernstein` where :math:`r_i^j`
and :math:`s_i^j` are computed on the initial sample :math:`(\\inputReal_1, \\dots, \\inputReal_\\sampleSize)`:

.. math::

     r_i^j & = \\lfloor m x_i^j \\rfloor \\\\
     s_i^j & = m-r_i^j+1

If :math:`m` does not divide :math:`\\sampleSize`, then :math:`C^B_\\sampleSize` is no longer a copula but still a valid core,
i.e a multivariate distribution whose range is included in :math:`[0,1]^d`. As a matter of fact, the marginals are not necessarily
uniform because some cells may have more points than others.

We still use the term *copula* even if it is not a copula.

See also
--------
BernsteinCopulaFactory

Examples
--------
Create a distribution:

>>> import openturns as ot
>>> sampleNormal = ot.Normal(2).getSample(10)
>>> m = 2
>>> empBernsteinCop = ot.EmpiricalBernsteinCopula(sampleNormal, m)

Draw a sample:

>>> sample = empBernsteinCop.getSample(5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::EmpiricalBernsteinCopula::getCopulaSample
"Get the empirical copula sample.

Returns
-------
sample : :class:`~openturns.Sample`
    The empirical copula sample."

// ---------------------------------------------------------------------

%feature("docstring") OT::EmpiricalBernsteinCopula::setCopulaSample
"Set the sample.

Parameters
----------
sample : 2-d sequence of float
    The sample from which the empirical copula sample is deduced.
isRankSample : bool
    Flag telling if the given sample is already a normalized rank sample.
    
    Default value is *False*."

// ---------------------------------------------------------------------

%feature("docstring") OT::EmpiricalBernsteinCopula::getBinNumber
"Get the bin number :math:`m`.

Returns
-------
m : int, :math:`0 < m \\leq \\sampleSize`
    The number of sub-intervals in which all the edges of the unit cube
    :math:`[0, 1]^d` are regularly partitioned."

// ---------------------------------------------------------------------

%feature("docstring") OT::EmpiricalBernsteinCopula::setBinNumber
"Set the bin number :math:`m`.

Parameters
----------
m : int, :math:`0 < m \\leq \\sampleSize`
    The number of sub-intervals in which all the edges of the unit cube
    :math:`[0, 1]^d` are regularly partitioned."
