/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useRef, useCallback, useEffect, useMemo } from 'react';
import { GeminiEventType as ServerGeminiEventType, getErrorMessage, isNodeError, MessageSenderType, logUserPrompt, GitService, UnauthorizedError, UserPromptEvent, DEFAULT_GEMINI_FLASH_MODEL, logConversationFinishedEvent, ConversationFinishedEvent, ApprovalMode, parseAndFormatApiError, ToolConfirmationOutcome, promptIdContext, } from '@google/gemini-cli-core';
import { FinishReason } from '@google/genai';
import { StreamingState, MessageType, ToolCallStatus } from '../types.js';
import { isAtCommand, isSlashCommand } from '../utils/commandUtils.js';
import { useShellCommandProcessor } from './shellCommandProcessor.js';
import { handleAtCommand } from './atCommandProcessor.js';
import { findLastSafeSplitPoint } from '../utils/markdownUtilities.js';
import { useStateAndRef } from './useStateAndRef.js';
import { useLogger } from './useLogger.js';
import { useReactToolScheduler, mapToDisplay as mapTrackedToolCallsToDisplay, } from './useReactToolScheduler.js';
import { promises as fs } from 'node:fs';
import path from 'node:path';
import { useSessionStats } from '../contexts/SessionContext.js';
import { useKeypress } from './useKeypress.js';
var StreamProcessingStatus;
(function (StreamProcessingStatus) {
    StreamProcessingStatus[StreamProcessingStatus["Completed"] = 0] = "Completed";
    StreamProcessingStatus[StreamProcessingStatus["UserCancelled"] = 1] = "UserCancelled";
    StreamProcessingStatus[StreamProcessingStatus["Error"] = 2] = "Error";
})(StreamProcessingStatus || (StreamProcessingStatus = {}));
const EDIT_TOOL_NAMES = new Set(['replace', 'write_file']);
function showCitations(settings) {
    const enabled = settings?.merged?.ui?.showCitations;
    if (enabled !== undefined) {
        return enabled;
    }
    return true;
}
/**
 * Manages the Gemini stream, including user input, command processing,
 * API interaction, and tool call lifecycle.
 */
export const useGeminiStream = (geminiClient, history, addItem, config, settings, onDebugMessage, handleSlashCommand, shellModeActive, getPreferredEditor, onAuthError, performMemoryRefresh, modelSwitchedFromQuotaError, setModelSwitchedFromQuotaError, onEditorClose, onCancelSubmit, setShellInputFocused, terminalWidth, terminalHeight, isShellFocused) => {
    const [initError, setInitError] = useState(null);
    const abortControllerRef = useRef(null);
    const turnCancelledRef = useRef(false);
    const [isResponding, setIsResponding] = useState(false);
    const [thought, setThought] = useState(null);
    const [pendingHistoryItemRef, setPendingHistoryItem] = useStateAndRef(null);
    const processedMemoryToolsRef = useRef(new Set());
    const { startNewPrompt, getPromptCount } = useSessionStats();
    const storage = config.storage;
    const logger = useLogger(storage);
    const gitService = useMemo(() => {
        if (!config.getProjectRoot()) {
            return;
        }
        return new GitService(config.getProjectRoot(), storage);
    }, [config, storage]);
    const [toolCalls, scheduleToolCalls, markToolsAsSubmitted] = useReactToolScheduler(async (completedToolCallsFromScheduler) => {
        // This onComplete is called when ALL scheduled tools for a given batch are done.
        if (completedToolCallsFromScheduler.length > 0) {
            // Add the final state of these tools to the history for display.
            addItem(mapTrackedToolCallsToDisplay(completedToolCallsFromScheduler), Date.now());
            // Handle tool response submission immediately when tools complete
            await handleCompletedTools(completedToolCallsFromScheduler);
        }
    }, config, getPreferredEditor, onEditorClose);
    const pendingToolCallGroupDisplay = useMemo(() => toolCalls.length ? mapTrackedToolCallsToDisplay(toolCalls) : undefined, [toolCalls]);
    const activeToolPtyId = useMemo(() => {
        const executingShellTool = toolCalls?.find((tc) => tc.status === 'executing' && tc.request.name === 'run_shell_command');
        if (executingShellTool) {
            return executingShellTool.pid;
        }
        return undefined;
    }, [toolCalls]);
    const loopDetectedRef = useRef(false);
    const [loopDetectionConfirmationRequest, setLoopDetectionConfirmationRequest,] = useState(null);
    const onExec = useCallback(async (done) => {
        setIsResponding(true);
        await done;
        setIsResponding(false);
    }, []);
    const { handleShellCommand, activeShellPtyId } = useShellCommandProcessor(addItem, setPendingHistoryItem, onExec, onDebugMessage, config, geminiClient, setShellInputFocused, terminalWidth, terminalHeight);
    const activePtyId = activeShellPtyId || activeToolPtyId;
    useEffect(() => {
        if (!activePtyId) {
            setShellInputFocused(false);
        }
    }, [activePtyId, setShellInputFocused]);
    const streamingState = useMemo(() => {
        if (toolCalls.some((tc) => tc.status === 'awaiting_approval')) {
            return StreamingState.WaitingForConfirmation;
        }
        if (isResponding ||
            toolCalls.some((tc) => tc.status === 'executing' ||
                tc.status === 'scheduled' ||
                tc.status === 'validating' ||
                ((tc.status === 'success' ||
                    tc.status === 'error' ||
                    tc.status === 'cancelled') &&
                    !tc
                        .responseSubmittedToGemini))) {
            return StreamingState.Responding;
        }
        return StreamingState.Idle;
    }, [isResponding, toolCalls]);
    useEffect(() => {
        if (config.getApprovalMode() === ApprovalMode.YOLO &&
            streamingState === StreamingState.Idle) {
            const lastUserMessageIndex = history.findLastIndex((item) => item.type === MessageType.USER);
            const turnCount = lastUserMessageIndex === -1 ? 0 : history.length - lastUserMessageIndex;
            if (turnCount > 0) {
                logConversationFinishedEvent(config, new ConversationFinishedEvent(config.getApprovalMode(), turnCount));
            }
        }
    }, [streamingState, config, history]);
    const cancelOngoingRequest = useCallback(() => {
        if (streamingState !== StreamingState.Responding) {
            return;
        }
        if (turnCancelledRef.current) {
            return;
        }
        turnCancelledRef.current = true;
        abortControllerRef.current?.abort();
        if (pendingHistoryItemRef.current) {
            addItem(pendingHistoryItemRef.current, Date.now());
        }
        addItem({
            type: MessageType.INFO,
            text: 'Request cancelled.',
        }, Date.now());
        setPendingHistoryItem(null);
        onCancelSubmit();
        setIsResponding(false);
        setShellInputFocused(false);
    }, [
        streamingState,
        addItem,
        setPendingHistoryItem,
        onCancelSubmit,
        pendingHistoryItemRef,
        setShellInputFocused,
    ]);
    useKeypress((key) => {
        if (key.name === 'escape' && !isShellFocused) {
            cancelOngoingRequest();
        }
    }, { isActive: streamingState === StreamingState.Responding });
    const prepareQueryForGemini = useCallback(async (query, userMessageTimestamp, abortSignal, prompt_id) => {
        if (turnCancelledRef.current) {
            return { queryToSend: null, shouldProceed: false };
        }
        if (typeof query === 'string' && query.trim().length === 0) {
            return { queryToSend: null, shouldProceed: false };
        }
        let localQueryToSendToGemini = null;
        if (typeof query === 'string') {
            const trimmedQuery = query.trim();
            logUserPrompt(config, new UserPromptEvent(trimmedQuery.length, prompt_id, config.getContentGeneratorConfig()?.authType, trimmedQuery));
            onDebugMessage(`User query: '${trimmedQuery}'`);
            await logger?.logMessage(MessageSenderType.USER, trimmedQuery);
            // Handle UI-only commands first
            const slashCommandResult = isSlashCommand(trimmedQuery)
                ? await handleSlashCommand(trimmedQuery)
                : false;
            if (slashCommandResult) {
                switch (slashCommandResult.type) {
                    case 'schedule_tool': {
                        const { toolName, toolArgs } = slashCommandResult;
                        const toolCallRequest = {
                            callId: `${toolName}-${Date.now()}-${Math.random().toString(16).slice(2)}`,
                            name: toolName,
                            args: toolArgs,
                            isClientInitiated: true,
                            prompt_id,
                        };
                        scheduleToolCalls([toolCallRequest], abortSignal);
                        return { queryToSend: null, shouldProceed: false };
                    }
                    case 'submit_prompt': {
                        localQueryToSendToGemini = slashCommandResult.content;
                        return {
                            queryToSend: localQueryToSendToGemini,
                            shouldProceed: true,
                        };
                    }
                    case 'handled': {
                        return { queryToSend: null, shouldProceed: false };
                    }
                    default: {
                        const unreachable = slashCommandResult;
                        throw new Error(`Unhandled slash command result type: ${unreachable}`);
                    }
                }
            }
            if (shellModeActive && handleShellCommand(trimmedQuery, abortSignal)) {
                return { queryToSend: null, shouldProceed: false };
            }
            // Handle @-commands (which might involve tool calls)
            if (isAtCommand(trimmedQuery)) {
                const atCommandResult = await handleAtCommand({
                    query: trimmedQuery,
                    config,
                    addItem,
                    onDebugMessage,
                    messageId: userMessageTimestamp,
                    signal: abortSignal,
                });
                // Add user's turn after @ command processing is done.
                addItem({ type: MessageType.USER, text: trimmedQuery }, userMessageTimestamp);
                if (!atCommandResult.shouldProceed) {
                    return { queryToSend: null, shouldProceed: false };
                }
                localQueryToSendToGemini = atCommandResult.processedQuery;
            }
            else {
                // Normal query for Gemini
                addItem({ type: MessageType.USER, text: trimmedQuery }, userMessageTimestamp);
                localQueryToSendToGemini = trimmedQuery;
            }
        }
        else {
            // It's a function response (PartListUnion that isn't a string)
            localQueryToSendToGemini = query;
        }
        if (localQueryToSendToGemini === null) {
            onDebugMessage('Query processing resulted in null, not sending to Gemini.');
            return { queryToSend: null, shouldProceed: false };
        }
        return { queryToSend: localQueryToSendToGemini, shouldProceed: true };
    }, [
        config,
        addItem,
        onDebugMessage,
        handleShellCommand,
        handleSlashCommand,
        logger,
        shellModeActive,
        scheduleToolCalls,
    ]);
    // --- Stream Event Handlers ---
    const handleContentEvent = useCallback((eventValue, currentGeminiMessageBuffer, userMessageTimestamp) => {
        if (turnCancelledRef.current) {
            // Prevents additional output after a user initiated cancel.
            return '';
        }
        let newGeminiMessageBuffer = currentGeminiMessageBuffer + eventValue;
        if (pendingHistoryItemRef.current?.type !== 'gemini' &&
            pendingHistoryItemRef.current?.type !== 'gemini_content') {
            if (pendingHistoryItemRef.current) {
                addItem(pendingHistoryItemRef.current, userMessageTimestamp);
            }
            setPendingHistoryItem({ type: 'gemini', text: '' });
            newGeminiMessageBuffer = eventValue;
        }
        // Split large messages for better rendering performance. Ideally,
        // we should maximize the amount of output sent to <Static />.
        const splitPoint = findLastSafeSplitPoint(newGeminiMessageBuffer);
        if (splitPoint === newGeminiMessageBuffer.length) {
            // Update the existing message with accumulated content
            setPendingHistoryItem((item) => ({
                type: item?.type,
                text: newGeminiMessageBuffer,
            }));
        }
        else {
            // This indicates that we need to split up this Gemini Message.
            // Splitting a message is primarily a performance consideration. There is a
            // <Static> component at the root of App.tsx which takes care of rendering
            // content statically or dynamically. Everything but the last message is
            // treated as static in order to prevent re-rendering an entire message history
            // multiple times per-second (as streaming occurs). Prior to this change you'd
            // see heavy flickering of the terminal. This ensures that larger messages get
            // broken up so that there are more "statically" rendered.
            const beforeText = newGeminiMessageBuffer.substring(0, splitPoint);
            const afterText = newGeminiMessageBuffer.substring(splitPoint);
            addItem({
                type: pendingHistoryItemRef.current?.type,
                text: beforeText,
            }, userMessageTimestamp);
            setPendingHistoryItem({ type: 'gemini_content', text: afterText });
            newGeminiMessageBuffer = afterText;
        }
        return newGeminiMessageBuffer;
    }, [addItem, pendingHistoryItemRef, setPendingHistoryItem]);
    const handleUserCancelledEvent = useCallback((userMessageTimestamp) => {
        if (turnCancelledRef.current) {
            return;
        }
        if (pendingHistoryItemRef.current) {
            if (pendingHistoryItemRef.current.type === 'tool_group') {
                const updatedTools = pendingHistoryItemRef.current.tools.map((tool) => tool.status === ToolCallStatus.Pending ||
                    tool.status === ToolCallStatus.Confirming ||
                    tool.status === ToolCallStatus.Executing
                    ? { ...tool, status: ToolCallStatus.Canceled }
                    : tool);
                const pendingItem = {
                    ...pendingHistoryItemRef.current,
                    tools: updatedTools,
                };
                addItem(pendingItem, userMessageTimestamp);
            }
            else {
                addItem(pendingHistoryItemRef.current, userMessageTimestamp);
            }
            setPendingHistoryItem(null);
        }
        addItem({ type: MessageType.INFO, text: 'User cancelled the request.' }, userMessageTimestamp);
        setIsResponding(false);
        setThought(null); // Reset thought when user cancels
    }, [addItem, pendingHistoryItemRef, setPendingHistoryItem, setThought]);
    const handleErrorEvent = useCallback((eventValue, userMessageTimestamp) => {
        if (pendingHistoryItemRef.current) {
            addItem(pendingHistoryItemRef.current, userMessageTimestamp);
            setPendingHistoryItem(null);
        }
        addItem({
            type: MessageType.ERROR,
            text: parseAndFormatApiError(eventValue.error, config.getContentGeneratorConfig()?.authType, undefined, config.getModel(), DEFAULT_GEMINI_FLASH_MODEL),
        }, userMessageTimestamp);
        setThought(null); // Reset thought when there's an error
    }, [addItem, pendingHistoryItemRef, setPendingHistoryItem, config, setThought]);
    const handleCitationEvent = useCallback((text, userMessageTimestamp) => {
        if (!showCitations(settings)) {
            return;
        }
        if (pendingHistoryItemRef.current) {
            addItem(pendingHistoryItemRef.current, userMessageTimestamp);
            setPendingHistoryItem(null);
        }
        addItem({ type: MessageType.INFO, text }, userMessageTimestamp);
    }, [addItem, pendingHistoryItemRef, setPendingHistoryItem, settings]);
    const handleFinishedEvent = useCallback((event, userMessageTimestamp) => {
        const finishReason = event.value.reason;
        if (!finishReason) {
            return;
        }
        const finishReasonMessages = {
            [FinishReason.FINISH_REASON_UNSPECIFIED]: undefined,
            [FinishReason.STOP]: undefined,
            [FinishReason.MAX_TOKENS]: 'Response truncated due to token limits.',
            [FinishReason.SAFETY]: 'Response stopped due to safety reasons.',
            [FinishReason.RECITATION]: 'Response stopped due to recitation policy.',
            [FinishReason.LANGUAGE]: 'Response stopped due to unsupported language.',
            [FinishReason.BLOCKLIST]: 'Response stopped due to forbidden terms.',
            [FinishReason.PROHIBITED_CONTENT]: 'Response stopped due to prohibited content.',
            [FinishReason.SPII]: 'Response stopped due to sensitive personally identifiable information.',
            [FinishReason.OTHER]: 'Response stopped for other reasons.',
            [FinishReason.MALFORMED_FUNCTION_CALL]: 'Response stopped due to malformed function call.',
            [FinishReason.IMAGE_SAFETY]: 'Response stopped due to image safety violations.',
            [FinishReason.UNEXPECTED_TOOL_CALL]: 'Response stopped due to unexpected tool call.',
        };
        const message = finishReasonMessages[finishReason];
        if (message) {
            addItem({
                type: 'info',
                text: `⚠️  ${message}`,
            }, userMessageTimestamp);
        }
    }, [addItem]);
    const handleChatCompressionEvent = useCallback((eventValue, userMessageTimestamp) => {
        if (pendingHistoryItemRef.current) {
            addItem(pendingHistoryItemRef.current, userMessageTimestamp);
            setPendingHistoryItem(null);
        }
        return addItem({
            type: 'info',
            text: `IMPORTANT: This conversation approached the input token limit for ${config.getModel()}. ` +
                `A compressed context will be sent for future messages (compressed from: ` +
                `${eventValue?.originalTokenCount ?? 'unknown'} to ` +
                `${eventValue?.newTokenCount ?? 'unknown'} tokens).`,
        }, Date.now());
    }, [addItem, config, pendingHistoryItemRef, setPendingHistoryItem]);
    const handleMaxSessionTurnsEvent = useCallback(() => addItem({
        type: 'info',
        text: `The session has reached the maximum number of turns: ${config.getMaxSessionTurns()}. ` +
            `Please update this limit in your setting.json file.`,
    }, Date.now()), [addItem, config]);
    const handleLoopDetectionConfirmation = useCallback((result) => {
        setLoopDetectionConfirmationRequest(null);
        if (result.userSelection === 'disable') {
            config.getGeminiClient().getLoopDetectionService().disableForSession();
            addItem({
                type: 'info',
                text: `Loop detection has been disabled for this session. Please try your request again.`,
            }, Date.now());
        }
        else {
            addItem({
                type: 'info',
                text: `A potential loop was detected. This can happen due to repetitive tool calls or other model behavior. The request has been halted.`,
            }, Date.now());
        }
    }, [config, addItem]);
    const handleLoopDetectedEvent = useCallback(() => {
        // Show the confirmation dialog to choose whether to disable loop detection
        setLoopDetectionConfirmationRequest({
            onComplete: handleLoopDetectionConfirmation,
        });
    }, [handleLoopDetectionConfirmation]);
    const processGeminiStreamEvents = useCallback(async (stream, userMessageTimestamp, signal) => {
        let geminiMessageBuffer = '';
        const toolCallRequests = [];
        for await (const event of stream) {
            switch (event.type) {
                case ServerGeminiEventType.Thought:
                    setThought(event.value);
                    break;
                case ServerGeminiEventType.Content:
                    geminiMessageBuffer = handleContentEvent(event.value, geminiMessageBuffer, userMessageTimestamp);
                    break;
                case ServerGeminiEventType.ToolCallRequest:
                    toolCallRequests.push(event.value);
                    break;
                case ServerGeminiEventType.UserCancelled:
                    handleUserCancelledEvent(userMessageTimestamp);
                    break;
                case ServerGeminiEventType.Error:
                    handleErrorEvent(event.value, userMessageTimestamp);
                    break;
                case ServerGeminiEventType.ChatCompressed:
                    handleChatCompressionEvent(event.value, userMessageTimestamp);
                    break;
                case ServerGeminiEventType.ToolCallConfirmation:
                case ServerGeminiEventType.ToolCallResponse:
                    // do nothing
                    break;
                case ServerGeminiEventType.MaxSessionTurns:
                    handleMaxSessionTurnsEvent();
                    break;
                case ServerGeminiEventType.Finished:
                    handleFinishedEvent(event, userMessageTimestamp);
                    break;
                case ServerGeminiEventType.Citation:
                    handleCitationEvent(event.value, userMessageTimestamp);
                    break;
                case ServerGeminiEventType.LoopDetected:
                    // handle later because we want to move pending history to history
                    // before we add loop detected message to history
                    loopDetectedRef.current = true;
                    break;
                case ServerGeminiEventType.Retry:
                    // Will add the missing logic later
                    break;
                default: {
                    // enforces exhaustive switch-case
                    const unreachable = event;
                    return unreachable;
                }
            }
        }
        if (toolCallRequests.length > 0) {
            scheduleToolCalls(toolCallRequests, signal);
        }
        return StreamProcessingStatus.Completed;
    }, [
        handleContentEvent,
        handleUserCancelledEvent,
        handleErrorEvent,
        scheduleToolCalls,
        handleChatCompressionEvent,
        handleFinishedEvent,
        handleMaxSessionTurnsEvent,
        handleCitationEvent,
    ]);
    const submitQuery = useCallback(async (query, options, prompt_id) => {
        if ((streamingState === StreamingState.Responding ||
            streamingState === StreamingState.WaitingForConfirmation) &&
            !options?.isContinuation)
            return;
        const userMessageTimestamp = Date.now();
        // Reset quota error flag when starting a new query (not a continuation)
        if (!options?.isContinuation) {
            setModelSwitchedFromQuotaError(false);
            config.setQuotaErrorOccurred(false);
        }
        abortControllerRef.current = new AbortController();
        const abortSignal = abortControllerRef.current.signal;
        turnCancelledRef.current = false;
        if (!prompt_id) {
            prompt_id = config.getSessionId() + '########' + getPromptCount();
        }
        return promptIdContext.run(prompt_id, async () => {
            const { queryToSend, shouldProceed } = await prepareQueryForGemini(query, userMessageTimestamp, abortSignal, prompt_id);
            if (!shouldProceed || queryToSend === null) {
                return;
            }
            if (!options?.isContinuation) {
                startNewPrompt();
                setThought(null); // Reset thought when starting a new prompt
            }
            setIsResponding(true);
            setInitError(null);
            try {
                const stream = geminiClient.sendMessageStream(queryToSend, abortSignal, prompt_id);
                const processingStatus = await processGeminiStreamEvents(stream, userMessageTimestamp, abortSignal);
                if (processingStatus === StreamProcessingStatus.UserCancelled) {
                    return;
                }
                if (pendingHistoryItemRef.current) {
                    addItem(pendingHistoryItemRef.current, userMessageTimestamp);
                    setPendingHistoryItem(null);
                }
                if (loopDetectedRef.current) {
                    loopDetectedRef.current = false;
                    handleLoopDetectedEvent();
                }
            }
            catch (error) {
                if (error instanceof UnauthorizedError) {
                    onAuthError('Session expired or is unauthorized.');
                }
                else if (!isNodeError(error) || error.name !== 'AbortError') {
                    addItem({
                        type: MessageType.ERROR,
                        text: parseAndFormatApiError(getErrorMessage(error) || 'Unknown error', config.getContentGeneratorConfig()?.authType, undefined, config.getModel(), DEFAULT_GEMINI_FLASH_MODEL),
                    }, userMessageTimestamp);
                }
            }
            finally {
                setIsResponding(false);
            }
        });
    }, [
        streamingState,
        setModelSwitchedFromQuotaError,
        prepareQueryForGemini,
        processGeminiStreamEvents,
        pendingHistoryItemRef,
        addItem,
        setPendingHistoryItem,
        setInitError,
        geminiClient,
        onAuthError,
        config,
        startNewPrompt,
        getPromptCount,
        handleLoopDetectedEvent,
    ]);
    const handleApprovalModeChange = useCallback(async (newApprovalMode) => {
        // Auto-approve pending tool calls when switching to auto-approval modes
        if (newApprovalMode === ApprovalMode.YOLO ||
            newApprovalMode === ApprovalMode.AUTO_EDIT) {
            let awaitingApprovalCalls = toolCalls.filter((call) => call.status === 'awaiting_approval');
            // For AUTO_EDIT mode, only approve edit tools (replace, write_file)
            if (newApprovalMode === ApprovalMode.AUTO_EDIT) {
                awaitingApprovalCalls = awaitingApprovalCalls.filter((call) => EDIT_TOOL_NAMES.has(call.request.name));
            }
            // Process pending tool calls sequentially to reduce UI chaos
            for (const call of awaitingApprovalCalls) {
                if (call.confirmationDetails?.onConfirm) {
                    try {
                        await call.confirmationDetails.onConfirm(ToolConfirmationOutcome.ProceedOnce);
                    }
                    catch (error) {
                        console.error(`Failed to auto-approve tool call ${call.request.callId}:`, error);
                    }
                }
            }
        }
    }, [toolCalls]);
    const handleCompletedTools = useCallback(async (completedToolCallsFromScheduler) => {
        if (isResponding) {
            return;
        }
        const completedAndReadyToSubmitTools = completedToolCallsFromScheduler.filter((tc) => {
            const isTerminalState = tc.status === 'success' ||
                tc.status === 'error' ||
                tc.status === 'cancelled';
            if (isTerminalState) {
                const completedOrCancelledCall = tc;
                return (completedOrCancelledCall.response?.responseParts !== undefined);
            }
            return false;
        });
        // Finalize any client-initiated tools as soon as they are done.
        const clientTools = completedAndReadyToSubmitTools.filter((t) => t.request.isClientInitiated);
        if (clientTools.length > 0) {
            markToolsAsSubmitted(clientTools.map((t) => t.request.callId));
        }
        // Identify new, successful save_memory calls that we haven't processed yet.
        const newSuccessfulMemorySaves = completedAndReadyToSubmitTools.filter((t) => t.request.name === 'save_memory' &&
            t.status === 'success' &&
            !processedMemoryToolsRef.current.has(t.request.callId));
        if (newSuccessfulMemorySaves.length > 0) {
            // Perform the refresh only if there are new ones.
            void performMemoryRefresh();
            // Mark them as processed so we don't do this again on the next render.
            newSuccessfulMemorySaves.forEach((t) => processedMemoryToolsRef.current.add(t.request.callId));
        }
        const geminiTools = completedAndReadyToSubmitTools.filter((t) => !t.request.isClientInitiated);
        if (geminiTools.length === 0) {
            return;
        }
        // If all the tools were cancelled, don't submit a response to Gemini.
        const allToolsCancelled = geminiTools.every((tc) => tc.status === 'cancelled');
        if (allToolsCancelled) {
            if (geminiClient) {
                // We need to manually add the function responses to the history
                // so the model knows the tools were cancelled.
                const combinedParts = geminiTools.flatMap((toolCall) => toolCall.response.responseParts);
                geminiClient.addHistory({
                    role: 'user',
                    parts: combinedParts,
                });
            }
            const callIdsToMarkAsSubmitted = geminiTools.map((toolCall) => toolCall.request.callId);
            markToolsAsSubmitted(callIdsToMarkAsSubmitted);
            return;
        }
        const responsesToSend = geminiTools.flatMap((toolCall) => toolCall.response.responseParts);
        const callIdsToMarkAsSubmitted = geminiTools.map((toolCall) => toolCall.request.callId);
        const prompt_ids = geminiTools.map((toolCall) => toolCall.request.prompt_id);
        markToolsAsSubmitted(callIdsToMarkAsSubmitted);
        // Don't continue if model was switched due to quota error
        if (modelSwitchedFromQuotaError) {
            return;
        }
        submitQuery(responsesToSend, {
            isContinuation: true,
        }, prompt_ids[0]);
    }, [
        isResponding,
        submitQuery,
        markToolsAsSubmitted,
        geminiClient,
        performMemoryRefresh,
        modelSwitchedFromQuotaError,
    ]);
    const pendingHistoryItems = [
        pendingHistoryItemRef.current,
        pendingToolCallGroupDisplay,
    ].filter((i) => i !== undefined && i !== null);
    useEffect(() => {
        const saveRestorableToolCalls = async () => {
            if (!config.getCheckpointingEnabled()) {
                return;
            }
            const restorableToolCalls = toolCalls.filter((toolCall) => EDIT_TOOL_NAMES.has(toolCall.request.name) &&
                toolCall.status === 'awaiting_approval');
            if (restorableToolCalls.length > 0) {
                const checkpointDir = storage.getProjectTempCheckpointsDir();
                if (!checkpointDir) {
                    return;
                }
                try {
                    await fs.mkdir(checkpointDir, { recursive: true });
                }
                catch (error) {
                    if (!isNodeError(error) || error.code !== 'EEXIST') {
                        onDebugMessage(`Failed to create checkpoint directory: ${getErrorMessage(error)}`);
                        return;
                    }
                }
                for (const toolCall of restorableToolCalls) {
                    const filePath = toolCall.request.args['file_path'];
                    if (!filePath) {
                        onDebugMessage(`Skipping restorable tool call due to missing file_path: ${toolCall.request.name}`);
                        continue;
                    }
                    try {
                        if (!gitService) {
                            onDebugMessage(`Checkpointing is enabled but Git service is not available. Failed to create snapshot for ${filePath}. Ensure Git is installed and working properly.`);
                            continue;
                        }
                        let commitHash;
                        try {
                            commitHash = await gitService.createFileSnapshot(`Snapshot for ${toolCall.request.name}`);
                        }
                        catch (error) {
                            onDebugMessage(`Failed to create new snapshot: ${getErrorMessage(error)}. Attempting to use current commit.`);
                        }
                        if (!commitHash) {
                            commitHash = await gitService.getCurrentCommitHash();
                        }
                        if (!commitHash) {
                            onDebugMessage(`Failed to create snapshot for ${filePath}. Checkpointing may not be working properly. Ensure Git is installed and the project directory is accessible.`);
                            continue;
                        }
                        const timestamp = new Date()
                            .toISOString()
                            .replace(/:/g, '-')
                            .replace(/\./g, '_');
                        const toolName = toolCall.request.name;
                        const fileName = path.basename(filePath);
                        const toolCallWithSnapshotFileName = `${timestamp}-${fileName}-${toolName}.json`;
                        const clientHistory = await geminiClient?.getHistory();
                        const toolCallWithSnapshotFilePath = path.join(checkpointDir, toolCallWithSnapshotFileName);
                        await fs.writeFile(toolCallWithSnapshotFilePath, JSON.stringify({
                            history,
                            clientHistory,
                            toolCall: {
                                name: toolCall.request.name,
                                args: toolCall.request.args,
                            },
                            commitHash,
                            filePath,
                        }, null, 2));
                    }
                    catch (error) {
                        onDebugMessage(`Failed to create checkpoint for ${filePath}: ${getErrorMessage(error)}. This may indicate a problem with Git or file system permissions.`);
                    }
                }
            }
        };
        saveRestorableToolCalls();
    }, [
        toolCalls,
        config,
        onDebugMessage,
        gitService,
        history,
        geminiClient,
        storage,
    ]);
    return {
        streamingState,
        submitQuery,
        initError,
        pendingHistoryItems,
        thought,
        cancelOngoingRequest,
        pendingToolCalls: toolCalls,
        handleApprovalModeChange,
        activePtyId,
        loopDetectionConfirmationRequest,
    };
};
//# sourceMappingURL=useGeminiStream.js.map