/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import fs from 'node:fs';
import path from 'node:path';
import * as Diff from 'diff';
import { ApprovalMode } from '../config/config.js';
import { BaseDeclarativeTool, BaseToolInvocation, Kind, ToolConfirmationOutcome, } from './tools.js';
import { ToolErrorType } from './tool-error.js';
import { makeRelative, shortenPath } from '../utils/paths.js';
import { getErrorMessage, isNodeError } from '../utils/errors.js';
import { DEFAULT_DIFF_OPTIONS, getDiffStat } from './diffOptions.js';
import { ToolNames } from './tool-names.js';
import { getSpecificMimeType } from '../utils/fileUtils.js';
import { FileOperation } from '../telemetry/metrics.js';
import { IDEConnectionStatus } from '../ide/ide-client.js';
import { getProgrammingLanguage } from '../telemetry/telemetry-utils.js';
import { logFileOperation } from '../telemetry/loggers.js';
import { FileOperationEvent } from '../telemetry/types.js';
export async function getCorrectedFileContent(config, filePath, proposedContent) {
    let originalContent = '';
    let fileExists = false;
    const correctedContent = proposedContent;
    try {
        originalContent = await config
            .getFileSystemService()
            .readTextFile(filePath);
        fileExists = true; // File exists and was read
    }
    catch (err) {
        if (isNodeError(err) && err.code === 'ENOENT') {
            fileExists = false;
            originalContent = '';
        }
        else {
            // File exists but could not be read (permissions, etc.)
            fileExists = true; // Mark as existing but problematic
            originalContent = ''; // Can't use its content
            const error = {
                message: getErrorMessage(err),
                code: isNodeError(err) ? err.code : undefined,
            };
            // Return early as we can't proceed with content correction meaningfully
            return { originalContent, correctedContent, fileExists, error };
        }
    }
    return { originalContent, correctedContent, fileExists };
}
class WriteFileToolInvocation extends BaseToolInvocation {
    config;
    constructor(config, params) {
        super(params);
        this.config = config;
    }
    toolLocations() {
        return [{ path: this.params.file_path }];
    }
    getDescription() {
        const relativePath = makeRelative(this.params.file_path, this.config.getTargetDir());
        return `Writing to ${shortenPath(relativePath)}`;
    }
    async shouldConfirmExecute(_abortSignal) {
        if (this.config.getApprovalMode() === ApprovalMode.AUTO_EDIT) {
            return false;
        }
        const correctedContentResult = await getCorrectedFileContent(this.config, this.params.file_path, this.params.content);
        if (correctedContentResult.error) {
            // If file exists but couldn't be read, we can't show a diff for confirmation.
            return false;
        }
        const { originalContent, correctedContent } = correctedContentResult;
        const relativePath = makeRelative(this.params.file_path, this.config.getTargetDir());
        const fileName = path.basename(this.params.file_path);
        const fileDiff = Diff.createPatch(fileName, originalContent, // Original content (empty if new file or unreadable)
        correctedContent, // Content after potential correction
        'Current', 'Proposed', DEFAULT_DIFF_OPTIONS);
        const ideClient = this.config.getIdeClient();
        const ideConfirmation = this.config.getIdeMode() &&
            ideClient.getConnectionStatus().status === IDEConnectionStatus.Connected
            ? ideClient.openDiff(this.params.file_path, correctedContent)
            : undefined;
        const confirmationDetails = {
            type: 'edit',
            title: `Confirm Write: ${shortenPath(relativePath)}`,
            fileName,
            filePath: this.params.file_path,
            fileDiff,
            originalContent,
            newContent: correctedContent,
            onConfirm: async (outcome) => {
                if (outcome === ToolConfirmationOutcome.ProceedAlways) {
                    this.config.setApprovalMode(ApprovalMode.AUTO_EDIT);
                }
                if (ideConfirmation) {
                    const result = await ideConfirmation;
                    if (result.status === 'accepted' && result.content) {
                        this.params.content = result.content;
                    }
                }
            },
            ideConfirmation,
        };
        return confirmationDetails;
    }
    async execute(_abortSignal) {
        const { file_path, content, ai_proposed_content, modified_by_user } = this.params;
        const correctedContentResult = await getCorrectedFileContent(this.config, file_path, content);
        if (correctedContentResult.error) {
            const errDetails = correctedContentResult.error;
            const errorMsg = errDetails.code
                ? `Error checking existing file '${file_path}': ${errDetails.message} (${errDetails.code})`
                : `Error checking existing file: ${errDetails.message}`;
            return {
                llmContent: errorMsg,
                returnDisplay: errorMsg,
                error: {
                    message: errorMsg,
                    type: ToolErrorType.FILE_WRITE_FAILURE,
                },
            };
        }
        const { originalContent, correctedContent: fileContent, fileExists, } = correctedContentResult;
        // fileExists is true if the file existed (and was readable or unreadable but caught by readError).
        // fileExists is false if the file did not exist (ENOENT).
        const isNewFile = !fileExists ||
            (correctedContentResult.error !== undefined &&
                !correctedContentResult.fileExists);
        try {
            const dirName = path.dirname(file_path);
            if (!fs.existsSync(dirName)) {
                fs.mkdirSync(dirName, { recursive: true });
            }
            await this.config
                .getFileSystemService()
                .writeTextFile(file_path, fileContent);
            // Generate diff for display result
            const fileName = path.basename(file_path);
            // If there was a readError, originalContent in correctedContentResult is '',
            // but for the diff, we want to show the original content as it was before the write if possible.
            // However, if it was unreadable, currentContentForDiff will be empty.
            const currentContentForDiff = correctedContentResult.error
                ? '' // Or some indicator of unreadable content
                : originalContent;
            const fileDiff = Diff.createPatch(fileName, currentContentForDiff, fileContent, 'Original', 'Written', DEFAULT_DIFF_OPTIONS);
            const originallyProposedContent = ai_proposed_content || content;
            const diffStat = getDiffStat(fileName, currentContentForDiff, originallyProposedContent, content);
            const llmSuccessMessageParts = [
                isNewFile
                    ? `Successfully created and wrote to new file: ${file_path}.`
                    : `Successfully overwrote file: ${file_path}.`,
            ];
            if (modified_by_user) {
                llmSuccessMessageParts.push(`User modified the \`content\` to be: ${content}`);
            }
            const displayResult = {
                fileDiff,
                fileName,
                originalContent: correctedContentResult.originalContent,
                newContent: correctedContentResult.correctedContent,
                diffStat,
            };
            const lines = fileContent.split('\n').length;
            const mimetype = getSpecificMimeType(file_path);
            const extension = path.extname(file_path); // Get extension
            const programming_language = getProgrammingLanguage({ file_path });
            if (isNewFile) {
                logFileOperation(this.config, new FileOperationEvent(WriteFileTool.Name, FileOperation.CREATE, lines, mimetype, extension, diffStat, programming_language));
            }
            else {
                logFileOperation(this.config, new FileOperationEvent(WriteFileTool.Name, FileOperation.UPDATE, lines, mimetype, extension, diffStat, programming_language));
            }
            return {
                llmContent: llmSuccessMessageParts.join(' '),
                returnDisplay: displayResult,
            };
        }
        catch (error) {
            // Capture detailed error information for debugging
            let errorMsg;
            let errorType = ToolErrorType.FILE_WRITE_FAILURE;
            if (isNodeError(error)) {
                // Handle specific Node.js errors with their error codes
                errorMsg = `Error writing to file '${file_path}': ${error.message} (${error.code})`;
                // Log specific error types for better debugging
                if (error.code === 'EACCES') {
                    errorMsg = `Permission denied writing to file: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.PERMISSION_DENIED;
                }
                else if (error.code === 'ENOSPC') {
                    errorMsg = `No space left on device: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.NO_SPACE_LEFT;
                }
                else if (error.code === 'EISDIR') {
                    errorMsg = `Target is a directory, not a file: ${file_path} (${error.code})`;
                    errorType = ToolErrorType.TARGET_IS_DIRECTORY;
                }
                // Include stack trace in debug mode for better troubleshooting
                if (this.config.getDebugMode() && error.stack) {
                    console.error('Write file error stack:', error.stack);
                }
            }
            else if (error instanceof Error) {
                errorMsg = `Error writing to file: ${error.message}`;
            }
            else {
                errorMsg = `Error writing to file: ${String(error)}`;
            }
            return {
                llmContent: errorMsg,
                returnDisplay: errorMsg,
                error: {
                    message: errorMsg,
                    type: errorType,
                },
            };
        }
    }
}
/**
 * Implementation of the WriteFile tool logic
 */
export class WriteFileTool extends BaseDeclarativeTool {
    config;
    static Name = ToolNames.WRITE_FILE;
    constructor(config) {
        super(WriteFileTool.Name, 'WriteFile', `Writes content to a specified file in the local filesystem.

      The user has the ability to modify \`content\`. If modified, this will be stated in the response.`, Kind.Edit, {
            properties: {
                file_path: {
                    description: "The absolute path to the file to write to (e.g., '/home/user/project/file.txt'). Relative paths are not supported.",
                    type: 'string',
                },
                content: {
                    description: 'The content to write to the file.',
                    type: 'string',
                },
            },
            required: ['file_path', 'content'],
            type: 'object',
        });
        this.config = config;
    }
    validateToolParamValues(params) {
        const filePath = params.file_path;
        if (!filePath) {
            return `Missing or empty "file_path"`;
        }
        if (!path.isAbsolute(filePath)) {
            return `File path must be absolute: ${filePath}`;
        }
        const workspaceContext = this.config.getWorkspaceContext();
        if (!workspaceContext.isPathWithinWorkspace(filePath)) {
            const directories = workspaceContext.getDirectories();
            return `File path must be within one of the workspace directories: ${directories.join(', ')}`;
        }
        try {
            if (fs.existsSync(filePath)) {
                const stats = fs.lstatSync(filePath);
                if (stats.isDirectory()) {
                    return `Path is a directory, not a file: ${filePath}`;
                }
            }
        }
        catch (statError) {
            return `Error accessing path properties for validation: ${filePath}. Reason: ${statError instanceof Error ? statError.message : String(statError)}`;
        }
        return null;
    }
    createInvocation(params) {
        return new WriteFileToolInvocation(this.config, params);
    }
    getModifyContext(_abortSignal) {
        return {
            getFilePath: (params) => params.file_path,
            getCurrentContent: async (params) => {
                const correctedContentResult = await getCorrectedFileContent(this.config, params.file_path, params.content);
                return correctedContentResult.originalContent;
            },
            getProposedContent: async (params) => {
                const correctedContentResult = await getCorrectedFileContent(this.config, params.file_path, params.content);
                return correctedContentResult.correctedContent;
            },
            createUpdatedParams: (_oldContent, modifiedProposedContent, originalParams) => {
                const content = originalParams.content;
                return {
                    ...originalParams,
                    ai_proposed_content: content,
                    content: modifiedProposedContent,
                    modified_by_user: true,
                };
            },
        };
    }
}
//# sourceMappingURL=write-file.js.map