%feature("docstring") OT::ValueFunction
"Function mapping a field to a field.

Parameters
----------
g : :class:`~openturns.Function`
    Function :math:`g: \\Rset^d \\rightarrow \\Rset^{d'}`.
mesh : :class:`~openturns.Mesh`
    The mesh of the domain :math:`\\cD`.

Notes
-----
Value functions act on fields to produce fields such that:

.. math::

    f: \\left| \\begin{array}{rcl}
               \\cM_N \\times (\\Rset^d)^N & \\rightarrow & \\cM_N \\times (\\Rset^{d'})^{N} \\\\
                F & \\mapsto & F'
              \\end{array} \\right.

with :math:`F = (\\vect{t}_i, \\vect{v}_i)_{1 \\leq i \\leq N}`, :math:`F' = (\\vect{t}_i, \\vect{v}'_i)_{1 \\leq i \\leq N}` and :math:`\\cM_{N}` a mesh of :math:`\\cD \\subset \\Rset^{n}`.

A value function keeps the mesh unchanged: the input mesh is equal to the output mesh.

The  field :math:`F'` is defined by the function :math:`g: \\Rset^d \\rightarrow \\Rset^{d'}`:

.. math::

    \\forall \\vect{t}_i \\in \\cM_N, \\quad \\vect{v}'_i = g(\\vect{v}_i)
 

The first constructor builds an object which evaluation operator is not defined (it throws a *NotYetImplementedException*).
The instantiation of such an object is used to extract an actual :class:`~openturns.ValueFunction` from a  :class:`~openturns.Study`.

See also
--------
FieldFunction, VertexValueFunction, OpenTURNSPythonFunction, PythonFunction

Examples
--------
>>> import openturns as ot

Create the function :math:`g : \\Rset \\rightarrow \\Rset` defined by:

.. math::

    g: \\left|\\begin{array}{rcl}
                \\Rset & \\rightarrow & \\Rset \\\\
                x & \\mapsto & x^2
            \\end{array}\\right.

>>> g = ot.SymbolicFunction('x', 'x^2')

Convert :math:`g` into a value function with :math:`n=1` the dimension of the
mesh of the field on which :math:`g` will be applied:

>>> n = 1
>>> tg = ot.RegularGrid(0.0, 0.2, 6)
>>> myValueFunction = ot.ValueFunction(g, tg)
>>> # Create a TimeSeries
>>> data = ot.Sample(tg.getN(), g.getInputDimension())
>>> for i in range(data.getSize()):
...     for j in range(data.getDimension()):
...         data[i, j] = i * data.getDimension() + j
>>> ts = ot.TimeSeries(tg, data)
>>> print(ts)
    [ t   v0  ]
0 : [ 0   0   ]
1 : [ 0.2 1   ]
2 : [ 0.4 2   ]
3 : [ 0.6 3   ]
4 : [ 0.8 4   ]
5 : [ 1   5   ]
>>> print(myValueFunction(ts))
    [ y0 ]
0 : [  0 ]
1 : [  1 ]
2 : [  4 ]
3 : [  9 ]
4 : [ 16 ]
5 : [ 25 ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::ValueFunction::getFunction
"Get the function :math:`g`.

Returns
-------
g : :class:`~openturns.Function`
    Function :math:`g: \\Rset^d \\rightarrow \\Rset^{d'}`.

Examples
--------
>>> import openturns as ot
>>> g = ot.SymbolicFunction('x', 'x^2')
>>> n = 1
>>> mesh = ot.Mesh(n)
>>> myValueFunction = ot.ValueFunction(g, mesh)
>>> print(myValueFunction.getFunction())
[x]->[x^2]"

// ---------------------------------------------------------------------

%feature("docstring") OT::ValueFunction::getMarginal
"Get the marginal(s) at given indice(s).

Parameters
----------
i : int or list of ints, :math:`0 \\leq i < d'`
    Indice(s) of the marginal(s) to be extracted.

Returns
-------
function : :class:`~openturns.ValueFunction`
    The initial function restricted to the concerned marginal(s) at the indice(s)
    :math:`i`."

// ---------------------------------------------------------------------

%feature("docstring") OT::ValueFunction::getOutputMesh
"Get the mesh associated to the output domain.

Returns
-------
outputMesh : :class:`~openturns.Mesh`
    The output mesh which is equal to the input one: :math:`\\cM_{N}`."

