%feature("docstring") OT::ResourceMap
"The ResourceMap allows one to manage constants that are associated to key strings.

Find below the ResourceMap's content:

.. execforresourcemap::

Examples
--------
Get some keys:

>>> import openturns as ot
>>> for key in ot.ResourceMap.GetKeys()[:4]:
...     print(key)
ARMA-MeanEpsilon
ARMALikelihoodFactory-DefaultMaximumCallsNumber
ARMALikelihoodFactory-DefaultRhoBeg
ARMALikelihoodFactory-DefaultRhoEnd"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::Get
"Access a parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
value : str
    The value associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.Get('WeightedExperiment-DefaultSize')
'100'"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetType
"Access the type of a parameter.

Parameters
----------
key : str, possible values are 'str','int','float' or 'bool'
    An identifier associated to the parameter.

Returns
-------
type : str
    The type associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.GetType('Distribution-SupportEpsilon')
'float'"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::Set
"Access an existing parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : str
    The value associated to that key. If the key already exists as a *float*,
    an *int* or a *bool*, the given string is converted into the
    corresponding type.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.Set('Distribution-SupportEpsilon', '1.0e-14')
"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetAsBool
"Access a boolean parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
value : bool
    The value associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.GetAsBool('MaximumEntropyOrderStatisticsDistribution-CheckMarginals')
True"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::SetAsBool
"Set an existing boolean parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : bool
    The value associated to that key. The key is added to the bool map even if
    it already exists in another map (float, int or str).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.SetAsBool('MaximumEntropyOrderStatisticsDistribution-CheckMarginals', True)"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::AddAsBool
"Add a new boolean parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : bool
    The value associated to that key. The key is added to the bool map even if
    it already exists in another map (float, int or str).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.AddAsBool('MaximumEntropyOrderStatisticsDistribution-CheckMarginals2', True)
>>> ot.ResourceMap.RemoveKey('MaximumEntropyOrderStatisticsDistribution-CheckMarginals2')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::SetAsScalar
"Set an existing floating-point parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : float
    The value associated to that key. The key is added to the float map even if
    it already exists in another map (str, int or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.SetAsScalar('SQP-DefaultTau', 0.5)"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::AddAsScalar
"Add a new floating-point parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : float
    The value associated to that key. The key is added to the float map even if
    it already exists in another map (str, int or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.AddAsScalar('SQP-DefaultTau2', 0.5)
>>> ot.ResourceMap.RemoveKey('SQP-DefaultTau2')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetAsScalar
"Access a floating-point parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
value : float
    The value associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.GetAsScalar('SQP-DefaultTau')
0.5"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::SetAsString
"Set an existing string parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : str
    The value associated to that key. The key is added to the string map even if
    it already exists in another map (float, int or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.SetAsString('View-ImageFormat', 'png')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::AddAsString
"Add a new string parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : str
    The value associated to that key. The key is added to the string map even if
    it already exists in another map (float, int or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.AddAsString('View-ImageFormat2', 'png')
>>> ot.ResourceMap.RemoveKey('View-ImageFormat2')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetAsString
"Access a string parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
value : str
    The value associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.Get('WeightedExperiment-DefaultSize')
'100'"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::SetAsUnsignedInteger
"Set an existing integer parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : int
    The value associated to that key. The key is added to the integer
    map even if it already exists in another map (float, str or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.SetAsUnsignedInteger('RandomWalkMetropolisHastings-DefaultBurnIn', 1000)"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::AddAsUnsignedInteger
"Add a new integer parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.
value : int
    The value associated to that key. The key is added to the integer
    map even if it already exists in another map (float, str or bool).

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.AddAsUnsignedInteger('RandomWalkMetropolisHastings-DefaultBurnIn2', 0)
>>> ot.ResourceMap.RemoveKey('RandomWalkMetropolisHastings-DefaultBurnIn2')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetAsUnsignedInteger
"Access an integer parameter.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
value : int
    The value associated to that key.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.GetAsUnsignedInteger('RandomWalkMetropolisHastings-DefaultBurnIn')
1000"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetSize
"Get the size of the ResourceMap.

Returns
-------
size : positive int
    The number of constants defined in the ResourceMap."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetBoolSize
"Get the size of the bool map in ResourceMap.

Returns
-------
size : positive int
    The number of *bool* constants defined in the ResourceMap."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetScalarSize
"Get the size of the float map in ResourceMap.

Returns
-------
size : positive int
    The number of *float* constants defined in the ResourceMap."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetStringSize
"Get the size of the string map in ResourceMap.

Returns
-------
size : positive int
    The number of *string* constants defined in the ResourceMap."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetUnsignedIntegerSize
"Get the size of the unsigned integer map in ResourceMap.

Returns
-------
size : positive int
    The number of *int* constants defined in the ResourceMap."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetKeys
"Get the keys of the ResourceMap.

Returns
-------
keys : tuple of str
    The list of identifiers."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetBoolKeys
"Get the bool keys of the ResourceMap.

Returns
-------
keys : tuple of str
    The list of *bool* identifiers."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetScalarKeys
"Get the float keys of the ResourceMap.

Returns
-------
keys : tuple of str
    The list of *float* identifiers."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetStringKeys
"Get the string keys of the ResourceMap.

Returns
-------
keys : tuple of str
    The list of *string* identifiers."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::GetUnsignedIntegerKeys
"Get the unsigned integer keys of the ResourceMap.

Returns
-------
keys : tuple of str
    The list of *int* identifiers."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::HasKey
"Check if an entry exists.

Parameters
----------
key : str
    An identifier associated to the parameter.

Returns
-------
has_key : bool
    Whether an entry with that key exists."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::Reload
"Reload the configuration.

Reverts the values of the default keys and erases new keys."

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::RemoveKey
"Remove an entry.

Parameters
----------
key : str
    An identifier associated to the parameter.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.AddAsString('dummy_key', 'key_to_be_removed')
>>> ot.ResourceMap.RemoveKey('dummy_key')"

// ---------------------------------------------------------------------

%feature("docstring") OT::ResourceMap::FindKeys
"Find keys matching a pattern.

Parameters
----------
substr : str
    A pattern to search.

Returns
-------
value : sequence of str
    The keys matching the pattern.

Examples
--------
>>> import openturns as ot
>>> ot.ResourceMap.FindKeys('Multinomial')
('Multinomial-eta', 'Multinomial-smallA')"
