/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType } from '@google/gemini-cli-core';
import { USER_SETTINGS_PATH } from './config/settings.js';
import { validateAuthMethod } from './config/auth.js';
import {} from './config/settings.js';
function getAuthTypeFromEnv() {
    if (process.env['GOOGLE_GENAI_USE_GCA'] === 'true') {
        return AuthType.LOGIN_WITH_GOOGLE;
    }
    if (process.env['GOOGLE_GENAI_USE_VERTEXAI'] === 'true') {
        return AuthType.USE_VERTEX_AI;
    }
    if (process.env['GEMINI_API_KEY']) {
        return AuthType.USE_GEMINI;
    }
    return undefined;
}
export async function validateNonInteractiveAuth(configuredAuthType, useExternalAuth, nonInteractiveConfig, settings) {
    const enforcedType = settings.merged.security?.auth?.enforcedType;
    if (enforcedType) {
        const currentAuthType = getAuthTypeFromEnv();
        if (currentAuthType !== enforcedType) {
            console.error(`The configured auth type is ${enforcedType}, but the current auth type is ${currentAuthType}. Please re-authenticate with the correct type.`);
            process.exit(1);
        }
    }
    const effectiveAuthType = enforcedType || getAuthTypeFromEnv() || configuredAuthType;
    if (!effectiveAuthType) {
        console.error(`Please set an Auth method in your ${USER_SETTINGS_PATH} or specify one of the following environment variables before running: GEMINI_API_KEY, GOOGLE_GENAI_USE_VERTEXAI, GOOGLE_GENAI_USE_GCA`);
        process.exit(1);
    }
    if (!useExternalAuth) {
        const err = validateAuthMethod(effectiveAuthType);
        if (err != null) {
            console.error(err);
            process.exit(1);
        }
    }
    await nonInteractiveConfig.refreshAuth(effectiveAuthType);
    return nonInteractiveConfig;
}
//# sourceMappingURL=validateNonInterActiveAuth.js.map