﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>
#include <aws/pinpoint/model/MessageType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pinpoint {
namespace Model {

/**
 * <p>Specifies the default settings for a one-time SMS message that's sent
 * directly to an endpoint.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pinpoint-2016-12-01/SMSMessage">AWS
 * API Reference</a></p>
 */
class SMSMessage {
 public:
  AWS_PINPOINT_API SMSMessage() = default;
  AWS_PINPOINT_API SMSMessage(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API SMSMessage& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The body of the SMS message.</p>
   */
  inline const Aws::String& GetBody() const { return m_body; }
  inline bool BodyHasBeenSet() const { return m_bodyHasBeenSet; }
  template <typename BodyT = Aws::String>
  void SetBody(BodyT&& value) {
    m_bodyHasBeenSet = true;
    m_body = std::forward<BodyT>(value);
  }
  template <typename BodyT = Aws::String>
  SMSMessage& WithBody(BodyT&& value) {
    SetBody(std::forward<BodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SMS program name that you provided to AWS Support when you requested your
   * dedicated number.</p>
   */
  inline const Aws::String& GetKeyword() const { return m_keyword; }
  inline bool KeywordHasBeenSet() const { return m_keywordHasBeenSet; }
  template <typename KeywordT = Aws::String>
  void SetKeyword(KeywordT&& value) {
    m_keywordHasBeenSet = true;
    m_keyword = std::forward<KeywordT>(value);
  }
  template <typename KeywordT = Aws::String>
  SMSMessage& WithKeyword(KeywordT&& value) {
    SetKeyword(std::forward<KeywordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This field is reserved for future use.</p>
   */
  inline const Aws::String& GetMediaUrl() const { return m_mediaUrl; }
  inline bool MediaUrlHasBeenSet() const { return m_mediaUrlHasBeenSet; }
  template <typename MediaUrlT = Aws::String>
  void SetMediaUrl(MediaUrlT&& value) {
    m_mediaUrlHasBeenSet = true;
    m_mediaUrl = std::forward<MediaUrlT>(value);
  }
  template <typename MediaUrlT = Aws::String>
  SMSMessage& WithMediaUrl(MediaUrlT&& value) {
    SetMediaUrl(std::forward<MediaUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SMS message type. Valid values are TRANSACTIONAL (for messages that are
   * critical or time-sensitive, such as a one-time passwords) and PROMOTIONAL (for
   * messsages that aren't critical or time-sensitive, such as marketing
   * messages).</p>
   */
  inline MessageType GetMessageType() const { return m_messageType; }
  inline bool MessageTypeHasBeenSet() const { return m_messageTypeHasBeenSet; }
  inline void SetMessageType(MessageType value) {
    m_messageTypeHasBeenSet = true;
    m_messageType = value;
  }
  inline SMSMessage& WithMessageType(MessageType value) {
    SetMessageType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number to send the SMS message from. This value should be one of the
   * dedicated long or short codes that's assigned to your AWS account. If you don't
   * specify a long or short code, Amazon Pinpoint assigns a random long code to the
   * SMS message and sends the message from that code.</p>
   */
  inline const Aws::String& GetOriginationNumber() const { return m_originationNumber; }
  inline bool OriginationNumberHasBeenSet() const { return m_originationNumberHasBeenSet; }
  template <typename OriginationNumberT = Aws::String>
  void SetOriginationNumber(OriginationNumberT&& value) {
    m_originationNumberHasBeenSet = true;
    m_originationNumber = std::forward<OriginationNumberT>(value);
  }
  template <typename OriginationNumberT = Aws::String>
  SMSMessage& WithOriginationNumber(OriginationNumberT&& value) {
    SetOriginationNumber(std::forward<OriginationNumberT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sender ID to display as the sender of the message on a recipient's
   * device. Support for sender IDs varies by country or region.</p>
   */
  inline const Aws::String& GetSenderId() const { return m_senderId; }
  inline bool SenderIdHasBeenSet() const { return m_senderIdHasBeenSet; }
  template <typename SenderIdT = Aws::String>
  void SetSenderId(SenderIdT&& value) {
    m_senderIdHasBeenSet = true;
    m_senderId = std::forward<SenderIdT>(value);
  }
  template <typename SenderIdT = Aws::String>
  SMSMessage& WithSenderId(SenderIdT&& value) {
    SetSenderId(std::forward<SenderIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The message variables to use in the SMS message. You can override the default
   * variables with individual address variables.</p>
   */
  inline const Aws::Map<Aws::String, Aws::Vector<Aws::String>>& GetSubstitutions() const { return m_substitutions; }
  inline bool SubstitutionsHasBeenSet() const { return m_substitutionsHasBeenSet; }
  template <typename SubstitutionsT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  void SetSubstitutions(SubstitutionsT&& value) {
    m_substitutionsHasBeenSet = true;
    m_substitutions = std::forward<SubstitutionsT>(value);
  }
  template <typename SubstitutionsT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  SMSMessage& WithSubstitutions(SubstitutionsT&& value) {
    SetSubstitutions(std::forward<SubstitutionsT>(value));
    return *this;
  }
  template <typename SubstitutionsKeyT = Aws::String, typename SubstitutionsValueT = Aws::Vector<Aws::String>>
  SMSMessage& AddSubstitutions(SubstitutionsKeyT&& key, SubstitutionsValueT&& value) {
    m_substitutionsHasBeenSet = true;
    m_substitutions.emplace(std::forward<SubstitutionsKeyT>(key), std::forward<SubstitutionsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The entity ID or Principal Entity (PE) id received from the regulatory body
   * for sending SMS in your country.</p>
   */
  inline const Aws::String& GetEntityId() const { return m_entityId; }
  inline bool EntityIdHasBeenSet() const { return m_entityIdHasBeenSet; }
  template <typename EntityIdT = Aws::String>
  void SetEntityId(EntityIdT&& value) {
    m_entityIdHasBeenSet = true;
    m_entityId = std::forward<EntityIdT>(value);
  }
  template <typename EntityIdT = Aws::String>
  SMSMessage& WithEntityId(EntityIdT&& value) {
    SetEntityId(std::forward<EntityIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The template ID received from the regulatory body for sending SMS in your
   * country.</p>
   */
  inline const Aws::String& GetTemplateId() const { return m_templateId; }
  inline bool TemplateIdHasBeenSet() const { return m_templateIdHasBeenSet; }
  template <typename TemplateIdT = Aws::String>
  void SetTemplateId(TemplateIdT&& value) {
    m_templateIdHasBeenSet = true;
    m_templateId = std::forward<TemplateIdT>(value);
  }
  template <typename TemplateIdT = Aws::String>
  SMSMessage& WithTemplateId(TemplateIdT&& value) {
    SetTemplateId(std::forward<TemplateIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_body;
  bool m_bodyHasBeenSet = false;

  Aws::String m_keyword;
  bool m_keywordHasBeenSet = false;

  Aws::String m_mediaUrl;
  bool m_mediaUrlHasBeenSet = false;

  MessageType m_messageType{MessageType::NOT_SET};
  bool m_messageTypeHasBeenSet = false;

  Aws::String m_originationNumber;
  bool m_originationNumberHasBeenSet = false;

  Aws::String m_senderId;
  bool m_senderIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::Vector<Aws::String>> m_substitutions;
  bool m_substitutionsHasBeenSet = false;

  Aws::String m_entityId;
  bool m_entityIdHasBeenSet = false;

  Aws::String m_templateId;
  bool m_templateIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
