﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/DescribeOptionGroupOptionsMessage">AWS
 * API Reference</a></p>
 */
class DescribeOptionGroupOptionsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeOptionGroupOptionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeOptionGroupOptions"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the engine to describe options for.</p> <p>Valid Values:</p> <ul>
   * <li> <p> <code>db2-ae</code> </p> </li> <li> <p> <code>db2-se</code> </p> </li>
   * <li> <p> <code>mariadb</code> </p> </li> <li> <p> <code>mysql</code> </p> </li>
   * <li> <p> <code>oracle-ee</code> </p> </li> <li> <p> <code>oracle-ee-cdb</code>
   * </p> </li> <li> <p> <code>oracle-se2</code> </p> </li> <li> <p>
   * <code>oracle-se2-cdb</code> </p> </li> <li> <p> <code>postgres</code> </p> </li>
   * <li> <p> <code>sqlserver-ee</code> </p> </li> <li> <p> <code>sqlserver-se</code>
   * </p> </li> <li> <p> <code>sqlserver-ex</code> </p> </li> <li> <p>
   * <code>sqlserver-web</code> </p> </li> </ul>
   */
  inline const Aws::String& GetEngineName() const { return m_engineName; }
  inline bool EngineNameHasBeenSet() const { return m_engineNameHasBeenSet; }
  template <typename EngineNameT = Aws::String>
  void SetEngineName(EngineNameT&& value) {
    m_engineNameHasBeenSet = true;
    m_engineName = std::forward<EngineNameT>(value);
  }
  template <typename EngineNameT = Aws::String>
  DescribeOptionGroupOptionsRequest& WithEngineName(EngineNameT&& value) {
    SetEngineName(std::forward<EngineNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If specified, filters the results to include only options for the specified
   * major engine version.</p>
   */
  inline const Aws::String& GetMajorEngineVersion() const { return m_majorEngineVersion; }
  inline bool MajorEngineVersionHasBeenSet() const { return m_majorEngineVersionHasBeenSet; }
  template <typename MajorEngineVersionT = Aws::String>
  void SetMajorEngineVersion(MajorEngineVersionT&& value) {
    m_majorEngineVersionHasBeenSet = true;
    m_majorEngineVersion = std::forward<MajorEngineVersionT>(value);
  }
  template <typename MajorEngineVersionT = Aws::String>
  DescribeOptionGroupOptionsRequest& WithMajorEngineVersion(MajorEngineVersionT&& value) {
    SetMajorEngineVersion(std::forward<MajorEngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter isn't currently supported.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeOptionGroupOptionsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeOptionGroupOptionsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that you can retrieve the
   * remaining results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum
   * 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeOptionGroupOptionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous request. If this
   * parameter is specified, the response includes only records beyond the marker, up
   * to the value specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeOptionGroupOptionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_engineName;

  Aws::String m_majorEngineVersion;

  Aws::Vector<Filter> m_filters;

  int m_maxRecords{0};

  Aws::String m_marker;
  bool m_engineNameHasBeenSet = false;
  bool m_majorEngineVersionHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
