﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogsRequest.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchLogs {
namespace Model {

/**
 */
class DeleteLogAnomalyDetectorRequest : public CloudWatchLogsRequest {
 public:
  AWS_CLOUDWATCHLOGS_API DeleteLogAnomalyDetectorRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteLogAnomalyDetector"; }

  AWS_CLOUDWATCHLOGS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHLOGS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the anomaly detector to delete. You can find the ARNs of log
   * anomaly detectors in your account by using the <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListLogAnomalyDetectors.html">ListLogAnomalyDetectors</a>
   * operation.</p>
   */
  inline const Aws::String& GetAnomalyDetectorArn() const { return m_anomalyDetectorArn; }
  inline bool AnomalyDetectorArnHasBeenSet() const { return m_anomalyDetectorArnHasBeenSet; }
  template <typename AnomalyDetectorArnT = Aws::String>
  void SetAnomalyDetectorArn(AnomalyDetectorArnT&& value) {
    m_anomalyDetectorArnHasBeenSet = true;
    m_anomalyDetectorArn = std::forward<AnomalyDetectorArnT>(value);
  }
  template <typename AnomalyDetectorArnT = Aws::String>
  DeleteLogAnomalyDetectorRequest& WithAnomalyDetectorArn(AnomalyDetectorArnT&& value) {
    SetAnomalyDetectorArn(std::forward<AnomalyDetectorArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_anomalyDetectorArn;
  bool m_anomalyDetectorArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
