/***************************************************************************
                          kpl.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPL_H
#define KPL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dcopobject.h>
#include <kmainwindow.h>
#include <kurl.h>

class QStrList;
class KAccel;
class KAction;
class KRecentFilesAction;
class KToggleAction;
class KURL;
class ArrayItem;
class FunItem;
class ItemDlg;
class KplDoc;
class KplView;
class PSGraph;

/**
  * Kpl application.
  * Provides the kpl top level main window with menubar, toolbar and statusbar.
  * @author Werner Stille
  */
class KplApp : public KMainWindow, public DCOPObject
{
  K_DCOP
  Q_OBJECT
public:
  /** Constructor. Creates a kpl window. */
  KplApp();
  /** Destructor. */
  ~KplApp();
  /**
   * Returns a pointer to the current document connected to the KMainWindow
   * instance and is used by the View class to access the document object's
   * methods.
   * @return pointer to the current document.
   */
  KplDoc* getDocument() const; 	
  /**
   * Opens files from URL list.
   * @param list pointer to list with URLs.
   * @param idec 0 for decimal point, 1 for comma, -1 for auto settings.
   * @param type one of the following: Unknown, Data, Plot.
   */
  void openList(KURL::List* list, int idec, int type);

k_dcop:
  /** Creates a new plot in the current view. */
  void newPlot();
  /**
   * Opens a plot file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  int openPlot(const QString& url);
  /**
   * Opens a data file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  int openData(const QString& url);
  /** Performs plot. */
  void doPlot();
  /**
   * Saves a plot file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  int savePlot(const QString& url);
  /**
   * Generates PostScript file.
   * @param url URL of the file.
   * @param orientation true for landscape orientation.
   * @return true when when succeeded.
   */
  int savePS(const QString& url, int orientation);
  /**
   * Prints plot.
   * @param printerName name of the printer.
   * @param fileName name of the output file (empty for output to printer).
   * @param orientation true for landscape orientation.
   * @param numCopies number of copies.
   * @return true when when succeeded.
   */
  int printPlot(const QString& printerName, const QString & fileName,
                int orientation, int numCopies);
  /**
   * Sets autoplot option.
   * @param setting true to plot automatically after changes.
   */
  void setAutoplotOption(int setting);
  /**
   * Sets add data option.
   * @param setting true to add data and plot files to current plot.
   */
  void setAddFilesOption(int setting);
  /**
   * Sets path saving option.
   * @param setting true to save absolute paths.
   */
  void setPathOption(int setting);
  /**
   * Sets calculate PS bounding path option.
   * @param setting true to calculate automatically the box.
   */
  void setBoundingBoxOption(int setting);
  /**
   * Sets page size.
   * @param w width in cm.
   * @param h height in cm.
   */
  void setPageFormat(double w, double h);
  /**
   * Sets default colors.
   * @param colFrame color for frame and labels.
   * @param colGrid color for ticks and grid lines.
   * @param colData color for lines and markers.
   */
  void setColors(const QString& colFrame, const QString& colGrid,
                 const QString& colData);
  /**
   * Sets format for numeric output.
   * @param pres printf() presentation character.
   * @param prec precision.
   */
  void setOutputFormat(const QString& pres, int prec);
  /**
   * Sets Autoplot parameters.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels, 3 for grid, 4 for grid with labels.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sh heading text.
   * @param pathHeading true for heading = path.
   * @param ix x column index.
   * @param iy y column index.
   * @param ie y error column index.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param errbars true to plot error bars.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param autoNorm true for automatic normalization.
   * @param autoFit true for automatic parameter fit.
   */
  void setAutoplot(double x0, double w, double y0, double h,
                   int gridmode, const QString& sx, const QString& sy,
                   const QString& sh, int pathHeading, int ix, int iy,
                   int ie, int logx, int logy, int errbars, int symb,
                   int autoNorm, int autoFit);
  /**
   * Sets Autofit parameters.
   * @param path absolute path of the library for the fit function.
   * @param name fit function name.
   * @param nonLin true for nonlinear parameter fits.
   * @param errCol true for data weighing by error column, false for weighing
   * by error model function.
   * @param errModPath absolute path of the library for the error model
   * function.
   * @param errModName error model function name.
   * @param itMax maximum number of iterations (nonlinear fits only).
   * @param tol tolerance level (nonlinear fits only).
   * @param symb function plot: 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param showDlg true to display parameter fit dialog.
   * @param savePar true to save resulting parameter files.
   * @param follow true to follow parameter sets in fit series.
   */
  void setAutofit(const QString& path, const QString& name,
                  int nonLin, int errCol,
                  const QString& errModPath, const QString& errModName,
                  int itMax, double tol, int symb, const QString& color,
                  int showDlg, int savePar, int follow);
  /**
   * Loads Autofit function parameters from file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  int loadAutofitPar(const QString& url);
  /**
   * Sets Autofit function parameter value.
   * @param iPar parameter index.
   * @param value parameter value.
   * @return true when when succeeded.
   */
  int setAutofitPar(int iPar, double value);
  /**
   * Enables/disables (Auto)fit function parameter.
   * @param iPar parameter index.
   * @param fit true enables parameter fitting.
   * @return true when when succeeded.
   */
  int enableFitPar(int iPar, int fit);
  /**
   * Loads error model function parameters from file.
   * @param url URL of the file.
   * @param iArr array list index.
   * @return true when when succeeded.
   */
  int loadErrModPar(const QString& url, int iArr);
  /**
   * Sets error model function parameter value.
   * @param iPar parameter index.
   * @param value parameter value.
   * @param iArr array list index.
   * @return true when when succeeded.
   */
  int setErrModPar(int iPar, double value, int iArr);
  /**
   * Deletes plot item.
   * @param iItem item index.
   * @return true when when succeeded.
   */
  int deleteItem(int iItem);
  /**
   * Moves plot item.
   * @param is source item index.
   * @param id destination item index.
   * @return true when when succeeded.
   */
  int moveItem(int is, int id);
  /**
   * Activates or deactivates plot item.
   * @param iItem item index.
   * @param active true for activating item (making visible).
   * @return true when when succeeded.
   */
  int activateItem(int iItem, int active);
  /**
   * Appends new frame item corresponding to arguments.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param xmin x value of left data window margin in user units.
   * @param xmax x value of right data window margin in user units.
   * @param ymin y value of bottom data window margin in user units.
   * @param ymax y value of top window data margin in user units.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param ndigx number of digits for x axis.
   * @param ndigy number of digits for y axis.
   * @param colFrame color for frame and labels.
   * @param colGrid color for grids and tics.
   * @param xtic distance between major tics of x axis.
   * @param ytic distance between major tics of y axis.
   * @param mticx number of intervals between major tics of x axis.
   * @param mticy number of intervals between major tics of y axis.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels, 3 for grid, 4 for grid with labels.
   * @param iex logarithm of normalization factor for x axis.
   * @param iey logarithm of normalization factor for y axis.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sh heading text.
   * @param relSize relative size of text, markers and lines.
   * @param active true for visible item.
   */
  void newFrameItem(double x0, double w, double y0, double h,
                    double xmin, double xmax, double ymin, double ymax,
                    int logx, int logy, int ndigx, int ndigy,
                    const QString& colFrame, const QString& colGrid,
                    double xtic, double ytic, int mticx, int mticy,
                    int gridmode, int iex, int iey,
                    const QString& sx, const QString& sy, const QString& sh,
                    double relSize, int active);
  /** Appends new frame item corresponding to autoplot settings. */
  void newAutoFrameItem();
  /**
   * Appends new array item.
   * @param path absolute path of the data file.
   * @param ix x column index.
   * @param iy y column index.
   * @param ie y error column index.
   * @param istart start row index.
   * @param n number of data points.
   * @param errbars true to plot error bars.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for data plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
  int newArrayItem(const QString& path, int ix, int iy, int ie, int istart,
                   int n, int errbars, int symb, const QString& color,
                   double fx, double fy, int active);
  /**
   * Appends new function item.
   * @param path absolute path of the function library.
   * @param name function name.
   * @param xmin minimum argument value.
   * @param xmax maximum argument value.
   * @param dx argument stepsize.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
  int newFunItem(const QString& path, const QString& name,
                 double xmin, double xmax, double dx,
                 int symb, const QString& color,
                  double fx, double fy, int active);
  /**
   * Appends new parametric function item.
   * @param pathx absolute path of the x function library.
   * @param namex x function name.
   * @param pathy absolute path of the y function library.
   * @param namey y function name.
   * @param tmin minimum argument value.
   * @param tmax maximum argument value.
   * @param dt argument stepsize.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
  int newParFunItem(const QString& pathx, const QString& namex,
                    const QString& pathy, const QString& namey,
                    double tmin, double tmax, double dt,
                    int symb, const QString& color,
                    double fx, double fy, int active);
  /**
   * Appends new spline item.
   * @param deriv order of derivative.
   * @param low lower limit for spline integral.
   * @param xmin minimum argument value.
   * @param xmax maximum argument value.
   * @param dx argument stepsize.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param relSize relative size of lines or symbols.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   */
  void newSplineItem(int deriv, double low, double xmin, double xmax,
                     double dx, int symb, const QString& color,
                     double relSize, double fx, double fy, int active);
  /**
   * Appends new 3D array item.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param xmin minimum x value.
   * @param xmax maximum x value.
   * @param ymin minimum y value.
   * @param ymax maximum y value.
   * @param zmin minimum z value.
   * @param zmax maximum z value.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param logz true for logarithmic z axis.
   * @param ndigx number of digits for x axis.
   * @param ndigy number of digits for y axis.
   * @param ndigz number of digits for z axis.
   * @param colFrame color for frame and labels.
   * @param colGrid color for grids and tics.
   * @param colData color for data.
   * @param xtic distance between major tics of x axis.
   * @param ytic distance between major tics of y axis.
   * @param ztic distance between major tics of z axis.
   * @param mticx number of intervals between major tics of x axis.
   * @param mticy number of intervals between major tics of y axis.
   * @param mticz number of intervals between major tics of z axis.
   * @param frame true to show frame.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels.
   * @param gridmode3D 0 for lines parallel to x, 1 for lines parallel to z,
   * 2 for x,z grid.
   * @param xBox x box width in cm.
   * @param yBox y box width in cm.
   * @param zBox z box width in cm.
   * @param xRef x position of center of box bottom plane.
   * @param yRef y position of center of box bottom plane.
   * @param phi angle of rotation about y axis in deg.
   * @param theta tilt angle of y axis in deg.
   * @param iex logarithm of normalization factor for x axis.
   * @param iey logarithm of normalization factor for y axis.
   * @param iez logarithm of normalization factor for z axis.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sz z axis text.
   * @param sh heading text.
   * @param relSize relative size of text, markers and lines.
   * @param path absolute path of the data file.
   * @param ix x column index.
   * @param iy y column index.
   * @param iz z column index.
   * @param ie y error column index.
   * @param istart start row index.
   * @param n number of data points.
   * @param errbars true to plot error bars.
   * @param smf smoothing factor.
   * @param dx x argument stepsize.
   * @param dz z argument stepsize.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param fz normalization factor for z values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
  int newArray3DItem(double x0, double w, double y0, double h,
                     double xmin, double xmax, double ymin, double ymax,
                     double zmin, double zmax, int logx, int logy,
                     int logz, int ndigx, int ndigy, int ndigz,
                     const QString& colFrame, const QString& colGrid,
                     const QString& colData, double xtic, double ytic,
                     double ztic, int mticx, int mticy, int mticz,
                     int frame, int gridmode, int gridmode3D,
                     double xBox, double yBox, double zBox, double xRef,
                     double yRef, double phi, double theta, int iex,
                     int iey, int iez, const QString& sx,
                     const QString& sy, const QString& sz,
                     const QString& sh, double relSize,
                     const QString& path, int ix, int iy, int iz, int ie,
                     int istart, int n, int errbars,  double smf, double dx,
                     double dz, double fx, double fy, double fz,
                     int active);
  /**
   * Appends new 3D function item.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param xmin minimum x value.
   * @param xmax maximum x value.
   * @param ymin minimum y value.
   * @param ymax maximum y value.
   * @param zmin minimum z value.
   * @param zmax maximum z value.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param logz true for logarithmic z axis.
   * @param ndigx number of digits for x axis.
   * @param ndigy number of digits for y axis.
   * @param ndigz number of digits for z axis.
   * @param colFrame color for frame and labels.
   * @param colGrid color for grids and tics.
   * @param colData color for data.
   * @param xtic distance between major tics of x axis.
   * @param ytic distance between major tics of y axis.
   * @param ztic distance between major tics of z axis.
   * @param mticx number of intervals between major tics of x axis.
   * @param mticy number of intervals between major tics of y axis.
   * @param mticz number of intervals between major tics of z axis.
   * @param frame true to show frame.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels.
   * @param gridmode3D 0 for lines parallel to x, 1 for lines parallel to z,
   * 2 for x,z grid.
   * @param xBox x box width in cm.
   * @param yBox y box width in cm.
   * @param zBox z box width in cm.
   * @param xRef x position of center of box bottom plane.
   * @param yRef y position of center of box bottom plane.
   * @param phi angle of rotation about y axis in deg.
   * @param theta tilt angle of y axis in deg.
   * @param iex logarithm of normalization factor for x axis.
   * @param iey logarithm of normalization factor for y axis.
   * @param iez logarithm of normalization factor for z axis.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sz z axis text.
   * @param sh heading text.
   * @param relSize relative size of text, markers and lines.
   * @param path absolute path of the function library.
   * @param name function name.
   * @param dx x argument stepsize.
   * @param dz z argument stepsize.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param fz normalization factor for z values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
  int newFun3DItem(double x0, double w, double y0, double h,
                   double xmin, double xmax, double ymin, double ymax,
                   double zmin, double zmax, int logx, int logy,
                   int logz, int ndigx, int ndigy, int ndigz,
                   const QString& colFrame, const QString& colGrid,
                   const QString& colData, double xtic, double ytic,
                   double ztic, int mticx, int mticy, int mticz,
                   int frame, int gridmode, int gridmode3D, double xBox,
                   double yBox, double zBox, double xRef, double yRef,
                   double phi, double theta, int iex, int iey, int iez,
                   const QString& sx, const QString& sy,
                   const QString& sz, const QString& sh, double relSize,
                   const QString& path, const QString& name,
                   double dx, double dz, double fx, double fy,
                   double fz, int active);
  /**
   * Appends new image item.
   * @param path absolute path of the EPS file.
   * @param x x position in cm.
   * @param y y position in cm.
   * @param fx x scaling factor.
   * @param fy y scaling factor.
   * @param active true for visible item.
   */
  void newImageItem(const QString& path, double x, double y, double fx,
                    double fy, int active);
  /**
   * Appends new legend item.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param colSymb color for marker or line.
   * @param x x position in cm.
   * @param y y position in cm.
   * @param text legend text.
   * @param colText color for text.
   * @param xoff x offset for text in cm.
   * @param yoff y offset for text in cm.
   * @param relSize relative size of text.
   * @param active true for visible item.
   */
  void newLegendItem(int symb, const QString& colSymb, double x, double y,
                     const QString& text, const QString& colText,
                     double xoff, double yoff, double relSize,
                     int active);
  /**
   * Appends new text item.
   * @param text text.
   * @param color color for text.
   * @param align 1: left aligned, 2: centered, 3: right aligned.
   * @param x x start position in cm.
   * @param y y start position in cm.
   * @param dir direction in degree.
   * @param relSize relative line width.
   * @param active true for visible item.
   */
  void newTextItem(const QString& text, const QString& color, int align,
                   double x, double y, double dir, double relSize,
                   int active);
  /**
   * Appends new line item.
   * @param symb 1 to 9 for different line types.
   * @param color color for line.
   * @param x x start position in cm.
   * @param y y start position in cm.
   * @param dir direction in degree.
   * @param len length in cm.
   * @param relSize relative line width.
   * @param active true for visible item.
   */
  void newLineItem(int symb, const QString& color, double x, double y,
                   double dir, double len, double relSize, int active);
  /**
   * Appends new arrow item.
   * @param color color for arrow.
   * @param x x position of head in cm.
   * @param y y position of head in cm.
   * @param dir direction in degree.
   * @param len total length in cm.
   * @param relSize relative size of head.
   * @param active true for visible item.
   */
  void newArrowItem(const QString& color, double x, double y,
                    double dir, double len, double relSize, int active);
  /**
    * Appends new arc item.
    * @param color color for arc.
    * @param x x position in cm.
    * @param y y position in cm.
    * @param w width in cm.
    * @param h height in cm.
    * @param a start angle in degree.
    * @param alen length in degree.
    * @param relSize relative line width.
    * @param ang rotation angle in degree.
    * @param active true for visible item.
    */
  void newArcItem(const QString& color, double x, double y, double w,
                  double h, double a, double alen, double relSize, double ang,
                  int active);
  /**
   * Appends new rectangle item.
   * @param color color for rectangle.
   * @param x x position of left side in cm.
   * @param y y position of bottom side in cm.
   * @param w width in cm.
   * @param h height in cm.
   * @param relSize relative line width.
   * @param fillStyle > 0 for pattern filled rectangle.
   * @param active true for visible item.
   */
  void newRectangleItem(const QString& color, double x, double y,
                        double w, double h, double relSize, int fillStyle,
                        int active);
  /**
   * Appends new ellipse item.
   * @param color color for ellipse.
   * @param x x position of center in cm.
   * @param y y position of center in cm.
   * @param w width in cm.
   * @param h height in cm.
   * @param relSize relative line width.
   * @param fillStyle > 0 for pattern filled ellipse.
   * @param active true for visible item.
   */
  void newEllipseItem(const QString& color, double x, double y,
                      double w, double h, double relSize, int fillStyle,
                      int active);
  /**
   * Appends new scale bar item.
   * @param str text.
   * @param color color for scale bar.
   * @param x x position of bar in cm.
   * @param y y position of bar in cm.
   * @param len width in cm.
   * @param relSize relative line width.
   * @param orientation 0 for horizontal, 1 for vertical.
   * @param active true for visible item.
   */
  void newScaleBarItem(const QString& str, const QString& color,
                       double x, double y, double len, double relSize,
                       int orientation, int active);
  /**
   * Scales automatically all child items of a frame item.
   * @param iItem item index.
   * @param autoNorm true for automatic normalization.
   * @return true when when succeeded.
   */
  int autoScaleFrame(int iItem, int autoNorm);
  /**
   * Loads function parameter file.
   * @param url URL of the file.
   * @param iItem item index.
   * @param yFun true for y function parameter, false for x function parameter
   *  (only for parametric functions).
   * @return true when when succeeded.
   */
  int loadPar(const QString& url, int iItem, int yFun);
  /**
   * Sets function parameter value.
   * @param iItem item index.
   * @param iPar parameter index.
   * @param value parameter value.
   * @param yFun true for y function parameter, false for x function parameter
   *  (only for parametric functions).
   * @return true when when succeeded.
   */
  int setPar(int iItem, int iPar, double value, int yFun);
  /**
   * Sets array row range.
   * @param iItem item index.
   * @param iStart start row index.
   * @param n number of data points.
   * @return true when when succeeded.
   */
  int setArrayRange(int iItem, int iStart, int n);
  /**
   * Adds array and function items to item lists for (multidimensional)
   * parameter fitting.
   * @param erase erases lists first when true.
   * @param iArr array item index.
   * @param iFun function item index.
   * @param errCol true for data weighing by error column, false for weighing
   * by error model function.
   * @param errModPath absolute path of the library for the error model
   * function.
   * @param errModName error model function name.
   * @return true when when succeeded.
   */
  int addFitItems(int erase, int iArr, int iFun, int errCol,
                  const QString& errModPath, const QString& errModName);
  /**
   * Performs parameter fit.
   * @param nonLin true for nonlinear parameter fits.
   * @param savePar true to save resulting parameter files.
   * @param follow true to follow parameter sets in fit series.
   */
  int fit(int nonLin, int savePar, int follow);
  /**
   * Performs smoothing spline fit.
   * @param iArr array item index.
   * @param iSpl spline item index.
   * @param deg degree of spline.
   * @param smf smoothing factor.
   * @param xmin lower boundary of approximation interval.
   * @param xmax upper boundary of approximation interval.
   * @param errCol true for data weighing by error column, false for weighing
   * by error model function.
   * @param errModPath absolute path of the library for the error model
   * function.
   * @param errModName error model function name.
   * @return true when when succeeded.
   */
  int splineFit(int iArr, int iSpl, int deg, double smf, double xmin,
                double xmax, int errCol, const QString& errModPath,
                const QString& errModName);
  /**
   * Returns chi-square of last fit.
   * @return chi-square.
   */
  double chiSquare();
  /**
   * Saves parameters and their errors resulting from a fit to a file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  int saveFitPar(const QString& url);
  /**
   * Exports function value table to file.
   * @param url URL of the file.
   * @param iItem item index.
   * @return true when when succeeded.
   */
  int exportValues(const QString& url, int iItem);

protected:
  /** Initializes menu and toolbar. */
  void initActions();
  /** Calculates precision for statusbar. */
  void calcStatusBar();
  /** Inserts statusbar items. */
  void initStatusBar();
  /** Creates and initialize new statusbar. */
  void newStatusBar();
  /**
   * initKeyAccel creates the keyboard accelerator items for the available slots and changes the menu accelerators.
   * @see KAccel
   */
  void initKeyAccel();
  /**
   * Initializes the document object of the main window that is connected to the view in initView().
   * @see initView();
   */
  void initDocument();
  /**
   * Creates the centerwidget of the KMainWindow instance and sets it as the view.
   */
  void initView();
  /**
   * Saves the options.
   * @param saveAll save everything when true.
   */
  void saveOptions(bool saveAll);
  /** Reads the options on init. */
  void readOptions();
  /**
   * Opens an existing URL or local file for reading.
   * @param type one of the following: Unknown, Data, Plot.
   */
  void openRead(int type);
  /**
   * Saves plot file using current or new name.
   * @param current true for save under current name.
   * @return true when file is saved.
   */
  bool save(bool current);
  /**
   * Saves plot file.
   * @param url URL of the file.
   * @return true when file is saved.
   */
  bool saveAs(const KURL &url);
  /**
   * Draws items for PostScript file.
   * @param g pointer to PSGraph instance.
   */
  void drawPS(PSGraph* g);
  /**
   * Generates PostScript file.
   * @param url URL of the file.
   * @param orientation != 0 for landscape orientation.
   * @return true when file is saved.
   */
  bool savePSURL(const KURL& url, bool orientation);
  /**
   * Warns at unsaved changes.
   * @return true when ready to close file.
   */
  virtual bool queryClose();
  /** Reimplemented for message box on last window exit. */
  virtual bool queryExit();
  /**
   * Saves the window properties for each open window during session end
   * to the session config file, including saving the currently
   * opened file by a temporary filename provided by KApplication.
   * @see KMainWindow#saveProperties
   */
  virtual void saveProperties(KConfig*);
  /**
   * Reads the session config file and restores the application's state
   * including the last opened files and documents by reading the
   * temporary files saved by saveProperties().
   * @see KMainWindow#readProperties
   */
  virtual void readProperties(KConfig*);
  KAction* fileSave;
  KAction* fileSaveAs;
  KAction* fileClose;
  KAction* filePrint;
  KAction* filePlot;
  KAction* filePSP;
  KAction* filePSL;
  KAction* filePreviewP;
  KAction* filePreviewL;
  KAction* editUndo;
  KAction* editRedo;
  KAction* viewReload;
  KToggleAction* showToolbar;
  KToggleAction* showStatusbar;
  KToggleAction* autoplot;
  KToggleAction* addData;
  KToggleAction* showSource;
  KToggleAction* boundingBox;
  KToggleAction* printPS;
  KToggleAction* saveAbsPath;
  KToggleAction* warning;
  KToggleAction* saveAtEnd;
  KRecentFilesAction* recentFiles;
  KplView* view;
  KplDoc* doc;
  ItemDlg* itemDlg;
  KConfig* config;
  KAccel* key_accel;
  QTimer* timer;
  KToolBar::BarPosition toolBarPos;
  QList<ArrayItem> arr;
  QList<FunItem> fun;
  QStringList previews;
  enum StatusId{StatusMsg = 1, StatusAdd, StatusPos, StatusVal};
  bool plot;
  int tRefresh, xPrec, yPrec;
  double fz;

  friend class KplView;

public slots:
  /**
   * Opens a file.
   * @param url URL of the file.
   */
  void openDocumentURL(const KURL& url);
  /**
   * Opens files from URL list.
   * @param list pointer to list with URLs.
   */
  void slotOpenList(KURL::List* list);

protected slots:
  /** Creates a new plot in the current view. */
  void slotFileNew();
  /** Opens an existing plot file. */
  void slotFileOpenPlot();
  /** Opens an existing data file. */
  void slotFileOpenData();
  /** Saves the current plot file. */
  void slotFileSave();
  /** Saves the current plot as a file with different name. */
  void slotFileSaveAs();
  /** Closes the current file. */
  void slotFileClose();
  /** Plots the current file. */
  void slotFilePlot();
  /** Plots the current file conditionally. */
  void slotPlotCond();
  /** Generates PostScript file (portrait orientation). */
  void slotFilePSP();
  /** Generates PostScript file (landscape orientation). */
  void slotFilePSL();
  /**
   * Generates PostScript file.
   * @param orientation true for landscape orientation.
   */
  void slotFilePS(bool orientation);
  /** Displays PostScript preview (portrait orientation). */
  void slotFilePreviewP();
  /** Displays PostScript preview (landscape orientation). */
  void slotFilePreviewL();
  /**
   * Displays PostScript preview.
   * @param orientation true for landscape orientation.
   */
  void slotFilePreview(bool orientation);
  /** Prints the current file. */
  void slotFilePrint();
  /** Opens a new kpl window. */
  void slotFileNewWindow();
  /** Closes the current window. */
  void slotFileCloseWindow();
  /** Exits the application. */
  void slotFileQuit();
  /** Restores previous item from backup. */
  void slotEditUndo();
  /** Restores next item from backup. */
  void slotEditRedo();
  /** Edits plot items. */
  void slotEditItems();
  /** Increases magnification. */
  void slotViewZoomIn();
  /** Decreases magnification. */
  void slotViewZoomOut();
  /** Sets magnification. */
  void slotViewZoom();
  /** Watches file. */
  void slotWatchFile();
  /** Reloads file. */
  void slotViewReload();
  /** Reloads file periodically. */
  void slotViewAutoReload();
  /** Toggles the visibility of the toolbar. */
  void slotOptionsToolBar();
  /** Toggles the visibility of the statusbar. */
  void slotOptionsStatusBar();
  /** Toggles the autoplot option. */
  void slotOptionsAutoPlot();
  /** Toggles the add data option. */
  void slotOptionsAddData();
  /** Toggles the show function source option. */
  void slotOptionsShowSource();
  /** Displays the key bindings dialog. */
  void slotOptionsKeys();
  /** Displays the toolbars configure dialog. */
  void slotEditToolbars();
  /** Displays the basic settings dialog. */
  void slotOptionsBasic();
  /** Displays the autoplot settings dialog. */
  void slotOptionsAuto();
  /** Displays the autofit settings dialog. */
  void slotOptionsAutoFit();
  /** Saves options. */
  void slotOptionsSave();
  /**
   * Changes the status message.
   * @param text pointer to new message text.
   */
  void slotStatusMsg(const QString& text);
  /**
   * Changes the status message of the whole statusbar temporarily.
   * @param text pointer to temporary message text.
   */
  void slotStatusHelpMsg(const QString& text);
  /** Updates titlebar, menus, plot, and item list.
   * @param updPlot true when plot should be uddated.
   * @param updList true when item list should be uddated.
   */
  void slotUpdate(bool updPlot, bool updList);
  /** Updates titlebar and menus. */
  void updWin();
  /**
   * Updates mouse position on statusbar.
   * @param pos position of the mouse cursor.
   * @param showValues true to show Values in data units.
   */
  void updStatus(const QPoint& pos, bool showValues);
  /** Clears mouse position information on statusbar. */
  void clearStatus();
  /** Shrinks window. */
  void shrink();
  /** Shows main popup menu. */
  void showPopupMenu();
};

#endif // KPL_H
