/****************************************************************/
/*                                                              */
/*      fkt.c                                                   */
/*                                                              */
/*      This file is a part of kpl - a program for graphical    */
/*      presentation of data sets and functions.                */
/*                                                              */
/*      Copyright (C) 2004 by Werner Stille                     */
/*      <stille@uni-freiburg.de>                                */
/*                                                              */
/*      Released under the GPL; see file LICENSE for details.   */
/*                                                              */
/****************************************************************/

/* 2D example functions for kpl */

#include <math.h>

/****************************************************************/
/*                                                              */
/*    polynom(x, p)                                             */
/*      calculates polynomial                                   */
/*                                                              */
/*    Returns: p[0] + p[1]*x * p[2]*x*x + p[3]*x*x*x + ...      */
/*                                                              */
/****************************************************************/

double polynom(double x, const double* p)
{
  int i;
  double f = 0.0;

  if (x)
    for (i = 19; i >= 0; i--) {
      if (f)
        f *= x;
      if (p[i])
        f += p[i];
    }
  else
    f = p[0];
  return f;
}

/****************************************************************/
/*                                                              */
/*    foursynth(x, p)                                           */
/*      calculates Fourier synthesis                            */
/*                                                              */
/*    Returns: p[0] + p[2]*cos(x-p[1]) + p[3]*sin(x-p[1])       */
/*               + p[4]*cos(2*(x-p[1])) + p[5]*sin(2*(x-p[1]))  */
/*               + p[6]*cos(3*(x-p[1])) + p[7]*sin(3*(x-p[1]))  */
/*               + p[8]*cos(4*(x-p[1])) + ...                   */
/*                                                              */
/****************************************************************/

double foursynth(double x, const double* p)
{
  int i, j;
  double f, a;

  x -= p[1];
  f = p[0];
  for (i = 2; i < 19; i += 2) {
    j = i + 1;
    a = (j >> 1) * x;
    if (p[i])
      f += p[i] * cos(a);
    if (p[j])
      f += p[j] * sin(a);
  }
  return f;
}

/****************************************************************/
/*                                                              */
/*    gaussian(x, p)                                            */
/*      calculates sum of Gaussians                             */
/*                                                              */
/*    Returns: p[0] + p[1]*exp(-((x-p[2])/p[3])^2)              */
/*               + p[4]*exp(-((x-p[5])/p[6])^2)                 */
/*               + p[7]*exp(-((x-p[8])/p[9])^2) + ...           */
/*                                                              */
/****************************************************************/

double gaussian(double x, const double* p)
{
  int i;
  double f, t;

  f = p[0];
  for (i = 1; i < 18; i += 3) {
    if (p[i] && p[i + 2]) {
      t = (x - p[i + 1]) / p[i + 2];
      if (fabs(t) < 37.0)
        f += p[i] * exp(-t * t);
    }
  }
  return f;
}

/****************************************************************/
/*                                                              */
/*    exponential(x, p)                                         */
/*      calculates sum of exponentials                          */
/*                                                              */
/*    Returns: p[0] + p[1]*exp(p[2]*x) + p[3]*exp(p[4]*x)       */
/*                + p[5]*exp(p[6]*x) + ...                      */
/*                                                              */
/****************************************************************/

double exponential(double x, const double* p)
{
  int i;
  double f;

  f = p[0];
  for (i = 1; i < 18; i += 2)
    if (p[i])
      f += p[i] * exp(p[i + 1] * x);
  return f;
}

/****************************************************************/
/*                                                              */
/*    powers(x, p)                                              */
/*      calculates sum of powers of x                           */
/*                                                              */
/*    Returns: p[0]*x^p[1] + p[2]*x^p[3] + p[4]*x^p[5] + ...    */
/*                                                              */
/****************************************************************/

double powers(double x, const double* p)
{
  int i;
  double f = 0.0;

  for (i = 0; i < 19; i += 2)
    if (p[i])
      f += p[i] * pow(x, p[i + 1]);
  return f;
}

/****************************************************************/
/*                                                              */
/*    math(x, p)                                                */
/*      functions of math.h                                     */
/*                                                              */
/*    Returns: p[0] + p[1] * f(p[3] * (x - p[2]))               */
/*                                                              */
/*    p[4]:    selects function f(x) (see case...)              */
/*                                                              */
/****************************************************************/

double math(double x, const double* p)
{
  double a, f;

  a = p[3] * (x - p[2]);
  switch ((int) (p[4] + 0.5)) {
    case 1:
      f = sqrt(a);
      break;
    case 2:
      f = exp(a);
      break;
    case 3:
      f = log(a);
      break;
    case 4:
      f = log10(a);
      break;
    case 5:
      f = sin(a);
      break;
    case 6:
      f = cos(a);
      break;
    case 7:
      f = tan(a);
      break;
    case 8:
      f = asin(a);
      break;
    case 9:
      f = acos(a);
      break;
    case 10:
      f = atan(a);
      break;
    case 11:
      f = sinh(a);
      break;
    case 12:
      f = cosh(a);
      break;
    case 13:
      f = tanh(a);
      break;
    default:
      f = a;
  }
  return p[0] + p[1] * f;
}

/* 3D example functions for kpl */

/****************************************************************/
/*                                                              */
/*    f3d1(x, y, p)                                             */
/*      calculates sin(r) / r                                   */
/*                                                              */
/****************************************************************/

double f3d1(double x, double z, const double* p)
{
  double r = sqrt(x * x + z * z);
  return (fabs(r) > 1.0e-8) ? (sin(r) / r) : 1.0;
}

/****************************************************************/
/*                                                              */
/*    f3d2(x, y, p)                                             */
/*      calculates x^2 - z^2                                    */
/*                                                              */
/****************************************************************/

double f3d2(double x, double z, const double* p)
{
  return x * x - z * z;
}
