//
// This file is part of the Marble Desktop Globe.
//
// This program is free software licensed under the GNU LGPL. You can
// find a copy of this license in LICENSE.txt in the top directory of
// the source code.
//
// Copyright 2007      Murad Tagirov <tmurad@gmail.com>
// Copyright 2007      Inge Wallin   <ingwa@kde.org>
//


#ifndef MARBLE_GEODATALABELSTYLE_H
#define MARBLE_GEODATALABELSTYLE_H


#include <QtGui/QFont>

#include "GeoDataColorStyle.h"

#include "geodata_export.h"

namespace Marble
{

class GeoDataLabelStylePrivate;

/**
 * @short specifies how the <b>name</b> of a GeoDataFeature is drawn
 *
 * A GeoDataLabelStyle specifies how the <b>name</b> of a
 * GeoDataFeature is drawn in the viewer. A custom color, color mode
 * (both inherited from GeoDataColorStyle, and scale for the label
 * (name) can be specified.
 */
class GEODATA_EXPORT GeoDataLabelStyle : public GeoDataColorStyle
{
  public:
    enum Alignment{Corner, Center /*, Left, Right, Below  */};

    /// Construct a new GeoDataLabelStyle
    GeoDataLabelStyle();
    GeoDataLabelStyle( const GeoDataLabelStyle& other );
    /**
     * @brief Construct a new GeoDataLabelStyle
     * @param  font   the font to use when showing the name
     * @param  color  the color to use when showing the name @see GeoDataColorStyle
     */
    GeoDataLabelStyle( const QFont &font, const QColor &color );

    ~GeoDataLabelStyle();
    
    /**
    * @brief assignment operator
    */
    GeoDataLabelStyle& operator=( const GeoDataLabelStyle& other );

    /// Provides type information for downcasting a GeoData
    virtual QString nodeType() const;

    /**
     * @brief Set the scale of the label
     * @param  scale  the new scale
     */
    void setScale( const float &scale );
    /**
     * @brief Return the current scale of the label
     * @return  the current scale
     */
    float scale() const;
    /**
     * @brief Set the alignment of the label
     * @param  int the alignment
     */
    void setAlignment ( GeoDataLabelStyle::Alignment alignment );
    /**
     * @brief Return the alignment of the label
     * @return  the current alignment
     */
    GeoDataLabelStyle::Alignment alignment() const;
    /**
     * @brief Set the font of the label
     * @param  font  the new font
     */
    void setFont( const QFont &font );
    /**
     * @brief Return the current font of the label
     * @return  the current font
     */
    QFont font() const;

    /**
     * @brief  Serialize the style to a stream.
     * @param  stream  the stream
     */
    virtual void pack( QDataStream& stream ) const;
    /**
     * @brief  Unserialize the style from a stream
     * @param  stream  the stream
     */
    virtual void unpack( QDataStream& stream );

  private:
    GeoDataLabelStylePrivate * const d;
};

}

#endif
