/* 
 *  arch-tag: Implementation of Song History List
 *
 *  Copyright (C) 2003 Jeffrey Yasskin <jyasskin@mail.utexas.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  
 */ 

#include "rb-history.h"

#include "rhythmdb.h"

struct RBHistoryPrivate
{
	GList *head;
	GList *tail;
	GList *current;
	guint size;

	RhythmDB *db;

	GHashTable *entry_to_link;
};

#define MAX_HISTORY_SIZE 50

static void rb_history_class_init (RBHistoryClass *klass);
static void rb_history_init (RBHistory *shell_player);
static void rb_history_finalize (GObject *object);

static void rb_history_set_property (GObject *object,
				     guint prop_id,
				     const GValue *value,
				     GParamSpec *pspec);

static void rb_history_limit_size (RBHistory *hist, gboolean cut_from_beginning, guint max_size);
static void rb_history_song_deleted_cb (RhythmDB *view, RhythmDBEntry *entry, RBHistory *hist);

/**
 * If the entry is in the history, removes all instances of it. Unrefs the entry and decrements the list size.
 */
static void rb_history_remove_entry (RBHistory *hist, RhythmDBEntry *entry);
/**
 * Removes a link from the history and updates all pointers. Doesn't unref the entry, but does change the size of the list.
 */
static void rb_history_delete_link (RBHistory *hist, GList *to_delete);

enum
{
	PROP_0,
	PROP_DB,
};

static GObjectClass *parent_class = NULL;

GType
rb_history_get_type (void)
{
	static GType rb_history_type = 0;

	if (rb_history_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (RBHistoryClass),
			NULL,
			NULL,
			(GClassInitFunc) rb_history_class_init,
			NULL,
			NULL,
			sizeof (RBHistory),
			0,
			(GInstanceInitFunc) rb_history_init
		};

		rb_history_type = g_type_register_static (G_TYPE_OBJECT,
							  "RBHistory",
							  &our_info, 0);
	}

	return rb_history_type;
}

static void
rb_history_class_init (RBHistoryClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = rb_history_finalize;

	object_class->set_property = rb_history_set_property;

	g_object_class_install_property (object_class,
					 PROP_DB,
					 g_param_spec_object ("db",
							      "RhythmDB",
							      "RhythmDB object",
							      RHYTHMDB_TYPE,
							      G_PARAM_WRITABLE));

}

static void
rb_history_init (RBHistory *hist)
{
	hist->priv = g_new0 (RBHistoryPrivate, 1);

	hist->priv->entry_to_link = g_hash_table_new (g_direct_hash,
						      g_direct_equal);
}

RBHistory *
rb_history_new (RhythmDB *db)
{
	RBHistory *history;

	history = g_object_new (RB_TYPE_HISTORY, 
				"db", db,
				NULL);

	g_return_val_if_fail (history->priv != NULL, NULL);

	return history;
}

static void 
rb_history_finalize (GObject *object)
{
	RBHistory *hist;
	g_return_if_fail (object != NULL);
	g_return_if_fail (RB_IS_HISTORY (object));

	hist = RB_HISTORY (object);

	g_list_free (hist->priv->head);
	g_free (hist->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rb_history_set_property (GObject *object,
			      guint prop_id,
			      const GValue *value,
			      GParamSpec *pspec)
{
	RBHistory *hist = RB_HISTORY (object);

	switch (prop_id)
	{
	case PROP_DB: {
		RhythmDB *db = g_value_get_object (value);
		if (db != hist->priv->db) {
			if (hist->priv->db) {
				g_signal_handlers_disconnect_by_func (G_OBJECT (hist->priv->db),
						G_CALLBACK (rb_history_song_deleted_cb),
						hist);
			}
			hist->priv->db = db;
			if (hist->priv->db) {
				g_signal_connect (G_OBJECT (hist->priv->db),
						"entry_deleted",
						G_CALLBACK (rb_history_song_deleted_cb),
						hist);
			}
		}
		} break;
	default:
		 G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		 break;
	}
}

RhythmDBEntry * 
rb_history_forward (RBHistory *hist)
{
	if (hist->priv->current == hist->priv->tail)
		return NULL;
	
	hist->priv->current = g_list_next (hist->priv->current);
	return hist->priv->current->data;
}

RhythmDBEntry *
rb_history_back (RBHistory *hist)
{
	if (hist->priv->current == hist->priv->head)
		return NULL;
	
	hist->priv->current = g_list_previous (hist->priv->current);
	return hist->priv->current->data;
}

void
rb_history_set_playing (RBHistory *hist, RhythmDBEntry *entry)
{
	if (hist->priv->head == NULL) {
		rhythmdb_entry_ref (hist->priv->db, entry);
		hist->priv->size++;
		hist->priv->head = hist->priv->tail = hist->priv->current = g_list_append (NULL, entry);
		g_hash_table_insert (hist->priv->entry_to_link, entry, hist->priv->current);
	} else if (hist->priv->current->data == entry) {
	} else {
		rhythmdb_entry_ref (hist->priv->db, entry);
		rb_history_remove_entry (hist, entry);
		hist->priv->size++;
		g_list_insert_before (hist->priv->head, g_list_next (hist->priv->current), entry);
		hist->priv->current = g_list_next (hist->priv->current);
		g_hash_table_insert (hist->priv->entry_to_link, entry, hist->priv->current);
		/* This loop could be more efficient if there were a g_list_truncate */
		while (g_list_next (hist->priv->current))
			g_list_delete_link (hist->priv->head, g_list_next (hist->priv->current));
		hist->priv->tail = hist->priv->current;
	}
	rb_history_limit_size (hist, TRUE, MAX_HISTORY_SIZE);
}

void
rb_history_enqueue (RBHistory *hist, RhythmDBEntry *entry)
{
	rhythmdb_entry_ref (hist->priv->db, entry);
	rb_history_remove_entry (hist, entry);
	hist->priv->size++;
	if (hist->priv->head == NULL)
		hist->priv->head = hist->priv->tail = hist->priv->current = g_list_append (NULL, entry);
	else {
		g_list_append (hist->priv->tail, entry);
		hist->priv->tail = g_list_next (hist->priv->tail);
	}
	g_hash_table_insert (hist->priv->entry_to_link, entry, hist->priv->tail);
	rb_history_limit_size (hist, TRUE, MAX_HISTORY_SIZE);
}

void
rb_history_prepend (RBHistory *hist, RhythmDBEntry *entry)
{
	rhythmdb_entry_ref (hist->priv->db, entry);
	rb_history_remove_entry (hist, entry);
	hist->priv->size++;
	if (hist->priv->head == NULL)
		hist->priv->head = hist->priv->tail = hist->priv->current = g_list_prepend (NULL, entry);
	else {
		hist->priv->head = g_list_insert_before (hist->priv->head, hist->priv->current, entry);
		hist->priv->current = g_list_previous (hist->priv->current);
	}
	g_hash_table_insert (hist->priv->entry_to_link, entry, hist->priv->current);
	rb_history_limit_size (hist, FALSE, MAX_HISTORY_SIZE);
}

/**
 * Cuts nodes off of the history from the desired end until it is smaller than max_size. 
 * Never cuts off the current node.
 */
static void
rb_history_limit_size (RBHistory *hist, gboolean cut_from_beginning, guint max_size)
{
	while (hist->priv->size > max_size) {
		if (cut_from_beginning 
				|| hist->priv->current == hist->priv->tail) {
			rhythmdb_entry_unref (hist->priv->db, hist->priv->head->data);
			g_hash_table_remove (hist->priv->entry_to_link, hist->priv->head->data);
			hist->priv->head = g_list_delete_link (hist->priv->head, hist->priv->head);
			hist->priv->size--;
		} else {
			rhythmdb_entry_unref (hist->priv->db, hist->priv->tail->data);
			g_hash_table_remove (hist->priv->entry_to_link, hist->priv->tail->data);
			hist->priv->tail = g_list_delete_link (g_list_previous (hist->priv->tail), hist->priv->tail);
			hist->priv->size--;
		}
	}
}

static void
rb_history_song_deleted_cb (RhythmDB *db, RhythmDBEntry *entry,
			    RBHistory *hist)
{
	rb_history_remove_entry (hist, entry);
}

static void rb_history_remove_entry (RBHistory *hist, RhythmDBEntry *entry)
{
	GList *to_delete = g_hash_table_lookup (hist->priv->entry_to_link, entry);
	if (to_delete) {
		g_hash_table_remove (hist->priv->entry_to_link, entry);
		rhythmdb_entry_unref (hist->priv->db, entry);

		rb_history_delete_link (hist, to_delete);
	}
}

static void rb_history_delete_link (RBHistory *hist, GList *to_delete)
{
	/* There are 3 pointers into the GList, and we need to
	 * make sure they still point to the right places after
	 * deleting *to_delete.
	 * It's pretty messy. If you see a better structure for
	 * the if structure, please fix it.
	 */
	if (to_delete == hist->priv->head) {
		if (to_delete == hist->priv->tail) {
			hist->priv->head = 
				hist->priv->current =
				hist->priv->tail = g_list_delete_link (hist->priv->head, to_delete);
		} else if (to_delete == hist->priv->current) {
			hist->priv->head = 
				hist->priv->current = g_list_delete_link (hist->priv->head, to_delete);
		} else {
			hist->priv->head = g_list_delete_link (hist->priv->head, to_delete);
		}
	} else if (to_delete == hist->priv->current) {
		if (to_delete == hist->priv->tail) {
			hist->priv->current = hist->priv->tail = g_list_previous (hist->priv->tail);
			g_list_delete_link (hist->priv->head, to_delete);
		} else {
			hist->priv->current = g_list_next (hist->priv->current);
			g_list_delete_link (hist->priv->head, to_delete);
		}
	} else if (to_delete == hist->priv->tail) {
		hist->priv->tail = g_list_previous (hist->priv->tail);
		g_list_delete_link (hist->priv->head, to_delete);
	} else {
		g_list_delete_link (hist->priv->head, to_delete);
	}

	hist->priv->size--;
}
