# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: pybibtex.py,v 1.7 1998/11/09 13:01:48 gobry Exp $

# Module d'extension pour les fichiers BibTeX

import pyb
import bibtex
import tempfile
import os
import pwd
import time
import traceback
import re
import string
import basics
import pybopen

from types import *


# --------------------------------------------------
# Register a method to open BibTeX files
# --------------------------------------------------

def __bibtex_opener (entity):
	
	method, address, file, p, q, f = entity
	base = None
	
	if method == 'file' and file [-4:] == '.bib':
		try:
			base = DBase (file)
		except:
			traceback.print_exc ()
		
	return base

pybopen.register_method (__bibtex_opener)


# --------------------------------------------------
# La classe Entry herite de pyb
# --------------------------------------------------

class Entry (pyb.Entry):
	def __getitem__ (self, key):
		item = self.__dict [key]
		if item [0] == '"' or item [0] == '{':
			item = item [1:-1]
		return item

	def __setitem__ (self, key, value):
		value = string.strip (value)
		
		if string.find (value, '"'):
			value = '{' + value + '}'
		else:
			value = '"' + value + '"'

		self.__dict [key] = value


	def quoted (self, key):
		return self.__dict [key]
	
# --------------------------------------------------
# Une base de references bibliographiques,
# comme un dictionnaire avec des extensions...
# --------------------------------------------------

class DBase (pyb.DBase):

	def __parsefile__ (self):
		self.__entry = {}
		self.__parser = 0

		# Stockage temporaire avant un "update"
		self.__to_delete = {}
		self.__to_modify = {}
		
		# Ouvrir le fichier associe
		self.__parser = bibtex.open (self.__name)
		
		finished = 0

		# Creer la base de cles
		while not finished:
			entry = bibtex.next (self.__parser)
			if entry == None:
				finished = 1
			else:
				# Sauter les definitions de strings
				if entry.has_key (' name '):
					name = entry [' name ']
					if self.has_key (name):
						raise KeyError, 'key `' + name + '\' defined more than once'
					else:
						self.__entry [name] = entry [' offset ']
		
	def __init__ (self, basename):
		"Initialisation"
		self.__name = basename

		self.__parsefile__ ()

	def __del__ (self):
		"Destruction de la base"
		self.update ()

	def keys (self):
		"Retourne la liste des cles"
		return self.__entry.keys ()

	def __len__ (self):
		""
		return len (self.__entry)

	def __repr__ (self):
		""
		return "BibTeX database `%s' (%d entries)" % (self.__name, len (self))
	
	def has_key (self, key):
		"Presence d'une cle"
		return self.__entry.has_key (key)

	def __getitem__ (self, name):
		"Retourne une entree en fonction de la cle"

		ref = self.__entry [name]
		type = __builtins__['type']
		
		# Si on recupere directement une classe...
		if type (ref) is InstanceType:
			return ref
		else:
			# Sinon il faut aller lire les donnees dans le fichier
			bibtex.offset (self.__parser, ref)
			entry = bibtex.next (self.__parser)
			
			del entry[' offset ']
			type = entry [' type ']
			name = entry [' name ']
			del entry[' type ']
			del entry[' name ']
			del entry[' body ']
			del entry[' length ']

			for e in entry.keys ():
				entry [e] = string.strip (entry [e])
				
			return Entry (name, type, entry);
	
	def __setitem__ (self, key, value):
		"Modifier ou ajouter une entree"

		if value.name () <> None:
			key = value.name ()
		
		self.__entry [key] = value

		# Sauver la version modifiee aussi
		self.__to_modify [key] = value
		
	def __delitem__ (self, key):
		"Detruire une entree"
		if self.has_key (key):
			# Noter la destruction
			self.__to_delete [key] = 1
			# Enlever du dictionnaire
			del self.__entry [key]
		else:
			raise KeyError, "no such entry"


	# ==================================================

	def foreach (self, function, args = None):
		"Parcours de toutes les entrees"
		bibtex.first (self.__parser)
		finished = 0

		# Parcourir les entrees du fichier
		while not finished:
			entry = bibtex.next (self.__parser)
			if entry == None:
				finished = 1
			else:
				if entry.has_key (' name '):
					name = entry [' name ']

					# Version modifiee ??
					if self.__to_modify.has_key (name):
						continue
					else:
						del entry[' offset ']
						
						type = entry [' type ']

						del entry[' type ']
						del entry[' name ']
						del entry[' body ']
						del entry[' length ']
						
						for e in entry.keys ():
							entry [e] = string.strip (entry [e])
		
						function (Entry (name, type, entry), args)

		# Les nouvelles ou les modifiees
		for entry in self.__to_modify.keys ():
			function (self.__to_modify [entry], args)
			

	def update (self):
		"Flusher les entrees detruites ou modifiees"

		# Y a-t-il eu des modifs ?
		if len (self.__to_modify) == 0 and len (self.__to_delete) == 0:
			return
		
		# Faire un backup
		os.rename (self.__name, self.__name + '.bak')

		try:
			
			# Et recreer le fichier
			file = open (self.__name, 'w')
		
			bibtex.first (self.__parser)
			finished = 0
			offset = 0
			
			while not finished:
				entry = bibtex.next (self.__parser)
				if entry == None:
					finished = 1
				else:
					# Garder les 'string'
					if not entry.has_key (' name '):
						file.write (entry [' body '])
						continue
					
					name = entry [' name ']
					
					# L'entree est elle modifiee ou supprimee ??
					if self.__to_modify.has_key (name) or self.__to_delete.has_key (name):
						continue
					
					# Sinon on sauve l'ancienne definition
					file.write (entry [' body '])
					
			# Ajouter ceux qui restent dans __to_modify, parce qu'ils
			# sont nouveaux.
			if len (self.__to_modify) > 0:
				
				for entry in self.__to_modify.keys ():
					file.write (self.__to_modify [entry].__str__ ())
					file.write ('\n\n')
					
			file.close ()
			self.__parsefile__ ()
			
		except:
			traceback.print_exc ()
			# Trying to reverse operation
			file.close ()
			os.rename (self.__name + '.bak' , self.__name)
			raise IOError, "can't save file " + self.__name


	def where (self, key):
		"Chercher une entree selon des criteres"
		result = []

		def search_function (entry, param):
			"the real search function "
			key = param [0]
			result = param [1]

			if key.match (entry):
				result.append (entry.name ())
						
		self.foreach (search_function, (key, result))

		# Creer la base de references
		refs = pyb.Reference ()

		if len (result) > 0:
			refs.add (self, result)

		return refs
