/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include <gnome.h>

#include <pan/acache.h>
#include <pan/article.h>
#include <pan/base-prefs.h>
#include <pan/debug.h>
#include <pan/decode.h>
#include <pan/pan-glib-extensions.h>

#include <pan/nntp.h>
#include <pan/prefs.h> /* for auto_part_selection */
#include <pan/queue.h>
#include <pan/task-decode.h>
#include <pan/util.h> /* for pan_lock/unlock */

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint task_decode_run (Task* task);

static char* task_decode_describe (const StatusItem* item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static char*
task_decode_describe (const StatusItem* item)
{
	TaskDecode *decode = TASK_DECODE(item);

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);
	g_return_val_if_fail (decode->readable_name!=NULL, NULL);

        return g_strdup_printf (_("Saving \"%s\""), decode->readable_name);
}

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

static void
task_decode_destructor (PanObject* object)
{
	TaskDecode * task = TASK_DECODE(object);

	/* destruct the TaskDecode parts */
        debug1 (DEBUG_PAN_OBJECT, "task_decode destructor: %p", object);
	g_free (task->readable_name);
	g_free (task->path);
	g_free (task->filename);

	/* deflag */
	if (1) {
		GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=task->articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, l->data);
		articles_remove_flag ((Article**)a->pdata, a->len, STATE_DECODE_QUEUED);
		g_ptr_array_free (a, TRUE);
	}

	/* checkin the article bodies */
	if (1) {
		GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=task->articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, (gpointer)article_get_message_id(ARTICLE(l->data)));
		acache_file_checkin ((const gchar**)a->pdata, a->len);
		if (a->len>1 && task->runval==TASK_SUCCESS)
			acache_expire_messages ((const gchar**)a->pdata, a->len);
		g_ptr_array_free (a, TRUE);
	}

	g_slist_foreach (task->articles, (GFunc)pan_object_unref, NULL);
	g_slist_free (task->articles);
	group_unref_articles (task->group, NULL);

	/* destruct the superclass */
	task_destructor (PAN_OBJECT(object));
}

PanObject*
task_decode_new (Group          * group,
                 const char     * readable_name, /* for description */
                 const char     * path,
                 const char     * filename,
                 const GSList   * articles)
{
	GSList * l;
	GPtrArray * a;
	TaskDecode * task;

	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (readable_name!=NULL, NULL);
	g_return_val_if_fail (articles->data!=NULL, NULL);

	/* ref the group's articles */
	group_ref_articles (group, NULL);
	g_slist_foreach ((GSList*)articles, (GFunc)pan_object_ref, NULL);

	/* checkout the article bodies */
	if (1) {
		const GSList * l;
		GPtrArray * a = g_ptr_array_new ();
		for (l=articles; l!=NULL; l=l->next)
			g_ptr_array_add (a, (gpointer)article_get_message_id(ARTICLE(l->data)));
		acache_file_checkout ((const gchar**)a->pdata, a->len);
		g_ptr_array_free (a, TRUE);
	}

	/* create the task */
       	task = g_new0 (TaskDecode, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_decode_new: %p", task);
	task_constructor (TASK(task),
			  task_decode_destructor,
			  task_decode_describe,
			  task_decode_run,
			  group->server, FALSE, TRUE);

	/* create the task-decode */
	task->runval = TASK_FAIL;
	task->group = group;
	task->readable_name = g_strdup(readable_name);
	task->path = g_strdup (path);
	task->filename = g_strdup (filename);
	task->articles = g_slist_copy ((GSList*)articles);

	/* flag the articles */
	a = g_ptr_array_new ();
	for (l=task->articles; l!=NULL; l=l->next)
		g_ptr_array_add (a, l->data);
	articles_add_flag ((Article**)a->pdata, a->len, STATE_DECODE_QUEUED);
	g_ptr_array_free (a, TRUE);

	/* if no path specified, fall back on Pan group and global defaults */
	if (task->path == NULL)
		task->path = g_strdup (group->download_dir);
	if (task->path == NULL)
		task->path = g_strdup (download_dir);

	/* replace "%g" with the group name, "%s" with the server name */
	replace_gstr (&task->path, pan_substitute(task->path,"%s",group->server->name));
	replace_gstr (&task->path, pan_substitute(task->path,"%g",group->name));

	return PAN_OBJECT(task);
}

/*****
******
*****/

static gint
task_decode_run (Task * task)
{
	gint result;
	GSList * articles;
	Server * server;
	Group * group;
	Article * article;
	decode_data * dd = NULL;

	/* sanity clause */
	g_return_val_if_fail (task!=NULL, TASK_DECODE(task)->runval=TASK_FAIL_HOPELESS);
	g_return_val_if_fail (TASK_DECODE(task)->articles!=NULL, TASK_DECODE(task)->runval=TASK_FAIL_HOPELESS);

	server = task->server;
	group = TASK_DECODE(task)->group;
	articles = TASK_DECODE(task)->articles;

	/* get the articles bodies */
	if (1)
	{
		GSList * l;
		gint index = 0;
		GPtrArray * squarepeg = g_ptr_array_new ();
		for (l=articles; l!=NULL; l=l->next)
			g_ptr_array_add (squarepeg, l->data);
		result = nntp_download_bodies (STATUS_ITEM(task),
		                               TASK_DECODE(task)->group,
		                               task->sock,
		                               &task->hint_abort,
		                               squarepeg,
		                               &index,
		                               FALSE, TRUE);
		g_ptr_array_free (squarepeg, TRUE);
	}
	if (result != TASK_SUCCESS) {
		status_item_emit_error_va (STATUS_ITEM(task),
		                           _("Unable to decode `%s': the article(s) couldn't be downloaded."),
					   article_get_subject(ARTICLE(articles->data)));
		return TASK_DECODE(task)->runval = result;
	}


	/* update "read" attribute */
       	article = ARTICLE(articles->data);
	articles_set_read (&article, 1, TRUE); 

	/* pass it off to article_decode */
	dd = g_new0 (decode_data, 1);
	dd->server = server;
	dd->group = group;
	dd->articles = articles;
	dd->item = STATUS_ITEM(task);
	dd->path = TASK_DECODE(task)->path;
	dd->filename = TASK_DECODE(task)->filename;
	decode_article (dd);
	g_free (dd);


	return TASK_DECODE(task)->runval = TASK_SUCCESS;
}


/*****
******
******  VALIDATE-AND-QUEUE or SELF-DESTRUCT
******
*****/

typedef struct
{
	GtkWidget * clist;
	GtkWidget * window;
	TaskDecode * task;
	int queue_index;
}
ResolvePartsUI;

static void
select_articles_clicked_cb (GnomeDialog   * dialog,
                            int             button,
                            gpointer        user_data)
{
	ResolvePartsUI *ui = (ResolvePartsUI*) user_data;

	if (button==0) /* "ok" */
	{
		GList* sel = GTK_CLIST(ui->clist)->selection;
		const int rows = GTK_CLIST(ui->clist)->rows;
		int row;

		/* remove the non-selected articles */
		for (row=rows-1; row>=0; --row)
		{
			gboolean row_selected = g_list_find (sel,GINT_TO_POINTER(row))!=NULL;

			if (!row_selected)
			{
				GSList* n = g_slist_nth (ui->task->articles, row);
				ui->task->articles = g_slist_remove_link(ui->task->articles,n);
				g_slist_free_1 (n);
			}
		}

		queue_task_insert (TASK(ui->task), ui->queue_index);
	}
	else
	{
		pan_object_unref (PAN_OBJECT(ui->task));
	}

	/* cleanup */
	pan_lock ();
	gtk_widget_destroy (ui->window);
	pan_unlock ();
	g_free (ui);
}

static void
choose_article_parts_auto (TaskDecode  * task,
                           int           queue_index)
{
	GSList* l;
	Article *last_article = NULL;
	
	l = task->articles;
	while (l != NULL)
	{
		Article *article = ARTICLE (l->data);

		/* increment l here so that removing article
		   from task->articles doesn't invalidate it */
		l = l->next;

		/* If this is a reply sitting in a multipart thread,
		 * don't use the reply. */
		if (article->part == 0)
		{
			task->articles = g_slist_remove (task->articles, article);
		}
		else if (last_article && (last_article->part == article->part))
		{
			/* Select the better of the two assuming
			 * that more lines is better */
			if (last_article->linecount > article->linecount)
			{
				task->articles = g_slist_remove (task->articles, article);
			}
			else if (last_article->linecount < article->linecount)
			{
				task->articles = g_slist_remove (task->articles, last_article);
				last_article = article;
			}
			else
			{
				/* Check the date next if lines match,
				 * last posted is preferred */
				if (last_article->date > article->date)
				{
					task->articles = g_slist_remove (task->articles, article);
				}
				else
				{ 
					/* if article is newer, or the same, kill the other one */
					task->articles = g_slist_remove (task->articles, last_article);
					last_article = article;
				}	
			}
		}
		else
		{
			last_article = article;
		}
	}

	queue_task_insert (TASK(task), queue_index);
}

static void
choose_article_parts_manual (TaskDecode  * task,
                             int           queue_index)
{
	GSList* l;
	GtkWidget* w;
	GtkWidget* list;
	GtkWidget* vbox;
	GtkWidget* label;
	char* columns [4];
	int last_part = -1;
	ResolvePartsUI* ui = g_new0 (ResolvePartsUI, 1);

	ui->queue_index = queue_index;
	ui->task = task;

	pan_lock ();

	ui->window = gnome_dialog_new (
		_("Decode Multipart Message"),
		GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL,
		NULL);
	gtk_signal_connect (
		GTK_OBJECT(ui->window), "clicked",
		GTK_SIGNAL_FUNC(select_articles_clicked_cb), ui);

	gtk_window_set_policy (GTK_WINDOW (ui->window), TRUE, TRUE, TRUE);
	gnome_dialog_set_default (GNOME_DIALOG(ui->window), 1);
	vbox = GNOME_DIALOG(ui->window)->vbox;

	label = gtk_label_new (
		_("Pan found multiple copies of some of the parts of this post.\n"
		  "Please select the parts to use for this decode."));
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, GNOME_PAD_SMALL);

	columns[0] = _("Subject");
	columns[1] = _("Lines");
	columns[2] = _("Author");
	columns[3] = _("Date");
	ui->clist = list = gtk_clist_new_with_titles (4, columns);
	gtk_clist_set_selection_mode (GTK_CLIST(list), GTK_SELECTION_MULTIPLE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(list), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(list), 1, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(list), 2, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(list), 3, TRUE);

	for (l=task->articles; l!=NULL; l=l->next)
	{
		char datebuf[32];
		char lines[8];
		Article * article = ARTICLE (l->data);
		const struct tm local_tm = *localtime (&article->date);
		gchar * text[4];
		gchar * author = article_get_author_str (article);
		int row;
		sprintf (lines, "%d", article->linecount);
		strftime (datebuf, sizeof(datebuf), "%Y/%m/%d %H:%M", &local_tm);

	       	text[0] = (gchar*) article_get_subject (article);
		text[1] = lines;
		text[2] = author;
		text[3] = datebuf;

		row = gtk_clist_append (GTK_CLIST(list), text);
		gtk_clist_set_row_data (GTK_CLIST(list), row, article);
		if (last_part != article->part)
		{
			gtk_clist_select_row (GTK_CLIST(list), row, 0);
			last_part = article->part;
		}

		g_free (author);
	}
	w = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(w), ui->clist);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, GNOME_PAD_SMALL);

	gtk_window_set_default_size (GTK_WINDOW (ui->window), 500, 400);
	gtk_widget_show_all (ui->window);

	pan_unlock ();
}

void
task_decode_validate_and_queue_or_self_destruct (TaskDecode  * task,
                                                 int           queue_index)
{
	const int length = g_slist_length(task->articles);

	/**
	 * (1) If we have a number of articles in the decode list, and
	 * (2) the number of parts we expected doesn't match it, and
	 * (3) if we have a clue what the number of parts is supposed to be
	 *     (we may not, if user didn't post with a [1/24]-style header),
	 * then try to weed out the ones we don't want.
	 */
	if ((length>1)
		&& (length!=ARTICLE(task->articles->data)->parts)
		&& ARTICLE(task->articles->data)->parts)
	{
		if (auto_part_selection)
		{
			choose_article_parts_auto (task, queue_index);	
		}
		else
		{
			choose_article_parts_manual (task, queue_index);
		}
	}
	else
	{
		queue_task_insert (TASK(task), queue_index);
	}
}
