/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <pthread.h>

#include <gnome.h>

#include <string.h>

#include <pan/article-thread.h>
#include <pan/debug.h>
#include <pan/pan-glib-extensions.h>
#include <pan/status-item.h>

#include <pan/articlelist.h>
#include <pan/article-toolbar.h>
#include <pan/gui.h>
#include <pan/util.h>

#include <xpm/article_read.xpm>
#include <xpm/article_unread.xpm>
#include <xpm/article_new.xpm>
#include <xpm/disk.xpm>
#include <xpm/favorite.xpm>
#include <xpm/filter.xpm>
#include <xpm/bomb.xpm>
#include <xpm/binary_complete.xpm>
#include <xpm/binary_incomplete.xpm>
#include <xpm/binary.xpm>
#include <xpm/bluecheck.xpm>
#include <xpm/mini_page.xpm>

static GtkWidget * toolbar = NULL;

static GtkWidget * show_unread_button = NULL;
static GtkWidget * show_read_button = NULL;
static GtkWidget * show_new_button = NULL;
static GtkWidget * show_complete_binary_button = NULL;
static GtkWidget * show_incomplete_binary_button = NULL;
static GtkWidget * show_nonbinary_button = NULL;
static GtkWidget * show_watched_button = NULL;
static GtkWidget * show_killfile_button = NULL;
static GtkWidget * show_normal_rank_button = NULL;
static GtkWidget * show_saved_button = NULL;
static GtkWidget * show_queued_button = NULL;
static GtkWidget * show_idle_button = NULL;
static GtkWidget * show_cached_button = NULL;
static GtkWidget * show_non_cached_button = NULL;

/*****
******  SUBJECT FILTER
*****/

static ArticleFilterMode _filter_mode = ARTICLE_FILTER_SUBJECT;
static gchar * _subject_filter_str = NULL;

void
article_toolbar_get_filter (gchar ** str_allocme, ArticleFilterMode * mode)
{
	g_return_if_fail (str_allocme!=NULL);

	*str_allocme = g_strdup (_subject_filter_str);
	*mode = _filter_mode;
}

static gchar*
filter_selected_describe (const StatusItem* item)
{
	return g_strdup (_("Filtering Articles"));
}

static void
filter_selected_thread (void * unused)
{
	Group *group = NULL;
	StatusItem *item = NULL;
	debug_enter ("filter_selected_thread");

	group = articlelist_get_group ();
	if (!group) { /* articlelist is idle */
		debug_exit ("filter_selected_thread");
		return;
	}

	/* create a new status item to get thread messages */
	item = STATUS_ITEM(status_item_new(filter_selected_describe));
	status_item_set_active (item, TRUE);

	/* update the articlelist */
	articlelist_refresh (item);

	/* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));

	debug_exit ("filter_selected_thread");
}

static void
set_filter_mode (GtkWidget * unused, ArticleFilterMode mode)
{
	debug_enter ("set_filter_mode");

	if (mode != _filter_mode)
	{
		pthread_t thread;

		_filter_mode = mode;

		pthread_create (&thread, NULL, (void*)filter_selected_thread, NULL);
		pthread_detach (thread);
	}

	debug_exit ("set_filter_mode");
}

static void
article_filter_cb (GtkWidget* w, gpointer data)
{
	gchar * s;
	gchar * pch;
	debug_enter ("article_filter_cb");

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	s = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	s = g_strstrip (s);
	if (strchr(s,'*'))
		pch = g_strdup (s);
	else if (*s)
		pch = g_strdup_printf("*%s*",s);
	else
		pch = NULL;
	g_free (s);

	/* if the new string differs from the old one, then update. */
	if (!pan_strcmp (_subject_filter_str, pch))
		g_free (pch);
	else {
		pthread_t thread;
		replace_gstr (&_subject_filter_str, pch);
		pthread_create (&thread, NULL, (void*)filter_selected_thread, NULL);
		pthread_detach (thread);
	}

	debug_exit ("article_filter_cb");
}

/*****
******  STATE FILTER
*****/

static gboolean dampen_feedback_loop = FALSE;

static gint
state_filter_changed_idle (gpointer data)
{
	guint state_filter = GPOINTER_TO_UINT(data);
	debug_enter ("state_filter_changed_idle");

	pan_lock_unconditional ();
	dampen_feedback_loop = TRUE;

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_new_button),
	                                state_filter & STATE_FILTER_NEW);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_unread_button),
	                                state_filter & STATE_FILTER_UNREAD);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_read_button),
	                                state_filter & STATE_FILTER_READ);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_complete_binary_button),
	                                state_filter & STATE_FILTER_COMPLETE_BINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_incomplete_binary_button),
	                                state_filter & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_nonbinary_button),
	                                state_filter & STATE_FILTER_NONBINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_watched_button),
	                                state_filter & STATE_FILTER_WATCHED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_killfile_button),
	                                state_filter & STATE_FILTER_IGNORED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_normal_rank_button),
	                                state_filter & STATE_FILTER_NORMAL_RANK);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_saved_button),
	                                state_filter & STATE_FILTER_SAVED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_queued_button),
	                                state_filter & STATE_FILTER_QUEUED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_idle_button),
	                                state_filter & STATE_FILTER_IDLE);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_cached_button),
	                                state_filter & STATE_FILTER_CACHED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_non_cached_button),
	                                state_filter & STATE_FILTER_NOT_CACHED);

	dampen_feedback_loop = FALSE;
	pan_unlock_unconditional ();

	debug_exit ("state_filter_changed_idle");
	return 0;
}
static gint
state_filter_changed_cb (gpointer call_object,
                         gpointer call_arg,
                         gpointer user_data)
{
	debug_enter ("state_filter_changed_cb");

	pan_lock ();
	gtk_idle_add (state_filter_changed_idle, call_arg);
	pan_unlock ();

	debug_exit ("state_filter_changed_cb");
	return 0;
}


/*****
******  CREATE TOOLBAR
*****/

extern GtkTooltips *ttips;

static void
state_filter_button_toggled_cb (GtkCheckMenuItem * tb, gpointer user_data)
{
        const guint flag = GPOINTER_TO_UINT(user_data);
	const gboolean on = tb->active;

	if (!dampen_feedback_loop)
		articlelist_poke_state_filter (flag, on);
}

static GtkWidget*
create_menu_separator (void)
{
	GtkWidget * w = gtk_menu_item_new ();
	gtk_widget_show (w);
	return w;
}

static GtkWidget*
create_toggle_menu_item (const gchar * label, gchar ** xpm, gint filter, GtkWidget ** setme)
{
	GtkWidget * l;
	GtkWidget * p;
	GtkWidget * w;
	GtkWidget * box;

	box = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	p = gnome_pixmap_new_from_xpm_d (xpm);
	l = gtk_label_new (label);
	gtk_box_pack_start (GTK_BOX(box), p, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(box), l, FALSE, FALSE, 0);

	w = gtk_check_menu_item_new ();
	gtk_container_add (GTK_CONTAINER(w), box);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", state_filter_button_toggled_cb, GUINT_TO_POINTER(filter));
	gtk_widget_show_all (w);
	*setme = w;
	return w;
}

static GtkWidget*
foo (void)
{
	GtkWidget * w;
	GtkMenu * menu;
       
	menu = GTK_MENU(gtk_menu_new());
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show New Articles"),
		article_new_xpm,
		STATE_FILTER_NEW,
		&show_new_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Unread Articles"),
		article_unread_xpm,
		STATE_FILTER_UNREAD,
		&show_unread_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Read Articles"),
		article_read_xpm,
		STATE_FILTER_READ,
		&show_read_button));
	gtk_menu_append (menu, create_menu_separator());
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Watched Threads"),
		favorite_xpm,
		STATE_FILTER_WATCHED,
		&show_watched_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Ignored Threads"),
		bomb_xpm,
		STATE_FILTER_IGNORED,
		&show_killfile_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Threads Neither Watched nor Ignored"),
		mini_page_xpm,
		STATE_FILTER_NORMAL_RANK,
		&show_normal_rank_button));
	gtk_menu_append (menu, create_menu_separator());
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Complete Binary Articles"),
		binary_complete_xpm,
		STATE_FILTER_COMPLETE_BINARIES,
		&show_complete_binary_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Incomplete Binary Articles"),
		binary_incomplete_xpm,
		STATE_FILTER_INCOMPLETE_BINARIES,
		&show_incomplete_binary_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Text Articles"),
		mini_page_xpm,
		STATE_FILTER_NONBINARIES,
		&show_nonbinary_button));
	gtk_menu_append (menu, create_menu_separator());
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Saved Articles"),
		binary_xpm,
		STATE_FILTER_SAVED,
		&show_saved_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Queued Articles"),
		bluecheck_xpm,
		STATE_FILTER_QUEUED,
		&show_queued_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Articles Neither Queued nor Saved"),
		mini_page_xpm,
		STATE_FILTER_IDLE,
		&show_idle_button));
	gtk_menu_append (menu, create_menu_separator());
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Articles which are Cached Locally"),
		disk_xpm,
		STATE_FILTER_CACHED,
		&show_cached_button));
	gtk_menu_append (menu, create_toggle_menu_item(
		_("Show Articles which are not Cached Locally"),
		mini_page_xpm,
		STATE_FILTER_NOT_CACHED,
		&show_non_cached_button));

	w = gtk_menu_item_new ();
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (filter_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Filters"), "");
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(w), GTK_WIDGET(menu));
	return w;                                                                                                                                            
}
typedef struct
{
	const gchar * name;
	ArticleFilterMode mode;
}
ArticleToolbarMenuStruct;

GtkWidget*
article_toolbar_new (void)
{
	GtkWidget * w = NULL;

	toolbar = gtk_hbox_new (FALSE, GNOME_PAD);

	/* filter menu */
	w = gtk_menu_bar_new ();
	gtk_menu_bar_append (GTK_MENU_BAR(w), foo());
	gtk_box_pack_start (GTK_BOX(toolbar), w, FALSE, FALSE, 0);

	/* Filter label */
	if (1) {
		GtkWidget * option_menu = gtk_option_menu_new ();
		GtkWidget *menu = gtk_menu_new ();
		int index = 0;
		int i;
		ArticleToolbarMenuStruct foo[] = {
			{NULL, ARTICLE_FILTER_SUBJECT},
			{NULL, ARTICLE_FILTER_AUTHOR}
		};
		const int row_qty = sizeof(foo) / sizeof(foo[0]);
		foo[0].name = _("Subject");
		foo[1].name = _("Author");
		for (i=0; i<row_qty; ++i) {
			GtkWidget * item = gtk_menu_item_new_with_label (foo[i].name);
			gtk_signal_connect (GTK_OBJECT(item), "activate",
				(GtkSignalFunc)set_filter_mode,
				GINT_TO_POINTER(foo[i].mode));
			gtk_menu_append (GTK_MENU (menu), item);
			gtk_widget_show (item);
		}
		gtk_menu_set_active (GTK_MENU(menu), index);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
		gtk_widget_show_all (GTK_WIDGET(option_menu));
		/*gtk_widget_set (option_menu,
			"width", 60,
			"height", 25,
			NULL); FIXME: If we don't do this the om doesn't show up*/
		w = option_menu;
	}
        //w = gtk_label_new (_("Find:"));
	gtk_box_pack_start (GTK_BOX(toolbar), w, FALSE, FALSE, 0);

	/* Filter */
	w = gtk_entry_new();
	gtk_signal_connect (GTK_OBJECT (w), "activate", GTK_SIGNAL_FUNC(article_filter_cb), w);
	gtk_signal_connect (GTK_OBJECT (w), "focus-out-event", GTK_SIGNAL_FUNC(article_filter_cb), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
	        _("Type in a search string and press ENTER. "
	          "Wildcards are allowed; see http://pan.rebelbase.com/wildmat.html "
	          "for more information."), "");
	gtk_box_pack_start (GTK_BOX(toolbar), w, TRUE, TRUE, 0);

	/* show the toolbar */
	gtk_widget_show_all (toolbar);

	/* listen for changes to the articlelist "state filter"
	   so that we can update the menus accordingly */
	pan_callback_add (articlelist_state_filter_changed,
	                  state_filter_changed_cb,
	                  NULL);

	return toolbar;
}
