/*
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/base-prefs.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-ui.h>
#include <pan/identities/identity-edit-ui.h>
#include <pan/identities/identity-manager.h>

#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget * identity_clist;
	GtkWidget * dialog;
	GPtrArray * identities;
	gchar     * default_news;
	gchar     * default_mail;

	GtkWidget * copy_button;
	GtkWidget * remove_button;
	GtkWidget * edit_button;
}
IdentityListDialog;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

static gint
get_selected_index (IdentityListDialog * d)
{
	gint sel;
	GtkCList * clist;

	/* find the selected identity */
	pan_lock ();
	clist = GTK_CLIST(d->identity_clist);
	sel = -1;
	if (clist->selection != NULL)
		sel = GPOINTER_TO_INT(clist->selection->data);
	pan_unlock ();

	return sel;
}                                                                                                                               

static void
button_refresh (IdentityListDialog * d)
{
	const gint sel = get_selected_index (d);
	const gboolean have_sel = sel != -1;

	pan_lock ();
	gtk_widget_set_sensitive (d->copy_button, have_sel);
	gtk_widget_set_sensitive (d->edit_button, have_sel);
	gtk_widget_set_sensitive (d->remove_button, have_sel);
	pan_unlock ();
}

static void
clist_refresh (IdentityListDialog * d)
{
	gint i;
	const gint sel = get_selected_index (d);
	GtkCList * clist = GTK_CLIST(d->identity_clist);

	pan_lock ();
	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);
	for (i=0; i<d->identities->len; ++i)
	{
		gint row;
		Identity * id = IDENTITY(g_ptr_array_index(d->identities,i));
	       	row = gtk_clist_insert (clist, -1, &id->name);
		gtk_clist_set_row_data (clist, row, id);
	}
	gtk_clist_select_row (clist, sel, 0);
	gtk_clist_thaw (clist);
	pan_unlock ();
}

static void
mark_default_or_not (IdentityListDialog   * d,
                     gchar               ** default_id_name,
                     gchar                * id_name,
                     gchar                  is_default)
{
	/* If there's only one identity, it's always the default */

	if (d->identities->len == 1)
		is_default = TRUE;

	if (is_default)
		replace_gstr (default_id_name, g_strdup (id_name));
	else
	if (!pan_strcmp (*default_id_name, id_name))
		replace_gstr (default_id_name, NULL);

}
/*****
******
*****/

static gint
identity_dialog_close_cb (GnomeDialog * dialog, gpointer data)
{
	IdentityListDialog * d = (IdentityListDialog*) data;

	/* save changed identities */
	identity_manager_set_identities (d->identities, d->default_news,
		d->default_mail);

	/* cleanup identities */
	if (d->identities != NULL) {
		pan_g_ptr_array_foreach (d->identities, (GFunc)pan_object_unref, NULL);
		g_ptr_array_free (d->identities, TRUE);
		d->identities = NULL;
	}
	g_free (d->default_news);
	g_free (d->default_mail);

	/* cleanup dialog */
	g_free (d);
	return FALSE;
}

static void
identity_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	gnome_dialog_close (dialog);
}


/**
*** Remove
**/

static void
remove_button_clicked_cb (GtkButton * button, gpointer data)
{
	IdentityListDialog * d = (IdentityListDialog*) data;
	gint sel = get_selected_index (d);
	if (sel != -1)
	{
		if (d->identities->len <= 1)
		{
			pan_error_dialog_parented (d->dialog,
				_("You need at least one profile"));
		}
		else
		{
			Identity * id;

			id = IDENTITY(g_ptr_array_index(d->identities, sel));

			mark_default_or_not (d, &d->default_news, id->name, FALSE);
			mark_default_or_not (d, &d->default_mail, id->name, FALSE);

			g_ptr_array_remove_index (d->identities, sel);
			pan_object_unref (PAN_OBJECT(id));

			clist_refresh (d);
			button_refresh (d);

		}
	}
}

/**
*** Add
**/

static void
add_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	if (index != IDENTITY_EDIT_DIALOG_CLICKED_CANCEL)
	{
		guint                i;
		GtkWidget *          w = GTK_WIDGET(dialog);
		IdentityListDialog * d = (IdentityListDialog*) data;
		gboolean             default_news, default_mail;
		Identity           * id;

		id = identity_edit_dialog_get_identity (w,
			&default_news, &default_mail);

		/* make sure we have an identity */
		if (id == NULL) {
			pan_error_dialog_parented (dialog, _("Incomplete Profile."));
			return;
		}

		/* see if we've already got this identity */
		for (i=0; i!=d->identities->len; ++i)
		{
			Identity * id_old;

			id_old = IDENTITY(g_ptr_array_index(d->identities,i));
			if (!pan_strcmp(id->name, id_old->name))
					break;
		}

		/* either insert or update */
		if (i == d->identities->len)
		{
			g_ptr_array_add (d->identities, id);
		}
		else
		{
			Identity * id_old;

			id_old = IDENTITY(g_ptr_array_index(d->identities,i));
			g_ptr_array_index (d->identities,i) = id;
			pan_object_unref (PAN_OBJECT(id_old));
		}

		/* mark defaults */
		mark_default_or_not (d, &d->default_news, id->name, default_news);
		mark_default_or_not (d, &d->default_mail, id->name, default_mail);

		/* rebuild the clist */
		clist_refresh (d);
	}

	gnome_dialog_close (dialog);
}

static void
add_button_clicked_cb (GtkButton * button, gpointer data)
{
	IdentityListDialog * f = (IdentityListDialog*) data;
	GtkWidget * dialog = identity_edit_dialog_new (NULL, FALSE, FALSE);

	gtk_signal_connect (GTK_OBJECT(dialog), "clicked", 
		GTK_SIGNAL_FUNC(add_dialog_clicked_cb), f);
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(f->dialog));
	gtk_widget_show_all (dialog);
}

/**
*** Copy 
**/

static void
copy_button_clicked_cb (GtkButton *button, gpointer data)
{
	IdentityListDialog * d = (IdentityListDialog*) data;
	gint sel = get_selected_index (d);
	if (sel != -1)
	{
		const Identity * id_old;
		Identity       * id_new;
		GtkWidget      * dialog;

		id_old = IDENTITY(g_ptr_array_index(d->identities,sel));
		id_new = identity_dup(id_old);
		replace_gstr (&id_new->name, g_strdup(_("New Profile")));
		dialog = identity_edit_dialog_new (id_new, FALSE, FALSE);
		pan_object_unref (PAN_OBJECT (id_new));
		gtk_signal_connect (GTK_OBJECT(dialog), "clicked", 
			GTK_SIGNAL_FUNC(add_dialog_clicked_cb), d);
		gnome_dialog_set_parent (GNOME_DIALOG(dialog), 
			GTK_WINDOW(d->dialog));
		gtk_widget_show_all (dialog);

	}
}

/**
***  Edit
**/

static void
edit_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	if (index != IDENTITY_EDIT_DIALOG_CLICKED_CANCEL)
	{
		IdentityListDialog * d = (IdentityListDialog*) data;
		gint sel = get_selected_index (d);
		if (sel != -1)
		{ 
			gboolean   default_news, default_mail;
			GtkWidget * w = GTK_WIDGET(dialog);
			Identity * id_old = IDENTITY(g_ptr_array_index(d->identities,sel));
			Identity * id_new = identity_dup (identity_edit_dialog_get_identity(w, &default_news, &default_mail));
			g_ptr_array_index(d->identities,sel) = id_new;

			mark_default_or_not (d, &d->default_news, id_new->name, 
				default_news);
			mark_default_or_not (d, &d->default_mail, id_new->name, 
				default_mail);
		
			pan_object_unref (PAN_OBJECT(id_old));

			clist_refresh (d);
		}
	}

	gnome_dialog_close (dialog);
}
static void
edit_button_clicked_cb (GtkButton * button, gpointer data)
{
	IdentityListDialog * d = (IdentityListDialog*) data;
	gint sel = get_selected_index (d);
	if (sel != -1)
	{
		const Identity * id_old = IDENTITY(g_ptr_array_index(d->identities,sel));
		GtkWidget * dialog;
		gboolean         default_news, default_mail;

		default_news = pan_strcmp (id_old->name, d->default_news) ? FALSE : TRUE;
		default_mail = pan_strcmp (id_old->name, d->default_mail) ? FALSE : TRUE;
		dialog = identity_edit_dialog_new (id_old, default_news, default_mail);
		gtk_signal_connect (GTK_OBJECT(dialog), "clicked", GTK_SIGNAL_FUNC(edit_dialog_clicked_cb), d);
		gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(d->dialog));
		gtk_widget_show_all (dialog);
	}
}
static gboolean
identity_clist_button_press_cb (GtkWidget * w, GdkEventButton * b, gpointer data)
{
	if (b->button==1 && b->type==GDK_2BUTTON_PRESS)
		edit_button_clicked_cb (NULL, data);
	return FALSE;
}

static void     
list_selection_changed_cb (GtkCList          * clist,
                           gint                row,
                           gint                column,
                           GdkEventButton    * event,
                           gpointer            user_data)
{
	button_refresh ((IdentityListDialog*)user_data);
}


/************
*************  PROTECTED
************/

/************
*************  PUBLIC
************/

GtkWidget*
identity_dialog_new (void)
{
	GtkWidget * w;
	GtkWidget * hbox;
	GtkWidget * bbox;
	GtkTooltips * tips = gtk_tooltips_new ();
	IdentityListDialog * d = g_new0 (IdentityListDialog, 1);
	Identity  * id;
	gchar * titles [1];


	/* load identities */
	d->identities = g_ptr_array_new ();
	identity_manager_get_identities (d->identities);
	id = identity_manager_get_default(ID_NEWS_DEFAULT);
	if (id)
		replace_gstr(&d->default_news, g_strdup (id->name));
	id = identity_manager_get_default(ID_MAIL_DEFAULT);
	if (id)
		replace_gstr(&d->default_mail, g_strdup (id->name));

	/* dialog */
	w = d->dialog = gnome_dialog_new (_("Pan: Profile"), GNOME_STOCK_BUTTON_OK, NULL);
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT(w), "close", GTK_SIGNAL_FUNC(identity_dialog_close_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(identity_dialog_clicked_cb), d);

	/* workarea */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(w)->vbox), hbox, TRUE, TRUE, 0);

	/* clist */
	titles[0] = _("Profile");
	w = d->identity_clist= gtk_clist_new_with_titles (1, titles);
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event",
	                    GTK_SIGNAL_FUNC(identity_clist_button_press_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "select_row",
	                    GTK_SIGNAL_FUNC(list_selection_changed_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "unselect_row",
	                    GTK_SIGNAL_FUNC(list_selection_changed_cb), d);


        w = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER(w), d->identity_clist);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);
	gtk_widget_set_usize (w, 300, 300);

	/* button box */
	bbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, FALSE, 0);

	/* add button */
	w = gtk_button_new_with_label (_("Add New"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Add a new profile"), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(add_button_clicked_cb), d);

	/* copy from button */
	w = gtk_button_new_with_label (_("Copy From"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Use the selected profile to create a new profile"), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(copy_button_clicked_cb), d);
	d->copy_button = w;

	/* edit button */
	w = gtk_button_new_with_label (_("Edit"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Edit the selected profile"), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(edit_button_clicked_cb), d);
	d->edit_button = w;

	/* remove button */
	w = gtk_button_new_with_label (_("Remove"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Remove the selected profile"), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(remove_button_clicked_cb), d);
	d->remove_button = w;

	clist_refresh (d);
	button_refresh (d);
	return d->dialog;
}
