/* query-druid.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "query-druid.h"
#include <libgnomedb/libgnomedb.h>
#include "workspace-page.h"
#include "query-fields-editor.h"

static void query_druid_class_init (QueryDruidClass * class);
static void query_druid_init (QueryDruid * wid);
static void query_druid_dispose (GObject   * object);


struct _QueryDruidPriv
{
	GnomeDbDict      *dict;
	GnomeDbQuery     *build_query;
	
	/* druid pages */
	GtkWidget   *d_start;
	GtkWidget   *d_query_type;
	GtkWidget   *d_select_targets;
	GtkWidget   *d_select_joins;
	GtkWidget   *d_select_fields;
	GtkWidget   *d_select_where;
	GtkWidget   *d_select_order_by;
	GtkWidget   *d_select_finish;
	GtkWidget   *d_sql;
	GtkWidget   *d_not_yet_impl;

	/* extra data for each page */
	GtkWidget   *query_name;
	GtkWidget   *query_descr;
	GtkWidget   *query_type_table;
	GtkWidget   *sql_editor;

	/* SELECT query pages */
	GtkWidget   *target_tables_selector;
	GtkWidget   *target_targets_selector;
	GtkWidget   *target_add_button;
	GtkWidget   *target_up_button;
	GtkWidget   *target_dn_button;
	GtkWidget   *target_del_button;
	GtkWidget   *fields_targets_selector;
	GtkWidget   *fields_fields_selector;
	GtkWidget   *field_add_button;
	GtkWidget   *field_up_button;
	GtkWidget   *field_dn_button;
	GtkWidget   *field_del_button;
	GtkWidget   *expr_entry;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
query_druid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryDruidClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_druid_class_init,
			NULL,
			NULL,
			sizeof (QueryDruid),
			0,
			(GInstanceInitFunc) query_druid_init
		};		
		
		type = g_type_register_static (GNOME_TYPE_DRUID, "QueryDruid", &info, 0);
	}

	return type;
}

static void
query_druid_class_init (QueryDruidClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = query_druid_dispose;
}

static void
query_druid_init (QueryDruid * wid)
{
	wid->priv = g_new0 (QueryDruidPriv, 1);
	wid->priv->dict = NULL;
}

static void query_druid_initialize (QueryDruid *mgsel);


static void object_weak_notify (QueryDruid *mgsel, GObject *obj);
/**
 * query_druid_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new #QueryDruid widget.
 *
 * Returns: the new widget
 */
GtkWidget *
query_druid_new (GnomeDbDict *dict)
{
	GObject    *obj;
	QueryDruid *qdruid;

	g_return_val_if_fail (dict && IS_GNOME_DB_DICT (dict), NULL);
		
	obj = g_object_new (QUERY_DRUID_TYPE, NULL);
	qdruid = QUERY_DRUID (obj);

	qdruid->priv->dict = dict;

	g_object_weak_ref (G_OBJECT (qdruid->priv->dict),
			   (GWeakNotify) object_weak_notify, qdruid);
	
	query_druid_initialize (qdruid);

	return GTK_WIDGET (obj);
}

static void
object_weak_notify (QueryDruid *qdruid, GObject *obj)
{
	if (obj == (GObject*) qdruid->priv->dict)
		/* Tell that we don't need to weak unref the GnomeDbDict */
		qdruid->priv->dict = NULL;
}

static void build_query_fields_changed (GnomeDbQuery *query, GnomeDbQfield *field, QueryDruid *qdruid);
static void build_query_fields_order_changed (GnomeDbQuery *query, QueryDruid *qdruid);

static void
query_druid_dispose (GObject *object)
{
	QueryDruid *qdruid;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_DRUID (object));
	qdruid = QUERY_DRUID (object);

	if (qdruid->priv) {
		if (qdruid->priv->build_query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (qdruid->priv->build_query),
							      G_CALLBACK (build_query_fields_changed), qdruid);
			g_signal_handlers_disconnect_by_func (G_OBJECT (qdruid->priv->build_query),
							      G_CALLBACK (build_query_fields_order_changed), qdruid);
			g_object_unref (qdruid->priv->build_query);
			qdruid->priv->build_query = NULL;
		}

		/* Weak unref the GnomeDbDict if necessary */
		if (qdruid->priv->dict)
			g_object_weak_unref (G_OBJECT (qdruid->priv->dict),
					     (GWeakNotify) object_weak_notify, qdruid);
		
		/* the private area itself */
		g_free (qdruid->priv);
		qdruid->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void make_start_page (QueryDruid *qdruid);
static void make_query_attributes_page (QueryDruid *qdruid);
static void make_sql_query_page (QueryDruid *qdruid);
static void make_select_targets_page (QueryDruid *qdruid);
static void make_select_joins_page (QueryDruid *qdruid);
static void make_select_fields_page (QueryDruid *qdruid);
static void make_select_where_page (QueryDruid *qdruid);
static void make_select_params_page (QueryDruid *qdruid);
static void make_select_finish_page (QueryDruid *qdruid);
static void make_nyi_page (QueryDruid *qdruid);


static gboolean prev_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);
static gboolean next_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);
static void finish_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid);

static void
query_druid_initialize (QueryDruid *qdruid)
{	
	/* druid itself */
	gtk_container_set_border_width (GTK_CONTAINER (qdruid), 4);

	/* query which is built during the druid usage */
	qdruid->priv->build_query = GNOME_DB_QUERY (gnome_db_query_new (qdruid->priv->dict));
	gnome_db_query_set_query_type (qdruid->priv->build_query, GNOME_DB_QUERY_TYPE_SELECT);

	g_signal_connect (G_OBJECT (qdruid->priv->build_query), "field_added",
			  G_CALLBACK (build_query_fields_changed), qdruid);
	g_signal_connect (G_OBJECT (qdruid->priv->build_query), "field_removed",
			  G_CALLBACK (build_query_fields_changed), qdruid);
	g_signal_connect (G_OBJECT (qdruid->priv->build_query), "fields_order_changed",
			  G_CALLBACK (build_query_fields_order_changed), qdruid);

	/* pages making */
	make_start_page (qdruid);
	make_query_attributes_page (qdruid);

	make_select_targets_page (qdruid);
	make_select_joins_page (qdruid);
	make_select_fields_page (qdruid);
	make_select_where_page (qdruid);
	make_select_params_page (qdruid);
	make_select_finish_page (qdruid);

	make_sql_query_page (qdruid);

	make_nyi_page (qdruid);
}

static void
build_query_fields_changed (GnomeDbQuery *query, GnomeDbQfield *field, QueryDruid *qdruid)
{
	build_query_fields_order_changed (query, qdruid);
}

static void
build_query_fields_order_changed (GnomeDbQuery *query, QueryDruid *qdruid)
{
	/* compute up and down status in the fields selection page */
	GnomeDbField *selfield;
	gboolean canup = FALSE, candn = FALSE;

	selfield = (GnomeDbField *) gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->fields_fields_selector));
	if (selfield && IS_GNOME_DB_FIELD (selfield)) {
		GSList *fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (qdruid->priv->build_query));
		GSList *last = g_slist_last (fields);
		candn = TRUE;
		if (last && (last->data == (gpointer) selfield))
			candn = FALSE;
		g_slist_free (fields);
		canup = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (qdruid->priv->build_query), selfield) != 0;

		gtk_widget_set_sensitive (qdruid->priv->field_up_button, canup);
		gtk_widget_set_sensitive (qdruid->priv->field_dn_button, candn);
	}	
}


/*
 * Start page
 */ 
static void
make_start_page (QueryDruid *qdruid)
{
	GtkWidget *wid;

	wid = gnome_druid_page_edge_new (GNOME_EDGE_START);
	qdruid->priv->d_start = wid;
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_edge_set_title (GNOME_DRUID_PAGE_EDGE (wid), _("New query"));
	gnome_druid_page_edge_set_text (GNOME_DRUID_PAGE_EDGE (wid),
					_("This druid allows you to create a new query.\n\n"
					  "The query can be a data manipulation query of any type "
					  "(SELECT, INSERT, UPDATE or DELETE) and can also be an aggregation "
					  "query (UNION, INTERSECT or EXCEPT).\n\n"
					  "The query can be created either with a point and click interface "
					  "or entering the corresponding SQL statement."));
}


/*
 * Query attributes page
 */ 
static void query_type_rb_toggled_cb (GtkToggleButton *btn, QueryDruid *qdruid);
static void
make_query_attributes_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *table, *rb;
	GSList *rb_group1 = NULL;
	GSList *rb_group2 = NULL;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	qdruid->priv->d_query_type = wid;
	gtk_widget_show_all (wid);
	g_signal_connect (G_OBJECT (wid), "back", G_CALLBACK (prev_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("Query attributes"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	str = g_strdup_printf ("<b>%s</b>", _("Name and description:"));
	wid = gtk_label_new (str);
	g_free (str);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	table = gtk_table_new (2, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 2);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
	
	wid = gtk_label_new (_("Name:"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);
	
	wid = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (wid), _("Untitled"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	qdruid->priv->query_name = wid;

	
	wid = gtk_label_new (_("Description:"));
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);
	
	wid = gtk_entry_new ();
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	qdruid->priv->query_descr = wid;

	str = g_strdup_printf ("<b>%s</b>", _("Type of query:"));
	wid = gtk_label_new (str);
	g_free (str);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("Point & click wizard"));
	gtk_widget_show (rb);
	gtk_box_pack_start (GTK_BOX (vbox), rb, FALSE, FALSE, 0);
	rb_group1 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (-1));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rb), TRUE);
	
	wid = gtk_table_new (5, 2, FALSE);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_table_set_col_spacings (GTK_TABLE (wid), 20);
	qdruid->priv->query_type_table = wid;

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("SELECT query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 0, 1, GTK_FILL, 0, 0, 0);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_SELECT));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);
	g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1", 
			   GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_SELECT));

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("INSERT query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_INSERT));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("UPDATE query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 2, 3, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_UPDATE));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("DELETE query"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 3, 4, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_DELETE));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("Aggreation query (union, intersect or except)"));
	gtk_widget_show (rb);
	gtk_table_attach (GTK_TABLE (wid), rb, 1, 2, 4, 5, GTK_FILL, 0, 0, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group2);
	rb_group2 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_UNION));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	rb = gtk_radio_button_new_with_mnemonic (NULL, _("SQL query (of any type)"));
	gtk_widget_show (rb);
	gtk_box_pack_start (GTK_BOX (vbox), rb, FALSE, FALSE, 0);
	gtk_radio_button_set_group (GTK_RADIO_BUTTON (rb), rb_group1);
	rb_group1 = gtk_radio_button_get_group (GTK_RADIO_BUTTON (rb));
	g_object_set_data (G_OBJECT (rb), "action", GINT_TO_POINTER (-2));
	g_signal_connect (G_OBJECT (rb), "toggled", G_CALLBACK (query_type_rb_toggled_cb), qdruid);

	/* default action */
	g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", 
			   GINT_TO_POINTER (GNOME_DB_QUERY_TYPE_SELECT));
}
static void
query_type_rb_toggled_cb (GtkToggleButton *btn, QueryDruid *qdruid)
{
	gint action;

	if (! gtk_toggle_button_get_active (btn))
		return;

	action = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (btn), "action"));

	switch (action) {
	case -1:
		gtk_widget_set_sensitive (qdruid->priv->query_type_table, TRUE);
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", 
				   g_object_get_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1"));
		break;
	case -2:
		gtk_widget_set_sensitive (qdruid->priv->query_type_table, FALSE);
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", GINT_TO_POINTER (action));
		break;
	default:
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type_1", GINT_TO_POINTER (action));
		g_object_set_data (G_OBJECT (qdruid->priv->d_query_type), "query_type", GINT_TO_POINTER (action));
		gnome_db_query_set_query_type (qdruid->priv->build_query, action);
		g_print ("Query type set to %d\n", action);
		break;
	}
}

/*
 * SELECT query: targets page
 */
static void select_targets_table_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbTable *table, QueryDruid *qdruid);
static void select_targets_target_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbTarget *target, QueryDruid *qdruid);
static void select_targets_add_table (GtkButton *button, QueryDruid *qdruid);
static void select_targets_up_target (GtkButton *button, QueryDruid *qdruid);
static void select_targets_dn_target (GtkButton *button, QueryDruid *qdruid);
static void select_targets_del_target (GtkButton *button, QueryDruid *qdruid);
static void
make_select_targets_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *table, *label, *bbox, *button;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_targets = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SELECT query: targets"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s:</b>\n%s", _("Target selection:"),
			       _("select the tables and views from which "
				 "data will be extracted."));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);

	table = gtk_table_new (2, 4, FALSE);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Tables / Views:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Query targets:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	bbox = gtk_vbutton_box_new ();
	gtk_widget_show (bbox);
	gtk_table_attach (GTK_TABLE (table), bbox, 1, 2, 1, 2,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	
	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	wid = gtk_image_new_from_stock ("gtk-go-forward", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_targets_add_table), qdruid);
	qdruid->priv->target_add_button = button;

	bbox = gtk_vbutton_box_new ();
	gtk_widget_show (bbox);
	gtk_table_attach (GTK_TABLE (table), bbox, 3, 4, 1, 2,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);

	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	wid = gtk_image_new_from_stock ("gtk-go-up", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_targets_up_target), qdruid);
	qdruid->priv->target_up_button = button;
	
	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	wid = gtk_image_new_from_stock ("gtk-go-down", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_targets_dn_target), qdruid);
	qdruid->priv->target_dn_button = button;

	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	wid = gtk_image_new_from_stock ("gtk-cancel", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_targets_del_target), qdruid);
	qdruid->priv->target_del_button = button;
	
	wid = gnome_db_selector_new (qdruid->priv->dict, NULL, GNOME_DB_SELECTOR_TABLES, 0);
	qdruid->priv->target_tables_selector = wid;
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_widget_show (wid);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1, 2);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (select_targets_table_select_changed_cb), qdruid);

	wid = gnome_db_selector_new (qdruid->priv->dict, G_OBJECT (qdruid->priv->build_query),
			       GNOME_DB_SELECTOR_TARGETS, 0);
	qdruid->priv->target_targets_selector = wid;
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_widget_show (wid);	
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 2, 3, 1, 2);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (select_targets_target_select_changed_cb), qdruid);
}

static void
select_targets_table_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbTable *table, QueryDruid *qdruid)
{
	gtk_widget_set_sensitive (qdruid->priv->target_add_button, table ? TRUE : FALSE);
}

static void
select_targets_target_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbTarget *target, QueryDruid *qdruid)
{
	/* gtk_widget_set_sensitive (qdruid->priv->target_up_button, target ? TRUE : FALSE); */
	/* gtk_widget_set_sensitive (qdruid->priv->target_dn_button, target ? TRUE : FALSE); */
	gtk_widget_set_sensitive (qdruid->priv->target_del_button, target ? TRUE : FALSE);
}

static void
select_targets_add_table (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbTarget *target;
	GnomeDbTable *table;
	GSList *constraints, *list;
	GSList *targets = gnome_db_query_get_targets (qdruid->priv->build_query), *tlist;

	/* actual target creation */
	table = (GnomeDbTable *) 
		gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->target_tables_selector));
	g_assert (table && IS_GNOME_DB_TABLE (table));

	target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (qdruid->priv->build_query, GNOME_DB_ENTITY (table)));
	gnome_db_query_add_target (qdruid->priv->build_query, target, NULL);
	g_object_unref (target);

	/* using database FK constraints to create joins */
	constraints = gnome_db_database_get_tables_fk_constraints (gnome_db_dict_get_database (qdruid->priv->dict),
							     table, NULL, FALSE);
	list = constraints;
	while (list) {
		GnomeDbConstraint *cons = GNOME_DB_CONSTRAINT (list->data);
		GnomeDbTable *otable;
		GnomeDbTarget *otarget = NULL;
		gboolean table_is_pkey = TRUE;

		otable = gnome_db_constraint_get_table (cons);
		if (otable == table) {
			table_is_pkey = FALSE;
			otable = gnome_db_constraint_fkey_get_ref_table (cons);
		}
		
		/* find a suitable target to make a join with */
		tlist = targets;
		while (tlist && !otarget) {
			if (gnome_db_target_get_represented_entity (GNOME_DB_TARGET (tlist->data)) == 
			    (GnomeDbEntity *) otable)
				otarget = GNOME_DB_TARGET (tlist->data);
			tlist = g_slist_next (tlist);
		}

		if (otarget) {
			GnomeDbJoin *join;

			/* actual join */
			join = GNOME_DB_JOIN (gnome_db_join_new_with_targets (qdruid->priv->build_query, otarget, target));
			gnome_db_join_set_join_type (join, GNOME_DB_JOIN_TYPE_INNER);
			gnome_db_join_set_condition_from_fkcons (join);
			gnome_db_query_add_join (qdruid->priv->build_query, join);
			g_object_unref (join);
		}

		list = g_slist_next (list);
	}
	g_slist_free (constraints);
	g_slist_free (targets);

#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}

static void
select_targets_up_target (GtkButton *button, QueryDruid *qdruid)
{
	g_print ("%s() called\n", __FUNCTION__);
}

static void
select_targets_dn_target (GtkButton *button, QueryDruid *qdruid)
{
	g_print ("%s() called\n", __FUNCTION__);
}

static void
select_targets_del_target (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbTarget *target;
	
	target = (GnomeDbTarget *)
		gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->target_targets_selector));
	gnome_db_query_del_target (qdruid->priv->build_query, target);
#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}

static void
make_select_joins_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *label;
	GnomeDbGraph *graph;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_joins = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SELECT query: joins"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>\n%s", _("Targets' joins:"),
			       _("choose the joins that should be taken into account in the query "
				 "(most of the time there are not any target not joined to any other target)"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);

	graph = GNOME_DB_GRAPH (gnome_db_graph_query_new (qdruid->priv->build_query));
	wid = gnome_db_canvas_query_struct_new (qdruid->priv->build_query, graph);
	gnome_db_canvas_set_zoom_factor (GNOME_DB_CANVAS (wid), 0.8);
	g_object_unref (graph);
	wid = gnome_db_canvas_set_in_scrolled_window (GNOME_DB_CANVAS (wid));
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
}

static void select_fields_tfield_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbField *field, QueryDruid *qdruid);
static void select_fields_qfield_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbQfield *field, QueryDruid *qdruid);
static void select_fields_add_field (GtkButton *button, QueryDruid *qdruid);
static void select_fields_add_expr (GtkButton *button, QueryDruid *qdruid);
static void select_fields_up_field (GtkButton *button, QueryDruid *qdruid);
static void select_fields_dn_field (GtkButton *button, QueryDruid *qdruid);
static void select_fields_del_field (GtkButton *button, QueryDruid *qdruid);
static void
make_select_fields_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *table, *label, *bbox, *button;
	gchar *str;
	
	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_fields = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "back", G_CALLBACK (prev_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SELECT query: fields"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>\n%s", _("Fields selection:"),
			       _("select the fields from the already selected targets. "
				 "Alternatively fields can be entered as textual expressions for "
				 "functions, constants, etc."));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);

	table = gtk_table_new (4, 4, FALSE);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Possible fields:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Query fields:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Textual\nexpression:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	wid = gtk_entry_new ();
	gtk_widget_show (wid);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 3, 4,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	qdruid->priv->expr_entry = wid;

	bbox = gtk_vbutton_box_new ();
	gtk_widget_show (bbox);
	gtk_table_attach (GTK_TABLE (table), bbox, 1, 2, 1, 2,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	
	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	wid = gtk_image_new_from_stock (GTK_STOCK_GO_FORWARD, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_fields_add_field), qdruid);
	qdruid->priv->field_add_button = button;

	bbox = gtk_vbutton_box_new ();
	gtk_widget_show (bbox);
	gtk_table_attach (GTK_TABLE (table), bbox, 1, 2, 3, 4,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	
	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	wid = gtk_image_new_from_stock (GTK_STOCK_GO_FORWARD, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_fields_add_expr), qdruid);

	bbox = gtk_vbutton_box_new ();
	gtk_widget_show (bbox);
	gtk_table_attach (GTK_TABLE (table), bbox, 3, 4, 1, 2,
			  (GtkAttachOptions) (0),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);

	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	wid = gtk_image_new_from_stock (GTK_STOCK_GO_UP, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_fields_up_field), qdruid);
	qdruid->priv->field_up_button = button;
	
	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	wid = gtk_image_new_from_stock (GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_fields_dn_field), qdruid);
	qdruid->priv->field_dn_button = button;

	button = gtk_button_new ();
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	wid = gtk_image_new_from_stock ("gtk-cancel", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (wid);
	gtk_container_add (GTK_CONTAINER (button), wid);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (select_fields_del_field), qdruid);
	qdruid->priv->field_del_button = button;
	
	wid = gnome_db_selector_new (qdruid->priv->dict, G_OBJECT (qdruid->priv->build_query), 
			       GNOME_DB_SELECTOR_TARGETS_CTS, 0);
	qdruid->priv->fields_targets_selector = wid;
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_widget_show (wid);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 1, 2);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (select_fields_tfield_select_changed_cb), qdruid);

	wid = gnome_db_selector_new (qdruid->priv->dict, G_OBJECT (qdruid->priv->build_query),
			       GNOME_DB_SELECTOR_QVIS_FIELDS, 0);
	qdruid->priv->fields_fields_selector = wid;
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_widget_show (wid);	
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 2, 3, 1, 4);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (select_fields_qfield_select_changed_cb), qdruid);
}

static void
select_fields_tfield_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbField *field, QueryDruid *qdruid)
{
	gtk_widget_set_sensitive (qdruid->priv->field_add_button, field && IS_GNOME_DB_FIELD (field) ? TRUE : FALSE);
}

static void
select_fields_qfield_select_changed_cb (GnomeDbSelector *mgsel, GnomeDbQfield *field, QueryDruid *qdruid)
{
	gboolean sensitive = FALSE, candn = FALSE;

	if (field && IS_GNOME_DB_FIELD (field))
		sensitive = TRUE;

	if (sensitive) {
		GSList *fields = gnome_db_entity_get_fields (GNOME_DB_ENTITY (qdruid->priv->build_query));
		GSList *last = g_slist_last (fields);
		candn = TRUE;
		if (last && (last->data == (gpointer) field))
			candn = FALSE;
		g_slist_free (fields);
	}
	gtk_widget_set_sensitive (qdruid->priv->field_up_button, 
				  sensitive && 
				  (gnome_db_entity_get_field_index (GNOME_DB_ENTITY (qdruid->priv->build_query), GNOME_DB_FIELD (field)) != 0));
	gtk_widget_set_sensitive (qdruid->priv->field_dn_button, candn);
	gtk_widget_set_sensitive (qdruid->priv->field_del_button, sensitive);
}

static void
select_fields_add_field (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbField *field;
	GnomeDbTarget *target;
	GnomeDbQfield *newfield;

	field = (GnomeDbField *) gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->fields_targets_selector));
	target = (GnomeDbTarget *) gnome_db_selector_get_selected_object_parent (GNOME_DB_SELECTOR (qdruid->priv->fields_targets_selector));

	g_assert (field && IS_GNOME_DB_FIELD (field));
	g_assert (target && IS_GNOME_DB_TARGET (target));

	newfield = GNOME_DB_QFIELD (gnome_db_qf_field_new_with_objects (qdruid->priv->build_query, target, field));
	gnome_db_base_set_name (GNOME_DB_BASE (newfield), gnome_db_base_get_name (GNOME_DB_BASE (field)));
	gnome_db_entity_add_field (GNOME_DB_ENTITY (qdruid->priv->build_query), GNOME_DB_FIELD (newfield));
	g_object_unref (newfield);
#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}

static void
select_fields_add_expr (GtkButton *button, QueryDruid *qdruid)
{
	GError *error = NULL;
	const gchar *str;
	gchar *msg;

	str = gtk_entry_get_text (GTK_ENTRY (qdruid->priv->expr_entry));
	if (! gnome_db_query_add_field_from_sql (qdruid->priv->build_query, str, &error)) {
		GtkWidget *dlg;

		if (error) {
			msg = g_strdup_printf ("<b>%s</b>\n%s '%s':\n\n%s",
					       _("Error parsing/analysing field expression:"),
					       _("while parsing"),
					       str,
					       error->message);
					       
			g_error_free (error);
		}
		else
			msg = g_strdup_printf ("<b>%s</b>\n%s '%s'",
					       _("Error parsing/analysing field expression:"),
					       _("while parsing"),
					       str);

		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
							  GTK_BUTTONS_CLOSE, msg);
		g_free (msg);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}
}

static void
select_fields_up_field (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbField *field;
	gint pos;

	field = (GnomeDbField *) gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->fields_fields_selector));
	g_assert (field && IS_GNOME_DB_FIELD (field));
	pos = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (qdruid->priv->build_query), field);
	if (pos > 0) {
		GnomeDbField *ofield;
		ofield = gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (qdruid->priv->build_query), pos - 1);
		gnome_db_entity_swap_fields (GNOME_DB_ENTITY (qdruid->priv->build_query), field, ofield);
	}

#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}

static void
select_fields_dn_field (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbField *field;
	gint pos;
	GnomeDbField *ofield;

	field = (GnomeDbField *) gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->fields_fields_selector));
	g_assert (field && IS_GNOME_DB_FIELD (field));
	pos = gnome_db_entity_get_field_index (GNOME_DB_ENTITY (qdruid->priv->build_query), field);
	ofield = gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (qdruid->priv->build_query), pos + 1);
	if (ofield)
		gnome_db_entity_swap_fields (GNOME_DB_ENTITY (qdruid->priv->build_query), field, ofield);

#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}

static void
select_fields_del_field (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbField *field;

	field = (GnomeDbField *) gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (qdruid->priv->fields_fields_selector));
	g_assert (field && IS_GNOME_DB_FIELD (field));
	gnome_db_entity_remove_field (GNOME_DB_ENTITY (qdruid->priv->build_query), field);
	
#ifdef debug
	gnome_db_base_dump (GNOME_DB_BASE (qdruid->priv->build_query), 0);
#endif
}


static void
make_select_where_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox, *label;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_where = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SELECT query: condition and ordering"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	label = gtk_label_new (NULL);
	str = g_strdup_printf ("<b>%s</b>\n%s", _("Selection condition and ordering:"),
			       _("Enter the conditions to apply on the selected rows "
				 "to filter the final rows, and the ordering"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 0);

	wid = query_fields_editor_new (qdruid->priv->build_query, QEF_FILTER_OPTION | QEF_ORDER_OPTION);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
}

static void
make_select_params_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox;

	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_order_by = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SELECT query: parameters definition"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	wid = gtk_label_new ("Parameters");
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
}

static void
make_select_finish_page (QueryDruid *qdruid)
{
	GtkWidget *wid;

	wid = gnome_druid_page_edge_new (GNOME_EDGE_FINISH);
	gtk_widget_show_all (wid);
	qdruid->priv->d_select_finish = wid;
	g_signal_connect (G_OBJECT (wid), "finish", G_CALLBACK (finish_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_edge_set_title (GNOME_DRUID_PAGE_EDGE (wid), _("SELECT query: ready"));
	gnome_druid_page_edge_set_text (GNOME_DRUID_PAGE_EDGE (wid), _("The selection query is now ready."));
}


/*
 * Query from SQL page
 */
static void sql_query_text_changed_cb (GtkButton *button, QueryDruid *qdruid);
static void
make_sql_query_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	qdruid->priv->d_sql = wid;
	gtk_widget_show_all (wid);
	g_signal_connect (G_OBJECT (wid), "back", G_CALLBACK (prev_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	g_signal_connect (G_OBJECT (wid), "finish", G_CALLBACK (finish_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("SQL statement Query"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	str = g_strdup_printf ("<b>%s</b>", _("Enter the SQL statement representing the query:"));
	wid = gtk_label_new (str);
	g_free (str);
	gtk_widget_show (wid);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_label_set_justify (GTK_LABEL (wid), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0.5);

	wid = gnome_db_editor_new ();
	qdruid->priv->sql_editor = wid;
	gtk_widget_show (wid);
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), TRUE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), TRUE);
        gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);

	wid = gtk_button_new_with_label (_("Test query validity"));
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "clicked", G_CALLBACK (sql_query_text_changed_cb), qdruid);
}


static void
sql_query_text_changed_cb (GtkButton *button, QueryDruid *qdruid)
{
	GnomeDbQuery *query;
	gchar *sql;
	GError *error = NULL;
	gchar *str = NULL;
	GtkWidget *dlg;

	sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (qdruid->priv->sql_editor));
	query = GNOME_DB_QUERY (gnome_db_query_new_from_sql (qdruid->priv->dict, sql, &error));
	if (gnome_db_query_get_query_type (query) ==  GNOME_DB_QUERY_TYPE_NON_PARSED_SQL) {
		if (error) {
			str = g_strdup_printf ("<b>%s</b>\n%s", _("Parsing/analyzing error:"), error->message);
			g_error_free (error);
		}
		else
			str = g_strdup_printf ("<b>%s</b>\n%s",_("Parsing/analyzing error:"),
					       _("The query has an error"));
	}
	
	g_object_unref (G_OBJECT (query));

	if (str) {
		dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE, "");
		gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dlg), str);
		g_free (str);
	}
	else {
		str = g_strdup_printf ("<b>%s</b>", _("No error parsing and analyzing the query"));
		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE, str);
		g_free (str);
	}
	
	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);
}


/*
 * Not yet implemented Page
 */
static void
make_nyi_page (QueryDruid *qdruid)
{
	GtkWidget *wid, *vbox;
	gchar *str;

	wid = gnome_druid_page_standard_new ();
	gtk_widget_show_all (wid);
	qdruid->priv->d_not_yet_impl = wid;
	g_signal_connect (G_OBJECT (wid), "next", G_CALLBACK (next_page_cb), qdruid);
	gnome_druid_append_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (wid));
	gnome_druid_page_standard_set_title (GNOME_DRUID_PAGE_STANDARD (wid), _("To be implemented"));
	vbox = GNOME_DRUID_PAGE_STANDARD (wid)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 16);

	str = g_strdup_printf ("<b>%s</b>", _("This functionnality is not yet implemented"));
	wid = gtk_label_new (str);
	g_free (str);
	gtk_widget_show (wid);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
}




/* 
 * General callback for all the pages when the BACK button is pressed
 * Returns: TRUE if a page change has occurred.
 */
static gboolean 
prev_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;
	gboolean retval = FALSE;


	/* page to select the query type */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_query_type)) {
		gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_start));
		gnome_druid_set_show_finish (GNOME_DRUID (qdruid), FALSE);
		retval = TRUE;
		done = TRUE;
	}

	/* fields page: if there are less than 2 targets, then skip the joins page */
	if (page == (GnomeDruidPage *) qdruid->priv->d_select_fields) {
		GSList *targets = gnome_db_query_get_targets (qdruid->priv->build_query);
		if (g_slist_length (targets) < 2) {
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_select_targets));
			retval = TRUE;
		}
		g_slist_free (targets);
		done = TRUE;
	}


	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		done = TRUE;
		gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_query_type));
		gnome_druid_set_show_finish (GNOME_DRUID (qdruid), FALSE);
		retval = TRUE;
	}	
	
	if (!done)
		g_warning ("Unhandled 'back' druid page signal");

	return retval;
}


/* 
 * General callback for all the pages when the NEXT button is pressed
 * Returns: TRUE if a page change has been forced.
 */
static gboolean 
next_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;
	gboolean retval = FALSE;

	/* Start page */
	if ((page == (GnomeDruidPage *) qdruid->priv->d_start) ||
	    (page == (GnomeDruidPage *) qdruid->priv->d_select_joins) ||
	    (page == (GnomeDruidPage *) qdruid->priv->d_select_fields) ||
	    (page == (GnomeDruidPage *) qdruid->priv->d_select_where))
		/* nothing to be done here: use the default druid pages order */
		done = TRUE;

	/* page to select the query type */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_query_type)) {
		gint query_type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (page), "query_type"));
	
		switch (query_type) {
		case -2:
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_sql));
			gnome_druid_set_show_finish (GNOME_DRUID (qdruid), TRUE);
			break;
		case GNOME_DB_QUERY_TYPE_SELECT:
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_select_targets));
			gnome_druid_set_show_finish (GNOME_DRUID (qdruid), FALSE);
			break;			
		default:
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_not_yet_impl));
			gnome_druid_set_show_finish (GNOME_DRUID (qdruid), TRUE);
			gnome_druid_set_buttons_sensitive (GNOME_DRUID (qdruid), FALSE, TRUE, TRUE, FALSE);
			break;
		}
		retval = TRUE;
		done = TRUE;
	}
	
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_select_order_by)) {
		
		gnome_druid_set_show_finish (GNOME_DRUID (qdruid), TRUE);
		done = TRUE;
	}

	/* targets page: if there are less than 2 targets, then skip the joins page */
	if (page == (GnomeDruidPage *) qdruid->priv->d_select_targets) {
		GSList *targets = gnome_db_query_get_targets (qdruid->priv->build_query);
		if (g_slist_length (targets) < 2) {
			gnome_druid_set_page (GNOME_DRUID (qdruid), GNOME_DRUID_PAGE (qdruid->priv->d_select_fields));
			retval = TRUE;
		}
		g_slist_free (targets);
		done = TRUE;
	}
	
	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		done = TRUE;
		g_assert_not_reached ();
	}	

	if (!done)
		g_warning ("Unhandled 'next' druid page signal");

	return retval;
}

static void
finish_page_cb (GnomeDruidPage *page, GtkWidget *druid, QueryDruid *qdruid)
{
	gboolean done = FALSE;
	GnomeDbQuery *created_query = NULL;

	/* page for SQL queries */
	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_sql)) {
		GnomeDbQuery *query;
		gchar *sql;
		const gchar *str;

		sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (qdruid->priv->sql_editor));
		query = GNOME_DB_QUERY (gnome_db_query_new_from_sql (qdruid->priv->dict, sql, NULL));
		str = gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_name));
		if (str && *str)
			gnome_db_base_set_name (GNOME_DB_BASE (query), str);
		str = gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_descr));
		if (str && *str)
			gnome_db_base_set_description (GNOME_DB_BASE (query), str);

		gnome_db_dict_assume_query (qdruid->priv->dict, query);
		created_query = query;
		g_object_unref (G_OBJECT (query));

		done = TRUE;
	}

	if (!done && (page == (GnomeDruidPage *) qdruid->priv->d_select_finish)) {
		GnomeDbGraph *graph;
		gnome_db_dict_assume_query (qdruid->priv->dict, qdruid->priv->build_query);
		created_query = qdruid->priv->build_query;

		graph = gnome_db_dict_get_graph_for_object (qdruid->priv->dict, G_OBJECT (qdruid->priv->build_query));
		gnome_db_dict_assume_graph (qdruid->priv->dict, graph);

		done = TRUE;
	}

	if (created_query) {
		gnome_db_base_set_name (GNOME_DB_BASE (created_query), gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_name)));
		gnome_db_base_set_description (GNOME_DB_BASE (created_query), gtk_entry_get_text (GTK_ENTRY (qdruid->priv->query_descr)));
	}

	if (!done)
		g_warning ("Unhandled 'finish' druid page signal");

	gtk_widget_destroy (GTK_WIDGET (qdruid));
}

