'''
Provides basic magnification services.

@author: Eitan Isaacson <eitan@ascender.com>
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corp
@license: The BSD License

All rights reserved. This program and the accompanying materials are made 
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}

Current revision $Revision: 454 $
Latest change by $Author: parente $ on $Date: 2007-01-10 10:16:27 -0500 (Wed, 10 Jan 2007) $
'''
# import useful modules for Perks
import Perk, Task, AEConstants
from i18n import bind, _
from AEConstants import CMD_GOTO, CMD_GET_ROI
from math import cos, sin, atan, pi, sqrt

# metadata describing this Perk
__uie__ = dict(kind='perk', tier=None, all_tiers=True)

class ZoomMove(object):
  '''
  Class that deals with seamless panning and repositioning of the magnifier.

  @cvar pan_rate: Duation in milliseconds between "frames"
  @type pan_rate: int
  @cvar timeout_id: GObject timeout id.
  @type timeout_id: int
  @cvar end_coord: Coordinate that we are trying to get to
  @type end_coord: tuple
  @cvar last_track_type: The last type of tracking we did
  @type last_track_type: int
  @ivar accel: Acceleration in px/sec**2
  @type accel: int
  @ivar velocity: Velocity in px/sec
  @type velocity: int
  @ivar smooth_pan: Allow smooth panning
  @type smooth_pan: boolean
  '''
  pan_rate = 40
  timeout_id = None
  smooth_pan = None
  accel = None
  velocity = None
  _curr_velocity = None
  _curr_coord = None
  _vect_length = None
  _end_coord = None
  
  def goTo(self, end_coord, start_coord=None, smooth=False):
    '''
    Main public method for putting the magnifier on a coordinate.

    @param end_coord: Coordinate we want to reach.
    @type end_coord: tuple
    @param start_coord: Coordinate we are starting at, if none is given, 
      there will be no panning.
    @type start_coord: tuple
    @param smooth: Use smooth panning?
    @type smooth: boolean
    '''
    if start_coord and (self.smooth_pan or smooth):
      self._panTo(end_coord, start_coord)
    else:
      self._hopTo(end_coord)

  def _hopTo(self, end_coord):
    '''
    Stop panning and hop to given coordinate.

    @param end_coord: Coordinate we want to reach.
    @type end_coord: tuple
    '''
    self.stopPan()
    self.goToPos(end_coord)

  def _panTo(self, end_coord, start_coord):
    '''
    Pan to given coordinate.

    @param end_coord: Coordinate we want to reach.
    @type end_coord: tuple
    @param start_coord: Coordinate we are starting at.
    @type start_coord: tuple
    '''
    if end_coord == self._end_coord:
      return
        
    self.stopPan()
        
    self._curr_coord = float(start_coord[0]), float(start_coord[1])
    
    self._end_coord = end_coord
    
    x = self._end_coord[0] - self._curr_coord[0]
    y = self._end_coord[1] - self._curr_coord[1]

    self._vect_length = sqrt(x**2 + y**2)

    if (x, y) == (0, 0):
      return
    elif y == 0 and x > 0:
      angle = pi/2
    elif y == 0 and x < 0:
      angle = -pi/2
    elif y < 0:
      angle = atan (float(x)/float(y)) + pi
    else:
      angle = atan (float(x)/float(y))

    self._curr_velocity = 0
      
    self._x_step = sin(angle)
    self._y_step = cos(angle)
            
    self.startPan()

  def _panStep (self):
    '''
    Increment X and Y coordinates in a certain vector.
    Accelerate, deccelerate accordingly.
    '''
    # This is neccesarry to avoid jerks when tracking is enabled both with
    # mouse and anything else.
    if not self.smooth_pan:
      self.goToPos (self._end_coord)
      return False

    # If accel is set to zero it is disabled.
    if self.accel == 0:
      self._curr_velocity = self.velocity
      # currentvelocity**2/(2*acceleration) = The distance needed to get
      # to a complete stop at the current velocity and deceleration.
    elif (self._curr_velocity**2/(2*self.accel)) >= self._vect_length:
      self._curr_velocity -= self.accel
      # If the current velocity is smaller than the final velocity 
      # then accelerate.
    elif self._curr_velocity < self.velocity:
      self._curr_velocity += self.accel

    x = self._curr_velocity*self._x_step
    y = self._curr_velocity*self._y_step
    
    self._curr_coord = (x + self._curr_coord[0], 
                        y + self._curr_coord[1])
      
    progress = sqrt(x**2 + y**2)
    self._vect_length -= progress

    if self._vect_length < 0:
      self.goToPos (self._end_coord)
      return False
    else:
      self.goToPos (tuple(map(int,map(round,self._curr_coord))))
      return True

  def stopPan(self):
    '''
    Stop current pan motion.
    '''
    raise NotImplementedError

  def startPan(self):
    '''
    Start a timeout and execute _panstep
    '''
    raise NotImplementedError
    

  def goToPos(self, pos):
    '''
    Actual method that translates the zoomer, to be implemented by perk.
    '''
    raise NotImplementedError

class BasicMagPerkState(Perk.PerkState):
  '''  
  Defines a single set of state variables for all instances of the
  L{BasicMagPerk} and their L{Task}s. The variables are configuration settings
  that should be respected in all instances of L{BasicMagPerk}.

  MouseTrack (bool): Track cursor
  FocusTrack (bool): Track focus
  CaretTrack (bool): Track caret
  SmoothPan (bool): Smooth pan
  CaretPan (bool): Smooth caret panning
  PanAccel (int): Pan acceleration
  PanVelocity (int): Pan velocity
  '''
  def init(self):
    self.newBool('MouseTrack', True, _('Track cursor'),
                 _('Magnifier is to track mouse movemenself.'))
    self.newBool('FocusTrack', True, _('Track focus'),
                 _('Magnifier is to track focused area'))
    self.newBool('CaretTrack', True, _('Track caret'),
                 _('Magnifier is to track text caret'))
    self.newBool('SmoothPan', False, _('Smooth panning'),
                 _('Magnifier will pan to next selection'))
    self.newBool('CaretPan', False, _('Smooth caret panning'),
                 _('Magnifier will smoothly pan on caret movements, by '
                   'default it only pans on new lines.'))
    self.newRange('PanAccel', 40, _('Pan Acceleration'), 1, 100, 0,
       _('Acceleration rate of magnifier pan motion (pixels per second '
         'squared).'))
    self.newRange('PanVelocity', 600, _('Pan Velocity'), 1, 1000, 0,
       _('Velocity of magnifier pan motion (pixels per second).'))
    
  def getGroups(self):
    g = self.newGroup()
    t = g.newGroup(_('Tracking'))
    t.extend(['MouseTrack', 
              'FocusTrack', 
              'CaretTrack'])
    p = g.newGroup(_('Kinematics'))
    p.extend(['SmoothPan',
              'CaretPan',
              'PanAccel', 
              'PanVelocity'])
    return g

class BasicMagPerk(Perk.Perk, ZoomMove):
  '''
  A perk to provide magnification for a user.
  '''
  STATE=BasicMagPerkState

  def init(self):
    self.setPerkIdealOutput('magnifier')

    self.registerTask(HandleMouseChange())
    self.registerTask(HandleSelectorChange('mag selector'))
    self.registerTask(HandleFocusChange('mag focus'))
    self.registerTask(HandleCaretChange('mag caret'))
    self.registerTask(HandleReviewChange('mag review'))

    self.accel = self.perk_state.PanAccel*(self.pan_rate/1000.0)
    self.velocity = self.perk_state.PanVelocity*(self.pan_rate/1000.0)
    self.smooth_pan = self.perk_state.SmoothPan

    for setting in self.state:
      setting.addObserver(self._updateSetting)

    self.focused_por = self.task_por
    self.last_caret_offset = 0
    
    # link to review mode tasks
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review previous item')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review current item')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review next item')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review previous word')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review current word')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review next word')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review previous char')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review current char')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER, 
                   'review next char')
    self.chainTask('mag review', AEConstants.CHAIN_AFTER,
                   'pointer to focus')
    
  def _updateSetting(self, state, setting):
    name = setting.name
    value = setting.value
    if name == 'PanAccel':
      self.accel = value*(self.pan_rate/1000.0)
    elif name == 'PanVelocity':
      self.velocity = value*(self.pan_rate/1000.0)
    elif name == 'SmoothPan':
      self.smooth_pan = value
      
  def getName(self):
    return _('Basic magnifier')
    
  def getDescription(self):
    return _('Manages basic magnification services')
  
  def startPan(self):
    '''
    Register a TimerTask for the pan animation
    '''
    self.registerTask(PanTimer(self.pan_rate, 'pan step'))

  def stopPan(self):
    '''
    Unregister pan animation timer task
    '''
    task = self.getNamedTask('pan step')
    if task is not None:
      self.unregisterTask(task)
    
  def goToPos(self, pos):
    '''
    Send command to magnifier device to center zoomer on given coordinate.
    
    @param pos: Coordinate
    @type pos: tuple
    '''
    roi = self.send(CMD_GET_ROI, None)
    if roi is None:
      return
    x1, y1, x2, y2 = roi
    w, h = x2 - x1, y2 - y1
    x1 = pos[0] - w/2
    y1 = pos[1] - h/2
    x2, y2 = x1 + w, y1 + h
    self.send(CMD_GOTO, (x1, y1, x2, y2))
  
  def getPos(self):
    '''
    Get ROI from magnifier and return center coordinate
    
    @return: magnifier's focal point.
    @rtype: tuple
    '''
    roi = self.send(CMD_GET_ROI, None)
    if roi is None:
      return (0, 0, 0, 0)
    else:
      return  roi
  
  def getDesiredFocalPoint(self, por, x_padding=5):
    '''
    Calculates the desired focal point for a specific POR.
    And retuns current focal point, and desired focal point.

    @param por: POR that's focal point needs to be determined.
    @type por: L{POR.POR}
    @param x_padding: X axis padding
    @type x_padding: integer
    '''

    width, height = self.getAccVisualExtents(por)
    end_x, end_y = self.getAccVisualPoint(por)
    roi_x1, roi_y1, roi_x2, roi_y2 = self.perk.getPos()
    roi_width = (roi_x2 - roi_x1)/2

    start_pos = ((roi_x1 + roi_x2)/2, (roi_y1 + roi_y2)/2)

    x_offset = width/2 - roi_width + x_padding

    if width < roi_x2 - roi_x1:
      pass
    elif end_x - x_offset < roi_width:
      end_x = roi_width
    else:
      end_x = end_x - x_offset
 
    return start_pos, (end_x, end_y)

class HandleFocusChange(Task.FocusTask):
  '''
  Moves the magnifier to coordinate of this focused element.
  '''
  def executeGained(self, por, layer, **kwargs):
    if not self.perk.perk_state.FocusTrack:
      return True

    self.perk.focused_por = self.task_por
    start_pos, end_pos = self.perk.getDesiredFocalPoint(self.task_por)
    self.perk.goTo(end_pos, start_pos)

class HandleCaretChange(Task.CaretTask):
  '''
  Moves the magnifier to coordinate of this caret
  '''
  def execute(self, por, **kwargs):
    if not self.perk.perk_state.CaretTrack:
      return True

    start_pos, end_pos = self.perk.getDesiredFocalPoint(self.task_por)

    if (self.perk.focused_por == self.task_por or 
        por.item_offset != self.perk.last_caret_offset
        or self.perk_state.CaretPan):
      self.perk.goTo(end_pos, start_pos)
    else:
      self.perk.goTo(end_pos)

    self.perk.last_caret_offset =  por.item_offset

class HandleSelectorChange(Task.SelectorTask):
  '''
  Moves the magnifier to coordinate of this selected element
  '''
  def execute(self, por, **kwargs):
    if not self.perk.perk_state.FocusTrack:
      return True

    start_pos, end_pos = self.perk.getDesiredFocalPoint(self.task_por)
    self.perk.goTo(end_pos, start_pos)

class HandleMouseChange(Task.MouseTask):
  '''
  Moves the magnifier to coordinate of mouse
  '''
  def executeMoved(self, pos, **kwargs):
    '''
    Execute on each mouse event, for now we only look at cursor movements.
    '''
    if not self.perk.perk_state.MouseTrack:
      return True

    self.perk.goTo(pos)

class PanTimer(Task.TimerTask):
  '''
  Execute on timed intervals. Used to animate pan magnifier movement.
  '''
  def execute(self, **kwargs):
    rv = self.perk._panStep()
    if not rv:
      self.unregisterTask(self)

class HandleReviewChange(Task.InputTask):
  '''  
  Synchronizes the magnified area to the text at the pointer when reviewing.
  '''
  def execute(self, **kwargs):
    result = self.getTempVal('review')
    if result in (AEConstants.REVIEW_OK, AEConstants.REVIEW_WRAP):
      start_pos, end_pos = self.perk.getDesiredFocalPoint(self.task_por)
      self.perk.goTo(end_pos, start_pos)
