/* gnome-db-goo-entity.c
 *
 * Copyright (C) 2002 - 2007 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgda/libgda.h>
#include "gnome-db-goo.h"
#include "gnome-db-goo-entity.h"
#include "gnome-db-goo-field.h"
#include <glib/gi18n-lib.h>

static void gnome_db_goo_entity_class_init (GnomeDbGooEntityClass * class);
static void gnome_db_goo_entity_init       (GnomeDbGooEntity * drag);
static void gnome_db_goo_entity_dispose    (GObject   * object);
static void gnome_db_goo_entity_finalize   (GObject   * object);

static void gnome_db_goo_entity_set_property (GObject *object,
					   guint param_id,
					   const GValue *value,
					   GParamSpec *pspec);
static void gnome_db_goo_entity_get_property (GObject *object,
					   guint param_id,
					   GValue *value,
					   GParamSpec *pspec);

enum
{
	PROP_0,
	PROP_ENTITY,
	PROP_TARGET,
	PROP_MENU_FUNC
};

struct _GnomeDbGooEntityPrivate
{
	GdaQueryTarget      *target; /* NULL if the GnomeDbGooEntity is a generic entity */
	GdaEntity           *entity;

	/* UI building information */
        GSList             *field_items; /* list of GooCanvasItem for the fields */
	GSList             *other_items; /* list of GooCanvasItem for other purposes */
	gdouble            *field_ypos; /* array for each field's Y position in this canvas group */
	GtkWidget          *(*popup_menu_func) (GnomeDbGooEntity *ce);

	/* presentation parameters */
        gdouble             x_text_space;
        gdouble             y_text_space;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *entity_parent_class = NULL;

GType
gnome_db_goo_entity_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGooEntityClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_goo_entity_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGooEntity),
			0,
			(GInstanceInitFunc) gnome_db_goo_entity_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_GOO_ITEM, "GnomeDbGooEntity", &info, 0);
	}

	return type;
}

	

static void
gnome_db_goo_entity_class_init (GnomeDbGooEntityClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	entity_parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_goo_entity_dispose;
	object_class->finalize = gnome_db_goo_entity_finalize;

	/* Properties */
	object_class->set_property = gnome_db_goo_entity_set_property;
	object_class->get_property = gnome_db_goo_entity_get_property;

	g_object_class_install_property
                (object_class, PROP_ENTITY,
                 g_param_spec_object ("entity", NULL, NULL, GDA_TYPE_ENTITY, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property
                (object_class, PROP_TARGET,
                 g_param_spec_object ("target", NULL, NULL, GDA_TYPE_QUERY_TARGET, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property 
		(object_class, PROP_MENU_FUNC,
                 g_param_spec_pointer ("popup_menu_func", "Popup menu function", 
				       "Function to create a popup menu on each GnomeDbGooEntity",  G_PARAM_WRITABLE));
}

static gboolean button_press_event_cb (GnomeDbGooEntity *ce, GooCanvasItem  *target_item, GdkEventButton *event,
				       gpointer unused_data);

static void
gnome_db_goo_entity_init (GnomeDbGooEntity *entity)
{
	entity->priv = g_new0 (GnomeDbGooEntityPrivate, 1);
	entity->priv->entity = NULL;
	entity->priv->target = NULL;
	entity->priv->field_ypos = NULL;
	entity->priv->popup_menu_func = NULL;

	entity->priv->x_text_space = 3.;
	entity->priv->y_text_space = 3.;

	g_signal_connect (G_OBJECT (entity), "button-press-event",
			  G_CALLBACK (button_press_event_cb), NULL);
}

static void clean_items (GnomeDbGooEntity *ce);
static void create_items (GnomeDbGooEntity *ce);
static void object_destroyed_cb (GdaEntity *ent, GnomeDbGooEntity *ce);
static void entity_changed_cb (GdaEntity *ent, GnomeDbGooEntity *ce);

static void
gnome_db_goo_entity_dispose (GObject   * object)
{
	GnomeDbGooEntity *ce;

	g_return_if_fail (GNOME_DB_IS_GOO_ENTITY (object));

	ce = GNOME_DB_GOO_ENTITY (object);

	/* REM: let the GooCanvas library destroy the items itself */

	if (ce->priv->target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->target),
						      G_CALLBACK (object_destroyed_cb), ce);

                g_object_unref (ce->priv->target);
		ce->priv->target = NULL;
	}
	if (ce->priv->entity) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (object_destroyed_cb), ce);
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_changed_cb), ce);

                g_object_unref (ce->priv->entity);
		ce->priv->entity = NULL;
	}

	/* for the parent class */
	entity_parent_class->dispose (object);
}


static void
gnome_db_goo_entity_finalize (GObject   * object)
{
	GnomeDbGooEntity *ce;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_GOO_ENTITY (object));

	ce = GNOME_DB_GOO_ENTITY (object);
	if (ce->priv) {
		if (ce->priv->field_ypos)
			g_free (ce->priv->field_ypos);

		g_free (ce->priv);
		ce->priv = NULL;
	}

	/* for the parent class */
	entity_parent_class->finalize (object);
}

static void 
gnome_db_goo_entity_set_property    (GObject *object,
				  guint param_id,
				  const GValue *value,
				  GParamSpec *pspec)
{
	GnomeDbGooEntity *ce = NULL;
	GObject* propobject = NULL;

	ce = GNOME_DB_GOO_ENTITY (object);

	switch (param_id) {
	case PROP_TARGET:
		propobject = g_value_get_object (value);
		if (propobject == G_OBJECT (ce->priv->target))
			return;

		if (ce->priv->target) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->target),
							      G_CALLBACK (object_destroyed_cb), ce);
			g_object_set (G_OBJECT (object), "entity", NULL, NULL);

                        g_object_unref (ce->priv->target);
			ce->priv->target = NULL;
		}
		if (propobject) {
			g_return_if_fail (GDA_IS_QUERY_TARGET (propobject));
			ce->priv->target = GDA_QUERY_TARGET (propobject);
			g_object_ref (ce->priv->target);

			gda_object_connect_destroy (ce->priv->target, G_CALLBACK (object_destroyed_cb), ce);
		}
		g_object_set (G_OBJECT (object), "entity", 
			      gda_query_target_get_represented_entity (GDA_QUERY_TARGET (propobject)), NULL);
		break;
	case PROP_ENTITY:
		propobject = g_value_get_object (value);
		if (propobject && (propobject == G_OBJECT (ce->priv->entity)))
			return;

		if (ce->priv->entity) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (object_destroyed_cb), ce);
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (entity_changed_cb), ce);
			g_object_unref (ce->priv->entity);
			ce->priv->entity = NULL;
			clean_items (ce);
		}

		if (propobject) {
			g_return_if_fail (GDA_IS_ENTITY (propobject));
			ce->priv->entity = GDA_ENTITY(propobject);
			g_object_ref (ce->priv->entity);

			gda_object_connect_destroy (ce->priv->entity, G_CALLBACK (object_destroyed_cb), ce);
			/* Signals to keep the display up to date */
			g_signal_connect (G_OBJECT (ce->priv->entity), "changed",
					  G_CALLBACK (entity_changed_cb), ce);
		}
		create_items (ce);
		break;
	case PROP_MENU_FUNC:
		ce->priv->popup_menu_func = (GtkWidget *(*) (GnomeDbGooEntity *ce)) g_value_get_pointer (value);
		break;
	}
}

static void 
gnome_db_goo_entity_get_property    (GObject *object,
				    guint param_id,
				    GValue *value,
				    GParamSpec *pspec)
{
	TO_IMPLEMENT;
}

static void
object_destroyed_cb (GdaEntity *ent, GnomeDbGooEntity *ce)
{
	if ((ent == ce->priv->entity) && ce->priv->target) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (object_destroyed_cb), ce);
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_changed_cb), ce);
		ce->priv->entity = NULL;
	}
	else 
		goo_canvas_item_remove (GOO_CANVAS_ITEM (ce));
}

static void
entity_changed_cb (GdaEntity *ent, GnomeDbGooEntity *ce)
{
	create_items (ce);
}

/* 
 * destroy any existing GooCanvasItem obejcts 
 */
static void 
clean_items (GnomeDbGooEntity *ce)
{
	GSList *list;
	/* destroy all the items in the group */
	while (ce->priv->field_items)
		g_object_unref (G_OBJECT (ce->priv->field_items->data));

	for (list = ce->priv->other_items; list; list = list->next)
		g_object_unref (G_OBJECT (list->data));
	g_slist_free (ce->priv->other_items);
	ce->priv->other_items = NULL;

	/* free the fields positions */
	if (ce->priv->field_ypos) {
		g_free (ce->priv->field_ypos);
		ce->priv->field_ypos = NULL;
	}
}

static void field_item_destroy_cb (GnomeDbGooField *field, GnomeDbGooEntity *ce);

/*
 * create new GooCanvasItem objects
 */
static void 
create_items (GnomeDbGooEntity *ce)
{
	GooCanvasItem *item, *frame, *title;
        gdouble y, ysep;
#define Y_PAD 0.
#define X_PAD 3.
#define RADIUS_X 5.
#define RADIUS_Y 5.
#define MIN_HEIGHT 75.
        GooCanvasBounds border_bounds;
        GooCanvasBounds bounds;
	const gchar *cstr;
	gchar *tmpstr = NULL;
	GSList *fields, *list;
	gint fieldn;
	gdouble field_width;

	clean_items (ce);

        /* title */
	if (ce->priv->target) {
		gchar *_name;
		
		_name = gda_query_target_get_complete_name (ce->priv->target);
		tmpstr = g_markup_printf_escaped ("<b>%s</b>", _name);
		g_free (_name);
	}
	else {
		cstr = NULL;
		if (ce->priv->entity)
			cstr = gda_object_get_name (GDA_OBJECT (ce->priv->entity));
		if (cstr)
			tmpstr = g_markup_printf_escaped ("<b>%s</b>", cstr);
		else
			tmpstr = g_strdup_printf ("<b>%s</b>", _("No name"));	
	}
	y = RADIUS_Y;
        title = goo_canvas_text_new  (GOO_CANVAS_ITEM (ce), tmpstr, RADIUS_X + X_PAD, y, -1, GTK_ANCHOR_NORTH_WEST, 
				      "use-markup", TRUE, NULL);

	g_free (tmpstr);
        goo_canvas_item_get_bounds (title, &bounds);
        border_bounds = bounds;
        border_bounds.x1 = 0.;
        border_bounds.y1 = 0.;
        y += bounds.y2 - bounds.y1 + Y_PAD;

	/* separator's placeholder */
        ysep = y;
        y += Y_PAD;

	/* fields' vertical position */
	fields = NULL;
	if (ce->priv->entity)
		fields = gda_entity_get_fields (ce->priv->entity);
	ce->priv->field_ypos = g_new0 (gdouble, g_slist_length (fields) + 1);

	/* fields */
	for (fieldn = 0, list = fields; list; list = list->next, fieldn++) {
		ce->priv->field_ypos [fieldn] = y;
		item = gnome_db_goo_field_new (GOO_CANVAS_ITEM (ce),
					       GDA_ENTITY_FIELD (list->data),
					       X_PAD, ce->priv->field_ypos [fieldn], NULL);
		ce->priv->field_items = g_slist_append (ce->priv->field_items, item);
		
		/* the GooCanvas library need to tell when an item is destroyed */
		g_signal_connect (G_OBJECT (item), "destroy",
				  G_CALLBACK (field_item_destroy_cb), ce);

		goo_canvas_item_get_bounds (item, &bounds);
		border_bounds.x1 = MIN (border_bounds.x1, bounds.x1);
                border_bounds.x2 = MAX (border_bounds.x2, bounds.x2);
                border_bounds.y1 = MIN (border_bounds.y1, bounds.y1);
                border_bounds.y2 = MAX (border_bounds.y2, bounds.y2);

                y += bounds.y2 - bounds.y1 + Y_PAD;
	}
	if (!fields && (border_bounds.y2 - border_bounds.y1 < MIN_HEIGHT))
		border_bounds.y2 += MIN_HEIGHT - (border_bounds.y2 - border_bounds.y1);
	g_slist_free (fields);

	/* border */
	field_width = border_bounds.x2 - border_bounds.x1;
        border_bounds.y2 += RADIUS_Y;
        border_bounds.x2 += RADIUS_X;
        frame = goo_canvas_rect_new (GOO_CANVAS_ITEM (ce), border_bounds.x1, border_bounds.y1, 
				     border_bounds.x2, border_bounds.y2,
				    "radius-x", RADIUS_X, "radius-y", RADIUS_Y,
				    "fill-color", "white", NULL);		
	ce->priv->other_items = g_slist_prepend (ce->priv->other_items, frame);

	/* separator */
        item = goo_canvas_polyline_new_line (GOO_CANVAS_ITEM (ce), border_bounds.x1, ysep, border_bounds.x2, ysep,
					     "close-path", FALSE,
					     "line-width", .7, NULL);
	ce->priv->other_items = g_slist_prepend (ce->priv->other_items, item);

	goo_canvas_item_lower (frame, NULL);

	/* setting the fields' background width to be the same for all */
	for (list = ce->priv->field_items; list; list = list->next) 
		g_object_set (G_OBJECT (list->data), "width", field_width, NULL);
}

static void
field_item_destroy_cb (GnomeDbGooField *field, GnomeDbGooEntity *ce)
{
	g_assert (g_slist_find (ce->priv->field_items, field));
	ce->priv->field_items = g_slist_remove (ce->priv->field_items, field);
}

static gboolean
button_press_event_cb (GnomeDbGooEntity *ce, GooCanvasItem  *target_item, GdkEventButton *event,
		       gpointer unused_data)
{
	if (ce->priv->popup_menu_func) {
		GtkWidget *menu;
		menu = ce->priv->popup_menu_func (ce);
		gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
				NULL, NULL, ((GdkEventButton *)event)->button,
				((GdkEventButton *)event)->time);
		return TRUE;
	}

	return FALSE;	
}

/**
 * gnome_db_goo_entity_get_field_item
 * @ce: a #GnomeDbGooEntity object
 * @field: a #GdaEntityField object
 *
 * Get the #GnomeDbGooField object representing @field
 * in @ce.
 *
 * Returns: the corresponding #GnomeDbGooField
 */
GnomeDbGooField *
gnome_db_goo_entity_get_field_item (GnomeDbGooEntity *ce, GdaEntityField *field)
{
	GSList *fields;
	gint pos;

	g_return_val_if_fail (ce && GNOME_DB_IS_GOO_ENTITY (ce), NULL);
	g_return_val_if_fail (ce->priv, NULL);
	g_return_val_if_fail (ce->priv->entity, NULL);

	fields = gda_entity_get_fields (ce->priv->entity);
	pos = g_slist_index (fields, field);
	g_return_val_if_fail (pos >= 0, NULL);

	return g_slist_nth_data (ce->priv->field_items, pos);
}


/**
 * gnome_db_goo_entity_get_field_ypos
 * @ce: a #GnomeDbGooEntity object
 * @field: a #GdaEntityField object
 *
 * Get the Y position of the middle of the #GnomeDbGooField object representing @field
 * in @ce, in @ce's coordinates.
 *
 * Returns: the Y coordinate.
 */
gdouble
gnome_db_goo_entity_get_field_ypos (GnomeDbGooEntity *ce, GdaEntityField *field)
{
	gint pos;

	g_return_val_if_fail (ce && GNOME_DB_IS_GOO_ENTITY (ce), 0.);
	g_return_val_if_fail (ce->priv, 0.);
	g_return_val_if_fail (ce->priv->entity, 0.);
	g_return_val_if_fail (ce->priv->field_ypos, 0.);

	pos = gda_entity_get_field_index (ce->priv->entity, field);
	g_return_val_if_fail (pos >= 0, 0.);
	return (0.75 * ce->priv->field_ypos[pos+1] + 0.25 * ce->priv->field_ypos[pos]);
}


/**
 * gnome_db_goo_entity_new
 * @parent: the parent item, or NULL. 
 * @entity: a #GdaEntity to display
 * @x: the x coordinate
 * @y: the y coordinate
 * @...: optional pairs of property names and values, and a terminating NULL.
 *
 * Creates a new canvas item to display the @entity entity
 *
 * Returns: a new #GooCanvasItem object
 */
GooCanvasItem *
gnome_db_goo_entity_new (GooCanvasItem *parent, GdaEntity *entity, 
			 gdouble x, gdouble y, ...)
{
	GooCanvasItem *item;
	const char *first_property;
	va_list var_args;
		
	item = g_object_new (GNOME_DB_TYPE_GOO_ENTITY, NULL);

	if (parent) {
		goo_canvas_item_add_child (parent, item, -1);
		g_object_unref (item);
	}

	g_object_set (item, "entity", entity, NULL);

	va_start (var_args, y);
	first_property = va_arg (var_args, char*);
	if (first_property)
		g_object_set_valist ((GObject*) item, first_property, var_args);
	va_end (var_args);

	goo_canvas_item_translate (item, x, y);

	return item;
}

/**
 * gnome_db_goo_entity_new_target
 * @parent: the parent item, or NULL. 
 * @target: a #GdaQueryTarget to display
 * @x: the x coordinate
 * @y: the y coordinate
 * @...: optional pairs of property names and values, and a terminating NULL.
 *
 * Creates a new canvas item to display @target
 *
 * Returns: a new #GooCanvasItem object
 */
GooCanvasItem *
gnome_db_goo_entity_new_target (GooCanvasItem *parent, GdaQueryTarget *target, 
				gdouble x, gdouble y, ...)
{
	GooCanvasItem *item;
	const char *first_property;
	va_list var_args;
		
	item = g_object_new (GNOME_DB_TYPE_GOO_ENTITY, NULL);

	if (parent) {
		goo_canvas_item_add_child (parent, item, -1);
		g_object_unref (item);
	}

	g_object_set (item, "target", target, NULL);

	va_start (var_args, y);
	first_property = va_arg (var_args, char*);
	if (first_property)
		g_object_set_valist ((GObject*) item, first_property, var_args);
	va_end (var_args);

	goo_canvas_item_translate (item, x, y);

	return item;
}
