/* gnome-db-graph.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-graph.h"
#include <libgnomedb/gnome-db-ref-base.h>
#include <libgnomedb/gnome-db-xml-storage.h>
#include "gnome-db-graph-item.h"
#include "marshal.h"

/* 
 * Main static functions 
 */
static void gnome_db_graph_class_init (GnomeDbGraphClass *class);
static void gnome_db_graph_init (GnomeDbGraph *graph);
static void gnome_db_graph_dispose (GObject *object);
static void gnome_db_graph_finalize (GObject *object);

static void gnome_db_graph_set_property (GObject              *object,
				   guint                 param_id,
				   const GValue         *value,
				   GParamSpec           *pspec);
static void gnome_db_graph_get_property (GObject              *object,
				   guint                 param_id,
				   GValue               *value,
				   GParamSpec           *pspec);


static void init_ref_object (GnomeDbGraph *graph);
static void nullified_item_cb (GnomeDbGraphItem *item, GnomeDbGraph *graph);

static void graph_item_moved_cb (GnomeDbGraphItem *item, GnomeDbGraph *graph);


/* XML storage interface */
static void        gnome_db_graph_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_graph_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_graph_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_graph_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);


#ifdef debug
static void        gnome_db_graph_dump                (GnomeDbGraph *graph, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	ITEM_ADDED,
	ITEM_DROPPED,
	ITEM_MOVED,
	LAST_SIGNAL
};

static gint gnome_db_graph_signals[LAST_SIGNAL] = { 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_REF_OBJECT,
	PROP_GRAPH_TYPE
};


struct _GnomeDbGraphPrivate
{
	GnomeDbGraphType   type;
	GnomeDbRefBase    *ref_object;
	GSList       *graph_items;
};

/* module error */
GQuark gnome_db_graph_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_graph_error");
	return quark;
}


GType
gnome_db_graph_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGraphClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_graph_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGraph),
			0,
			(GInstanceInitFunc) gnome_db_graph_init
		};
		
		static const GInterfaceInfo xml_storage_info = {
                        (GInterfaceInitFunc) gnome_db_graph_xml_storage_init,
                        NULL,
                        NULL
                };

		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbGraph", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void
gnome_db_graph_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
        iface->get_xml_id = gnome_db_graph_get_xml_id;
        iface->save_to_xml = gnome_db_graph_save_to_xml;
        iface->load_from_xml = gnome_db_graph_load_from_xml;
}


static void
gnome_db_graph_class_init (GnomeDbGraphClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);
	
	gnome_db_graph_signals[ITEM_ADDED] =
		g_signal_new ("item_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbGraphClass, item_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_graph_signals[ITEM_DROPPED] =
		g_signal_new ("item_dropped",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbGraphClass, item_dropped),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	gnome_db_graph_signals[ITEM_MOVED] =
		g_signal_new ("item_moved",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbGraphClass, item_moved),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	
	class->item_added = NULL;
	class->item_dropped = NULL;
	class->item_moved = NULL;

	object_class->dispose = gnome_db_graph_dispose;
	object_class->finalize = gnome_db_graph_finalize;

	/* Properties */
	object_class->set_property = gnome_db_graph_set_property;
	object_class->get_property = gnome_db_graph_get_property;

	g_object_class_install_property (object_class, PROP_REF_OBJECT,
					 g_param_spec_pointer ("ref_object", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_GRAPH_TYPE,
					 g_param_spec_int ("graph_type", NULL, NULL, 
							   1, G_MAXINT, 1, G_PARAM_READABLE | G_PARAM_WRITABLE));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_graph_dump;
#endif
}

static void
gnome_db_graph_init (GnomeDbGraph *graph)
{
	graph->priv = g_new0 (GnomeDbGraphPrivate, 1);
	graph->priv->type = GNOME_DB_GRAPH_DB_RELATIONS;
	graph->priv->ref_object = NULL;
	graph->priv->graph_items = NULL;
}

static void ref_object_ref_lost_cb (GnomeDbRefBase *ref, GnomeDbGraph *graph);
static void
init_ref_object (GnomeDbGraph *graph)
{
	graph->priv->ref_object = GNOME_DB_REF_BASE (gnome_db_ref_base_new_no_ref_count (gnome_db_base_get_dict (GNOME_DB_BASE (graph))));
	g_signal_connect (G_OBJECT (graph->priv->ref_object), "ref_lost",
			  G_CALLBACK (ref_object_ref_lost_cb), graph);
}
static void
ref_object_ref_lost_cb (GnomeDbRefBase *ref, GnomeDbGraph *graph)
{
	gnome_db_base_nullify (GNOME_DB_BASE (graph));
}

/**
 * gnome_db_graph_new
 * @dict: a #GnomeDbDict object
 * @type: the graph type (one of #GnomeDbGraphType)
 *
 * Creates a new #GnomeDbGraph object. The graph type is used only to be able to sort out the
 * different types of graphs. It brings no special functionnality.
 *
 * Returns: the newly created object
 */
GObject   *
gnome_db_graph_new (GnomeDbDict *dict, GnomeDbGraphType type)
{
	GObject *obj = NULL;
	GnomeDbGraph *graph;
	guint id;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_GRAPH_TYPE, "dict", ASSERT_DICT (dict), NULL);
	graph = GNOME_DB_GRAPH (obj);

	g_object_get (G_OBJECT (ASSERT_DICT (dict)), "graph_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (obj), id);

	graph->priv->type = type;

	gnome_db_dict_declare_graph (ASSERT_DICT (dict), graph);

	return obj;
}

static void
gnome_db_graph_dispose (GObject *object)
{
	GnomeDbGraph *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH (object));

	graph = GNOME_DB_GRAPH (object);
	if (graph->priv) {		
		if (graph->priv->ref_object) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (graph->priv->ref_object),
							      G_CALLBACK (ref_object_ref_lost_cb), graph);
			g_object_unref (G_OBJECT (graph->priv->ref_object));
                        graph->priv->ref_object = NULL;
                }
		
		while (graph->priv->graph_items)
			nullified_item_cb (GNOME_DB_GRAPH_ITEM (graph->priv->graph_items->data), graph);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
nullified_item_cb (GnomeDbGraphItem *item, GnomeDbGraph *graph)
{
	g_assert (g_slist_find (graph->priv->graph_items, item));
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (nullified_item_cb) , graph);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item),
					      G_CALLBACK (graph_item_moved_cb) , graph);

	graph->priv->graph_items = g_slist_remove (graph->priv->graph_items, item);
#ifdef debug_signal
	g_print (">> 'ITEM_DROPPED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), gnome_db_graph_signals [ITEM_DROPPED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_DROPPED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_object_unref (G_OBJECT (item));
}


static void
gnome_db_graph_finalize (GObject   * object)
{
	GnomeDbGraph *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH (object));

	graph = GNOME_DB_GRAPH (object);
	if (graph->priv) {
		g_free (graph->priv);
		graph->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_graph_set_property (GObject              *object,
		       guint                 param_id,
		       const GValue         *value,
		       GParamSpec           *pspec)
{
	GnomeDbGraph *graph;
	gpointer ptr;

	graph = GNOME_DB_GRAPH (object);
	if (graph->priv) {
		switch (param_id) {
		case PROP_REF_OBJECT:
			if (!graph->priv->ref_object)
				init_ref_object (graph);

			if (graph->priv->ref_object) {
				ptr = g_value_get_pointer (value);
				g_signal_handlers_block_by_func (G_OBJECT (graph->priv->ref_object),
								 G_CALLBACK (ref_object_ref_lost_cb), graph);
				gnome_db_ref_base_set_ref_object (graph->priv->ref_object, ptr);
				g_signal_handlers_unblock_by_func (G_OBJECT (graph->priv->ref_object),
								   G_CALLBACK (ref_object_ref_lost_cb), graph);
			}
			break;
		case PROP_GRAPH_TYPE:
			graph->priv->type = g_value_get_int (value);
			break;
		}
	}
}

static void
gnome_db_graph_get_property (GObject              *object,
		       guint                 param_id,
		       GValue               *value,
		       GParamSpec           *pspec)
{
	GnomeDbGraph *graph;

	graph = GNOME_DB_GRAPH (object);
        if (graph->priv) {
                switch (param_id) {
                case PROP_REF_OBJECT:
			if (graph->priv->ref_object) 
				g_value_set_pointer (value, 
						     gnome_db_ref_base_get_ref_object (graph->priv->ref_object));
			else
				g_value_set_pointer (value, NULL);
                        break;
		case PROP_GRAPH_TYPE:
			g_value_set_int (value, graph->priv->type);
			break;
                }
        }
}


/**
 * gnome_db_graph_get_graph_type
 * @graph: a #GnomeDbGraph object
 *
 * Get the graph type of @graph.
 *
 * Returns: the type
 */
GnomeDbGraphType
gnome_db_graph_get_graph_type (GnomeDbGraph *graph)
{
	g_return_val_if_fail (graph && IS_GNOME_DB_GRAPH (graph), GNOME_DB_GRAPH_DB_RELATIONS);
	g_return_val_if_fail (graph->priv, GNOME_DB_GRAPH_DB_RELATIONS);

	return graph->priv->type;
}

/**
 * gnome_db_graph_get_items
 * @graph: a #GnomeDbGraph object
 *
 * Get a list of #GnomeDbGraphItem objects which are items of @graph
 *
 * Returns: a new list of #GnomeDbGraphItem objects
 */
GSList *
gnome_db_graph_get_items (GnomeDbGraph *graph)
{
	g_return_val_if_fail (graph && IS_GNOME_DB_GRAPH (graph), NULL);
	g_return_val_if_fail (graph->priv, NULL);

	if (graph->priv->graph_items)
		return g_slist_copy (graph->priv->graph_items);
	else
		return NULL;
}



/**
 * gnome_db_graph_get_item_from_obj
 * @graph: a #GnomeDbGraph object
 * @ref_obj: the #GnomeDbBase the returned item references
 * @create_if_needed:
 *
 * Get a pointer to a #GnomeDbGraphItem item from the object is represents.
 * If the searched #GnomeDbGraphItem is not found and @create_if_needed is TRUE, then a new
 * #GnomeDbGraphItem is created.
 *
 * Returns: the #GnomeDbGraphItem object, or %NULL if not found
 */
GnomeDbGraphItem *
gnome_db_graph_get_item_from_obj (GnomeDbGraph *graph, GnomeDbBase *ref_obj, gboolean create_if_needed)
{
	GnomeDbGraphItem *item = NULL;
	GSList *list;
	GnomeDbBase *obj;

	g_return_val_if_fail (graph && IS_GNOME_DB_GRAPH (graph), NULL);
	g_return_val_if_fail (graph->priv, NULL);
	g_return_val_if_fail (ref_obj, NULL);

	list = graph->priv->graph_items;
	while (list && !item) {
		g_object_get (G_OBJECT (list->data), "ref_object", &obj, NULL);
		if (obj == ref_obj)
			item = GNOME_DB_GRAPH_ITEM (list->data);
		list = g_slist_next (list);
	}

	if (!item && create_if_needed) {
		item = GNOME_DB_GRAPH_ITEM (gnome_db_graph_item_new (gnome_db_base_get_dict (GNOME_DB_BASE (graph)), ref_obj));
		
		gnome_db_graph_add_item (graph, item);
		g_object_unref (G_OBJECT (item));
	}

	return item;
}


/**
 * gnome_db_graph_add_item
 * @graph: a #GnomeDbGraph object
 * @item: a #GnomeDbGraphItem object
 *
 * Adds @item to @graph.
 */
void
gnome_db_graph_add_item (GnomeDbGraph *graph, GnomeDbGraphItem *item)
{
	g_return_if_fail (graph && IS_GNOME_DB_GRAPH (graph));
	g_return_if_fail (graph->priv);
	g_return_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item));

	g_object_ref (G_OBJECT (item));
	
	graph->priv->graph_items = g_slist_append (graph->priv->graph_items, item);
	gnome_db_base_connect_nullify (item, G_CALLBACK (nullified_item_cb), graph);
	g_signal_connect (G_OBJECT (item), "moved",
			  G_CALLBACK (graph_item_moved_cb), graph);
#ifdef debug_signal
	g_print (">> 'ITEM_ADDED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), gnome_db_graph_signals [ITEM_ADDED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_ADDED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

static void
graph_item_moved_cb (GnomeDbGraphItem *item, GnomeDbGraph *graph)
{
#ifdef debug_signal
	g_print (">> 'ITEM_MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (graph), gnome_db_graph_signals [ITEM_MOVED], 0, item);
#ifdef debug_signal
	g_print ("<< 'ITEM_MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

/**
 * gnome_db_graph_del_item
 * @graph: a #GnomeDbGraph object
 * @item: a #GnomeDbGraphItem object
 *
 * Removes @item from @graph
 */
void
gnome_db_graph_del_item (GnomeDbGraph *graph, GnomeDbGraphItem *item)
{
	g_return_if_fail (graph && IS_GNOME_DB_GRAPH (graph));
	g_return_if_fail (graph->priv);
	g_return_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item));

	nullified_item_cb (item, graph);
}

#ifdef debug
static void
gnome_db_graph_dump (GnomeDbGraph *graph, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (graph && IS_GNOME_DB_GRAPH (graph));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (graph->priv) {
		GSList *items;
		if (gnome_db_base_get_name (GNOME_DB_BASE (graph)))
			g_print ("%s" D_COL_H1 "GnomeDbGraph" D_COL_NOR "\"%s\" (%p) ",
				 str, gnome_db_base_get_name (GNOME_DB_BASE (graph)), graph);
		else
			g_print ("%s" D_COL_H1 "GnomeDbGraph" D_COL_NOR " (%p) ", str, graph);
		g_print ("\n");
		items = graph->priv->graph_items;
		while (items) {
			gnome_db_base_dump (GNOME_DB_BASE (items->data), offset+5);
			items = g_slist_next (items);
		}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, graph);
}
#endif

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_graph_get_xml_id (GnomeDbXmlStorage *iface)
{
        gchar *xml_id;
	
        g_return_val_if_fail (iface && IS_GNOME_DB_GRAPH (iface), NULL);
        g_return_val_if_fail (GNOME_DB_GRAPH (iface)->priv, NULL);
	
        xml_id = g_strdup_printf ("GR%d", gnome_db_base_get_id (GNOME_DB_BASE (iface)));

        return xml_id;
}

static xmlNodePtr
gnome_db_graph_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
        xmlNodePtr node = NULL;
	GnomeDbGraph *graph;
        gchar *str = NULL;
	GSList *list;

        g_return_val_if_fail (iface && IS_GNOME_DB_GRAPH (iface), NULL);
        g_return_val_if_fail (GNOME_DB_GRAPH (iface)->priv, NULL);

        graph = GNOME_DB_GRAPH (iface);

        node = xmlNewNode (NULL, "GNOME_DB_GRAPH");

        str = gnome_db_graph_get_xml_id (iface);
        xmlSetProp (node, "id", str);
        g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (graph)));
        xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (graph)));

	switch (graph->priv->type) {
	case GNOME_DB_GRAPH_DB_RELATIONS:
		str = "R";
		break;
	case GNOME_DB_GRAPH_QUERY_JOINS:
		str = "Q";
		break;
	case GNOME_DB_GRAPH_MODELLING:
		str = "M";
		break;
	default:
		g_assert_not_reached ();
		break;
	}
        xmlSetProp (node, "type", str);
	
	if (graph->priv->ref_object) {
		GnomeDbBase *base = gnome_db_ref_base_get_ref_object (graph->priv->ref_object);
		if (base) {
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
			xmlSetProp (node, "object", str);
			g_free (str);
		}
	}

	/* graph items */
	list = graph->priv->graph_items;
	while (list) {
		xmlNodePtr sub = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
		if (sub)
                        xmlAddChild (node, sub);
                else {
                        xmlFreeNode (node);
                        return NULL;
                }
		list = g_slist_next (list);
	}

        return node;
}

static gboolean
gnome_db_graph_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbGraph *graph;
        gchar *prop;
	xmlNodePtr children;
	gboolean id=FALSE;

        g_return_val_if_fail (iface && IS_GNOME_DB_GRAPH (iface), FALSE);
        g_return_val_if_fail (GNOME_DB_GRAPH (iface)->priv, FALSE);
        g_return_val_if_fail (node, FALSE);

	graph = GNOME_DB_GRAPH (iface);

	if (strcmp (node->name, "GNOME_DB_GRAPH")) {
                g_set_error (error,
                             GNOME_DB_GRAPH_ERROR,
                             GNOME_DB_GRAPH_XML_LOAD_ERROR,
                             _("XML Tag is not <GNOME_DB_GRAPH>"));
                return FALSE;
        }

	prop = xmlGetProp (node, "id");
        if (prop) {
                if (strlen (prop) <= 2) {
                        g_set_error (error,
                                     GNOME_DB_GRAPH_ERROR,
                                     GNOME_DB_GRAPH_XML_LOAD_ERROR,
                                     _("Wrong 'id' attribute in <GNOME_DB_GRAPH>"));
                        return FALSE;
                }
                gnome_db_base_set_id (GNOME_DB_BASE (graph), atoi (prop+2));
		id = TRUE;
                g_free (prop);
        }

	prop = xmlGetProp (node, "name");
        if (prop) {
                gnome_db_base_set_name (GNOME_DB_BASE (graph), prop);
                g_free (prop);
        }

        prop = xmlGetProp (node, "descr");
        if (prop) {
                gnome_db_base_set_description (GNOME_DB_BASE (graph), prop);
                g_free (prop);
        }

	prop = xmlGetProp (node, "type");
        if (prop) {
		switch (*prop) {
		case 'R':
			graph->priv->type = GNOME_DB_GRAPH_DB_RELATIONS;
			break;
		case 'Q':
			graph->priv->type = GNOME_DB_GRAPH_QUERY_JOINS;
			break;
		case 'M':
			graph->priv->type = GNOME_DB_GRAPH_MODELLING;
			break;
		default:
			g_set_error (error,
                                     GNOME_DB_GRAPH_ERROR,
                                     GNOME_DB_GRAPH_XML_LOAD_ERROR,
                                     _("Wrong 'type' attribute in <GNOME_DB_GRAPH>"));
                        return FALSE;
			break;
		}
                g_free (prop);
        }

	prop = xmlGetProp (node, "object");
	if (prop) {
		if (!graph->priv->ref_object)
			init_ref_object (graph);

		g_signal_handlers_block_by_func (G_OBJECT (graph->priv->ref_object),
						 G_CALLBACK (ref_object_ref_lost_cb), graph);
		gnome_db_ref_base_set_ref_name (graph->priv->ref_object, 0/* FIXME */, REFERENCE_BY_XML_ID, prop);
		g_signal_handlers_unblock_by_func (G_OBJECT (graph->priv->ref_object),
						   G_CALLBACK (ref_object_ref_lost_cb), graph);
		g_free (prop);
	}

	/* items nodes */
	children = node->children;
	while (children) {
		if (!strcmp (children->name, "GNOME_DB_GRAPH_ITEM")) {
			GnomeDbGraphItem *item;
			
			item = GNOME_DB_GRAPH_ITEM (gnome_db_graph_item_new (gnome_db_base_get_dict (GNOME_DB_BASE (graph)), NULL));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (item), children, error)) {
				gnome_db_graph_add_item (graph, item);
				g_object_unref (G_OBJECT (item));
			}
			else
				return FALSE;
                }

		children = children->next;
	}

	if (!id) {
		g_set_error (error,
			     GNOME_DB_GRAPH_ERROR,
			     GNOME_DB_GRAPH_XML_LOAD_ERROR,
			     _("Missing Id attribute in <GNOME_DB_GRAPH>"));
		return FALSE;
        }

        return TRUE;
}

