/* gnome-db-ref-base.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-ref-base.h"
#include "marshal.h"
#include <string.h>
#include "gnome-db-xml-storage.h"
#include "gnome-db-database.h"
#include "gnome-db-table.h"
#include "gnome-db-table-field.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-condition.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-server.h"
#include "gnome-db-server-function.h"
#include "gnome-db-server-aggregate.h"
#include "gnome-db-custom-layout.h"

#include "gnome-db-qf-all.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-func.h"


/* 
 * Main static functions 
 */
static void gnome_db_ref_base_class_init (GnomeDbRefBaseClass * class);
static void gnome_db_ref_base_init (GnomeDbRefBase * srv);
static void gnome_db_ref_base_dispose (GObject   * object);
static void gnome_db_ref_base_finalize (GObject   * object);

static void gnome_db_ref_base_set_property (GObject              *object,
				      guint                 param_id,
				      const GValue         *value,
				      GParamSpec           *pspec);
static void gnome_db_ref_base_get_property (GObject              *object,
				      guint                 param_id,
				      GValue               *value,
				      GParamSpec           *pspec);
static GType handled_object_type (GType type);


/* When the DbTable or GnomeDbServerDataType is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbRefBase *ref);

#ifdef debug
static void gnome_db_ref_base_dump (GnomeDbRefBase *field, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	REF_FOUND,
	REF_LOST,
	LAST_SIGNAL
};

static gint gnome_db_ref_base_signals[LAST_SIGNAL] = { 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbRefBasePrivate
{
	gboolean               increase_ref_object;
	GnomeDbBase           *ref_object;
	GType                  requested_type;
	GnomeDbRefBaseType     ref_type;
	gchar                 *name;
	gboolean               block_signals;
};


/* module error */
GQuark gnome_db_ref_base_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_ref_base_error");
	return quark;
}


GType
gnome_db_ref_base_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbRefBaseClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_ref_base_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbRefBase),
			0,
			(GInstanceInitFunc) gnome_db_ref_base_init
		};		
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbRefBase", &info, 0);
	}
	return type;
}

static void
gnome_db_ref_base_class_init (GnomeDbRefBaseClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_ref_base_signals[REF_FOUND] =
		g_signal_new ("ref_found",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbRefBaseClass, ref_found),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	gnome_db_ref_base_signals[REF_LOST] =
		g_signal_new ("ref_lost",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbRefBaseClass, ref_lost),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->ref_found = NULL;
	class->ref_lost = NULL;

	object_class->dispose = gnome_db_ref_base_dispose;
	object_class->finalize = gnome_db_ref_base_finalize;

	/* Properties */
	object_class->set_property = gnome_db_ref_base_set_property;
	object_class->get_property = gnome_db_ref_base_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_ref_base_dump;
#endif
}

static void
gnome_db_ref_base_init (GnomeDbRefBase * gnome_db_ref_base)
{
	gnome_db_ref_base->priv = g_new0 (GnomeDbRefBasePrivate, 1);
	gnome_db_ref_base->priv->increase_ref_object = TRUE; /* the default is to use g_object_ref() and g_object_unref() on object */
	gnome_db_ref_base->priv->ref_object = NULL;
	gnome_db_ref_base->priv->requested_type = 0;
	gnome_db_ref_base->priv->ref_type = REFERENCE_BY_XML_ID;
	gnome_db_ref_base->priv->name = NULL;
	gnome_db_ref_base->priv->block_signals = FALSE;
}


/**
 * gnome_db_ref_base_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new GnomeDbRefBase object. This #GnomeDbRefBase object does itself increase the
 * reference count of the referenced object, so if all the reference count holders call g_object_unref()
 * on there referenced object, then that object will not be destroyed because this #GnomeDbRefBase still
 * has a reference on it. Use gnome_db_ref_base_new_no_ref_count() if you don't want to increase the reference
 * count of the referenced object.
 *
 * Returns: the new object
 */
GObject*
gnome_db_ref_base_new (GnomeDbDict *dict)
{
	GObject   *obj;
	GnomeDbRefBase *gnome_db_ref_base;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_REF_BASE_TYPE, "dict", ASSERT_DICT (dict), NULL);
	gnome_db_ref_base = GNOME_DB_REF_BASE (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_ref_base), 0);

	return obj;
}

/**
 * gnome_db_ref_base_new_no_ref_count
 * @dict: a #GnomeDbDict object
 *
 * Creates a new GnomeDbRefBase object. This #GnomeDbRefBase object does not
 * itself increase the reference count of the object it keeps a reference to,
 * which means that if all the reference count holders call g_object_unref()
 * on there referenced object, then that object will actually be destroyed
 * and a "ref_lost" signal will be emitted. Use gnome_db_ref_base_new() if you want to
 * increase the count of the referenced object.
 *
 * Returns: the new object
 */
GObject *
gnome_db_ref_base_new_no_ref_count (GnomeDbDict *dict)
{
	GObject *obj;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	obj = gnome_db_ref_base_new (ASSERT_DICT (dict));
	GNOME_DB_REF_BASE (obj)->priv->increase_ref_object = FALSE;

	return obj;
}

/**
 * gnome_db_ref_base_new_copy
 * @orig: a #GnomeDbRefBase object
 *
 * Creates a new GnomeDbRefBase object which is a copy of @orig. This is a copy constructor.
 *
 * Returns: the new object
 */
GObject *
gnome_db_ref_base_new_copy (GnomeDbRefBase *orig)
{
	GObject   *obj;
	GnomeDbRefBase *gnome_db_ref_base;

	g_return_val_if_fail (orig && IS_GNOME_DB_REF_BASE (orig), NULL);

	obj = g_object_new (GNOME_DB_REF_BASE_TYPE, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (orig)), NULL);
	gnome_db_ref_base = GNOME_DB_REF_BASE (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_ref_base), 0);

	if (orig->priv->ref_object) {
		GObject *obj = G_OBJECT (orig->priv->ref_object);

		if (orig->priv->increase_ref_object)
			g_object_ref (obj);

		gnome_db_base_connect_nullify (obj, G_CALLBACK (nullified_object_cb), gnome_db_ref_base);
		gnome_db_ref_base->priv->ref_object = GNOME_DB_BASE (obj);
		if (! gnome_db_ref_base->priv->block_signals) {
#ifdef debug_signal
			g_print (">> 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
			g_signal_emit (G_OBJECT (gnome_db_ref_base), gnome_db_ref_base_signals[REF_FOUND], 0);
#ifdef debug_signal
			g_print ("<< 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
		}
	}

	gnome_db_ref_base->priv->increase_ref_object = orig->priv->increase_ref_object;
	gnome_db_ref_base->priv->requested_type = orig->priv->requested_type;
	gnome_db_ref_base->priv->ref_type = orig->priv->ref_type;
	if (orig->priv->name) 
		gnome_db_ref_base->priv->name = g_strdup (orig->priv->name);

	return obj;	
}

static void 
nullified_object_cb (GObject *obj, GnomeDbRefBase *ref)
{
	g_return_if_fail (ref->priv->ref_object && (G_OBJECT (ref->priv->ref_object) == obj));

	g_signal_handlers_disconnect_by_func (G_OBJECT (ref->priv->ref_object),
					      G_CALLBACK (nullified_object_cb), ref);
	if (ref->priv->increase_ref_object)
		g_object_unref (ref->priv->ref_object);

	ref->priv->ref_object = NULL;
#ifdef debug_signal
	g_print (">> 'REF_LOST' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (ref), gnome_db_ref_base_signals[REF_LOST], 0);
#ifdef debug_signal
	g_print ("<< 'REF_LOST' from %s\n", __FUNCTION__);
#endif
}

static void
gnome_db_ref_base_dispose (GObject *object)
{
	GnomeDbRefBase *gnome_db_ref_base;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_REF_BASE (object));

	gnome_db_ref_base = GNOME_DB_REF_BASE (object);
	if (gnome_db_ref_base->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_ref_base->priv->ref_object)
			nullified_object_cb (G_OBJECT (gnome_db_ref_base->priv->ref_object), gnome_db_ref_base);
		if (gnome_db_ref_base->priv->name) {
			g_free (gnome_db_ref_base->priv->name);
			gnome_db_ref_base->priv->name = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_ref_base_finalize (GObject   * object)
{
	GnomeDbRefBase *gnome_db_ref_base;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_REF_BASE (object));

	gnome_db_ref_base = GNOME_DB_REF_BASE (object);
	if (gnome_db_ref_base->priv) {

		g_free (gnome_db_ref_base->priv);
		gnome_db_ref_base->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_ref_base_set_property (GObject              *object,
			  guint                 param_id,
			  const GValue         *value,
			  GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbRefBase *gnome_db_ref_base;

	gnome_db_ref_base = GNOME_DB_REF_BASE (object);
	if (gnome_db_ref_base->priv) {
		switch (param_id) {
		case PROP:
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_ref_base_get_property (GObject              *object,
			  guint                 param_id,
			  GValue               *value,
			  GParamSpec           *pspec)
{
	GnomeDbRefBase *gnome_db_ref_base;
	gnome_db_ref_base = GNOME_DB_REF_BASE (object);
	
	if (gnome_db_ref_base->priv) {
		switch (param_id) {
		case PROP:
			g_value_set_pointer (value, gnome_db_ref_base->priv->ref_object);
			break;
		}	
	}
}

/*
 * Returns 0 if @type is not an handled object type, and the real handled object type
 * if it is handled by the GnomeDbRefBase object
 */
static GType
handled_object_type (GType type)
{
	GType retval = 0;

	/* types accepted AS IS */
	if ((type == GNOME_DB_TABLE_TYPE) ||
	    (type == GNOME_DB_TABLE_FIELD_TYPE) ||
	    (type == GNOME_DB_QUERY_TYPE) ||
	    (type == GNOME_DB_TARGET_TYPE) ||
	    (type == GNOME_DB_FIELD_TYPE) ||
	    (type == GNOME_DB_SERVER_FUNCTION_TYPE) ||
	    (type == GNOME_DB_SERVER_AGGREGATE_TYPE) ||
	    (type == GNOME_DB_QFIELD_TYPE) ||
	    (type == GNOME_DB_CUSTOM_LAYOUT_TYPE))
		retval = type;

	/* type conversion */
	if ((type == GNOME_DB_QF_ALL_TYPE) ||
	    (type == GNOME_DB_QF_FIELD_TYPE) ||
	    (type == GNOME_DB_QF_VALUE_TYPE) ||
	    (type == GNOME_DB_QF_FUNC_TYPE))
		retval = GNOME_DB_QFIELD_TYPE;

	return retval;
}

/**
 * gnome_db_ref_base_set_ref_name
 * @ref: a #GnomeDbRefBase object
 * @ref_type: the requested referenced object's data type, or 0 if not specified and @type = REFERENCE_BY_XML_ID
 * @type: how to interpret the @name argument
 * @name: the name of the requested object
 *
 * Sets the type and XML Id of the object we want to reference. If any other object was already
 * referenced @ref is first reinitialized
 *
 * Rem: the name format is dependant on the type of object which is requested
 */
void
gnome_db_ref_base_set_ref_name (GnomeDbRefBase *ref, GType ref_type, GnomeDbRefBaseType type, const gchar *name)
{
	g_return_if_fail (ref && IS_GNOME_DB_REF_BASE (ref));
	g_return_if_fail (ref->priv);
	g_return_if_fail (name && *name);

	/* make sure we know how to retreive the requested object */
	if (!ref_type && (type == REFERENCE_BY_XML_ID) && (strlen (name) > 2)) {
		gchar *str = g_strdup (name), *ptr, *tok;
		gboolean has_sep = FALSE; /* TRUE if there is a ":" in the string */

		ptr = strtok_r (str, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (!ptr)
			ptr = str;
		else
			has_sep = TRUE;
		if ((strlen (ptr) > 2) || has_sep) {
			if ((*ptr == 'T') && (*(ptr+1) == 'V'))
				ref_type = GNOME_DB_TABLE_TYPE;
			if ((*ptr == 'F') && (*(ptr+1) == 'I'))
				ref_type = GNOME_DB_TABLE_FIELD_TYPE;
			if ((*ptr == 'D') && (*(ptr+1) == 'T'))
				ref_type = GNOME_DB_SERVER_DATA_TYPE_TYPE;
			if ((*ptr == 'P') && (*(ptr+1) == 'R'))
				ref_type = GNOME_DB_SERVER_FUNCTION_TYPE;
			if ((*ptr == 'A') && (*(ptr+1) == 'G'))
				ref_type = GNOME_DB_SERVER_AGGREGATE_TYPE;
			if ((*ptr == 'Q') && (*(ptr+1) == 'U'))
				ref_type = GNOME_DB_QUERY_TYPE;
			if ((*ptr == 'Q') && (*(ptr+1) == 'F'))
				ref_type = GNOME_DB_QFIELD_TYPE;
			if (has_sep && (*ptr == 'T'))
				ref_type = GNOME_DB_TARGET_TYPE;
			if (has_sep && (*ptr == 'C'))
				ref_type = GNOME_DB_CONDITION_TYPE;
			if ((*ptr == 'C') && (*(ptr+1) == 'L'))
				ref_type = GNOME_DB_CUSTOM_LAYOUT_TYPE;
		}
		g_free (str);
	}
	ref_type = handled_object_type (ref_type);
	g_return_if_fail (ref_type);

	/* Is there anything to change ? */
	if (ref->priv->name && name && !strcmp (ref->priv->name, name) &&
	    (ref_type == ref->priv->requested_type) && (ref->priv->ref_type == type)) {
		gnome_db_ref_base_activate (ref);
		return;
	}
	
	gnome_db_ref_base_deactivate (ref);
	
	ref->priv->ref_type = type;
	if (ref->priv->name) {
		g_free (ref->priv->name);
		ref->priv->name = NULL;
	}
	if (name)
		ref->priv->name = g_strdup (name);
	ref->priv->requested_type = ref_type;

	gnome_db_ref_base_activate (ref);
}

/**
 * gnome_db_ref_base_set_ref_object_type
 * @ref: a #GnomeDbRefBase object
 * @object: the object to keep a reference to
 * @type: the type of object requested: it must be a type in the class hierarchy of @object
 *
 * Rather than to set the XML Id of the object @ref has to reference, this function allows
 * to directly give the object, and specify the requested type, in case the object is known.
 */
void
gnome_db_ref_base_set_ref_object_type (GnomeDbRefBase *ref, GnomeDbBase *object, GType type)
{
	g_return_if_fail (ref && IS_GNOME_DB_REF_BASE (ref));
	g_return_if_fail (ref->priv);
	g_return_if_fail (object && IS_GNOME_DB_BASE (object));

	/* make sure we know how to retreive the requested object */
	type = handled_object_type (type);
	g_return_if_fail (type);

	gnome_db_ref_base_deactivate (ref);

	ref->priv->ref_type =  REFERENCE_BY_XML_ID;
	if (ref->priv->name) {
		g_free (ref->priv->name);
		ref->priv->name = NULL;
	}
	
	ref->priv->name = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (object));
	ref->priv->requested_type = type;

	/* Object treatment */
	if (ref->priv->increase_ref_object)
		g_object_ref (object);
	gnome_db_base_connect_nullify (object, G_CALLBACK (nullified_object_cb), ref);
	ref->priv->ref_object = object;
	if (! ref->priv->block_signals) {
#ifdef debug_signal
		g_print (">> 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (ref), gnome_db_ref_base_signals[REF_FOUND], 0);
#ifdef debug_signal
		g_print ("<< 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
	}
}

/**
 * gnome_db_ref_base_set_ref_object
 * @ref: a #GnomeDbRefBase object
 * @object: the object to keep a reference to
 *
 * Rather than to set the XML Id of the object @ref has to reference, this function allows
 * to directly give the object, in case the object is known.
 */
void
gnome_db_ref_base_set_ref_object (GnomeDbRefBase *ref, GnomeDbBase *object)
{
	GType ref_type;

	g_return_if_fail (object && IS_GNOME_DB_BASE (object));
	ref_type = G_OBJECT_TYPE (object);
	
	gnome_db_ref_base_set_ref_object_type (ref, object, ref_type);
}

/**
 * gnome_db_ref_base_replace_ref_object
 * @ref: a #GnomeDbRefBase object
 * @replacements: a #GHashTable
 *
 * Changes the referenced object with a new one: it looks into @replacements and if the
 * currently referenced object appears there as a key, then the reference is replaced with
 * the corresponding value.
 *
 * Nothing happens if @ref is not active, or if the referenced object is not found in @replacaments.
 */
void
gnome_db_ref_base_replace_ref_object (GnomeDbRefBase *ref, GHashTable *replacements)
{
	g_return_if_fail (ref && IS_GNOME_DB_REF_BASE (ref));
	g_return_if_fail (ref->priv);

	if (!replacements)
		return;
	
	if (ref->priv->ref_object) {
		GnomeDbBase *repl;
		repl = g_hash_table_lookup (replacements, ref->priv->ref_object);
		if (repl) {
			/* we don't want to send a "ref_dropped" signal here since we are just
			   changing the referenced object without really losing the reference */
			ref->priv->block_signals = TRUE;
			gnome_db_ref_base_set_ref_object_type (ref, repl, ref->priv->requested_type);		
			ref->priv->block_signals = FALSE;
		}
	}
}

/**
 * gnome_db_ref_base_get_ref_name
 * @ref: a #GnomeDbRefBase object
 * @ref_type: where to store the requested referenced object's data type, or NULL
 * @type: where to store how to interpret the returned name, or NULL
 *
 * Get the caracteristics of the requested object
 *
 * Returns: the name of the object (to be interpreted with @type)
 */
const gchar *
gnome_db_ref_base_get_ref_name (GnomeDbRefBase *ref, GType *ref_type, GnomeDbRefBaseType *type)
{
	g_return_val_if_fail (ref && IS_GNOME_DB_REF_BASE (ref), 0);
	g_return_val_if_fail (ref->priv, 0);

	if (ref_type)
		*ref_type = ref->priv->requested_type;
	if (type)
		*type = ref->priv->ref_type;

	return ref->priv->name;
}

/**
 * gnome_db_ref_base_get_ref_type
 * @ref: a #GnomeDbRefBase object
 *
 * Get the type of the referenced object by @ref (or the requested type if @ref is not active)
 *
 * Returns: the type
 */
GType
gnome_db_ref_base_get_ref_type (GnomeDbRefBase *ref)
{
	g_return_val_if_fail (ref && IS_GNOME_DB_REF_BASE (ref), 0);
	g_return_val_if_fail (ref->priv, 0);

	return ref->priv->requested_type;
}


/**
 * gnome_db_ref_base_get_ref_object
 * @ref: a #GnomeDbRefBase object
 *
 * Get the referenced object by @ref
 *
 * Returns: a pointer to the object, or NULL if the reference is not active
 */
GnomeDbBase *
gnome_db_ref_base_get_ref_object (GnomeDbRefBase *ref)
{
	g_return_val_if_fail (ref && IS_GNOME_DB_REF_BASE (ref), NULL);
	g_return_val_if_fail (ref->priv, NULL);

	if (!ref->priv->ref_object)
		gnome_db_ref_base_activate (ref);

	return ref->priv->ref_object;
}

/**
 * gnome_db_ref_base_activate
 * @ref: a #GnomeDbRefBase object
 *
 * Tries to "activate" a reference (to find the referenced object). Nothing happens if
 * the object is already activated
 *
 * Returns: TRUE on success
 */
gboolean
gnome_db_ref_base_activate (GnomeDbRefBase *ref)
{
	GnomeDbBase *obj = NULL;
	gboolean done = FALSE;

	g_return_val_if_fail (ref && IS_GNOME_DB_REF_BASE (ref), FALSE);
	g_return_val_if_fail (ref->priv, FALSE);

	if (!ref->priv->name)
		/* no object reference set, so we consider ourselve active */
		return TRUE;

	if (ref->priv->ref_object)
		return TRUE;

	/* Find the object */
	/* TABLE */
	if (!done && (ref->priv->requested_type == GNOME_DB_TABLE_TYPE)) {
		GnomeDbDatabase *db;
		GnomeDbTable *table;

		done = TRUE;
		db = gnome_db_dict_get_database (gnome_db_base_get_dict (GNOME_DB_BASE (ref)));
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			table = gnome_db_database_get_table_by_xml_id (db, ref->priv->name);
		else
			table = gnome_db_database_get_table_by_name (db, ref->priv->name);

		if (table)
			obj = GNOME_DB_BASE (table);
	}

	/* TABLE's FIELD */
	if (!done && (ref->priv->requested_type == GNOME_DB_TABLE_FIELD_TYPE)) {
		GnomeDbDatabase *db;
		GnomeDbTableField *field;

		done = TRUE;
		db = gnome_db_dict_get_database (gnome_db_base_get_dict (GNOME_DB_BASE (ref)));
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			field = gnome_db_database_get_field_by_xml_id (db, ref->priv->name);
		else
			field = gnome_db_database_get_field_by_name (db, ref->priv->name);

		if (field)
			obj = GNOME_DB_BASE (field);
	}

	/* QUERY */
	if (!done && (ref->priv->requested_type == GNOME_DB_QUERY_TYPE)) {
		GnomeDbQuery *query = NULL;

		done = TRUE;
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			query = gnome_db_dict_get_query_by_xml_id (gnome_db_base_get_dict (GNOME_DB_BASE (ref)), ref->priv->name);
		else
			TO_IMPLEMENT; /* not really needed, anyway */
		if (query)
			obj = GNOME_DB_BASE (query);
	}

	/* QUERY's FIELD */
	if (!done && (ref->priv->requested_type == GNOME_DB_QFIELD_TYPE)) {
		GnomeDbQuery *query;
		gchar *str, *ptr, *tok;

		str = g_strdup (ref->priv->name);
		ptr = strtok_r (str, ":", &tok);
		
		done = TRUE;
		query = gnome_db_dict_get_query_by_xml_id (gnome_db_base_get_dict (GNOME_DB_BASE (ref)), ptr);
		
		if (query) {
			GnomeDbField *field;
			
			field = gnome_db_entity_get_field_by_xml_id (GNOME_DB_ENTITY (query), ref->priv->name);
			if (field)
				obj = GNOME_DB_BASE (field);
		}
	}


	/* TARGET */
	if (!done && (ref->priv->requested_type == GNOME_DB_TARGET_TYPE)) {
		done = TRUE;

		if (ref->priv->ref_type == REFERENCE_BY_XML_ID) {
			gchar *str, *ptr, *tok;
			GnomeDbQuery *query;
			
			str = g_strdup (ref->priv->name);
			ptr = strtok_r (str, ":", &tok);
			query = gnome_db_dict_get_query_by_xml_id (gnome_db_base_get_dict (GNOME_DB_BASE (ref)), ptr);
			g_free (str);
			if (query) {
				GnomeDbTarget *target;
				
				target = gnome_db_query_get_target_by_xml_id (query, ref->priv->name);
				if (target)
					obj = GNOME_DB_BASE (target);
			}
		}
		else 
			TO_IMPLEMENT;
	}
	
	/* Generic FIELD (GnomeDbField interface)*/
	if (!done && (ref->priv->requested_type == GNOME_DB_FIELD_TYPE)) {
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID) {
			gchar *str, *ptr, *tok;
			str = g_strdup (ref->priv->name);
			ptr = strtok_r (str, ":", &tok);
			if ((*ptr == 'T') && (*(ptr+1) == 'V')) {
				/* we are really looking for a table's field */
				GnomeDbDatabase *db;
				GnomeDbTableField *field;
				
				done = TRUE;
				
				db = gnome_db_dict_get_database (gnome_db_base_get_dict (GNOME_DB_BASE (ref)));
				field = gnome_db_database_get_field_by_xml_id (db, ref->priv->name);
				
				if (field)
					obj = GNOME_DB_BASE (field);
			}
			
			if (!done && (*ptr == 'Q') && (*(ptr+1) == 'U')) {
				/* we are really looking for a query's field */
				GnomeDbQuery *query;
				
				done = TRUE;
				query = gnome_db_dict_get_query_by_xml_id (gnome_db_base_get_dict (GNOME_DB_BASE (ref)), ptr);
				
				if (query) {
					GnomeDbField *field;
					
					field = gnome_db_entity_get_field_by_xml_id (GNOME_DB_ENTITY (query), ref->priv->name);
					if (field)
						obj = GNOME_DB_BASE (field);
				}
			}
			g_free (str);		
		}
		else {
			done = TRUE;
			TO_IMPLEMENT;
		}
	}

	/* Server function */
	if (!done && (ref->priv->requested_type == GNOME_DB_SERVER_FUNCTION_TYPE)) {
		GnomeDbServerFunction *func = NULL;
		GnomeDbServer *srv;

		srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (ref)));
		done = TRUE;
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			func = gnome_db_server_get_function_by_xml_id (srv, ref->priv->name);
		else
			TO_IMPLEMENT; /* not really needed, anyway */
		if (func)
			obj = GNOME_DB_BASE (func);
	}

	/* Server aggregate */
	if (!done && (ref->priv->requested_type == GNOME_DB_SERVER_AGGREGATE_TYPE)) {
		GnomeDbServerAggregate *agg = NULL;
		GnomeDbServer *srv;

		srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (ref)));
		done = TRUE;
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			agg = gnome_db_server_get_aggregate_by_xml_id (srv, ref->priv->name);
		else
			TO_IMPLEMENT; /* not really needed, anyway */
		if (agg)
			obj = GNOME_DB_BASE (agg);
	}

	/* Custom Layout */
	if (!done && (ref->priv->requested_type == GNOME_DB_CUSTOM_LAYOUT_TYPE)) {
		GnomeDbCustomLayout *layout = NULL;

		done = TRUE;
		if (ref->priv->ref_type == REFERENCE_BY_XML_ID)
			layout = gnome_db_dict_get_layout_by_xml_id (gnome_db_base_get_dict (GNOME_DB_BASE (ref)), ref->priv->name);
		else
			TO_IMPLEMENT; /* not really needed, anyway */
		if (layout)
			obj = GNOME_DB_BASE (layout);
	}

	/* Object treatment */
	if (obj) {
		if (ref->priv->increase_ref_object)
			g_object_ref (obj);

		gnome_db_base_connect_nullify (obj, G_CALLBACK (nullified_object_cb), ref);
		ref->priv->ref_object = obj;
#ifdef debug_signal
		g_print (">> 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (ref), gnome_db_ref_base_signals[REF_FOUND], 0);
#ifdef debug_signal
		g_print ("<< 'REF_FOUND' from %s\n", __FUNCTION__);
#endif
	}

	return ref->priv->ref_object ? TRUE : FALSE;
}

/**
 * gnome_db_ref_base_deactivate
 * @ref: a #GnomeDbRefBase object
 *
 * Desctivates the object (loses the reference to the object)
 */
void
gnome_db_ref_base_deactivate (GnomeDbRefBase *ref)
{
	g_return_if_fail (ref && IS_GNOME_DB_REF_BASE (ref));
	g_return_if_fail (ref->priv);

	if (!ref->priv->name)
		return;
	
	if (! ref->priv->ref_object)
		return;

	g_signal_handlers_disconnect_by_func (G_OBJECT (ref->priv->ref_object),
					      G_CALLBACK (nullified_object_cb), ref);
	if (ref->priv->increase_ref_object)
		g_object_unref (ref->priv->ref_object);

	ref->priv->ref_object = NULL;

	if (! ref->priv->block_signals) {
#ifdef debug_signal
		g_print (">> 'REF_LOST' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (ref), gnome_db_ref_base_signals[REF_LOST], 0);
#ifdef debug_signal
		g_print ("<< 'REF_LOST' from %s\n", __FUNCTION__);
#endif
	}
}

/**
 * gnome_db_ref_base_is_active
 * @ref: a #GnomeDbRefBase object
 *
 * Find wether @ref is active
 *
 * Returns: TRUE if @ref is active
 */
gboolean
gnome_db_ref_base_is_active (GnomeDbRefBase *ref)
{
	g_return_val_if_fail (ref && IS_GNOME_DB_REF_BASE (ref), FALSE);
	g_return_val_if_fail (ref->priv, FALSE);
	
	if (!ref->priv->name)
		/* no object reference set, so we consider ourselve active */
		return TRUE;

	return ref->priv->ref_object ? TRUE : FALSE;
}


#ifdef debug
static void
gnome_db_ref_base_dump (GnomeDbRefBase *ref, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (ref && IS_GNOME_DB_REF_BASE (ref));
	g_return_if_fail (ref->priv);

	/* string for the offset */
	str = g_new0 (gchar, offset+1);
	for (i=0; i<offset; i++)
		str[i] = ' ';
	str[offset] = 0;

	if (ref->priv->ref_object) {
		g_print ("%s" D_COL_H1 "GnomeDbRefBase" D_COL_NOR " Active, points to id='%s': %p\n", str, 
			 ref->priv->name, ref->priv->ref_object);
		/*gnome_db_base_dump (GNOME_DB_BASE (ref->priv->ref_object), offset);*/
	}
	else 
		g_print ("%s" D_COL_ERR "BaseRef to id '%s' not active\n" D_COL_NOR, str, ref->priv->name);
}
#endif

