/* sel-functions.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-functions.h"
#include <libgnomedb/libgnomedb.h>
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-server-function.h"

/*
 *
 * Module for functions
 *
 */
static void         module_functions_fill_model (Module *module);
static void         module_functions_free (Module *module);
static const gchar *module_functions_col_name (Module *module, guint colno);

Module *
sel_module_functions_new (GnomeDbSelector *mgsel, gboolean insert_header, 
			  GtkTreeIter *iter, gpointer data)
{
	Module *module;

	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_functions_fill_model;
	module->free = module_functions_free;
	module->col_name = module_functions_col_name;
	module->obj_manager = NULL;
	module->model_store_data = NULL;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_PROCEDURES);

		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, NAME_COLUMN, 
				    _("Functions"), PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);		
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;	
}


static GSList *module_functions_get_objects_list (Module *module);
static gchar  *module_functions_get_extended_name (GObject *obj);
static void
module_functions_fill_model (Module *module)
{
	GnomeDbServer *srv;
	GtkTreeModel *model;
	GdkPixbuf *pixbuf_func = NULL;

	srv = gnome_db_dict_get_server (module->selector->priv->dict);
	pixbuf_func = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-procedures_16x16.png");

	/* Module's private data */
	module->mod_data = g_new0 (ModNameGroupData, 1);
	GROUP_DATA (module)->manager = G_OBJECT (srv);
	GROUP_DATA (module)->manager_weak_refed = FALSE;
	GROUP_DATA (module)->obj_pixbuf = pixbuf_func;
	GROUP_DATA (module)->get_objects_list = module_functions_get_objects_list;
	GROUP_DATA (module)->get_extended_name = module_functions_get_extended_name;
	
	/* Initial model filling */
	model = module->selector->priv->model;
	name_group_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (G_OBJECT (srv), "data_function_added",
			  G_CALLBACK (name_group_obj_added_cb), module);
	g_signal_connect (G_OBJECT (srv), "data_function_removed",
			  G_CALLBACK (name_group_obj_removed_cb), module);
	g_signal_connect (G_OBJECT (srv), "data_function_updated",
			  G_CALLBACK (name_group_obj_updated_cb), module);
	g_signal_connect (G_OBJECT (srv), "data_update_started",
			  G_CALLBACK (name_group_update_started_cb), module);
	g_signal_connect (G_OBJECT (srv), "data_update_finished",
			  G_CALLBACK (name_group_update_finished_cb), module);
}

static GSList *
module_functions_get_objects_list (Module *module)
{
	g_return_val_if_fail (GROUP_DATA (module)->manager, NULL);
	g_return_val_if_fail (IS_GNOME_DB_SERVER (GROUP_DATA (module)->manager), NULL);

	return gnome_db_server_get_functions (GNOME_DB_SERVER (GROUP_DATA (module)->manager));
}

static gchar *
module_functions_get_extended_name (GObject *obj)
{
	GnomeDbServerFunction *func;
	GString *string;
	const GSList *args;
	gchar *retval;
	gboolean firstarg = TRUE;

	g_return_val_if_fail (obj && IS_GNOME_DB_SERVER_FUNCTION (obj), NULL);

	func = GNOME_DB_SERVER_FUNCTION (obj);
	string = g_string_new (gnome_db_server_function_get_sqlname (GNOME_DB_SERVER_FUNCTION (func)));
	args = gnome_db_server_function_get_arg_types (func);
	g_string_append (string, " (");
	while (args) {
		if (firstarg)
			firstarg = FALSE;
		else
			g_string_append (string, ", ");
		if (args->data)
			g_string_append (string,
					 gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (args->data)));
		else
			g_string_append (string, "*");
		args = g_slist_next (args);
	}
	g_string_append (string, ")");
	retval = string->str;
	g_string_free (string, FALSE);

	return retval;
}

static void
module_functions_free (Module *module)
{
	GnomeDbServer *srv;
	srv = gnome_db_dict_get_server (module->selector->priv->dict);

	if (srv) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (srv),
						      G_CALLBACK (name_group_obj_added_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (srv),
						      G_CALLBACK (name_group_obj_removed_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (srv),
						      G_CALLBACK (name_group_obj_updated_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (srv),
						      G_CALLBACK (name_group_update_started_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (srv),
						      G_CALLBACK (name_group_update_finished_cb), module);
	}
	if (module->iter)
		gtk_tree_iter_free (module->iter);
	name_group_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_functions_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Function");
		break;
	default:
		return NULL;
		break;
	}
}
