/* gnome-db-model-wrapper.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-model-wrapper.h"
#include "gnome-db-data-model.h"
#include "gnome-db-parameter.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-wrapper-field.h"
#include "gnome-db-data-set.h"
#include <libgda/gda-data-model.h>

/* 
 * Main static functions 
 */
static void gnome_db_model_wrapper_class_init (GnomeDbModelWrapperClass * class);
static void gnome_db_model_wrapper_init (GnomeDbModelWrapper * srv);
static void gnome_db_model_wrapper_dispose (GObject   * object);
static void gnome_db_model_wrapper_finalize (GObject   * object);

static void gnome_db_model_wrapper_set_property (GObject              *object,
						 guint                 param_id,
						 const GValue         *value,
						 GParamSpec           *pspec);
static void gnome_db_model_wrapper_get_property (GObject              *object,
						 guint                 param_id,
						 GValue               *value,
						 GParamSpec           *pspec);
#ifdef debug
static void gnome_db_model_wrapper_dump (GnomeDbModelWrapper *wrapper, guint offset);
#endif

/* GnomeDbDataModel interface */
static void              gnome_db_model_wrapper_model_init             (GnomeDbDataModelIface *iface);
static GObject          *gnome_db_model_wrapper_model_copy             (GnomeDbDataModel *iface, GHashTable *replacements);
static gboolean          gnome_db_model_wrapper_model_refresh          (GnomeDbDataModel *iface, GError **error);
static GnomeDbDataSet   *gnome_db_model_wrapper_model_get_params       (GnomeDbDataModel *iface);
static GnomeDbDataSet   *gnome_db_model_wrapper_model_get_new_data_set (GnomeDbDataModel *iface);
static gboolean          gnome_db_model_wrapper_model_sync_data_set    (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint row);
static GnomeDbParameter *gnome_db_model_wrapper_model_get_param_at_col (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint col);
static gint              gnome_db_model_wrapper_model_get_col_at_param (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, 
									GnomeDbParameter *param);
static GSList           *gnome_db_model_wrapper_model_get_key_columns  (GnomeDbDataModel *iface, GnomeDbDataSet *data_set);


/* GdaDataModel interface */
static void                 gnome_db_model_wrapper_data_model_init (GdaDataModelIface *iface);
static gint                 gnome_db_model_wrapper_get_n_rows      (GdaDataModel *model);
static gint                 gnome_db_model_wrapper_get_n_columns   (GdaDataModel *model);
static GdaColumn  *gnome_db_model_wrapper_describe_column (GdaDataModel *model, gint col);
static const gchar         *gnome_db_model_wrapper_get_column_title(GdaDataModel *model, gint col);
static void                 gnome_db_model_wrapper_set_column_title(GdaDataModel *model, gint col, const gchar *title);
static gint                 gnome_db_model_wrapper_get_column_pos  (GdaDataModel *model, const gchar *title);
static const GdaRow        *gnome_db_model_wrapper_get_row         (GdaDataModel *model, gint row);
static const GdaValue      *gnome_db_model_wrapper_get_value_at    (GdaDataModel *model, gint col, gint row);
static gboolean             gnome_db_model_wrapper_is_updatable    (GdaDataModel *model);
static gboolean             gnome_db_model_wrapper_has_changed     (GdaDataModel *model);
static void                 gnome_db_model_wrapper_begin_changes   (GdaDataModel *model);
static gboolean             gnome_db_model_wrapper_commit_changes  (GdaDataModel *model);
static gboolean             gnome_db_model_wrapper_cancel_changes  (GdaDataModel *model);
static const GdaRow        *gnome_db_model_wrapper_append_row      (GdaDataModel *model, const GList *values);
static gboolean             gnome_db_model_wrapper_remove_row      (GdaDataModel *model, const GdaRow *row);
static gboolean             gnome_db_model_wrapper_update_row      (GdaDataModel *model, const GdaRow *row);
static gboolean             gnome_db_model_wrapper_append_column   (GdaDataModel *model, const GdaColumn *attrs);
static gboolean             gnome_db_model_wrapper_update_column   (GdaDataModel *model, gint col,
                                                                 const GdaColumn *attrs);
static gboolean             gnome_db_model_wrapper_remove_column   (GdaDataModel *model, gint col);
static void                 gnome_db_model_wrapper_set_notify      (GdaDataModel *model, gboolean do_notify_changes);
static gboolean             gnome_db_model_wrapper_get_notify      (GdaDataModel *model);
static gboolean             gnome_db_model_wrapper_set_command     (GdaDataModel *model, const gchar *txt, GdaCommandType type);
static const gchar         *gnome_db_model_wrapper_get_command     (GdaDataModel *model, GdaCommandType *type);

/* Entity interface */
static void             gnome_db_wrapper_entity_init         (GnomeDbEntityIface *iface);
static gboolean         gnome_db_wrapper_has_field           (GnomeDbEntity *iface, GnomeDbField *field);
static GSList          *gnome_db_wrapper_get_fields          (GnomeDbEntity *iface);
static GnomeDbField    *gnome_db_wrapper_get_field_by_name   (GnomeDbEntity *iface, const gchar *name);
static GnomeDbField    *gnome_db_wrapper_get_field_by_index  (GnomeDbEntity *iface, gint index);
static gint             gnome_db_wrapper_get_field_index     (GnomeDbEntity *iface, GnomeDbField *field);
static gboolean         gnome_db_wrapper_is_writable         (GnomeDbEntity *iface);
static GSList          *gnome_db_wrapper_get_parameters      (GnomeDbEntity *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass    *parent_class = NULL;

static void             make_entity_fields (GnomeDbModelWrapper *wrapper);


/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbModelWrapperPrivate
{
	GdaDataModel    *model;
	GSList          *fields; /* list of GnomeDbWrapperField */
};


/* module error */
GQuark gnome_db_model_wrapper_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_model_wrapper_error");
	return quark;
}


guint
gnome_db_model_wrapper_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbModelWrapperClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_model_wrapper_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbModelWrapper),
			0,
			(GInstanceInitFunc) gnome_db_model_wrapper_init
		};

		static const GInterfaceInfo gda_model_info = {
                        (GInterfaceInitFunc) gnome_db_model_wrapper_data_model_init,
                        NULL,
                        NULL
                };
		
		static const GInterfaceInfo model_info = {
                        (GInterfaceInitFunc) gnome_db_model_wrapper_model_init,
                        NULL,
                        NULL
                };

		static const GInterfaceInfo entity_info = {
			(GInterfaceInitFunc) gnome_db_wrapper_entity_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbModelWrapper", &info, 0);
		g_type_add_interface_static (type, GDA_TYPE_DATA_MODEL, &gda_model_info);
		g_type_add_interface_static (type, GNOME_DB_ENTITY_TYPE, &entity_info);
		g_type_add_interface_static (type, GNOME_DB_DATA_MODEL_TYPE, &model_info);
	}
	return type;
}

static void
gnome_db_model_wrapper_model_init (GnomeDbDataModelIface *iface)
{
	iface->copy = gnome_db_model_wrapper_model_copy;
	iface->refresh = gnome_db_model_wrapper_model_refresh;
	iface->get_params = gnome_db_model_wrapper_model_get_params;
	iface->get_new_data_set = gnome_db_model_wrapper_model_get_new_data_set;
	iface->sync_data_set = gnome_db_model_wrapper_model_sync_data_set;
	iface->get_param_at_col = gnome_db_model_wrapper_model_get_param_at_col;
	iface->get_col_at_param = gnome_db_model_wrapper_model_get_col_at_param;
	iface->get_key_columns = gnome_db_model_wrapper_model_get_key_columns;
	iface->get_status = NULL;
}

static void
gnome_db_model_wrapper_data_model_init (GdaDataModelIface *iface)
{
        iface->i_get_n_rows = gnome_db_model_wrapper_get_n_rows;
        iface->i_get_n_columns = gnome_db_model_wrapper_get_n_columns;
        iface->i_describe_column = gnome_db_model_wrapper_describe_column;
        iface->i_get_column_title = gnome_db_model_wrapper_get_column_title;
        iface->i_set_column_title = gnome_db_model_wrapper_set_column_title;
        iface->i_get_column_pos = gnome_db_model_wrapper_get_column_pos;
        iface->i_get_row = gnome_db_model_wrapper_get_row;
        iface->i_get_value_at = gnome_db_model_wrapper_get_value_at;
        iface->i_is_updatable = gnome_db_model_wrapper_is_updatable;
        iface->i_has_changed = gnome_db_model_wrapper_has_changed;
        iface->i_begin_changes = gnome_db_model_wrapper_begin_changes;
        iface->i_cancel_changes = gnome_db_model_wrapper_cancel_changes;
        iface->i_commit_changes = gnome_db_model_wrapper_commit_changes;
        iface->i_append_values = gnome_db_model_wrapper_append_row;
        iface->i_remove_row = gnome_db_model_wrapper_remove_row;
        iface->i_update_row = gnome_db_model_wrapper_update_row;
        iface->i_append_column = gnome_db_model_wrapper_append_column;
        iface->i_update_column = gnome_db_model_wrapper_update_column;
        iface->i_remove_column = gnome_db_model_wrapper_remove_column;
        iface->i_set_notify = gnome_db_model_wrapper_set_notify;
        iface->i_get_notify = gnome_db_model_wrapper_get_notify;
        iface->i_set_command = gnome_db_model_wrapper_set_command;
        iface->i_get_command = gnome_db_model_wrapper_get_command;
}

static void
gnome_db_wrapper_entity_init (GnomeDbEntityIface *iface)
{
	iface->has_field = gnome_db_wrapper_has_field;
	iface->get_fields = gnome_db_wrapper_get_fields;
	iface->get_field_by_name = gnome_db_wrapper_get_field_by_name;
	iface->get_field_by_xml_id = NULL;
	iface->get_field_by_index = gnome_db_wrapper_get_field_by_index;
	iface->get_field_index = gnome_db_wrapper_get_field_index;
	iface->add_field = NULL;
	iface->add_field_before = NULL;
	iface->swap_fields = NULL;
	iface->remove_field = NULL;
	iface->is_writable = gnome_db_wrapper_is_writable;
	iface->get_parameters = gnome_db_wrapper_get_parameters;
}

static void
gnome_db_model_wrapper_class_init (GnomeDbModelWrapperClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_model_wrapper_dispose;
	object_class->finalize = gnome_db_model_wrapper_finalize;

	/* Properties */
	object_class->set_property = gnome_db_model_wrapper_set_property;
	object_class->get_property = gnome_db_model_wrapper_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_model_wrapper_dump;
#endif
}

static void
gnome_db_model_wrapper_init (GnomeDbModelWrapper * wrapper)
{
	wrapper->priv = g_new0 (GnomeDbModelWrapperPrivate, 1);
	wrapper->priv->model = NULL;
	wrapper->priv->fields = NULL;
}

/**
 * gnome_db_model_wrapper_new
 * @dict: a #GnomeDbDict object
 * @model: the #GdaDataModel to wrap
 *
 * Creates a new #GnomeDbModelWrapper object
 *
 * Returns: the new object
 */
GObject   *
gnome_db_model_wrapper_new (GnomeDbDict *dict, GdaDataModel *model)
{
	GObject   *obj;
	GnomeDbModelWrapper *wrapper;
	
	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (model && GDA_IS_DATA_MODEL (model), NULL);

	obj = g_object_new (GNOME_DB_MODEL_WRAPPER_TYPE, "dict", ASSERT_DICT (dict), NULL);
	wrapper = GNOME_DB_MODEL_WRAPPER (obj);

	wrapper->priv->model = model;
	g_object_ref (G_OBJECT (model));

	make_entity_fields (wrapper);

	return obj;
}

static void
gnome_db_model_wrapper_dispose (GObject *object)
{
	GnomeDbModelWrapper *wrapper;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_MODEL_WRAPPER (object));

	wrapper = GNOME_DB_MODEL_WRAPPER (object);
	if (wrapper->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		/* model */
		if (wrapper->priv->model) {
			g_object_unref (wrapper->priv->model);
			wrapper->priv->model = NULL;
		}

		/* fields */
		while (wrapper->priv->fields) 
			gnome_db_base_nullify (GNOME_DB_BASE (wrapper->priv->fields->data));
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_model_wrapper_finalize (GObject   * object)
{
	GnomeDbModelWrapper *wrapper;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_MODEL_WRAPPER (object));

	wrapper = GNOME_DB_MODEL_WRAPPER (object);
	if (wrapper->priv) {
		g_free (wrapper->priv);
		wrapper->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_model_wrapper_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbModelWrapper *wrapper;

	wrapper = GNOME_DB_MODEL_WRAPPER (object);
	if (wrapper->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_model_wrapper_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbModelWrapper *wrapper;
	wrapper = GNOME_DB_MODEL_WRAPPER (object);
	
	if (wrapper->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			g_value_set_pointer (value, NULL);
			break;
		}	
	}
}

#ifdef debug
static void
gnome_db_model_wrapper_dump (GnomeDbModelWrapper *wrapper, guint offset)
{
	gchar *offstr, *str;
	gint n_cols, n_rows;
	gint *cols_size;
	gchar *sep_col  = " | ";
	gchar *sep_row  = "-+-";
	gchar sep_fill = '-';
	gint i, j;
	GdaDataModel *model;
	const GdaValue *value;
	
	g_return_if_fail (wrapper && IS_GNOME_DB_MODEL_WRAPPER (wrapper));
	g_return_if_fail (wrapper->priv);

        /* string for the offset */
        offstr = g_new0 (gchar, offset+1);
	memset (offstr, ' ', offset);

	g_print ("%s" D_COL_H1 "GnomeDbModelWrapper" D_COL_NOR " %p\n", offstr, wrapper);

	/* display fields information */
	if (wrapper->priv->fields) {
		GSList *list = wrapper->priv->fields;
		g_print ("%sFields: ", offstr);

		while (list) {
			g_print (" %p", list->data);
			list = g_slist_next (list);
		}
		g_print ("\n");
	}
	else
		g_print ("%sHas no field (yet)\n", offstr);
	

	model = wrapper->priv->model;
	if (model) {
		/* compute the columns widths: using column titles... */
		n_cols = gda_data_model_get_n_columns (model);
		n_rows = gda_data_model_get_n_rows (model);
		cols_size = g_new0 (gint, n_cols);
	
		for (i = 0; i < n_cols; i++) {
			str = gda_data_model_get_column_title (model, i);
			cols_size [i] = strlen (str);
		}

		/* ... and using column data */
		for (j = 0; j < n_rows; j++) {
			for (i = 0; i < n_cols; i++) {
				value = gda_data_model_get_value_at (model, i, j);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				cols_size [i] = MAX (cols_size [i], strlen (str));
				g_free (str);
			}
		}
	
		/* actual dumping of the contents: column titles...*/
		for (i = 0; i < n_cols; i++) {
			str = gda_data_model_get_column_title (model, i);
			if (i != 0)
				g_print ("%s", sep_col);
			g_print ("%*s", cols_size [i], str);
		}
		g_print ("\n");
		
		/* ... separation line ... */
		for (i = 0; i < n_cols; i++) {
			if (i != 0)
				g_print ("%s", sep_row);
			for (j = 0; j < cols_size [i]; j++)
				g_print ("%c", sep_fill);
		}
		g_print ("\n");

		/* ... and data */
		for (j = 0; j < n_rows; j++) {
			for (i = 0; i < n_cols; i++) {
				value = gda_data_model_get_value_at (model, i, j);
				str = value ? gda_value_stringify (value) : g_strdup ("_null_");
				if (i != 0)
					g_print ("%s", sep_col);
				g_print ("%*s", cols_size [i], str);
				g_free (str);
			}
			g_print ("\n");
		}
		g_free (cols_size);
	}
	else
		g_print ("%sNo data model present\n", offstr);

	g_free (offstr);
}
#endif


/*
 * GnomeDbDataModel interface implementation
 */
static GObject *
gnome_db_model_wrapper_model_copy (GnomeDbDataModel *iface, GHashTable *replacements)
{
	GObject *obj;
	GnomeDbModelWrapper *wrapper, *orig;

	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	orig = GNOME_DB_MODEL_WRAPPER (iface);
	g_return_val_if_fail (orig->priv, NULL);

	obj = g_object_new (GNOME_DB_MODEL_WRAPPER_TYPE, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (orig)), NULL);
	wrapper = GNOME_DB_MODEL_WRAPPER (obj);

	wrapper->priv->model = orig->priv->model;
	g_object_ref (wrapper->priv->model);

	return obj;
}

static gboolean
gnome_db_model_wrapper_model_refresh (GnomeDbDataModel *iface, GError **error)
{
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), FALSE);

	/* GdaDataModel does not support refresh */
	return TRUE;
}

static GnomeDbDataSet *
gnome_db_model_wrapper_model_get_params (GnomeDbDataModel *iface)
{
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);

	/* GdaDataModel does not support parameters */
	return NULL;
}

static GnomeDbDataSet *
gnome_db_model_wrapper_model_get_new_data_set (GnomeDbDataModel *iface)
{
	GnomeDbModelWrapper *wrapper;
	GnomeDbDataSet *data_set;
	GSList *params = NULL, *list;
	GObject *param;
	GnomeDbDict *dict;

	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);

	wrapper = GNOME_DB_MODEL_WRAPPER (iface);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (wrapper));
	
	make_entity_fields (wrapper);
	list = wrapper->priv->fields;
	while (list) {
		param = gnome_db_parameter_new (dict,
						gnome_db_field_get_data_type (GNOME_DB_FIELD (list->data)));
		g_object_set_data (G_OBJECT (param), "wrap:colfield", list->data);
		params = g_slist_prepend (params, param);
		/* set the parameter properties */
		gnome_db_base_set_name (GNOME_DB_BASE (param), gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
		gnome_db_base_set_description (GNOME_DB_BASE (param), 
					       gnome_db_base_get_description (GNOME_DB_BASE (list->data)));
		list = g_slist_next (list);
	}
	params = g_slist_reverse (params);
	data_set = GNOME_DB_DATA_SET (gnome_db_data_set_new (dict, params));

	/* get rid of the params list since we don't use them anymore */
	list = params;
        while (list) { 
                g_object_unref (G_OBJECT (list->data));
                list = g_slist_next (list);
        }
        g_slist_free (params);

	return data_set;
}

static gboolean
gnome_db_model_wrapper_model_sync_data_set (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint row)
{
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), FALSE);
	g_return_val_if_fail (IS_GNOME_DB_DATA_SET (data_set), FALSE);

	TO_IMPLEMENT;
	return TRUE;
}

static GnomeDbParameter *
gnome_db_model_wrapper_model_get_param_at_col (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, gint col)
{
	GnomeDbModelWrapper *wrapper;
	GnomeDbField *field;
	GnomeDbParameter *param = NULL;
	
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	wrapper = GNOME_DB_MODEL_WRAPPER (iface);
	g_return_val_if_fail (wrapper->priv, NULL);
	g_return_val_if_fail (data_set, NULL);

	make_entity_fields (wrapper);

	field = g_slist_nth_data (wrapper->priv->fields, col);
	if (field) {
		GSList *list = data_set->parameters;
		while (list && !param) {
			if (g_object_get_data (G_OBJECT (list->data), "wrap:colfield") == field)
				param = GNOME_DB_PARAMETER (list->data);
			list = g_slist_next (list);
		}
	}

	return param;
}

static gint
gnome_db_model_wrapper_model_get_col_at_param (GnomeDbDataModel *iface, GnomeDbDataSet *data_set, GnomeDbParameter *param)
{
	GnomeDbModelWrapper *wrapper;
	gint col = -1;
	GnomeDbField *field;

	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), -1);
	wrapper = GNOME_DB_MODEL_WRAPPER (iface);
	g_return_val_if_fail (wrapper->priv, -1);
	g_return_val_if_fail (data_set, -1);

	make_entity_fields (wrapper);
	
	field = g_object_get_data (G_OBJECT (param), "wrap:colfield");
	if (field) 
		col = g_slist_index (wrapper->priv->fields, field);

	if (col < 0)
		g_warning ("GnomeDbDataSet was not created using the gnome_db_data_model_get_new_data_set() method");
		
	return col;
}

static GSList *
gnome_db_model_wrapper_model_get_key_columns (GnomeDbDataModel *iface, GnomeDbDataSet *data_set)
{
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);

	TO_IMPLEMENT;
	return NULL;
}

/*
 * GdaDataModel interface implementation
 */
static gint
gnome_db_model_wrapper_get_n_rows (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), -1);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_rows)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_rows) (gdamodel);
	else
		return -1;
}

static gint
gnome_db_model_wrapper_get_n_columns (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), -1);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_columns)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_n_columns) (gdamodel);
	else
		return -1;
}

static GdaColumn *
gnome_db_model_wrapper_describe_column (GdaDataModel *model, gint col)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_describe_column)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_describe_column) (gdamodel, col);
	else
		return NULL;
}

static void
gnome_db_model_wrapper_set_column_title (GdaDataModel *model, gint col, const gchar *title)
{
	GdaDataModel *gdamodel;
	g_return_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model));

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_column_title)
		(GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_column_title) (gdamodel, col, title);
}

static const gchar *
gnome_db_model_wrapper_get_column_title (GdaDataModel *model, gint col)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_title)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_title) (gdamodel, col);
	else
		return NULL;
}

static gint
gnome_db_model_wrapper_get_column_pos (GdaDataModel *model, const gchar *title)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), -1);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_pos)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_column_pos) (gdamodel, title);
	else
		return -1;
}

static const GdaRow *
gnome_db_model_wrapper_get_row (GdaDataModel *model, gint row)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_row)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_row) (gdamodel, row);
	else
		return NULL;
}

static const GdaValue *
gnome_db_model_wrapper_get_value_at (GdaDataModel *model, gint col, gint row)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);
	
	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_value_at)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_value_at) (gdamodel, col, row);
	else
		return NULL;
}

static gboolean
gnome_db_model_wrapper_is_updatable (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_is_updatable)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_is_updatable) (gdamodel);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_has_changed (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_has_changed)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_has_changed) (gdamodel);
	else
		return FALSE;
}

static void
gnome_db_model_wrapper_begin_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model));

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_begin_changes)
		(GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_begin_changes) (gdamodel);
}

static gboolean
gnome_db_model_wrapper_commit_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);
	
	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_commit_changes)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_commit_changes) (gdamodel);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_cancel_changes (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_cancel_changes)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_cancel_changes) (gdamodel);
	else
		return FALSE;
}

static const GdaRow *
gnome_db_model_wrapper_append_row (GdaDataModel *model, const GList *values)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);
	
	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_append_values)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_append_values) (gdamodel, values);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_remove_row (GdaDataModel *model, const GdaRow *row)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_remove_row)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_remove_row) (gdamodel, row);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_update_row (GdaDataModel *model, const GdaRow *row)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_update_row)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_update_row) (gdamodel, row);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_append_column (GdaDataModel *model, const GdaColumn *attrs)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_append_column)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_append_column) (gdamodel, attrs);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_update_column (GdaDataModel *model, gint col,
				      const GdaColumn *attrs)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_update_column)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_update_column) (gdamodel, col, attrs);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_remove_column (GdaDataModel *model, gint col)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_remove_column)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_remove_column) (gdamodel, col);
	else
		return FALSE;
}


static void
gnome_db_model_wrapper_set_notify (GdaDataModel *model, gboolean do_notify_changes)
{
	GdaDataModel *gdamodel;
	g_return_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model));

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_notify)
		(GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_notify) (gdamodel, do_notify_changes);
}

static gboolean
gnome_db_model_wrapper_get_notify (GdaDataModel *model)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_notify)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_notify) (gdamodel);
	else
		return FALSE;
}

static gboolean
gnome_db_model_wrapper_set_command (GdaDataModel *model, const gchar *txt, GdaCommandType type)
{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), FALSE);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_command)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_set_command) (gdamodel, txt, type);
	else
		return FALSE;
}

static const gchar *
gnome_db_model_wrapper_get_command (GdaDataModel *model, GdaCommandType *type)

{
	GdaDataModel *gdamodel;
	g_return_val_if_fail (IS_GNOME_DB_MODEL_WRAPPER (model), NULL);

	gdamodel = GNOME_DB_MODEL_WRAPPER (model)->priv->model;
	if (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_command)
		return (GDA_DATA_MODEL_GET_IFACE (gdamodel)->i_get_command) (gdamodel, type);
	else
		return FALSE;
}



/* 
 * GnomeDbEntity interface implementation
 */
static void
nullified_field_cb (GnomeDbField *field, GnomeDbModelWrapper *wrapper)
{
	wrapper->priv->fields = g_slist_remove (wrapper->priv->fields, field);
        g_signal_handlers_disconnect_by_func (G_OBJECT (field),
                                              G_CALLBACK (nullified_field_cb), wrapper);
#ifdef debug_signal
        g_print (">> 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit_by_name (G_OBJECT (wrapper), "field_removed", field);
#ifdef debug_signal
        g_print ("<< 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_object_unref (field);
}

static void
make_entity_fields (GnomeDbModelWrapper *wrapper)
{
	GdaColumn *attrs;

	if (wrapper->priv->fields)
		return;

	if (wrapper->priv->model) {
		gint i, n_cols;
		GnomeDbField *field;

		n_cols = gda_data_model_get_n_columns (wrapper->priv->model);
		for (i = 0; i< n_cols; i++) {
			field = GNOME_DB_FIELD (gnome_db_wrapper_field_new_in_model (wrapper, i));
			wrapper->priv->fields = g_slist_prepend (wrapper->priv->fields, field);

			attrs = gda_data_model_describe_column (wrapper->priv->model, i);
			gnome_db_base_set_name (GNOME_DB_BASE (field), gda_column_get_name (attrs));
			gnome_db_base_set_description (GNOME_DB_BASE (field), gda_column_get_caption (attrs));

			gnome_db_base_connect_nullify (field, G_CALLBACK (nullified_field_cb), wrapper);
		}
		wrapper->priv->fields = g_slist_reverse (wrapper->priv->fields);
	}

#ifdef debug_NO
	gnome_db_base_dump (GNOME_DB_BASE (wrapper), 5);
#endif
}

static gboolean
gnome_db_wrapper_has_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, FALSE);

	TO_IMPLEMENT;
	return FALSE;
}

static GSList *
gnome_db_wrapper_get_fields (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, NULL);

	TO_IMPLEMENT;
	return NULL;
}

static GnomeDbField *
gnome_db_wrapper_get_field_by_name (GnomeDbEntity *iface, const gchar *name)
{
	GnomeDbField *field = NULL;
	GSList *list;
	gchar *lcname = g_utf8_strdown (name, -1);

	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, NULL);

	TO_IMPLEMENT;
	return field;
}

static GnomeDbField *
gnome_db_wrapper_get_field_by_index (GnomeDbEntity *iface, gint index)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, NULL);
	g_return_val_if_fail (index >= 0, NULL);
	g_return_val_if_fail (index < g_slist_length (GNOME_DB_MODEL_WRAPPER (iface)->priv->fields), NULL);
	
	TO_IMPLEMENT;
	return NULL;
}

static gint
gnome_db_wrapper_get_field_index (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), -1);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, -1);

	TO_IMPLEMENT;
	return -1;
}


static gboolean
gnome_db_wrapper_is_writable (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, FALSE);
	
	TO_IMPLEMENT;
	return FALSE;
}

static GSList *
gnome_db_wrapper_get_parameters (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_MODEL_WRAPPER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_MODEL_WRAPPER (iface)->priv, NULL);
	
	return NULL;
}
