/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-i18n.h"
#include <gtk/gtkentry.h>
#include <gtk/gtktable.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-connection-properties.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbConnectionPropertiesPrivate {
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *connection_name;
	GtkWidget *connection_string;
	GtkWidget *connection_provider;
	GtkWidget *connection_username;
	GtkWidget *feature_list;
};

static void gnome_db_connection_properties_class_init (GnomeDbConnectionPropertiesClass *klass);
static void gnome_db_connection_properties_init       (GnomeDbConnectionProperties *props,
						       GnomeDbConnectionPropertiesClass *klass);
static void gnome_db_connection_properties_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
refresh_widget (GnomeDbConnectionProperties *props)
{
	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	gtk_entry_set_text (GTK_ENTRY (props->priv->connection_name),
			    gda_connection_get_dsn (props->priv->cnc));
	gtk_entry_set_text (GTK_ENTRY (props->priv->connection_string),
			    gda_connection_get_cnc_string (props->priv->cnc));
	gtk_entry_set_text (GTK_ENTRY (props->priv->connection_provider),
			    gda_connection_get_provider (props->priv->cnc));
	gtk_entry_set_text (GTK_ENTRY (props->priv->connection_username),
			    gda_connection_get_username (props->priv->cnc));
}

/*
 * GnomeDbConnectionProperties class implementation
 */

static void
gnome_db_connection_properties_class_init (GnomeDbConnectionPropertiesClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_connection_properties_finalize;
}

static void
gnome_db_connection_properties_init (GnomeDbConnectionProperties *props,
				     GnomeDbConnectionPropertiesClass *klass)
{
	GtkWidget *frame;
	GtkWidget *table;
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	/* allocate internal structure */
	props->priv = g_new0 (GnomeDbConnectionPropertiesPrivate, 1);
	props->priv->cnc = NULL;

	/* create widgets */
	frame = gnome_db_new_frame_widget (_("Settings"));
	gtk_box_pack_start (GTK_BOX (props), frame, TRUE, TRUE, 2);

	table = gnome_db_new_table_widget (6, 3, FALSE);
	gtk_container_add (GTK_CONTAINER (frame), table);

	label = gnome_db_new_label_widget (_("Data source name"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	props->priv->connection_name = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (table), props->priv->connection_name,
			  1, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Connection string"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	props->priv->connection_string = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (table), props->priv->connection_string,
			  1, 2, 1, 2, GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Provider"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	props->priv->connection_provider = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (table), props->priv->connection_provider,
			  1, 2, 2, 3, GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("User name"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 2, 2);
	props->priv->connection_username = gnome_db_new_entry_widget (0, FALSE);
	gtk_table_attach (GTK_TABLE (table), props->priv->connection_username,
			  1, 2, 3, 4, GTK_FILL, GTK_FILL, 2, 2);

	/* create feature list */
	frame = gnome_db_new_frame_widget (_("Features"));
	gtk_box_pack_start (GTK_BOX (props), frame, TRUE, TRUE, 2);

	props->priv->feature_list = gnome_db_new_grid_widget (NULL);
	gtk_container_add (GTK_CONTAINER (frame), props->priv->feature_list);
}

static void
gnome_db_connection_properties_finalize (GObject *object)
{
	GnomeDbConnectionProperties *props = (GnomeDbConnectionProperties *) object;

	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	/* free memory */
	if (GDA_IS_CONNECTION (props->priv->cnc)) {
		g_object_unref (G_OBJECT (props->priv->cnc));
		props->priv->cnc = NULL;
	}

	g_free (props->priv);
	props->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_connection_properties_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbConnectionPropertiesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_connection_properties_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbConnectionProperties),
			0,
			(GInstanceInitFunc) gnome_db_connection_properties_init
		};
		type = g_type_register_static (PARENT_TYPE,
					       "GnomeDbConnectionProperties",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_connection_properties_new
 * @cnc: a #GdaConnection object.
 *
 * Create a new #GnomeDbConnectionProperties widget.
 */
GtkWidget *
gnome_db_connection_properties_new (GdaConnection *cnc)
{
	GnomeDbConnectionProperties *props;

	props = g_object_new (GNOME_DB_TYPE_CONNECTION_PROPERTIES, NULL);
	if (GDA_IS_CONNECTION (cnc)) {
		g_object_ref (G_OBJECT (cnc));
		props->priv->cnc = cnc;
		refresh_widget (props);
	}

	return GTK_WIDGET (props);
}

/**
 * gnome_db_connection_properties_get_connection
 */
GdaConnection *
gnome_db_connection_properties_get_connection (GnomeDbConnectionProperties *props)
{
	g_return_val_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props), NULL);
	return props->priv->cnc;
}

/**
 * gnome_db_connection_properties_set_connection
 */
void
gnome_db_connection_properties_set_connection (GnomeDbConnectionProperties *props,
					       GdaConnection *cnc)
{
	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	if (GDA_IS_CONNECTION (props->priv->cnc))
		g_object_unref (G_OBJECT (props->priv->cnc));

	props->priv->cnc = cnc;
	if (GDA_IS_CONNECTION (props->priv->cnc)) {
		g_object_ref (G_OBJECT (props->priv->cnc));
		refresh_widget (props);
	}
}
