/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-config.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-data-source-selector.h>
#include <libgnomedb/gnome-db-dsn-config-druid.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomedb/gnome-db-login.h>
#include <libgnomedb/gnome-db-util.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktable.h>
#include <gtk/gtkstock.h>

struct _GnomeDbLoginPrivate {
	gchar *dsn_name;
	GtkWidget *dsn_entry;
	GtkWidget *create_dsn_button;
	gboolean hiding_button;
	GtkWidget *username_entry;
	GtkWidget *password_entry;
};

static void gnome_db_login_class_init   (GnomeDbLoginClass *klass);
static void gnome_db_login_init         (GnomeDbLogin *login, GnomeDbLoginClass *klass);
static void gnome_db_login_set_property (GObject *object,
					 guint paramid,
					 const GValue *value,
					 GParamSpec *pspec);
static void gnome_db_login_get_property (GObject *object,
					 guint param_id,
					 GValue *value,
					 GParamSpec *pspec);
static void gnome_db_login_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_DSN,
	PROP_USERNAME,
	PROP_PASSWORD
};

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

/*
 * GnomeDbLogin class implementation
 */

static void
gnome_db_login_class_init (GnomeDbLoginClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_login_set_property;
	object_class->get_property = gnome_db_login_get_property;
	object_class->finalize = gnome_db_login_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_DSN,
		g_param_spec_string ("dsn", NULL, NULL, NULL, G_PARAM_READABLE));
	g_object_class_install_property (
		object_class, PROP_USERNAME,
		g_param_spec_string ("username", NULL, NULL, NULL, G_PARAM_READABLE));
	g_object_class_install_property (
		object_class, PROP_PASSWORD,
		g_param_spec_string ("password", NULL, NULL, NULL, G_PARAM_READABLE));
}

static void
gnome_db_login_init (GnomeDbLogin *login, GnomeDbLoginClass *klass)
{
	/* allocate the internal structure */
	login->priv = g_new0 (GnomeDbLoginPrivate, 1);
	login->priv->dsn_name = NULL;
	login->priv->hiding_button = TRUE;
}

static void
gnome_db_login_set_property (GObject *object,
			     guint param_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	switch (param_id) {
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_login_get_property (GObject *object,
			       guint param_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	switch (param_id) {
	case PROP_DSN :
		g_value_set_string (value, gnome_db_login_get_dsn (login));
		break;
	case PROP_USERNAME :
		g_value_set_string (value, gnome_db_login_get_username (login));
		break;
	case PROP_PASSWORD :
		g_value_set_string (value, gnome_db_login_get_password (login));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_login_finalize (GObject *object)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	/* free memory */
	if (login->priv->dsn_name != NULL) {
		g_free (login->priv->dsn_name);
		login->priv->dsn_name = NULL;
	}

	g_free (login->priv);
	login->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_login_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbLoginClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_login_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbLogin),
			0,
			(GInstanceInitFunc) gnome_db_login_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbLogin", &info, 0);
	}
	return type;
}

static void
druid_finished_cb (GnomeDbDsnConfigDruid *druid, gboolean error, GnomeDbLogin *login)
{
	const GdaDataSourceInfo *dsn_info;

	if (!error) {
		dsn_info = gnome_db_dsn_config_druid_get_dsn (druid);
		if (dsn_info) {
			gnome_db_config_save_data_source (dsn_info->name,
							  dsn_info->provider,
							  dsn_info->cnc_string,
							  dsn_info->description,
							  dsn_info->username,
							  dsn_info->password);
		}
	}

	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (druid)));
	gtk_widget_set_sensitive (gtk_widget_get_toplevel (GTK_WIDGET (login)), TRUE);
}

static void
add_dsn_cb (GtkButton *button, GnomeDbLogin *login)
{
	GtkWidget *druid, *dialog;
	GdkPixbuf *icon;

	gtk_widget_set_sensitive (gtk_widget_get_toplevel (GTK_WIDGET (login)), FALSE);

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (_("New data source"),
					      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (login))),
					      0, NULL);
	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}

	druid = gnome_db_dsn_config_druid_new ();
	g_signal_connect (G_OBJECT (druid), "finished",
			  G_CALLBACK (druid_finished_cb), login);
	gtk_widget_show (druid);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), druid, TRUE, TRUE, 6);

	/* run the dialog */
	gtk_widget_show (dialog);
}

static GtkWidget *
gnome_db_login_construct (GnomeDbLogin *login, const gchar *dsn)
{
	GtkWidget *table;
	GtkWidget *label;

	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), GTK_WIDGET (login));

	if (dsn != NULL)
		login->priv->dsn_name = g_strdup (dsn);
	else
		login->priv->dsn_name = NULL;

	table = gnome_db_new_table_widget (3, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (login), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Data Source:"));
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_label_set_selectable (GTK_LABEL (label), FALSE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	login->priv->dsn_entry = gnome_db_data_source_selector_new (dsn);
	gtk_widget_show (login->priv->dsn_entry);
	gtk_table_attach (GTK_TABLE (table), login->priv->dsn_entry, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);

	login->priv->create_dsn_button = gnome_db_new_button_widget_from_stock (GTK_STOCK_ADD);
	g_signal_connect (G_OBJECT (login->priv->create_dsn_button), "clicked",
			  G_CALLBACK (add_dsn_cb), login);
	gtk_widget_hide (login->priv->create_dsn_button);
	gtk_table_attach (GTK_TABLE (table), login->priv->create_dsn_button, 2, 3, 0, 1,
			  GTK_FILL, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("Username:"));
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_label_set_selectable (GTK_LABEL (label), FALSE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	login->priv->username_entry = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), login->priv->username_entry, 1, 3, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);

	label = gnome_db_new_label_widget (_("Password:"));
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_label_set_selectable (GTK_LABEL (label), FALSE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
	login->priv->password_entry = gnome_db_new_entry_widget (0, TRUE);
	gtk_entry_set_visibility (GTK_ENTRY (login->priv->password_entry), FALSE);
	gtk_table_attach (GTK_TABLE (table), login->priv->password_entry, 1, 3, 2, 3,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);

	return GTK_WIDGET (login);
}

/**
 * gnome_db_login_new
 */
GtkWidget *
gnome_db_login_new (const gchar *dsn)
{
	GnomeDbLogin *login;

	login = g_object_new (GNOME_DB_TYPE_LOGIN, NULL);
	return gnome_db_login_construct (login, dsn);
}

/**
 * gnome_db_login_get_dsn
 */
const gchar *
gnome_db_login_get_dsn (GnomeDbLogin *login)
{
	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);

	if (login->priv->dsn_name)
		return (const gchar *) login->priv->dsn_name;

	return gnome_db_option_menu_get_selection (GTK_OPTION_MENU (login->priv->dsn_entry));
}

/**
 * gnome_db_login_set_dsn
 * @login: A #GnomeDbLogin widget.
 * @dsn: Data source name.
 *
 * Set the data source name on the given #GnomeDbLogin widget.
 */
void
gnome_db_login_set_dsn (GnomeDbLogin *login, const gchar *dsn)
{
	g_return_if_fail (GNOME_DB_IS_LOGIN (login));
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (login->priv->dsn_entry), dsn);
}

/**
 * gnome_db_login_get_username
 * @login: A #GnomeDbLogin widget.
 *
 * Get the value currently entered in the username text entry.
 *
 * Returns: the username being entered in the login widget.
 */
const gchar *
gnome_db_login_get_username (GnomeDbLogin *login)
{
	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);
	return gtk_entry_get_text (GTK_ENTRY (login->priv->username_entry));
}

/**
 * gnome_db_login_set_username
 * @login: a #GnomeDbLogin widget.
 * @username: username to display in the widget.
 *
 * Sets the user name to be displayed in the given #GnomeDbLogin widget.
 */
void
gnome_db_login_set_username (GnomeDbLogin *login, const gchar *username)
{
	g_return_if_fail (GNOME_DB_IS_LOGIN (login));
	gtk_entry_set_text (GTK_ENTRY (login->priv->username_entry), username);
}

/**
 * gnome_db_login_get_password
 */
const gchar *
gnome_db_login_get_password (GnomeDbLogin *login)
{
	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);
	return gtk_entry_get_text (GTK_ENTRY (login->priv->password_entry));
}

/**
 * gnome_db_login_set_password
 * @login: a #GnomeDbLogin widget.
 * @password: password to display in the widget.
 *
 * Sets the password to be displayed in the given #GnomeDbLogin widget.
 */
void
gnome_db_login_set_password (GnomeDbLogin *login, const gchar *password)
{
	g_return_if_fail (GNOME_DB_IS_LOGIN (login));
	gtk_entry_set_text (GTK_ENTRY (login->priv->password_entry), password);
}

/**
 * gnome_db_login_set_enable_create_button
 * @login: A #GnomeDbLogin widget.
 * @enable: Flag to enable/disable the button.
 *
 * Enable/disable the button for allowing the user to create new data sources
 * from the login widget. This button is disabled by default, and when enabled,
 * allows the user to start the database properties configuration applet.
 */
void
gnome_db_login_set_enable_create_button (GnomeDbLogin *login, gboolean enable)
{
	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	if (login->priv->hiding_button && enable) {
		login->priv->hiding_button = FALSE;
		gtk_widget_show (login->priv->create_dsn_button);
	} else if (!login->priv->hiding_button && !enable) {
		login->priv->hiding_button = TRUE;
		gtk_widget_hide (login->priv->create_dsn_button);
	}
}
