/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <glib/gfileutils.h>
#include <libgda/gda-util.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkseparatormenuitem.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkwindow.h>
#include <gobject/gmarshal.h>
#include <libgnomedb/gnome-db-form.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomedb/gnome-db-model.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-table-editor.h>
#ifdef BUILD_WITH_GNOME
#include <libgnomeui/gnome-file-entry.h>
#endif
#include "gnome-db-i18n.h"
#include "libgnomedb-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbGridPrivate {
	GdaDataModel *model;

	/* widgets */
	GtkWidget *title;
	GtkWidget *scroll;
	GtkWidget *tree_view;

	gchar *title_string;
	gboolean show_title;
	GtkSelectionMode selection_mode;

	GHashTable *row_data;
};

static void gnome_db_grid_class_init   (GnomeDbGridClass *klass);
static void gnome_db_grid_init         (GnomeDbGrid *grid, GnomeDbGridClass *klass);
static void gnome_db_grid_set_property (GObject *object,
					guint paramid,
					const GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_get_property (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec);
static void gnome_db_grid_finalize   (GObject *object);

enum {
	ROW_SELECTED,
	SELECTION_CLEARED,
	DOUBLE_CLICKED,
	CREATE_POPUP_MENU,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_MODEL
};

static gint gnome_db_grid_signals[LAST_SIGNAL] = { 0 };
static GdkAtom clipboard_atom = GDK_NONE;
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

#if 0
static void
menu_describe_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *description;
	GdkPixbuf *icon;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (
		_("Recordset description"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
		NULL);

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}

	description = gnome_db_table_editor_new ();
	gtk_widget_show (description);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), description, TRUE, TRUE, 0);

	/* run the dialog */
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}
#endif

static gboolean
confirm_overwrite (GtkWindow *parent, const gchar *path)
{
	GtkWidget *dialog, *button;
	gboolean yes;
	gchar *msg;

	msg = g_strdup_printf (_("File '%s' already exists.\n"
				 "Do you want to overwrite it?"), path);

	/* build the dialog */
	dialog = gnome_db_new_alert (
		NULL,
		GTK_MESSAGE_QUESTION,
		msg,
		_("If you choose yes, the contents will be lost."));
	button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      button,
				      GTK_RESPONSE_NO);
	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      gtk_button_new_from_stock (GTK_STOCK_YES),
				      GTK_RESPONSE_YES);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 GTK_RESPONSE_NO);

	/* run the dialog */
	gtk_widget_show_all (dialog);
	yes = gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES;

	/* free memory */
	gtk_widget_destroy (dialog);
	g_free (msg);
	return yes;
}
					    
#ifdef BUILD_WITH_GNOME
static void
save_as_response_cb (GtkDialog *dialog, guint response_id, gpointer user_data)
{
	GtkWidget *types;
	GtkWidget *sel_type;
	GtkWidget *to_tab;
	GtkWidget *to_comma;
	GtkWidget *to_xml;
	GtkWidget *filename;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	if (response_id == GTK_RESPONSE_OK) {
		gchar *body;
		gchar *path;

		types = g_object_get_data (G_OBJECT (dialog), "types");
		to_tab = g_object_get_data (G_OBJECT (dialog), "to_tab");
		to_comma = g_object_get_data (G_OBJECT (dialog), "to_comma");
		to_xml = g_object_get_data (G_OBJECT (dialog), "to_xml");
		filename = g_object_get_data (G_OBJECT (dialog), "filename");

		sel_type = gtk_menu_get_active (
			GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (types))));

		if (sel_type == to_tab)
			body = gda_data_model_to_tab_separated (grid->priv->model);
		else if (sel_type == to_comma)
			body = gda_data_model_to_comma_separated (grid->priv->model);
		else if (sel_type == to_xml)
			body = gda_data_model_to_xml (grid->priv->model, TRUE);
		else
			body = NULL;

		if (body) {
			path = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (filename), FALSE);
			if (path) {
				if (g_file_test (path, G_FILE_TEST_EXISTS)) {
					if (!confirm_overwrite (GTK_WINDOW (dialog), path)) {
						g_free (body);
						g_free (path);
						return;
					}
				}

				if (!gda_file_save (path, body, strlen (body))) {
					gnome_db_show_error (_("Could not save file %s"), path);
					g_free (body);
					g_free (path);
					return;
				}
				g_free (path);
			} else {
				gnome_db_show_error (_("You must specify a file name"));
				g_free (body);
				return;
			}
			g_free (body);
		} else
			gnome_db_show_error (_("Got empty file while converting the data"));
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
filename_changed_cb (GtkWidget *widget, GtkDialog *dialog)
{
	gboolean ok = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (widget), FALSE) != NULL;
	gtk_dialog_set_response_sensitive (dialog, GTK_RESPONSE_OK, ok);
}

static void
menu_save_as_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *filename;
	GtkWidget *types;
	GtkWidget *to_tab;
	GtkWidget *to_comma;
	GtkWidget *to_xml;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (GDA_IS_DATA_MODEL (grid->priv->model));

	/* create dialog box */
	dialog = gtk_dialog_new_with_buttons (
		_("Saving Model"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_SAVE, GTK_RESPONSE_OK,
		NULL);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK, FALSE);

	table = gnome_db_new_table_widget (2, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 6);
	label = gtk_label_new_with_mnemonic (_("File _name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	filename = gnome_db_new_file_entry_widget ("gnome-db-grid-save-as",
						   _("Select File"));
	g_signal_connect (G_OBJECT (filename), "changed",
			  G_CALLBACK (filename_changed_cb), dialog);
	g_object_set_data (G_OBJECT (dialog), "filename", filename);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), filename);
	gtk_table_attach (GTK_TABLE (table), filename, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

	label = gtk_label_new_with_mnemonic (_("File _type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	types = gnome_db_new_option_menu_widget ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), types);
	gtk_table_attach (GTK_TABLE (table), types, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	g_object_set_data (G_OBJECT (dialog), "types", types);
	to_tab = gnome_db_option_menu_add_item (GTK_OPTION_MENU (types),
						_("Tab-delimited"));
	g_object_set_data (G_OBJECT (dialog), "to_tab", to_tab);
	to_comma = gnome_db_option_menu_add_item (GTK_OPTION_MENU (types),
						  _("Comma-delimited"));
	g_object_set_data (G_OBJECT (dialog), "to_comma", to_comma);
	to_xml = gnome_db_option_menu_add_item (GTK_OPTION_MENU (types),
						_("XML"));
	gnome_db_option_menu_set_selection (GTK_OPTION_MENU (types), _("XML"));
	g_object_set_data (G_OBJECT (dialog), "to_xml", to_xml);

	/* run the dialog */
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (save_as_response_cb), grid);
	gtk_widget_show_all (dialog);
}
#endif

static void
menu_select_all_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_select_all (grid);
}

static void
menu_show_columns_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid;
	GtkCheckMenuItem *item;

	grid = (GnomeDbGrid *) user_data;
	item = (GtkCheckMenuItem *) widget;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (GTK_IS_CHECK_MENU_ITEM (item));

	gnome_db_grid_set_column_titles_visible (
		grid, 
		gtk_check_menu_item_get_active (item));
}

static void
menu_unselect_all_cb (GtkWidget *widget, gpointer user_data)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gnome_db_grid_unselect_all (grid);
}

#if 0
static void
menu_view_detail_cb (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *form;
	GdkPixbuf *icon;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* create the dialog */
	dialog = gtk_dialog_new_with_buttons (
		_("View detail"),
		GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (grid))), 0,
		GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
		NULL);

	icon = gdk_pixbuf_new_from_file (LIBGNOMEDB_ICONSDIR "/gnome-db.png", NULL);
	if (icon) {
		gtk_window_set_icon (GTK_WINDOW (dialog), icon);
		g_object_unref (icon);
	}

	form = gnome_db_form_new ();
	gtk_widget_show (form);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), form, TRUE, TRUE, 0);
	gnome_db_form_set_form_type (GNOME_DB_FORM (form), GNOME_DB_FORM_TYPE_NAVIGATOR);
	gnome_db_form_set_model (GNOME_DB_FORM (form), grid->priv->model);

	/* run the dialog */
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}
#endif

static void
model_row_removed_cb (GdaDataModel *model, gint row, gpointer user_data)
{
	gint i, rows;
	gpointer key, value;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	rows = gda_data_model_get_n_rows (grid->priv->model);
	for (i = row; i <= rows; i++) {
		if (g_hash_table_lookup_extended (grid->priv->row_data, GINT_TO_POINTER (i + 1), &key, &value)) {
			g_hash_table_remove (grid->priv->row_data, GINT_TO_POINTER (i + 1));
			if (i > row) {
				/* update the rows */
				g_hash_table_insert (grid->priv->row_data, GINT_TO_POINTER (i), value);
			}
		}
	}
}

static gint
popup_button_pressed_cb (GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	GtkWidget *menu;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);
	
	if (event->button != 3)
		return FALSE;

	/* create the menu */
	menu = gtk_menu_new ();
	gtk_menu_append (
		GTK_MENU (menu),
		gnome_db_new_menu_item (_("Select _All"),
					FALSE,
					G_CALLBACK (menu_select_all_cb),
					grid));	
	gtk_menu_append (
		GTK_MENU (menu),
		gnome_db_new_menu_item (_("_Clear Selection"),
					FALSE,
					G_CALLBACK (menu_unselect_all_cb),
					grid));
	gtk_menu_append (
		GTK_MENU (menu),
		gnome_db_new_check_menu_item (
			_("Show Column _Titles"),
			gnome_db_grid_get_column_titles_visible (grid),
			G_CALLBACK (menu_show_columns_cb),
			grid));

#ifdef BUILD_WITH_GNOME
	gtk_menu_append (GTK_MENU (menu), gtk_separator_menu_item_new ());
	gtk_menu_append (
		GTK_MENU (menu),
		gnome_db_new_menu_item (GTK_STOCK_SAVE_AS,
					TRUE,
				      	G_CALLBACK (menu_save_as_cb),
					grid));
#endif
#if 0
	add_popup_menu_item (GTK_MENU (menu), _("Describe"), G_CALLBACK (menu_describe_cb), grid);
	add_popup_menu_item (GTK_MENU (menu), _("View detail..."), G_CALLBACK (menu_view_detail_cb), grid);
#endif

	/* allow listeners to add their custom menu items */
	g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[CREATE_POPUP_MENU], 0, GTK_MENU (menu));
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, event->button, event->time);
	gtk_widget_show_all (menu);

	return TRUE;
}

static void
selection_foreach (GtkTreeModel *tree_model,
		   GtkTreePath *path,
		   GtkTreeIter *iter,
		   gpointer user_data)
{
	DataModelInfo *iter_info;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (iter != NULL);

	gtk_tree_model_get (tree_model, iter, 0, &iter_info, -1);
	if (iter_info) {
		g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[ROW_SELECTED],
			       0, iter_info->row);
	}
}

static void
selection_changed_cb (GtkTreeSelection *selection, gpointer user_data)
{
	GtkTreeModel *tree_model;
	GtkTreeIter iter;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (gtk_tree_selection_get_mode (selection)) {
	case GTK_SELECTION_MULTIPLE :
		gtk_tree_selection_selected_foreach (
			selection,
			(GtkTreeSelectionForeachFunc) selection_foreach,
			grid);
		break;
	default :
		if (gtk_tree_selection_get_selected (selection, &tree_model, &iter)) {
			DataModelInfo *iter_info;

			gtk_tree_model_get (tree_model, &iter, 0, &iter_info, -1);
			g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[ROW_SELECTED],
				       0, iter_info->row);
		}
		else
			g_signal_emit (G_OBJECT (grid),
				       gnome_db_grid_signals[SELECTION_CLEARED], 0);
		break;
	}
}

static void
tree_view_row_activated_cb (GtkTreeView *tree_view, GtkTreePath *path,
			    GtkTreeViewColumn *column, gpointer user_data)
{
	GtkTreeIter iter;
	DataModelInfo *info = NULL;
	GnomeDbGrid *grid = (GnomeDbGrid *) user_data;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (!gtk_tree_model_get_iter (gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->tree_view)), &iter, path))
		return;

	gtk_tree_model_get (gtk_tree_view_get_model (GTK_TREE_VIEW (grid->priv->tree_view)), &iter, 0, &info, -1);
	if (!info)
		return;

	g_signal_emit (G_OBJECT (grid), gnome_db_grid_signals[DOUBLE_CLICKED], 0, info->row);
}

/*
 * Private functions
 */

static void
setup_grid (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_signal_connect (G_OBJECT (grid->priv->tree_view), "row_activated",
			  G_CALLBACK (tree_view_row_activated_cb), grid);
	g_signal_connect (G_OBJECT (grid->priv->tree_view), "button_press_event",
			  G_CALLBACK (popup_button_pressed_cb), grid);

	/* setup selection stuff */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_set_mode (selection, grid->priv->selection_mode);
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (selection_changed_cb), grid);
}

/*
 * GnomeDbGrid class implementation
 */

static void
gnome_db_grid_class_init (GnomeDbGridClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_grid_set_property;
	object_class->get_property = gnome_db_grid_get_property;
	object_class->finalize = gnome_db_grid_finalize;
	klass->row_selected = NULL;
	klass->selection_cleared = NULL;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_MODEL,
		g_param_spec_object ("model", NULL, NULL,
				     GDA_TYPE_DATA_MODEL,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	gnome_db_grid_signals[ROW_SELECTED] =
		g_signal_new ("row_selected",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, row_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	gnome_db_grid_signals[SELECTION_CLEARED] =
		g_signal_new ("selection_cleared",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, selection_cleared),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gnome_db_grid_signals[DOUBLE_CLICKED] =
		g_signal_new ("double_clicked",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, double_clicked),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1, G_TYPE_INT);
	gnome_db_grid_signals[CREATE_POPUP_MENU] =
		g_signal_new ("create_popup_menu",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbGridClass, create_popup_menu),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1, G_TYPE_POINTER);

	/* clipboard atom */
	if (!clipboard_atom)
		clipboard_atom = gdk_atom_intern ("CLIPBOARD", FALSE);
}

static void
gnome_db_grid_init (GnomeDbGrid *grid, GnomeDbGridClass *klass)
{
	/* allocate the internal structure */
	grid->priv = g_new0 (GnomeDbGridPrivate, 1);
	grid->priv->show_title = FALSE;
	grid->priv->title_string = g_strdup (_("Title"));
	grid->priv->row_data = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* set up widgets */
	grid->priv->title = gnome_db_gray_bar_new (grid->priv->title_string);
	gtk_box_pack_start (GTK_BOX (grid), grid->priv->title, 0, 1, 2);

	grid->priv->scroll = gnome_db_new_scrolled_window_widget ();
	gtk_box_pack_start (GTK_BOX (grid), grid->priv->scroll, 1, 1, 0);

	grid->priv->tree_view = gnome_db_new_tree_view_widget (NULL);
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	setup_grid (grid);

	gnome_db_grid_set_selection_mode (grid, GTK_SELECTION_MULTIPLE);
}

static void
gnome_db_grid_set_property (GObject *object,
			    guint param_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		gnome_db_grid_set_model (grid, GDA_DATA_MODEL (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_get_property (GObject *object,
			    guint param_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	switch (param_id) {
	case PROP_MODEL :
		g_value_set_object (value, G_OBJECT (grid->priv->model));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_grid_finalize (GObject *object)
{
	GnomeDbGrid *grid = (GnomeDbGrid *) object;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* free memory */
	if (grid->priv->title_string) {
		g_free (grid->priv->title_string);
		grid->priv->title_string = NULL;
	}

	if (grid->priv->model) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (grid->priv->model), model_row_removed_cb, grid);
		g_object_unref (G_OBJECT (grid->priv->model));
		grid->priv->model = NULL;
	}

	g_hash_table_destroy (grid->priv->row_data);
	grid->priv->row_data = NULL;

	g_free (grid->priv);
	grid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_grid_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGridClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_grid_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGrid),
			0,
			(GInstanceInitFunc) gnome_db_grid_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbGrid", &info, 0);
	}
	return type;
}

/**
 * gnome_db_grid_new
 * 
 * Create a #GnomeDbGrid.
 *
 * Returns: a GtkWidget widget.
 */
GtkWidget *
gnome_db_grid_new (void)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_new_with_model
 * @model: a #GdaDataModel variable.
 * 
 * Create a #GnomeDbGrid with a #GdaDataModel.
 *
 * Returns: a GtkWidget widget.
 */
GtkWidget *
gnome_db_grid_new_with_model (GdaDataModel *model)
{
	GnomeDbGrid *grid;

	grid = g_object_new (GNOME_DB_TYPE_GRID, NULL);
	gnome_db_grid_set_model (grid, model);
	return GTK_WIDGET (grid);
}

/**
 * gnome_db_grid_get_title
 * @grid: a #GnomeDbGrid widget.
 *
 * Get the title for the given grid.
 *
 * Returns: the title associated with the grid.
 */
const gchar *
gnome_db_grid_get_title (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return (const gchar *) grid->priv->title_string;
}

/**
 * gnome_db_grid_set_title
 * @grid: a #GnomeDbGrid widget.
 * @title: new title.
 *
 * Set the title for the given grid.
 */
void
gnome_db_grid_set_title (GnomeDbGrid *grid, const gchar *title)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	if (grid->priv->title_string)
		g_free (grid->priv->title_string);

	grid->priv->title_string = g_strdup (title);

	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (grid->priv->title), title);
}

/**
 * gnome_db_grid_set_column_title
 * @grid a #GnomeDbGrid widget
 * @col the number of the column to change the title
 * @title the new title for this column
 *
 * Sets the title for a column
 */
void 
gnome_db_grid_set_column_title (GnomeDbGrid *grid, gint col, const gchar *title)
{
	GtkTreeViewColumn *column = NULL;
	gint number_columns;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (title != NULL);

	number_columns = gda_data_model_get_n_columns (grid->priv->model);
	if (col >= 0 && col < number_columns) {
		column = gtk_tree_view_get_column ( GTK_TREE_VIEW(grid->priv->tree_view), col);
		gtk_tree_view_column_set_title (column, title);
	}
}

/** 
 * gnome_db_grid_get_column_title
 * @grid a #GnomeDbGrid widget
 * @col the number of the column to get the title
 *
 * Get the title of a column
 * Returns: the title of a column.
 */
const gchar *
gnome_db_grid_get_column_title (GnomeDbGrid *grid, gint col)
{
	GtkTreeViewColumn *column = NULL;
	gchar *title = NULL;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->tree_view), col);
	title = gtk_tree_view_column_get_title (column);

	return title;
}


/**
 * gnome_db_grid_get_show_title
 * @grid: a #GnomeDbGrid widget.
 *
 * Get whether the title is being shown for the given grid.
 *
 * Returns: TRUE if the title is shown, FALSE if not.
 */
gboolean
gnome_db_grid_get_show_title (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);
	return grid->priv->show_title;
}

/**
 * gnome_db_grid_set_show_title
 * @grid: a #GnomeDbGrid widget.
 * @show: whether to show the title or not.
 *
 * Set the title displaying mode for the given grid.
 */
void
gnome_db_grid_set_show_title (GnomeDbGrid *grid, gboolean show)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	grid->priv->show_title = show;
	if (show)
		gtk_widget_show (grid->priv->title);
	else
		gtk_widget_hide (grid->priv->title);
}

/**
 * gnome_db_grid_set_title_icon_from_file
 * @grid: a #GnomeDbGrid widget.
 * @file: image filename.
 *
 * Set the icon for the given grid.
 */
void
gnome_db_grid_set_title_icon_from_file (GnomeDbGrid *grid, const gchar *file)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	gnome_db_gray_bar_set_icon_from_file (GNOME_DB_GRAY_BAR (grid->priv->title),
					      file);
}

/**
 * gnome_db_grid_set_title_icon_from_stock
 * @grid: a #GnomeDbGrid widget.
 * @stock_id: a stock icon name.
 * @size: a stock icon size.
 *
 * Set the icon using a stock icon for the given grid.
 */
void
gnome_db_grid_set_title_icon_from_stock (GnomeDbGrid *grid,
					 const gchar *stock_id,
					 GtkIconSize  size)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	
	gnome_db_gray_bar_set_icon_from_stock (GNOME_DB_GRAY_BAR (grid->priv->title),
					       stock_id, size);
}

/**
 * gnome_db_grid_set_show_title_icon
 * @grid: a #GnomeDbGrid widget.
 * @show: whether to show the icon or not.
 *
 * Set the icon displaying mode for the given grid.
 */
void
gnome_db_grid_set_show_title_icon (GnomeDbGrid *grid, gboolean show)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	
	gnome_db_gray_bar_set_show_icon (GNOME_DB_GRAY_BAR (grid->priv->title),
					 show);
}

/**
 * gnome_db_grid_get_show_title_icon
 * @grid: a #GnomeDbGrid widget.
 *
 * Get whether the icon is being shown for the given grid.
 *
 * Returns: TRUE if the icon is shown, FALSE if not.
 */
gboolean
gnome_db_grid_get_show_title_icon (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);
	
	return gnome_db_gray_bar_get_show_icon (GNOME_DB_GRAY_BAR (grid->priv->title));
}

/**
 * gnome_db_grid_get_model
 * @grid: a #GnomeDbGrid widget.
 *
 * Retrieve the #GdaDataModel associated with this #GnomeDbGrid.
 * 
 * Returns: a #GdaDataModel widget.
 */
GdaDataModel *
gnome_db_grid_get_model (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return grid->priv->model;
}

static void
dummy_remove_hash (gpointer key, gpointer value, gpointer user_data)
{
}

/**
 * gnome_db_grid_set_model
 * @grid: a #GnomeDbGrid widget.
 * @model: a #GdaDataModel widget.
 *
 * Set the new model for the given grid.
 */
void
gnome_db_grid_set_model (GnomeDbGrid *grid, GdaDataModel *model)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	/* clean up current data */
	if (GDA_IS_DATA_MODEL (grid->priv->model)) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (grid->priv->model), model_row_removed_cb, grid);
		g_object_unref (G_OBJECT (grid->priv->model));
	}

	g_hash_table_foreach_remove (grid->priv->row_data, (GHRFunc) dummy_remove_hash, NULL);

	grid->priv->model = model;
	if (GDA_IS_DATA_MODEL (model)) {
		g_object_ref (G_OBJECT (model));
		g_signal_connect (G_OBJECT (grid->priv->model), "row_removed",
				  G_CALLBACK (model_row_removed_cb), grid);
	} else {
		/* just empty the grid and return */
		gtk_tree_view_set_model (GTK_TREE_VIEW (grid->priv->tree_view), NULL);
		return;
	}

	/* redisplay the GtkTreeView */
	gtk_widget_destroy (grid->priv->tree_view);
	grid->priv->tree_view = GTK_WIDGET (gnome_db_model_to_gtk_tree_view (grid->priv->model));
	gtk_container_add (GTK_CONTAINER (grid->priv->scroll), grid->priv->tree_view);
	gtk_widget_show (grid->priv->tree_view);
	setup_grid (grid);
}

static void
foreach_selected_cb (GtkTreeModel *tree_model,
		     GtkTreePath *path,
		     GtkTreeIter *iter,
		     gpointer user_data)
{
	DataModelInfo *iter_info;
	GList **list = (GList **) user_data;

	gtk_tree_model_get (tree_model, iter, 0, &iter_info, -1);
	if (iter_info)
		*list = g_list_append (*list, GINT_TO_POINTER (iter_info->row));
}

/**
 * gnome_db_grid_get_selection
 * @grid: a #GnomeDbGrid widget.
 *
 * Returns the list of the currently selected rows in a
 * #GnomeDbGrid widget. The returned value is a list of integers,
 * which represent each of the selected rows.
 *
 * Returns: a GList of integers. This list should be freed (by calling
 * #g_list_free) when no longer needed.
 */
GList *
gnome_db_grid_get_selection (GnomeDbGrid *grid)
{	
	GtkTreeSelection *selection;
	GList *list = NULL;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_selected_foreach (
		selection,
		(GtkTreeSelectionForeachFunc) foreach_selected_cb,
		&list);

	return list;
}

/**
 * gnome_db_grid_foreach_selected
 * @grid: a #GnomeDbGrid widget.
 * @foreach_func: function to be called for each selected row.
 * @user_data: data to pass over to @foreach_func.
 *
 * Call the given function for each selected row in the #GnomeDbGrid
 * widget.
 */
void
gnome_db_grid_foreach_selected (GnomeDbGrid *grid,
				GnomeDbGridForeachFunc foreach_func,
				gpointer user_data)
{
	GList *selection;
	GList *l;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (foreach_func != NULL);

	selection = gnome_db_grid_get_selection (grid);
	for (l = selection; l != NULL; l = l->next)
		foreach_func (grid, GPOINTER_TO_INT (l->data), user_data);
}

/**
 * gnome_db_grid_select_all
 * @grid: a #GnomeDbGrid widget.
 *
 * Select all rows in the #GnomeDbGrid widget.
 */
void
gnome_db_grid_select_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gnome_db_grid_set_selection_mode (grid, GTK_SELECTION_MULTIPLE);
	gtk_tree_selection_select_all (selection);
}

/**
 * gnome_db_grid_unselect_all
 * @grid: a #GnomeDbGrid widget.
 * 
 * Unselect all rows of the #GnomeDbGrid widget.
 */
void
gnome_db_grid_unselect_all (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_unselect_all (selection);
}

/**
 * gnome_db_grid_get_selection_mode
 * @grid: a #GnomeDbGrid widget.
 *
 * Retrieve the selection mode of a #GnomeDbGrid widget.
 *
 * Returns: a GtkSelectionMode widget.
 */
GtkSelectionMode
gnome_db_grid_get_selection_mode (GnomeDbGrid *grid)
{
	GtkTreeSelection *selection;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), GTK_SELECTION_NONE);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	return gtk_tree_selection_get_mode (selection);
}

/**
 * gnome_db_set_selection_mode
 * @grid: a #GnomeDbGrid widget.
 * @mode: a GtkSelectionMode widget.
 * 
 * Set the new selection mode for this #GnomeDbGrid widget.
 */
void
gnome_db_grid_set_selection_mode (GnomeDbGrid *grid, GtkSelectionMode mode)
{
	GtkTreeSelection *selection;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (grid->priv->tree_view));
	gtk_tree_selection_set_mode (selection, mode);

	grid->priv->selection_mode = mode;
}

/**
 * gnome_db_grid_set_column_titles_visible
 * @grid: a #GnomeDbGrid widget.
 * @visible: a gboolean.
 * 
 * Hide or Show the titles of the #GnomeDbGrid widget.
 */
void
gnome_db_grid_set_column_titles_visible (GnomeDbGrid *grid, gboolean visible)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view), visible);
}

/**
 * gnome_db_grid_get_column_titles_visible
 * @grid: a #GnomeDbGrid widget.
 *
 * Retrieve a gboolean ( TRUE or FALSE ).
 */
gboolean
gnome_db_grid_get_column_titles_visible (GnomeDbGrid *grid)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);
	return gtk_tree_view_get_headers_visible (GTK_TREE_VIEW (grid->priv->tree_view));
}

/**
 * gnome_db_grid_get_row_data
 * @grid: a #GnomeDbGrid widget.
 * @row: the row to retrieve data for.
 *
 * Get the data associated (see #gnome_db_grid_set_row_data) with the given row.
 *
 * Returns: the data associated with the given row.
 */
gpointer
gnome_db_grid_get_row_data (GnomeDbGrid *grid, gint row)
{
	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), NULL);
	return g_hash_table_lookup (grid->priv->row_data, GINT_TO_POINTER (row + 1));
}

/**
 * gnome_db_grid_set_row_data
 * @grid: a #GnomeDbGrid widget.
 * @row: the row to set the data for.
 * @data: the data.
 *
 * Associate a given data pointer with the given row in the grid widget. This
 * allows you to store context-specific data for each row on the grid.
 */
void
gnome_db_grid_set_row_data (GnomeDbGrid *grid, gint row, gpointer data)
{
	g_return_if_fail (GNOME_DB_IS_GRID (grid));
	g_return_if_fail (row >= 0);
	g_return_if_fail (grid->priv->model != NULL);
	g_return_if_fail (row < gda_data_model_get_n_rows (grid->priv->model));

	if (g_hash_table_lookup (grid->priv->row_data, GINT_TO_POINTER (row + 1)))
		g_hash_table_remove (grid->priv->row_data, GINT_TO_POINTER (row + 1));

	if (data)
		g_hash_table_insert (grid->priv->row_data, GINT_TO_POINTER (row + 1), data);
}

/**
 * gnome_db_grid_set_column_visible
 * @grid: a #GnomeDbGrid widget.
 * @col: column number.
 * @visible: a gboolean.
 * 
 * Hide or Show the column "col" of the #GnomeDbGrid widget.
 */
void
gnome_db_grid_set_column_visible (GnomeDbGrid *grid, gint col, gboolean visible)
{
	GtkTreeViewColumn *column = NULL;
	gint number_columns;

	g_return_if_fail (GNOME_DB_IS_GRID (grid));

	number_columns = gda_data_model_get_n_columns (grid->priv->model);
	if (col >= 0 && col < number_columns) {
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->tree_view), col);
		gtk_tree_view_column_set_visible (column, visible);
	}
}

/**
 * gnome_db_grid_get_column_visible
 * @grid: a #GnomeDbGrid widget.
 * @col: column number.
 *
 * Retrieve a gboolean ( TRUE or FALSE ).
 */
gboolean
gnome_db_grid_get_column_visible (GnomeDbGrid *grid, gint col)
{
	GtkTreeViewColumn *column = NULL;
	gint number_columns;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), FALSE);

	number_columns = gda_data_model_get_n_columns (grid->priv->model);
	if (col >= 0 && col < number_columns) {
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (grid->priv->tree_view), col);
		return gtk_tree_view_column_get_visible (column);
	}

	return FALSE;
}

typedef struct {
	gint row;
	gpointer search_data;
} find_data_t;

static void
find_by_data_in_hash (gpointer key, gpointer value, gpointer data)
{
	find_data_t *fdata = data;

	if (fdata->row == -1) {
		if (fdata->search_data == value)
			fdata->row = GPOINTER_TO_INT (key) - 1;
	}
}

/**
 * gnome_db_grid_find_row_from_data
 * @grid: a #GnomeDbGrid widget.
 * @data: data to search for.
 *
 * Find the row number for the row to which the given data is associated.
 *
 * Returns: the row number if found, or -1 if not.
 */
gint
gnome_db_grid_find_row_from_data (GnomeDbGrid *grid, gpointer data)
{
	find_data_t fdata;

	g_return_val_if_fail (GNOME_DB_IS_GRID (grid), -1);
	g_return_val_if_fail (data != NULL, -1);

	fdata.row = -1;
	fdata.search_data = data;
	g_hash_table_foreach (grid->priv->row_data, (GHFunc) find_by_data_in_hash, &fdata);
	return fdata.row;
}
