/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtktable.h>
#include <libgnomedb/gnome-db-dsn-config.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbDsnConfigPrivate {
	GtkWidget *wname;
	GtkWidget *wprovider;
	GtkWidget *wstring;
	GtkWidget *wdesc;
	GtkWidget *wuser;
};

static void gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass);
static void gnome_db_dsn_config_init       (GnomeDbDsnConfig *config,
					    GnomeDbDsnConfigClass *klass);
static void gnome_db_dsn_config_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
fill_provider_menu (GnomeDbDsnConfig *config)
{
	GList *list;
	GList *l;

	list = gda_config_get_provider_list ();

	for (l = list; l != NULL; l = l->next) {
		GdaProviderInfo *prov_info;

		prov_info = (GdaProviderInfo *) l->data;
		if (prov_info) {
			gnome_db_option_menu_add_item (
				GTK_OPTION_MENU (config->priv->wprovider),
				prov_info->id);
		}
	}

	gda_config_free_provider_list (list);
}

/*
 * GnomeDbDsnConfig class implementation
 */

static void
gnome_db_dsn_config_class_init (GnomeDbDsnConfigClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_dsn_config_finalize;
}

static void
gnome_db_dsn_config_init (GnomeDbDsnConfig *config, GnomeDbDsnConfigClass *klass)
{
	GtkWidget *table;
	GtkWidget *label;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));

	/* allocate private structure */
	config->priv = g_new0 (GnomeDbDsnConfigPrivate, 1);

	/* set up widgets */
	table = gnome_db_new_table_widget (6, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (config), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Data source name"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	config->priv->wname = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), config->priv->wname, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Provider"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	config->priv->wprovider = gnome_db_new_option_menu_widget ();
	fill_provider_menu (config);
	gtk_table_attach (GTK_TABLE (table), config->priv->wprovider, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Connection string"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	config->priv->wstring = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), config->priv->wstring, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Description"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 2, 2);
	config->priv->wdesc = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), config->priv->wdesc, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Username"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 2, 2);
	config->priv->wuser = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), config->priv->wuser, 1, 2, 4, 5,
			  GTK_FILL, GTK_FILL, 2, 2);
}

static void
gnome_db_dsn_config_finalize (GObject *object)
{
	GnomeDbDsnConfig *config = (GnomeDbDsnConfig *) object;

	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));

	/* free memory */
	g_free (config->priv);
	config->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_dsn_config_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDsnConfigClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_dsn_config_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDsnConfig),
			0,
			(GInstanceInitFunc) gnome_db_dsn_config_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbDsnConfig",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_dsn_config_new
 */
GtkWidget *
gnome_db_dsn_config_new (void)
{
	GnomeDbDsnConfig *config;

	config = g_object_new (GNOME_DB_TYPE_DSN_CONFIG, NULL);
	return GTK_WIDGET (config);
}

/**
 * gnome_db_dsn_config_get_info
 */
GdaDataSourceInfo *
gnome_db_dsn_config_get_info (GnomeDbDsnConfig *config)
{
	GdaDataSourceInfo *dsn_info;

	g_return_val_if_fail (GNOME_DB_IS_DSN_CONFIG (config), NULL);

	dsn_info = g_new0 (GdaDataSourceInfo, 1);
	dsn_info->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wname)));
	dsn_info->provider = g_strdup (
		gnome_db_option_menu_get_selection (GTK_OPTION_MENU (config->priv->wprovider)));
	dsn_info->cnc_string = g_strdup (
		gtk_entry_get_text (GTK_ENTRY (config->priv->wstring)));
	dsn_info->description = g_strdup (
		gtk_entry_get_text (GTK_ENTRY (config->priv->wdesc)));
	dsn_info->username = g_strdup (gtk_entry_get_text (GTK_ENTRY (config->priv->wuser)));

	return dsn_info;
}

/**
 * gnome_db_dsn_config_set_info
 */
void
gnome_db_dsn_config_set_info (GnomeDbDsnConfig *config, const GdaDataSourceInfo *dsn_info)
{
	g_return_if_fail (GNOME_DB_IS_DSN_CONFIG (config));
	g_return_if_fail (dsn_info != NULL);

	gtk_entry_set_text (GTK_ENTRY (config->priv->wname), dsn_info->name);
	//wprovider
	gtk_entry_set_text (GTK_ENTRY (config->priv->wstring), dsn_info->cnc_string);
	gtk_entry_set_text (GTK_ENTRY (config->priv->wdesc), dsn_info->description);
	gtk_entry_set_text (GTK_ENTRY (config->priv->wuser), dsn_info->username);
}
