/* GNOME DB library
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-widget.h>
#include <libgnomedb/gnome-db-control-widget.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbControlWidgetPrivate {
	GtkWidget *bonobo_widget;
	gchar *iid;
};

static void gnome_db_control_widget_class_init (GnomeDbControlWidgetClass *klass);
static void gnome_db_control_widget_init       (GnomeDbControlWidget *widget,
						GnomeDbControlWidgetClass *klass);
static void gnome_db_control_widget_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * GnomeDbControlWidget class implementation
 */

static void
gnome_db_control_widget_class_init (GnomeDbControlWidgetClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_control_widget_finalize;
}

static void
gnome_db_control_widget_init (GnomeDbControlWidget *wid, GnomeDbControlWidgetClass *klass)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));

	wid->priv = g_new0 (GnomeDbControlWidgetPrivate, 1);
	wid->priv->bonobo_widget = NULL;
	wid->priv->iid = NULL;
}

static void
gnome_db_control_widget_finalize (GObject *object)
{
	GnomeDbControlWidget *wid = (GnomeDbControlWidget *) object;

	g_return_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid));

	/* free memory */
	if (wid->priv->bonobo_widget != NULL)
		g_object_unref (wid->priv->bonobo_widget);
	g_free (wid->priv->iid);

	g_free (wid->priv);
	wid->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_control_widget_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbControlWidgetClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_control_widget_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbControlWidget),
			0,
			(GInstanceInitFunc) gnome_db_control_widget_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbControlWidget",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_control_widget_construct
 */
GtkWidget *
gnome_db_control_widget_construct (GnomeDbControlWidget *wid,
				   const gchar *id,
				   Bonobo_UIContainer uic)
{
	g_return_val_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid),
			      GTK_WIDGET (wid));

	wid->priv->iid = g_strdup(id);

	/* load the Bonobo control */
	wid->priv->bonobo_widget = bonobo_widget_new_control (id, uic);
	if (!BONOBO_IS_WIDGET (wid->priv->bonobo_widget)) {
		wid->priv->bonobo_widget = NULL;
		g_object_unref (G_OBJECT (wid));
		return NULL;
	}

	gtk_widget_show (wid->priv->bonobo_widget);
	gtk_box_pack_start (GTK_BOX (wid), wid->priv->bonobo_widget, TRUE, TRUE, 0);

	return GTK_WIDGET (wid);
}

/**
 * gnome_db_control_widget_new
 * @id: ID of the Bonobo control to be loaded
 * @uic: a reference to the Bonobo_UIContainer to use for the control
 *
 * Creates a new widget that loads the Bonobo control specified by the
 * @id parameter.
 * The second argument, @uic, is needed to associate a Bonobo container
 * with the new control, so that menus and toolbars are merged. This is
 * easily done by using a #GnomeDbWindow for all your Bonobo container needs,
 * and then use the #gnome_db_window_get_ui_container to retrieve the
 * Bonobo_UIContainer associated with the #GnomeDbWindow widget.
 *
 * Returns: a pointer to the newly-created widget
 */
GtkWidget *
gnome_db_control_widget_new (const gchar *id, Bonobo_UIContainer uic)
{
	GnomeDbControlWidget* wid;
	
	g_return_val_if_fail(id != NULL, NULL);
	
	wid = g_object_new (GNOME_DB_TYPE_CONTROL_WIDGET, NULL);
	return gnome_db_control_widget_construct (wid, id, uic);
}

/**
 * gnome_db_control_widget_is_alive
 * @wid: a #GnomeDbControlWidget widget
 *
 * Pings the given #GnomeDbControlWidget widget to see if the underlying
 * CORBA server is still alive.
 *
 * Returns: TRUE if alive, FALSE if dead
 */
gboolean
gnome_db_control_widget_is_alive (GnomeDbControlWidget *wid)
{
	BonoboControlFrame *control_frame;
	CORBA_Object corba_object;
	CORBA_Environment ev;
	gboolean is_dead;

	g_return_val_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid), FALSE);

	control_frame = bonobo_widget_get_control_frame (
		BONOBO_WIDGET (wid->priv->bonobo_widget));
	corba_object = bonobo_control_frame_get_control (control_frame);

	CORBA_exception_init (&ev);
	is_dead = CORBA_Object_non_existent (corba_object, &ev);
	CORBA_exception_free (&ev);

	return !is_dead;
}

/**
 * gnome_db_control_widget_activate
 * @wid: a #GnomeDbControlWidget widget
 *
 * Activates the given #GnomeDbControlWidget widget. This means that the
 * underlying control's menus and toolbars will be merged with the menus
 * and toolbars of the container being used, generally a #GnomeDbWindow
 * widget.
 */
void
gnome_db_control_widget_activate (GnomeDbControlWidget *wid)
{
	BonoboControlFrame *control_frame;
	
	g_return_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid));
	
	control_frame = bonobo_widget_get_control_frame (
		BONOBO_WIDGET (wid->priv->bonobo_widget));
	bonobo_control_frame_set_autoactivate (control_frame, FALSE);
	bonobo_control_frame_control_activate (control_frame);
}

/**
 * gnome_db_control_widget_deactivate
 * @wid: a #GnomeDbControlWidget widget
 *
 * Deactivates the given #GnomeDbControlWidget widget. This means that all
 * of the control's menus and toolbars are removed from the container.
 */
void
gnome_db_control_widget_deactivate (GnomeDbControlWidget *wid)
{
	BonoboControlFrame *control_frame;
	
	g_return_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid));
	
	control_frame = bonobo_widget_get_control_frame (
		BONOBO_WIDGET (wid->priv->bonobo_widget));
	bonobo_control_frame_set_autoactivate (control_frame, FALSE);
	bonobo_control_frame_control_deactivate (control_frame);
}

/**
 * gnome_db_control_widget_get_id
 * @wid: a #GnomeDbControlWidget widget.
 *
 * Returns the ID of the underlying control.
 */
const gchar *
gnome_db_control_widget_get_id (GnomeDbControlWidget *wid)
{
	g_return_val_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid), NULL);
	return wid->priv->iid;
}

/**
 * gnome_db_control_widget_get_objref
 */
Bonobo_Unknown
gnome_db_control_widget_get_objref (GnomeDbControlWidget *wid)
{
	g_return_val_if_fail (GNOME_DB_IS_CONTROL_WIDGET (wid), CORBA_OBJECT_NIL);
	g_return_val_if_fail (BONOBO_IS_WIDGET (wid->priv->bonobo_widget), CORBA_OBJECT_NIL);

	return bonobo_widget_get_objref (BONOBO_WIDGET (wid->priv->bonobo_widget));
}
