/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gnome;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.EventType;
import org.gnu.glib.Type;
import org.gnu.gnome.event.IconEntryEvent;
import org.gnu.gnome.event.IconEntryListener;
import org.gnu.gtk.VBox;

/**
 * A GnomeIconEntry is a button that pops up a window displaying a
 * collection of icons. The user can select one of the icons found in
 * /usr/share/pixmaps or browse the file system to find icons in other
 * directories. The pull-down list of directory names maintains a
 * history of previously opened directories.
 */
public class IconEntry extends VBox {
	
	private Vector listeners;
	
	/**
	 * Construct a new IconEntry
	 */
	public IconEntry(String historyId, String dialogTitle) {
		super(gnome_icon_entry_new(historyId, dialogTitle));
	}
	
	/**
	 * Construct a new IconEntry from a handle to a native resource.
	 */
	public IconEntry(int handle) {
	    super(handle);
	}
	
	/**
	 * Set the subdirectory in which to look for icons.
	 */
	public void setSubdirectory(String subdir) {
		gnome_icon_entry_set_pixmap_subdir(getHandle(), subdir);
	}
	
	/**
	 * Retrieve the filename of the icon selected
	 */
	public String getFilename() {
		return gnome_icon_entry_get_filename(getHandle());
	}
	
	/**
	 * Set the filename of an icon.
	 */
	public void setFilename(String filename) {
		gnome_icon_entry_set_filename(getHandle(), filename);
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gnome_icon_entry_get_type());
	}
	
	/**
	 * Set the title for the Browse Dialog.
	 */
	public void setBrowseDialogTitle(String browseDialogTitle) {
		gnome_icon_entry_set_browse_dialog_title(getHandle(), browseDialogTitle);
	}
	
	/**
	 * Set the history id
	 */
	public void setHistoryId(String id) {
		gnome_icon_entry_set_history_id(getHandle(), id);
	}
	
	/**
	 * Sets the maximum number of save entries in the browse dialog.
	 * @param maxSaved
	 */
	public void setMaxSaved(int maxSaved) {
		gnome_icon_entry_set_max_saved(getHandle(), maxSaved);
	}

	//////////////////////////////////////////////////////
	// Event handling
	//////////////////////////////////////////////////////

	/**
	 * Register an object to handle IconEntry events.
	 * @see org.gnu.gnome.event.IconEntryListener
	 */
	public void addListener(IconEntryListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(listeners, listener);
		if (i == -1) {
			if (null == listeners) {
				evtMap.initialize(this, IconEntryEvent.Type.BROWSE);
				evtMap.initialize(this, IconEntryEvent.Type.CHANGED);
				listeners = new Vector();
			}
			listeners.addElement(listener);
		}
	}
	/**
	 * Removes a listener
	 * @see #addListener(IconEntryListener)
	 */
	public void removeListener(IconEntryListener listener) {
		int i = findListener(listeners, listener);
		if (i > -1) {
			listeners.remove(i);
		}
		if (0 == listeners.size()) {
			evtMap.uninitialize(this, IconEntryEvent.Type.BROWSE);
			evtMap.uninitialize(this, IconEntryEvent.Type.CHANGED);
			listeners = null;
		}
	}

	protected void fireIconEntryEvent(IconEntryEvent event) {
		if (null == listeners) {
			return;
		}
		int size = listeners.size();
		int i = 0;
		while (i < size) {
			IconEntryListener lis = (IconEntryListener)listeners.elementAt(i);
			lis.iconEntryEvent(event);
			i++;
		}
	}

	private void handleChanged() {
		fireIconEntryEvent(new IconEntryEvent(this, IconEntryEvent.Type.CHANGED));
	}
	
	private void handleBrowse() {
		fireIconEntryEvent(new IconEntryEvent(this, IconEntryEvent.Type.BROWSE));
	}
	
	public Class getEventListenerClass(String signal) {
		Class cls = evtMap.getEventListenerClass(signal);
		if (cls == null) cls = super.getEventListenerClass(signal);
		return cls;
	}

	public EventType getEventType(String signal) {
		EventType et = evtMap.getEventType(signal);
		if (et == null) et = super.getEventType(signal);
		return et;
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("changed", "handleChanged", IconEntryEvent.Type.CHANGED, IconEntryListener.class);
		anEvtMap.addEvent("browse", "handleBrowse", IconEntryEvent.Type.BROWSE, IconEntryListener.class);
	}

	
	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected int gnome_icon_entry_get_type();
	native static final protected int gnome_icon_entry_new(String historyId, String browserDialogTitle);
	native static final protected void gnome_icon_entry_set_pixmap_subdir(int ientry, String subdir);
	native static final protected String gnome_icon_entry_get_filename(int ientry);
	native static final protected boolean gnome_icon_entry_set_filename(int ientry, String filename);
	native static final protected void gnome_icon_entry_set_browse_dialog_title(int ientry, String browseDialogTitle);
	native static final protected void gnome_icon_entry_set_history_id(int ientry, String historyId);
	native static final protected int gnome_icon_entry_pick_dialog(int ientry);
	native static final protected void gnome_icon_entry_set_max_saved(int ientry, int max);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
