/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003-2004 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003      Johan Dahlin <johan@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <stdlib.h>
#include <glib/gi18n.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>
#include <gconf/gconf-client.h>
#include "main-window.h"
#include "song-db.h"
#include "stock-icons.h"
#include "utils.h"

#ifdef HAVE_GST
#include <gst/gst.h>
#include <gst/control/control.h>
#endif

#ifdef HAVE_DBUS
#include "dbus.h"
#endif


static int arg_readonly = FALSE;
static int arg_version = FALSE;
static char *arg_db = NULL;

#ifdef HAVE_DBUS
static char *arg_remote = NULL;
#endif


GConfClient *gconf_client;

static struct poptOption options[] = {
#ifdef HAVE_GST
  {
    NULL,
    '\0',
    POPT_ARG_INCLUDE_TABLE,
    NULL, /* insert opts here */
    0,
    N_("Gstreamer options"),
    NULL
  },
#endif
  {
    "version",
    'v',
    POPT_ARG_NONE,
    &arg_version,
    0,
    N_("Print version"),
    NULL
  },
  {
    "readonly",
    '\0',
    POPT_ARG_NONE,
    &arg_readonly,
    0,
    N_("Don't write changes to the music database"),
    NULL,
  },
  {
    "db",
    '\0',
    POPT_ARG_STRING,
    &arg_db,
    0,
    N_("Use an alternative song database file"),
    NULL,
  },
#ifdef HAVE_DBUS
  { 
    "remote",
    'r',
    POPT_ARG_STRING,
    &arg_remote,
    0,
    N_("Specify a command to pass to an already running instance"),
    N_("play|stop|prev|next|push-mute|pop-mute|current-song|quit")
  },
#endif
  POPT_TABLEEND
};

int
main (int argc, char **argv)
{
  GnomeProgram *program;
  SongDB *db;
  GtkWidget *window;
  char *filename;
  gboolean crash = FALSE;

  bindtextdomain (GETTEXT_PACKAGE, JAMBOREE_LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  /* Not very nice, but it works. */
#ifdef HAVE_GST
  options[0].arg = (struct poptOptions*) gst_init_get_popt_table ();
#endif

  /* Temporary workaround until I figure out why I don't get a crash dialog. */
  if (g_getenv ("GNOME_DISABLE_CRASH_DIALOG") == NULL)
    crash = TRUE;
  
  program = gnome_program_init (PACKAGE, VERSION,
				LIBGNOMEUI_MODULE, argc, argv,
				GNOME_PARAM_HUMAN_READABLE_NAME, "Jamboree",
				GNOME_PARAM_APP_DATADIR, DATADIR,
				GNOME_PARAM_POPT_TABLE, options,
				LIBGNOMEUI_PARAM_CRASH_DIALOG, crash,
				NULL);

#ifdef HAVE_GST
  gst_init (&argc, &argv);
  gst_control_init (&argc, &argv);
#endif

  if (arg_version)
    {
      g_print ("Jamboree " VERSION "\n");
      exit (1);
    }
  
#ifdef HAVE_DBUS
  if (arg_remote)
    {
      gboolean invalid_arg;
      
      if (!dbus_send_remote_cmd (arg_remote, &invalid_arg))
	{
	  if (invalid_arg)
	    g_printerr (_("Run \"%s --help\" to see a full list "
			  "of available command line options.\n"), argv[0]);
	  else
	    g_printerr (_("Error while sending message to Jamboree.\n"));
	  
	  exit (1);
	}

      exit (0);
    }
#endif
  
  g_random_set_seed (time (NULL));
  
  ensure_dir ();

  if (arg_db != NULL)
    filename = g_strdup (arg_db);
  else
    filename = g_build_filename (g_get_home_dir (),
				 ".gnome2", "jamboree", "songs.db",
				 NULL);
  
  stock_icons_register ();
  string_entry_init ();
  
  db = song_db_new (filename, arg_readonly);
  if (!db)
    {
      g_printerr (_("Could not open \"%s\".\n"), filename);
      return 1;
    }
  
  if (song_db_get_version (db) < SONG_DB_VERSION)
    {
      /*g_print (_("Database format changed. Reading files...\n"));*/
      song_db_rebuild (db);
      song_db_set_version (db, SONG_DB_VERSION);
    }

  song_db_read_songs (db);
  
  /*song_db_dump (db);*/
  g_free (filename);

#ifdef HAVE_DBUS
    if (!dbus_init_service ())
    {
      g_printerr (_("Couldn't init Jamboree service.\n"));
      return FALSE;
    }
  
  dbus_init_handler ();
#endif

  gconf_client = gconf_client_get_default ();
  gconf_client_add_dir (gconf_client, GCONF_PATH,
			GCONF_CLIENT_PRELOAD_ONELEVEL,
			NULL);

  window = main_window_new (db);

  gtk_widget_show (window);

  g_signal_connect (window,
		    "destroy", G_CALLBACK (gtk_main_quit),
		    NULL);

  gtk_main ();

  g_object_unref (db);
  string_entry_shutdown ();
  g_object_unref (gconf_client);

  return 0;
}


  
