/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gnome-pilot-capplet.c
 *
 * Copyright (C) 1998 Red Hat Software       
 * Copyright (C) 1999-2000 Free Software Foundation
 * Copyright (C) 2001  Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Eskil Heyn Olsen
 *          Vadim Strizhevsky
 *          Michael Fulbright <msf@redhat.com>
 *          JP Rosevear <jpr@ximian.com>
 *
 */

#include <sys/stat.h>
#include <glade/glade.h>
#include "pilot.h"
#include "util.h"
#include "gnome-pilot-pdialog.h"
#include "gnome-pilot-cdialog.h"
#include "gnome-pilot-ddialog.h"
#include "gnome-pilot-capplet.h"

#define SMALL_ICON_SIZE 20

static GtkVBoxClass *parent_class = NULL;

struct _GnomePilotCappletPrivate 
{
	GladeXML *xml;

	GnomePilotClient *gpc;
	gint handle;
	
	PilotState *state;
	PilotState *orig_state;

	GPilotPilot *pilot;

	GHashTable *conduits;
	
	CappletWidget *parent;

	GtkWidget *notebook;

	GtkWidget *pilots_clist;
	GtkWidget *pilots_add;
	GtkWidget *pilots_edit;
	GtkWidget *pilots_delete;
	GtkWidget *pilots_popup;
	
	GtkWidget *devices_clist;
	GtkWidget *devices_add;
	GtkWidget *devices_edit;
	GtkWidget *devices_delete;
	GtkWidget *devices_popup;

	GtkWidget *pilots_menu;
	GtkWidget *pilots_username;
	GtkWidget *conduit_clist;
	GtkWidget *conduit_enable;
	GtkWidget *conduit_disable;
	GtkWidget *conduit_settings;
	GtkWidget *conduit_popup;
	GtkWidget *conduit_popup_enable;
	GtkWidget *conduit_popup_disable;
	GtkWidget *conduit_popup_settings;	
	GtkWidget *conduit_description;

	GtkWidget *sync_pcid;
};

static void class_init (GnomePilotCappletClass *klass);
static void init (GnomePilotCapplet *gpcap);

static gboolean get_widgets (GnomePilotCapplet *gpcap);
static void map_widgets (GnomePilotCapplet *gpcap);
static void init_widgets (GnomePilotCapplet *gpcap);
static void fill_widgets (GnomePilotCapplet *gpcap);

static void check_pilots_buttons (GnomePilotCapplet *gpcap);
static void check_devices_buttons (GnomePilotCapplet *gpcap);
static void check_conduits_buttons (GnomePilotCapplet *gpcap);

static void gpcap_pilots_add (GtkWidget *widget, gpointer user_data);
static void gpcap_pilots_edit (GtkWidget *widget, gpointer user_data);
static void gpcap_pilots_delete (GtkWidget *widget, gpointer user_data);
static void gpcap_pilots_popup (GtkCList *list, GdkEventButton *event, gpointer user_data);
static void gpcap_pilots_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);

static void gpcap_devices_add (GtkWidget *widget, gpointer user_data);
static void gpcap_devices_edit (GtkWidget *widget, gpointer user_data);
static void gpcap_devices_delete (GtkWidget *widget, gpointer user_data);
static void gpcap_devices_popup (GtkCList *list, GdkEventButton *event, gpointer user_data);
static void gpcap_devices_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);

static void gpcap_conduits_choose_pilot (GtkWidget *widget, gpointer user_data);
static void gpcap_conduits_enable (GtkWidget *widget, gpointer user_data);
static void gpcap_conduits_disable (GtkWidget *widget, gpointer user_data);
static void gpcap_conduits_settings (GtkWidget *widget, gpointer user_data);
static void gpcap_conduits_popup (GtkCList *list, GdkEventButton *event, gpointer user_data);
static void gpcap_conduits_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);
static void gpcap_conduits_unselected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);

static void gpcap_cap_try (GtkWidget *widget, gpointer user_data);
static void gpcap_cap_revert (GtkWidget *widget, gpointer user_data);
static void gpcap_cap_ok (GtkWidget *widget, gpointer user_data);
static void gpcap_cap_cancel (GtkWidget *widget, gpointer user_data);

static void gpcap_changed (GtkWidget *widget, gpointer user_data);

static void gpcap_destroy (GtkObject *object);

GtkType
gnome_pilot_capplet_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "GnomePilotCapplet",
        sizeof (GnomePilotCapplet),
        sizeof (GnomePilotCappletClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (GnomePilotCappletClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->destroy = gpcap_destroy;
}

static void
init (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	GtkWidget *window;
	
	priv = g_new0 (GnomePilotCappletPrivate, 1);

	gpcap->priv = priv;

	/* State information */
	loadPilotState (&priv->orig_state);
	priv->state = dupPilotState (priv->orig_state);

	/* Gui stuff */
	priv->xml = glade_xml_new (GLADEDATADIR "/gpilotd-capplet.glade", "Capplet");
	if (!priv->xml) {
		g_message ("gnome-pilot-capplet init(): Could not load the Glade XML file!");
		goto error;
	}

	if (!get_widgets (gpcap)) {
		g_message ("gnome-pilot-capplet init(): Could not find all widgets in the XML file!");
		goto error;
	}

	/* Ugly bits to suck the widget out */
	window = glade_xml_get_widget (priv->xml, "Capplet");
	gtk_object_ref (GTK_OBJECT (priv->notebook));
	gtk_container_remove (GTK_CONTAINER (window), priv->notebook);
	gtk_container_add (GTK_CONTAINER (gpcap), priv->notebook);
	
	map_widgets (gpcap);
	fill_widgets (gpcap);
	init_widgets (gpcap);

 error:
}

GtkWidget *
gnome_pilot_capplet_new (GnomePilotClient *gpc)
{
	GnomePilotCapplet *gpcap;
	GtkWidget *widget;
	
	widget = gtk_type_new (GNOME_PILOT_TYPE_CAPPLET);
	
	gpcap = GNOME_PILOT_CAPPLET (widget);
	gpcap->priv->gpc = gpc;
	
	return widget;
}

static gboolean
get_widgets (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;

#define GW(name) glade_xml_get_widget (priv->xml, name)

	priv->notebook = GW ("CappletMain");

	priv->pilots_clist = GW ("pilots_clist");
	priv->pilots_add = GW ("pilots_add_button");
	priv->pilots_edit = GW ("pilots_edit_button");
	priv->pilots_delete = GW ("pilots_delete_button");

	priv->devices_clist = GW ("devices_clist");
	priv->devices_add = GW ("devices_add_button");
	priv->devices_edit = GW ("devices_edit_button");
	priv->devices_delete = GW ("devices_delete_button");

	priv->pilots_menu = GW ("pilots_menu");
	priv->pilots_username = GW ("username_label");	
	priv->conduit_clist = GW ("conduit_list");
	priv->conduit_enable = GW ("conduit_enable_button");
	priv->conduit_disable = GW ("conduit_disable_button");
	priv->conduit_settings = GW ("conduit_settings_button");
	priv->conduit_description = GW ("description_label");
	
	priv->sync_pcid = GW ("syncpcid_entry");
	
#undef GW
	return (priv->notebook
		&& priv->pilots_clist
		&& priv->pilots_add
		&& priv->pilots_edit
		&& priv->pilots_delete
		&& priv->devices_clist
		&& priv->devices_add
		&& priv->devices_edit
		&& priv->devices_delete
		&& priv->pilots_menu
		&& priv->conduit_clist
		&& priv->conduit_enable
		&& priv->conduit_disable
		&& priv->conduit_settings
		&& priv->conduit_description		
		&& priv->sync_pcid);
}

static void
map_widgets (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	
	priv = gpcap->priv;

	
	gtk_object_set_data (GTK_OBJECT (gpcap), "syncpcid_entry", priv->sync_pcid);
}

static void 
init_widgets (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;

	/* Button signals */
	gtk_signal_connect (GTK_OBJECT (priv->pilots_add), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_pilots_add), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->pilots_edit), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_pilots_edit), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->pilots_delete), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_pilots_delete), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->devices_add), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_devices_add), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->devices_edit), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_devices_edit), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->devices_delete), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_devices_delete), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_enable), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_conduits_enable), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_disable), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_conduits_disable), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_settings), "clicked",
			    GTK_SIGNAL_FUNC (gpcap_conduits_settings), gpcap);
	
	/* Change signals */
	gtk_signal_connect (GTK_OBJECT (priv->sync_pcid), "changed",
			    GTK_SIGNAL_FUNC (gpcap_changed), gpcap);

	/* Row selection signals */
	gtk_signal_connect (GTK_OBJECT (priv->pilots_clist), "select-row",
			    GTK_SIGNAL_FUNC (gpcap_pilots_selected), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->pilots_clist), "unselect-row",
			    GTK_SIGNAL_FUNC (gpcap_pilots_selected), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->devices_clist), "select-row",
			    GTK_SIGNAL_FUNC (gpcap_devices_selected), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->devices_clist), "unselect-row",
			    GTK_SIGNAL_FUNC (gpcap_devices_selected), gpcap);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "select-row",
			    GTK_SIGNAL_FUNC (gpcap_conduits_selected), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "unselect-row",
			    GTK_SIGNAL_FUNC (gpcap_conduits_unselected), gpcap);

	/* Popup menu signals */
	gtk_signal_connect (GTK_OBJECT (priv->pilots_clist), "button_press_event",
			    GTK_SIGNAL_FUNC (gpcap_pilots_popup), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->devices_clist), "button_press_event",
			    GTK_SIGNAL_FUNC (gpcap_devices_popup), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "button_press_event",
			    GTK_SIGNAL_FUNC (gpcap_conduits_popup), gpcap);
}

static gint
append_pilots_clist (GnomePilotCapplet *gpcap, GPilotPilot *pilot)
{
	GnomePilotCappletPrivate *priv;
	gint row = 0;
	const gchar *row_text[4];
	gchar buf[20];
	
	priv = gpcap->priv;
		
	row_text[0]= pilot->name;
	g_snprintf (buf, sizeof (buf), "%d", pilot->pilot_id);
	row_text[1] = buf;
	row_text[2] = pilot->pilot_username;
	row_text[3] = sync_type_to_str (pilot->sync_options.default_sync_action);
	row = gtk_clist_append (GTK_CLIST (priv->pilots_clist), (gchar **)row_text);
	gtk_clist_set_row_data (GTK_CLIST (priv->pilots_clist), row, pilot);

	return row;
}

static void
fill_pilots_clist (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	GList *tmp;
	
	priv = gpcap->priv;
		
	gtk_clist_clear (GTK_CLIST (priv->pilots_clist));
	tmp = priv->state->pilots;
	while (tmp!= NULL) {
		GPilotPilot *pilot =(GPilotPilot*)tmp->data;

		append_pilots_clist (gpcap, pilot);
		tmp = tmp->next;
	}	
}

static gint
append_devices_clist (GnomePilotCapplet *gpcap, GPilotDevice *device)
{
	GnomePilotCappletPrivate *priv;
	gint row = 0;
	const gchar *row_text[4];
	gchar buf[20];
	
	priv = gpcap->priv;

	row_text[0] = device->name;
	row_text[1] = device->port;
	g_snprintf (buf, sizeof (buf), "%d", device->speed);
	row_text[2] = buf;
	row_text[3] = device_type_to_str (device->type);

	row = gtk_clist_append (GTK_CLIST (priv->devices_clist), (gchar **)row_text);
	gtk_clist_set_row_data (GTK_CLIST (priv->devices_clist), row, device);

	return row;
}

static void
fill_devices_clist (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	GList *tmp;
	
	priv = gpcap->priv;
	
	gtk_clist_clear (GTK_CLIST (priv->devices_clist));
	
	tmp = priv->state->devices;
	while (tmp!= NULL) {
		GPilotDevice *device =(GPilotDevice*)tmp->data;

		append_devices_clist (gpcap, device);
		tmp = tmp->next;
	}	
}

static gint
append_conduit_clist (GnomePilotCapplet *gpcap, ConduitState *conduit_state)
{
	GnomePilotCappletPrivate *priv;
	GnomePixmap *icon_gpixmap = NULL;
	GdkPixmap *pmap, *mask;
	const gchar *row_text[3];
	gint row = 0;
	
	priv = gpcap->priv;

	row_text[0] = conduit_state->name;
	row_text[1] = display_sync_type_name (conduit_state->enabled,conduit_state->sync_type);
	row = gtk_clist_append (GTK_CLIST (priv->conduit_clist), (gchar **)row_text);

	if (conduit_state->icon) {
		icon_gpixmap = (GnomePixmap *)gnome_stock_pixmap_widget_at_size (NULL,
										 conduit_state->icon,
										 SMALL_ICON_SIZE,
										 SMALL_ICON_SIZE);
	}
	if (icon_gpixmap) {
		pmap = icon_gpixmap->pixmap;
		mask   = icon_gpixmap->mask;
		gtk_clist_set_pixtext (GTK_CLIST (priv->conduit_clist), row, 0, conduit_state->name, 3, pmap, mask);
	}

	gtk_clist_set_row_data (GTK_CLIST (priv->conduit_clist), row, conduit_state);

	return row;
}

static void
fill_conduit_clist (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	ConduitState *conduit_state;
	GList *conduit_states;

	priv = gpcap->priv;

	gtk_clist_clear (GTK_CLIST (priv->conduit_clist));

	if (priv->pilot == NULL)
		return;
	
	if (priv->conduits == NULL )
		priv->conduits = g_hash_table_new (g_int_hash, g_int_equal);
	
	conduit_states = g_hash_table_lookup (priv->conduits, &priv->pilot->pilot_id);
	if (conduit_states == NULL) {
		conduit_states = load_conduit_list (priv->pilot);
		g_hash_table_insert (priv->conduits, &priv->pilot->pilot_id, conduit_states);
	}
	
	while (conduit_states != NULL) {
		conduit_state = (ConduitState*)conduit_states->data;

		append_conduit_clist (gpcap, conduit_state);
		conduit_states = conduit_states->next;
	}
}

static void
set_conduit_pilot (GnomePilotCapplet *gpcap, GPilotPilot *pilot)
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;

	priv->pilot = pilot;
	
	if (pilot)
		gtk_label_set_text (GTK_LABEL (priv->pilots_username), pilot->pilot_username);
	else
		gtk_label_set_text (GTK_LABEL (priv->pilots_username), "");

	fill_conduit_clist (gpcap);
	check_conduits_buttons (gpcap);
}

static void
fill_pilots_menu (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;
	GtkWidget *menu, *menu_item;
	GList *tmp;
	GPilotPilot *pilot = NULL;
	
	priv = gpcap->priv;
	
	menu = gtk_menu_new ();
	
	tmp = priv->state->pilots;
	while (tmp != NULL) {
		if (pilot == NULL)
			pilot = tmp->data;
		
		menu_item = gtk_menu_item_new_with_label (((GPilotPilot*)tmp->data)->name);
		gtk_widget_show (menu_item);
		gtk_object_set_data (GTK_OBJECT (menu_item), "pilot", tmp->data);
		gtk_signal_connect (GTK_OBJECT(menu_item),"activate",
				    GTK_SIGNAL_FUNC (gpcap_conduits_choose_pilot),
				    gpcap);
		gtk_menu_append (GTK_MENU (menu), menu_item);

		tmp = tmp->next;
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (priv->pilots_menu), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (priv->pilots_menu), 0);
	set_conduit_pilot (gpcap, pilot);
}

static void
fill_widgets (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;
	gchar buf[40];

	priv = gpcap->priv;
	
	/* Pilots page */
	fill_pilots_clist (gpcap);
	check_pilots_buttons (gpcap);
	
	/* Devices page */
	fill_devices_clist (gpcap);
	check_devices_buttons (gpcap);

	/* Conduits page */
	fill_pilots_menu (gpcap);

	/* Advanced page */
	g_snprintf (buf, sizeof (buf), "%d", priv->state ? priv->state->syncPCid : 0);
	gtk_entry_set_text (GTK_ENTRY (priv->sync_pcid), buf);
}

CappletWidget *
gnome_pilot_capplet_get_parent (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;

	return priv->parent;
}

void
gnome_pilot_capplet_set_parent (GnomePilotCapplet *gpcap, CappletWidget *cap)
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;
	
	priv->parent = cap;

	gtk_signal_connect (GTK_OBJECT (priv->parent), "try",
			   GTK_SIGNAL_FUNC (gpcap_cap_try), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "revert",
			   GTK_SIGNAL_FUNC (gpcap_cap_revert), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "ok",
			   GTK_SIGNAL_FUNC (gpcap_cap_ok), gpcap);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "cancel",
			   GTK_SIGNAL_FUNC (gpcap_cap_cancel), gpcap);

#if 0
	gtk_signal_connect (GTK_OBJECT (capplet), "help",
			   GTK_SIGNAL_FUNC (gnome_help_display), 
			   &help_entry);
#endif
}

void
gnome_pilot_capplet_update (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;
	
	priv = gpcap->priv;
	
	freePilotState (priv->orig_state);
	freePilotState (priv->state);

	loadPilotState (&priv->orig_state);
	priv->state = dupPilotState (priv->orig_state);

	fill_widgets (gpcap);
}

static void
gpcap_set_changed_state (GnomePilotCapplet *gpcap)
{
	GnomePilotCappletPrivate *priv;

	priv = gpcap->priv;
	
	if (priv->parent == NULL)
		return;
	
	capplet_widget_state_changed (CAPPLET_WIDGET (priv->parent), TRUE);
	
}

static void
check_pilots_buttons (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;
	gboolean test;
	
	priv = gpcap->priv;

	test = check_selection (GTK_CLIST (priv->pilots_clist));

	gtk_widget_set_sensitive (GTK_WIDGET (priv->pilots_edit), test);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->pilots_delete), test);
}

static GPilotPilot *
get_current_pilot (GnomePilotCapplet *gpcap, int *row)
{
	GnomePilotCappletPrivate *priv;
	gint curr_row;
	
	priv = gpcap->priv;

	curr_row = GPOINTER_TO_INT (GTK_CLIST (priv->pilots_clist)->selection->data);
	if (row)
		*row = curr_row;
	
	return (GPilotPilot*)gtk_clist_get_row_data (GTK_CLIST (priv->pilots_clist), curr_row);
}

static void 
gpcap_pilots_add (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GtkObject *dlg;
	GPilotPilot *pilot;
	gboolean res;
	
	priv = gpcap->priv;

	pilot = get_default_pilot (priv->state);
	
	dlg = gnome_pilot_pdialog_new (priv->gpc, priv->state, pilot);
	res = gnome_pilot_pdialog_run_and_close (GNOME_PILOT_PDIALOG (dlg));

	if (!res) {
		g_free (pilot);
	} else {
		gint row;
		
		priv->state->pilots = g_list_append (priv->state->pilots, pilot);
		row = append_pilots_clist (gpcap, pilot);
		gtk_clist_select_row (GTK_CLIST (priv->pilots_clist), row, 0);
		
		fill_pilots_menu (gpcap);
		
		gpcap_set_changed_state (gpcap);
	}
}

static void 
gpcap_pilots_edit (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GtkObject *dlg;
	GPilotPilot *pilot;
	gint row;
	gboolean res;
	
	priv = gpcap->priv;
	
	pilot = get_current_pilot (gpcap, &row);
	
	dlg = gnome_pilot_pdialog_new (priv->gpc, priv->state, pilot);
	res = gnome_pilot_pdialog_run_and_close (GNOME_PILOT_PDIALOG (dlg));
 
	if (res) {
		char buf[20];		
		
		gtk_clist_set_text (GTK_CLIST (priv->pilots_clist), row, 0, pilot->name);
		g_snprintf (buf, sizeof (buf), "%d", pilot->pilot_id);
		gtk_clist_set_text (GTK_CLIST (priv->pilots_clist), row, 1, buf);
		gtk_clist_set_text (GTK_CLIST (priv->pilots_clist), row, 2, pilot->pilot_username);
		gtk_clist_set_text (GTK_CLIST (priv->pilots_clist), row, 3, sync_type_to_str (pilot->sync_options.default_sync_action));
			
		fill_pilots_menu (gpcap);
		
		gpcap_set_changed_state (gpcap);
	}
}

static void 
gpcap_pilots_delete (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GPilotPilot *pilot;
	gint row;
	
	priv = gpcap->priv;
	
	pilot = get_current_pilot (gpcap, &row);

	if (yes_no_dialog (_("Are you sure you want to delete %s pilot?"), pilot->name)) {
		gtk_clist_remove (GTK_CLIST (priv->pilots_clist), row);
		priv->state->pilots = g_list_remove (priv->state->pilots, pilot);
		
		fill_pilots_menu (gpcap);
		
		gpcap_set_changed_state (gpcap);
	}

	check_pilots_buttons (gpcap);
}

static void
gpcap_pilots_popup (GtkCList *list, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;

	GnomeUIInfo popup [] = {
		GNOMEUIINFO_ITEM_DATA (N_("_Edit this pilot..."),
				       N_("Edit the currently selected pilot"),
				       gpcap_pilots_edit, gpcap, NULL),
		GNOMEUIINFO_ITEM_DATA (N_("_Delete this pilot"),
				       N_("Delete the currently selected pilot"),
				       gpcap_pilots_delete, gpcap, NULL),
		GNOMEUIINFO_END
	};
 
	priv = gpcap->priv;

	if (priv->pilots_popup == NULL) {
		priv->pilots_popup = gtk_menu_new ();
		gnome_app_fill_menu (GTK_MENU_SHELL (priv->pilots_popup), popup, NULL, TRUE, 0);
	}

	show_popup_menu (list, event, GTK_MENU (priv->pilots_popup));
}

static void
gpcap_pilots_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	
	check_pilots_buttons (gpcap);
}

static GPilotDevice *
get_current_device (GnomePilotCapplet *gpcap, int *row)
{
	GnomePilotCappletPrivate *priv;
	gint curr_row;
	
	priv = gpcap->priv;

	curr_row = GPOINTER_TO_INT (GTK_CLIST (priv->devices_clist)->selection->data);
	if (row)
		*row = curr_row;
	
	return (GPilotDevice *) gtk_clist_get_row_data (GTK_CLIST (priv->devices_clist), curr_row);
}

static void
check_devices_buttons (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;
	gboolean test;
	
	priv = gpcap->priv;

	test = check_selection (GTK_CLIST (priv->devices_clist));

	gtk_widget_set_sensitive (GTK_WIDGET (priv->devices_edit), test);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->devices_delete), test);
}

static void 
gpcap_devices_add (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GtkObject *dlg;
	GPilotDevice *device;
	gboolean res;
	
	priv = gpcap->priv;

	device = get_default_device (priv->state);
	
	dlg = gnome_pilot_ddialog_new (device);
	res = gnome_pilot_ddialog_run_and_close (GNOME_PILOT_DDIALOG (dlg));

	if (!res) {
		g_free (device);
	} else {
		gint row;
		
		priv->state->devices = g_list_append (priv->state->devices, device);
		row = append_devices_clist (gpcap, device);
		gtk_clist_select_row (GTK_CLIST (priv->devices_clist), row, 0);
		
		gpcap_set_changed_state (gpcap);
	}
}

static void 
gpcap_devices_edit (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GtkObject *dlg;
	GPilotDevice *device;
	gint row;
	gboolean res;
	
	priv = gpcap->priv;
	
	device = get_current_device (gpcap, &row);

	dlg = gnome_pilot_ddialog_new (device);
	res = gnome_pilot_ddialog_run_and_close (GNOME_PILOT_DDIALOG (dlg));
 
	if (res) {
		char buf[20];		
		
		gtk_clist_set_text (GTK_CLIST (priv->devices_clist), row, 0, device->name);
		gtk_clist_set_text (GTK_CLIST (priv->devices_clist), row, 1, device->port);
		g_snprintf (buf, sizeof (buf), "%d", device->speed);
		gtk_clist_set_text (GTK_CLIST (priv->devices_clist), row, 2, buf);
		gtk_clist_set_text (GTK_CLIST (priv->devices_clist), row, 3, device_type_to_str (device->type));
		
		gpcap_set_changed_state (gpcap);
	}
}

static void 
gpcap_devices_delete (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GPilotDevice *device;
	gint row;
	
	priv = gpcap->priv;
	
	device = get_current_device (gpcap, &row);

	if (yes_no_dialog (_("Are you sure you want to delete %s pilot?"), device->name)) {
		gtk_clist_remove (GTK_CLIST (priv->devices_clist), row);
		priv->state->devices = g_list_remove (priv->state->devices, device);
		gpcap_set_changed_state (gpcap);
	}

	check_devices_buttons (gpcap);
}

static void 
gpcap_devices_popup (GtkCList *list, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;

	GnomeUIInfo popup [] = {
		GNOMEUIINFO_ITEM_DATA (N_("_Edit this device..."),
				       N_("Edit the currently selected device"),
				       gpcap_devices_edit, gpcap, NULL),
		GNOMEUIINFO_ITEM_DATA (N_("_Delete this device"),
				       N_("Delete the currently selected device"),
				       gpcap_devices_delete, gpcap, NULL),
		GNOMEUIINFO_END
	};
 
	priv = gpcap->priv;

	if (priv->devices_popup == NULL) {
		priv->pilots_popup = gtk_menu_new ();
		gnome_app_fill_menu (GTK_MENU_SHELL (priv->pilots_popup), popup, NULL, TRUE, 0);
	}

	show_popup_menu (list, event, GTK_MENU (priv->pilots_popup));
}

static ConduitState *
get_state (GnomePilotCapplet *gpcap, int row)
{
	GnomePilotCappletPrivate *priv;
	
	priv = gpcap->priv;

	return (ConduitState*)gtk_clist_get_row_data (GTK_CLIST (priv->conduit_clist), row);
}


static ConduitState *
get_current_state (GnomePilotCapplet *gpcap, int *row)
{
	GnomePilotCappletPrivate *priv;
	gint curr_row;
	
	priv = gpcap->priv;

	curr_row = GPOINTER_TO_INT (GTK_CLIST (priv->conduit_clist)->selection->data);
	if (row)
		*row = curr_row;

	return get_state (gpcap, curr_row);
}

static void
check_conduits_buttons (GnomePilotCapplet *gpcap) 
{
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	gboolean test, enabled = FALSE;
	
	priv = gpcap->priv;

	test = check_selection (GTK_CLIST (priv->conduit_clist));
	if (test) {
		state = get_current_state (gpcap, NULL);
		enabled = state->enabled;
	}
	
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_enable), test && !enabled);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_disable), test && enabled);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_settings), test && enabled);
}

static void
gpcap_devices_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	
	check_devices_buttons (gpcap);
}

static void
gpcap_conduits_choose_pilot (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GPilotPilot *pilot;
	
	priv = gpcap->priv;

	pilot = gtk_object_get_data (GTK_OBJECT (widget), "pilot");

	set_conduit_pilot (gpcap, pilot);
}

static void 
gpcap_conduits_enable (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcap->priv;

	state = get_current_state (gpcap, &row);
	
	if (state->default_sync_type == GnomePilotConduitSyncTypeNotSet) {
		/* nothing? */
	} else if (state->default_sync_type == GnomePilotConduitSyncTypeCustom) {
		state->changed=TRUE;
		state->enabled=TRUE;
		state->sync_type = GnomePilotConduitSyncTypeCustom;
	} else {
		state->changed = TRUE;
		state->enabled = TRUE;
		state->sync_type = state->default_sync_type;
	}

	gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
			    display_sync_type_name (state->enabled, state->sync_type));

	check_conduits_buttons (gpcap);
	gpcap_set_changed_state (gpcap);
	gpcap_conduits_settings (widget, user_data);
}

static void 
gpcap_conduits_disable (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcap->priv;
	
	state = get_current_state (gpcap, &row);

	state->enabled=FALSE;
	state->changed=TRUE;

	gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
			    display_sync_type_name (FALSE, GnomePilotConduitSyncTypeNotSet));

	check_conduits_buttons (gpcap);
	gpcap_set_changed_state (gpcap);
}

static void 
gpcap_conduits_settings (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcap->priv;

	state = get_current_state (gpcap, &row);
	
	if (state->conduit == NULL) {
		/* must allocate conduit */
		if (gnome_pilot_conduit_management_instantiate_conduit (state->management,
									state->pilot, 
									&state->conduit)
		    != GNOME_PILOT_CONDUIT_MGMT_OK) {
			
			gchar *msg = _("Unable to instantiate %s conduit.");
			error_dialog (msg, state->name);
			return;
		}

		state->settings_widget2 = gnome_pilot_cdialog_new (state);
	}
	
	if (state->settings_widget2 != NULL) {
		if (gnome_pilot_cdialog_run_and_close (GNOME_PILOT_CDIALOG (state->settings_widget2))) {
			/* pressed ok */
			state->sync_type = gnome_pilot_cdialog_sync_type (GNOME_PILOT_CDIALOG (state->settings_widget2));
			state->first_sync_type = gnome_pilot_cdialog_first_sync_type (GNOME_PILOT_CDIALOG (state->settings_widget2));
			state->enabled = (state->sync_type != GnomePilotConduitSyncTypeNotSet);
			state->changed=TRUE;
			
			gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
					    display_sync_type_name (state->enabled, state->sync_type));
			
			gpcap_set_changed_state (gpcap);
		} else {
			/* pressed cancel */
			gnome_pilot_conduit_display_settings (state->conduit);
		}
	}
	
	check_conduits_buttons (gpcap);
}

static void
gpcap_conduits_popup (GtkCList *list, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	gint row;
	gboolean test = FALSE, enabled = FALSE;
	
	GnomeUIInfo popup [] = {
		GNOMEUIINFO_ITEM_DATA (N_("_Enable this conduit"),
				       N_("Disable the currently selected conduit"),
				       gpcap_conduits_enable, gpcap, NULL),
		GNOMEUIINFO_ITEM_DATA (N_("_Disable this conduit"),
				       N_("Enable the currently selected conduit"),
				       gpcap_conduits_disable, gpcap, NULL),

		GNOMEUIINFO_SEPARATOR,

		GNOMEUIINFO_ITEM_DATA (N_("_Settings..."),
				       N_("Modify the currently selected conduit's settings"),
				       gpcap_conduits_settings, gpcap, NULL),
		
		GNOMEUIINFO_END
	};
 
	priv = gpcap->priv;

	if (priv->conduit_popup == NULL) {
		priv->conduit_popup = gtk_menu_new ();
		gnome_app_fill_menu (GTK_MENU_SHELL (priv->conduit_popup), popup, NULL, TRUE, 0);

		priv->conduit_popup_enable = popup[0].widget;
		priv->conduit_popup_disable = popup[1].widget;
		priv->conduit_popup_settings = popup[3].widget;
		
	}
	
	if (gtk_clist_get_selection_info (list, event->x, event->y, &row, NULL)) {
		state = get_state (gpcap, row);
		test = TRUE;
		enabled = state->enabled;
	}

	gtk_widget_set_sensitive (priv->conduit_popup_enable, test && !enabled);
	gtk_widget_set_sensitive (priv->conduit_popup_disable, test && enabled);
	gtk_widget_set_sensitive (priv->conduit_popup_settings, test && enabled);

	show_popup_menu (list, event, GTK_MENU (priv->conduit_popup));
}

static void
gpcap_conduits_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	ConduitState *state;
	
	priv = gpcap->priv;
	
	state = get_current_state (gpcap, NULL);
	
	gtk_label_set_text (GTK_LABEL (priv->conduit_description), state->description);
	check_conduits_buttons (gpcap);
}

static void
gpcap_conduits_unselected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	
	check_conduits_buttons (gpcap);
}

static void 
conduit_try_foreach (gpointer key, gpointer value, gpointer func)
{
	GList *states = (GList*)value;
	
	while (states != NULL) {
		ConduitState *state = (ConduitState*)states->data;
		
		if (state->changed && state->enabled) {
			if (state->sync_type == GnomePilotConduitSyncTypeCustom)
				gnome_pilot_conduit_config_enable (state->config, GnomePilotConduitSyncTypeCustom);
			else
				gnome_pilot_conduit_config_enable_with_first_sync (state->config,
										   state->sync_type,
										   state->first_sync_type,
										   TRUE);
			if (state->first_sync_type == GnomePilotConduitSyncTypeNotSet)
				gnome_pilot_conduit_config_remove_first_sync (state->config);
		}
		
		if (state->changed && !state->enabled)
			gnome_pilot_conduit_config_disable (state->config);
		
		if (state->conduit && state->changed)
			gnome_pilot_conduit_save_settings (state->conduit);
		
		states = states->next;
	}
}

static void 
gpcap_cap_try (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	GList *tmp;
	
	priv = gpcap->priv;
	
	read_host_config (GTK_OBJECT (gpcap), priv->state);

	tmp = priv->state->devices;
	while (tmp!= NULL) {
		GPilotDevice *device = (GPilotDevice*)tmp->data;
		check_device_settings (device);
		tmp = tmp->next;
	}

	g_hash_table_foreach (priv->conduits, conduit_try_foreach, NULL);
	fill_conduit_clist (gpcap);
	fill_pilots_menu (gpcap);
	
	save_config_and_restart (priv->gpc, priv->state);
}

static void
conduit_revert_foreach (gpointer key, gpointer value, gpointer func)
{
	GList *states = (GList*)value;
	
	while (states != NULL) {
		ConduitState *state = (ConduitState*)states->data;
		
		if (state->changed && state->orig_enabled) {
			if (state->orig_sync_type == GnomePilotConduitSyncTypeCustom)
				gnome_pilot_conduit_config_enable (state->config, GnomePilotConduitSyncTypeCustom);
			else
				 gnome_pilot_conduit_config_enable (state->config, state->orig_sync_type);
			
			state->enabled = state->orig_enabled;
			state->sync_type = state->orig_sync_type;
			state->first_sync_type = state->orig_first_sync_type;
		}
		
		if (state->changed && !state->orig_enabled) {
			gnome_pilot_conduit_config_disable (state->config);
			
			state->enabled = state->orig_enabled;
		}
		
		if (state->conduit && state->changed)
			gnome_pilot_conduit_revert_settings (state->conduit);
		
		states = states->next;
	}
}

static void
gpcap_cap_revert (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);
	GnomePilotCappletPrivate *priv;
	
	priv = gpcap->priv;

	g_hash_table_foreach (priv->conduits, conduit_revert_foreach, gpcap);
	fill_conduit_clist (gpcap);

	save_config_and_restart (priv->gpc, priv->orig_state);
	freePilotState (priv->state);
	priv->state = dupPilotState (priv->orig_state);
	
	fill_widgets (gpcap);
}

static void 
gpcap_cap_ok (GtkWidget *widget, gpointer user_data)
{
	gpcap_cap_try (widget, user_data);
}

static void 
gpcap_cap_cancel (GtkWidget *widget, gpointer user_data)
{
	gpcap_cap_revert (widget, user_data);
}

static void
gpcap_changed (GtkWidget *widget, gpointer user_data)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (user_data);

	gpcap_set_changed_state (gpcap);
}

static void
gpcap_destroy (GtkObject *object)
{
	GnomePilotCapplet *gpcap = GNOME_PILOT_CAPPLET (object);
	GnomePilotCappletPrivate *priv;
	
	priv = gpcap->priv;

	freePilotState (priv->orig_state);
	freePilotState (priv->state);

	g_hash_table_destroy (priv->conduits);

	gtk_object_unref (GTK_OBJECT (priv->xml));
	
	if (priv->pilots_popup)
		gtk_widget_destroy (priv->pilots_popup);

	if (priv->devices_popup)
		gtk_widget_destroy (priv->devices_popup);

	if (priv->conduit_popup)
		gtk_widget_destroy (priv->conduit_popup);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}
