/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <config.h>
#include "add-printer.h"

#include <cups/cups.h>
#include <cups/http.h>
#include <cups/ipp.h>
#include <unistd.h>
#include <sys/types.h>

#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkcomboboxentry.h>
#include <glade/glade.h>
#include <glib/gi18n.h>
#include <libgnomeui/gnome-druid.h>
#include <libgnomeui/gnome-ui-init.h>

#include <libgnomecups/gnome-cups-ui-init.h>
#include <libgnomecups/gnome-cups-request.h>
#include <libgnomecups/gnome-cups-util.h>
#include <libgnomecups/gnome-cups-ui-util.h>
#include <libgnomecups/gnome-cups-ui-driver.h>

#include "druid-helper.h"
#include "snmpinter.h"

#include <stdarg.h>
#include <gnome-keyring.h>
#ifdef HAVE_LIBSMBCLIENT
#include <libgnomeui/gnome-password-dialog.h>
#include <libsmbclient.h>
#include <errno.h>
#endif

#define ENTRY_TEXT(xml,name) (gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, name))))

/* Keep in sync with the network_transports menu */
typedef enum {
	NETWORK_TYPE_IPP,
	NETWORK_TYPE_SMB,
	NETWORK_TYPE_LPD,
	NETWORK_TYPE_HP
} NetworkType;

enum {
	LOCAL_DETECTED_LABEL,
	LOCAL_DETECTED_PRINTER,
	LAST_LOCAL_DETECTED
};

typedef struct {
	char const *cannonical_name;
	char const *corporate_icon;
	GHashTable *aliases;
} Vendor;

static char *
entry_get_text_stripped (GladeXML *xml, char const *name)
{
	GtkWidget *w =  glade_xml_get_widget (xml, name);
	char const *content;

	if (GTK_IS_COMBO_BOX_ENTRY (w))
		/* cheesy hack, we should be able to do this more gracefully */
		w = gtk_bin_get_child (GTK_BIN (w));

	content = gtk_entry_get_text (GTK_ENTRY (w));
	if (content != NULL)
		return g_strstrip (g_strdup (content));
	return NULL;
}

/*****************************************************************************/

typedef struct {
	char *label;
	char *uri;
	char *location;
	char *vendor_and_model;
} LocalPrinter;

static void
local_printer_free (LocalPrinter *printer)
{
	g_free (printer->label);
	g_free (printer->location);
	g_free (printer->uri);
	g_free (printer->vendor_and_model);
	g_free (printer);
}

static void
local_printer_list_free (GSList *list)
{
	g_slist_foreach (list, (GFunc)local_printer_free, NULL);
	g_slist_free (list);
}

/*****************************************************************************/

static char *
combine_vendor_and_model (char const *vendor, char const *model)
{
	if (strstr (model, vendor) == model)
		return g_strdup (model);
	return g_strdup_printf ("%s %s", vendor, model);
}

static char *
parse_network_detect (char *line)
{
	char *vendor_and_model;
	char *vendor = NULL;
	char *model = NULL;
	char **fields;
	char **p;
	char *sep;

	sep = strchr (line, '\n');
	if (sep) {
		*sep = '\0';
	}

	line = g_strstrip (line);
	fields = g_strsplit (line, ";", -1);

	for (p = fields; *p != NULL; p++) {
		char **pieces = g_strsplit (*p, "=", -1);
		char *name = pieces[0];
		char *value = pieces[1];
		if (!name || !value) {
			g_strfreev (pieces);
			continue;
		}

		if (!strcmp (name, "vendor")) {
			vendor = g_strdup (value);
		} else if (!strcmp (name, "model")) {
			model = g_strdup (value);
		}

		g_strfreev (pieces);
	}
	g_strfreev (fields);

	if (!vendor || !model) {
		g_free (vendor);
		g_free (model);
		return NULL;
	}

	vendor_and_model = combine_vendor_and_model (vendor, model);

	g_free (vendor);
	g_free (model);

	return vendor_and_model;
}

/**********************************************************************/
/* Vendor/Model Page Helpers */

static void
set_selected_uri (GladeXML *xml, char const *uri)
{
	g_object_set_data_full (G_OBJECT (xml), "selected_uri",
				g_strdup (uri), g_free);
}

static char *
get_selected_uri (GladeXML *xml)
{
	char *str;

	str = g_object_get_data (G_OBJECT (xml), "selected_uri");
	g_return_val_if_fail (str != NULL, NULL);

	return g_strdup (str);
}

static void
set_selected_location (GladeXML *xml, char const *location)
{
	g_object_set_data_full (G_OBJECT (xml), "selected_location",
				g_strdup (location), g_free);
}

/************************************************************************/
/* Local or Remote  page */
static void
local_or_remote_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = FALSE;
	*next = TRUE;
}
static GtkWidget *
local_or_remote_next (GladeXML *xml)
{
	GtkWidget *local_radio = glade_xml_get_widget (xml, "local_radio");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (local_radio)))
		return glade_xml_get_widget (xml, "local_location_page");
	else
		return glade_xml_get_widget (xml, "network_location_page");
}

/*************************************************************************/
/* Local Location Page */
static LocalPrinter const *
local_port_get (GladeXML *xml)
{
	LocalPrinter *desc = NULL;
	if (!combo_selected_get (xml, "local_ports", 1, &desc, -1))
		return NULL;
	return desc;
}

static void
local_port_init (GladeXML *xml, GSList *devices)
{
	char *label;
	GSList		*ptr;
	LocalPrinter	*desc;
	GtkTreeIter	 iter;
        GtkComboBox	*combo;
        GtkCellRenderer *renderer;
	GtkListStore	*model = gtk_list_store_new (2,
		G_TYPE_STRING, G_TYPE_POINTER);

	for (ptr = devices ; ptr != NULL; ptr = ptr->next) {
		desc = ptr->data;
		label = (desc->vendor_and_model)
			? g_strdup_printf ("%s (%s)",
					   desc->label,
					   desc->vendor_and_model)
			: g_strdup (desc->label);
		gtk_list_store_append (model, &iter);
		gtk_list_store_set (model, &iter, 0, label, 1, desc, -1);
		g_free (label);
	}
	combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "local_ports")),
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (model));
	gtk_combo_box_set_active (combo, -1);
	if (devices != NULL)
		gtk_combo_box_set_active (combo, 0);

        renderer = gtk_cell_renderer_text_new ();
        gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, TRUE);
        gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer,
		"text", 0, NULL);
}

static void
update_local_location_sensitivities (GladeXML *xml)
{
	GtkWidget *view = glade_xml_get_widget (xml, "local_detected_view");
	GtkWidget *local_ports = glade_xml_get_widget (xml, "local_ports");

	if (toggle_button_is_active (xml, "local_use_detected_radio")) {
		gtk_widget_set_sensitive (view, TRUE);
		gtk_widget_set_sensitive (local_ports, FALSE);
	} else {
		gtk_widget_set_sensitive (view, FALSE);
		gtk_widget_set_sensitive (local_ports, TRUE);
	}
}

static GSList *
get_local_devices (void)
{
	GSList *ret;
	ipp_t *request;
	ipp_t *response;

	ret = NULL;

	request = gnome_cups_request_new_for_printer (CUPS_GET_DEVICES, NULL);
	response = gnome_cups_request_execute (request, NULL, "/", NULL);

	if (response) {
		ipp_attribute_t *attr;
		LocalPrinter *desc;
		char *device_class = NULL;

		desc = g_new0 (LocalPrinter, 1);
		for (attr = response->attrs; attr != NULL; attr = attr->next) {
			if (!attr->name) {
				if (device_class
				    && strcmp (device_class, "network")
				    && desc->label
				    && desc->uri) {
					ret = g_slist_prepend (ret, desc);
				} else {
					desc = g_new0 (LocalPrinter, 1);
				}
				desc = g_new0 (LocalPrinter, 1);

				g_free (device_class);
				device_class = NULL;

				continue;
			}

			if (!strcmp (attr->name, "device-class")) {
				g_free (device_class);
				device_class = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-info")) {
				g_free (desc->label);
				desc->label = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-uri")) {
				g_free (desc->uri);
				desc->uri = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-make-and-model") && strcmp (attr->values[0].string.text, "Unknown")) {
				g_free (desc->vendor_and_model);
				desc->vendor_and_model = g_strdup (attr->values[0].string.text);
			}
		}

		if (device_class
		    && strcmp (device_class, "network")
		    && desc->label
		    && desc->uri) {
			ret = g_slist_prepend (ret, desc);
		} else {
			local_printer_free (desc);
		}

		ret = g_slist_reverse (ret);

		g_free (device_class);

		ippDelete (response);
	}

	return ret;
}

static void
local_location_setup (GladeXML *xml)
{
	GtkTreeView	  *tree_view;
	GtkListStore	  *list_store;
	GtkTreeSelection  *selection;
	GtkCellRenderer	  *renderer;
	GtkTreeViewColumn *column;
	int num_detected;
	GSList *ptr, *devices;

	devices = get_local_devices ();
	g_object_set_data_full (G_OBJECT (xml),
		"local-devices", devices,
		(GDestroyNotify)local_printer_list_free);
	local_port_init (xml, devices);

	tree_view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Printer"),
							   renderer,
							   "markup",
							   LOCAL_DETECTED_LABEL,
							   NULL);
	gtk_tree_view_append_column (tree_view, column);

	list_store = gtk_list_store_new (LAST_LOCAL_DETECTED,
					 G_TYPE_STRING, G_TYPE_POINTER);
	gtk_tree_view_set_model (tree_view, GTK_TREE_MODEL (list_store));

	selection = gtk_tree_view_get_selection (tree_view);
	num_detected = 0;
	for (ptr = devices; ptr != NULL; ptr = ptr->next) {
		LocalPrinter *desc = ptr->data;
		if (desc->vendor_and_model
		    /* weed out two that are always returned
		     * regardless of what's there */
		    && strcmp (desc->vendor_and_model, "EPSON")
		    && strcmp (desc->vendor_and_model, "CANON")) {
			GtkTreeIter iter;

			gtk_list_store_append (list_store, &iter);
			gtk_list_store_set (list_store, &iter,
					    0, desc->vendor_and_model,
					    1, desc,
					    -1);
			if (num_detected == 0)
				gtk_tree_selection_select_iter (selection, &iter);
			num_detected++;
		}
	}

	if (num_detected == 0) {
		GtkTreeIter iter;
		gtk_list_store_append (list_store, &iter);
		gtk_list_store_set (list_store, &iter,
				    0, _("<i>No printers detected</i>"),
				    1, NULL,
				    -1);

		gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
		gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
		gtk_toggle_button_set_active ((GtkToggleButton *)
			glade_xml_get_widget (xml, "local_specify_port_radio"), TRUE);
	}

	update_local_location_sensitivities (xml);
	g_signal_connect_swapped (glade_xml_get_widget (xml, "local_use_detected_radio"),
		"toggled",
		G_CALLBACK (update_local_location_sensitivities), xml);
	g_signal_connect_swapped (glade_xml_get_widget (xml, "local_specify_port_radio"),
		"toggled",
		G_CALLBACK (update_local_location_sensitivities), xml);

	druid_watch_sensitivity_widget (xml, "local_use_detected_radio");
	druid_watch_sensitivity_widget (xml, "local_specify_port_radio");
}

static void
local_location_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	*back = TRUE;
	*next = toggle_button_is_active (xml, "local_specify_port_radio") ||
		tree_view_has_selection (xml, "local_detected_view");
}

static GtkWidget *
local_location_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "local_or_remote_page");
}

static GCupsDriverSelector *
driver_selector (GladeXML *xml)
{
	return (GCupsDriverSelector *) glade_xml_get_widget (xml, "driver_selector");
}

static GtkWidget *
local_location_next (GladeXML *xml)
{
	GtkTreeView *tree_view;
	GtkListStore *list_store;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean use_detected;
	GCupsDriverSelector *ds = driver_selector (xml);

	tree_view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));
	selection = gtk_tree_view_get_selection (tree_view);
	list_store = GTK_LIST_STORE (gtk_tree_view_get_model (tree_view));

	use_detected = !toggle_button_is_active (xml, "local_specify_port_radio");

	gcups_driver_selector_set_vendor_and_model (ds, NULL);

	if (use_detected) {
		if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
			char *location;
			LocalPrinter *desc;
			gtk_tree_model_get (GTK_TREE_MODEL (list_store),
					    &iter,
					    1, &desc, -1);
			location = g_strdup_printf (_("Local printer on %s"),
						       desc->location);
			gcups_driver_selector_set_vendor_and_model (ds, desc->vendor_and_model);
			set_selected_uri (xml, desc->uri);
			set_selected_location (xml, location);
			g_free (location);
		} else {
			g_warning ("no selected printer");
			return NULL;
		}
	} else {
		LocalPrinter const *lp = local_port_get (xml);
		if (NULL == lp)
			return NULL;
		set_selected_uri (xml, lp->uri);
		set_selected_location (xml, lp->location);
	}

	return glade_xml_get_widget (xml, "details_page");
}

/****************************************************************************/

static GHashTable *smb_servers;
static GSList *new_servers = NULL;
static GSList *new_printers = NULL;
static GStaticMutex server_mutex = G_STATIC_MUTEX_INIT;
#ifdef HAVE_LIBSMBCLIENT
static GStaticMutex smb_request_mutex = G_STATIC_MUTEX_INIT;
static GStaticMutex printer_request_mutex = G_STATIC_MUTEX_INIT;

typedef struct {
	GCond *cond;
	/* In */  char *server, *share;
		gboolean keyring;
	/* Out */ char *workgroup, *id, *passwd;
} SmbReqAuth;
typedef struct {
	char *name, *path;
} SmbNewServer;
typedef struct {
	char *server, *printer;
} SmbNewPrinter;
typedef struct {
	char *server;
} SmbGetPrinters;

static SmbReqAuth *auth_req = NULL;

/***************************************************************************/
/* On the SMB side */
static void 
smb_auth_fn (char const *server, char const *share,
	     char *workgroup, int workgroup_max,
	     char *id,	      int id_len,
	     char *passwd,    int passwd_len)
{
	SmbReqAuth *req;

	/* blah I need some extra user hooks here to store the keyring override
	 * flag safely */
	static gboolean used_keyring = FALSE;
	static char *used_keyring_for = NULL;

	if (share != NULL && 0 == strcmp (share, "IPC$"))
		return; /* don't bother */

	req = g_new0 (SmbReqAuth, 1);
	req->cond = g_cond_new ();
	req->server = g_strdup (server);
	req->share = g_strdup (share);

	/* cheesy hack to override the use of the keyring if the previous
	 * attmempt failed */
	req->keyring == (used_keyring_for == NULL ||
			 0 != strcmp (used_keyring_for, share));

	g_static_mutex_lock (&smb_request_mutex);
	if (auth_req != NULL) { g_warning ("dropping an auth req"); }
	auth_req = req;
	g_static_mutex_unlock (&smb_request_mutex);

	g_cond_wait (req->cond, g_static_mutex_get_mutex (&smb_request_mutex));

	g_static_mutex_lock (&smb_request_mutex);
	auth_req = NULL;
	g_static_mutex_unlock (&smb_request_mutex);

	strncpy (id,	 req->id     ? req->id     : "", id_len);
	strncpy (passwd, req->passwd ? req->passwd : "", passwd_len);

	used_keyring = req->keyring;
	g_free (used_keyring_for);
	used_keyring_for = g_strdup (server);

	g_cond_free (req->cond);
	g_free (req->server);
	g_free (req->share);
	g_free (req->workgroup);
	g_free (req->id);
	g_free (req->passwd);
	g_free (req);
}

static gpointer
cb_smb_thread (gpointer data)
{
	struct smbc_dirent const *dirent;
	unsigned wg_handle, serve_handle;

	if ((wg_handle = smbc_opendir ("smb://")) >= 0) {
		while ((dirent = smbc_readdir (wg_handle)))
			if (dirent->smbc_type == SMBC_WORKGROUP) {
				char *path = g_strconcat ("smb://", dirent->name, NULL);
				if ((serve_handle = smbc_opendir (path)) >= 0) {
					while ((dirent = smbc_readdir (serve_handle)))
						if (dirent->smbc_type == SMBC_SERVER) {
							char *path = g_strconcat ("smb://", dirent->name, "/", NULL);
							g_static_mutex_lock (&server_mutex);
							new_servers = g_slist_append (new_servers, g_strdup (dirent->name));
							new_servers = g_slist_append (new_servers, path);
							g_static_mutex_unlock (&server_mutex);
						}
					smbc_closedir (serve_handle);
				} else {
					g_warning ("Could not list %s : %s\n", path, strerror (errno));
				}
				g_free (path);
			}

		smbc_closedir (wg_handle);
	} else {
		g_warning ("Could not list %s : %s\n", "smb://", strerror (errno));
	}

	return NULL;
}

static gpointer
cb_smb_find_printers (char const *dir_url)
{
	struct smbc_dirent const *dirent;
	unsigned dir_handle;

reauth :
	if ((dir_handle = smbc_opendir (dir_url)) >= 0) {
		while ((dirent = smbc_readdir (dir_handle)))
			if (dirent->smbc_type == SMBC_PRINTER_SHARE) {
				g_static_mutex_lock (&printer_request_mutex);
				new_printers = g_slist_append (new_printers, g_strdup (dir_url));
				new_printers = g_slist_append (new_printers, g_strdup (dirent->name));
				g_static_mutex_unlock (&printer_request_mutex);
			}
		smbc_closedir (dir_handle);
	} else if (dir_handle == EACCES)
		goto reauth;
	else {
		g_warning ("Could not list %s : %s\n", dir_url, strerror (errno));
	}

	return NULL;
}

/***************************************************************************/
/* On the UI side */
static void
ui_auth_req_handler (GladeXML *xml)
{
	static char *default_id = NULL;
	GList *list = NULL;
	GtkWidget *w;
	GnomePasswordDialog *dialog;
	GnomeKeyringResult result = GNOME_KEYRING_RESULT_OK;
	GString *msg;
	gboolean found = FALSE;

	g_warning ("authenticating with %s for %s", auth_req->server, auth_req->share);

	if (default_id == NULL)
		default_id = g_strdup (g_getenv ("USER"));
	if (default_id == NULL)
		default_id = g_strdup (g_getenv ("LOGNAME"));

	if (default_id != NULL && auth_req->workgroup != NULL && auth_req->keyring == TRUE)
		result = gnome_keyring_find_network_password_sync (default_id,
			auth_req->workgroup, auth_req->server, auth_req->share,
			"smb", NULL, 0, &list);
	
	if (list != NULL) {
		if (result == GNOME_KEYRING_RESULT_OK) {
			GnomeKeyringNetworkPasswordData *pwd_data = list->data;
			auth_req->id = g_strdup (pwd_data->user);
			auth_req->passwd = g_strdup (pwd_data->password);
			found = TRUE;
		}
		gnome_keyring_network_password_list_free (list);
	}

	if (!found) {
		msg = g_string_new (_("Identity and Password for"));
		if (auth_req->share != NULL)
			g_string_append_printf (msg, _(" printer %s"), auth_req->share);
		if (auth_req->server != NULL)
			g_string_append_printf (msg, _(" on server %s"), auth_req->server);
		if (auth_req->workgroup != NULL)
			g_string_append_printf (msg, _(" in workgroup %s"), auth_req->workgroup);

		dialog = GNOME_PASSWORD_DIALOG (gnome_password_dialog_new (
			_("Authentication Required"), msg->str, "", "", FALSE));
		if (default_id != NULL)
			gnome_password_dialog_set_username	(dialog, default_id);
		gnome_password_dialog_set_show_userpass_buttons (dialog, FALSE);
		gnome_password_dialog_set_show_username		(dialog,  TRUE);
		gnome_password_dialog_set_show_domain		(dialog, FALSE);
		gnome_password_dialog_set_show_password		(dialog,  TRUE);
		gnome_password_dialog_set_show_remember		(dialog, FALSE);

		auth_req->keyring = FALSE;
		if (gnome_password_dialog_run_and_block (dialog)) {
			auth_req->id     = gnome_password_dialog_get_username (dialog);
			auth_req->passwd = gnome_password_dialog_get_password (dialog);
		}
		gtk_widget_destroy (GTK_WIDGET (dialog));
		g_string_free (msg, TRUE);
	}
	if (auth_req->id != NULL) {
		w = glade_xml_get_widget (xml, "smb_username_entry");
		gtk_entry_set_text (GTK_ENTRY (w), auth_req->id);
	}
	if (auth_req->passwd != NULL) {
		w = glade_xml_get_widget (xml, "smb_password_entry");
		gtk_entry_set_text (GTK_ENTRY (w), auth_req->passwd);
	}
}

static void
ui_add_server_handler (GladeXML *xml)
{
	GtkComboBox  *combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_host_entry"));
	GtkListStore *store = (GtkListStore *) gtk_combo_box_get_model (combo);
	GtkTreeIter   iter;
	char *name, *path;

	g_return_if_fail (new_servers != NULL);
	name = new_servers->data;
	new_servers = g_slist_remove (new_servers, name);

	g_return_if_fail (new_servers != NULL); /* minor leak of name */
	path = new_servers->data;
	new_servers = g_slist_remove (new_servers, path);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
		0, name,
		1, path,
		-1);
	g_free (name);
	g_free (path);
}

static void
ui_add_printer_handler (GladeXML *xml)
{
	GtkListStore *store;
	GtkTreeIter   iter;
	char *server, *printer;

	g_return_if_fail (new_printers != NULL);
	server = new_printers->data;
	new_printers = g_slist_remove (new_printers, server);

	g_return_if_fail (new_printers != NULL); /* minor leak of server */
	printer = new_printers->data;
	new_printers = g_slist_remove (new_printers, printer);

	store = g_hash_table_lookup (smb_servers, server);
	if (store != NULL) {
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
			0, printer,
			-1);
	} else {
		g_warning ("missing smb server model ??");
	}
	g_free (printer);
	g_free (server);
}
static gint
cb_smb_req_handler (GladeXML *xml)
{
	g_static_mutex_lock (&smb_request_mutex);
	if (auth_req != NULL) {
		ui_auth_req_handler (xml);
		g_cond_signal (auth_req->cond);
	}
	g_static_mutex_unlock (&smb_request_mutex);

	g_static_mutex_lock (&server_mutex);
	while (new_servers != NULL)
		ui_add_server_handler (xml);
	g_static_mutex_unlock (&server_mutex);

	g_static_mutex_lock (&printer_request_mutex);
	while (new_printers != NULL)
		ui_add_printer_handler (xml);
	g_static_mutex_unlock (&printer_request_mutex);
	return TRUE;
}
#endif

static void
cb_smb_host_changed (GtkComboBox *combo, GladeXML *xml)
{
	GtkTreeIter   iter;

	if (gtk_combo_box_get_active_iter (combo, &iter)) {
		gboolean make_req = FALSE;
		char *path;
		GtkTreeModel *model = gtk_combo_box_get_model (combo);
		gtk_tree_model_get (model, &iter, 1, &path, -1);
		if (smb_servers == NULL)
			smb_servers = g_hash_table_new_full (g_str_hash, g_str_equal,
					       g_free, NULL);
		if (NULL == (model = g_hash_table_lookup (smb_servers, path))) {
			model = (GtkTreeModel *)gtk_list_store_new (1, G_TYPE_STRING);
			g_hash_table_insert (smb_servers, g_strdup (path), model);
			make_req = TRUE;
		}

		gtk_combo_box_set_model (
			(GtkComboBox *)glade_xml_get_widget (xml, "smb_printer_entry"),
			GTK_TREE_MODEL (model));
#ifdef HAVE_LIBSMBCLIENT
		if (make_req)
			g_thread_create ((GThreadFunc)cb_smb_find_printers,
					 path, TRUE, NULL);
#endif
	}
}

static void
init_smb_combos (GladeXML *xml)
{
	GtkComboBox  *combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_host_entry"));
	GtkListStore *store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (store));
	gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (combo), 0);
	gtk_combo_box_set_active (combo, -1);
	g_signal_connect (GTK_WIDGET (combo),
		"changed",
		G_CALLBACK (cb_smb_host_changed), xml);

	combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_printer_entry"));
	store = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (store));
	gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (combo), 0);
	gtk_combo_box_set_active (combo, -1);
}

/* Network Location Page */
static char *
get_manual_lpd_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "lpd_host_entry");
	char *queue = entry_get_text_stripped (xml, "lpd_queue_entry");

	if (host[0])
		ret = g_strdup_printf ("lpd://%s/%s", host, queue);
	g_free (host);
	g_free (queue);
	return ret;
}

static char *
get_manual_smb_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "smb_host_entry");
	char *printer = entry_get_text_stripped (xml, "smb_printer_entry");
	char *username = entry_get_text_stripped (xml, "smb_username_entry");
	char *password = entry_get_text_stripped (xml, "smb_password_entry");

	if (host != NULL && host[0] && printer != NULL && printer[0]) {
		if (username != NULL && username[0])
			ret = g_strdup_printf ("smb://%s:%s@%s/%s",
				username, password, host, printer);
		else
			ret = g_strdup_printf ("smb://%s/%s", host, printer);
	}

	g_free (host);
	g_free (printer);
	g_free (username);
	g_free (password);

	return ret;
}

static char *
get_manual_ipp_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *uri = entry_get_text_stripped (xml, "ipp_uri_entry");

	if (uri[0])
		ret = g_strdup (uri);
	g_free (uri);

	return ret;
}

static char *
get_manual_hp_uri (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "hp_host_entry");
	char *port = entry_get_text_stripped (xml, "hp_port_entry");

	if (port == NULL || !port[0]) {
		g_free (port);
		port = g_strdup ("9100");
	}

	if (host != NULL && host[0])
		ret = g_strdup_printf ("socket://%s:%s", host, port);

	g_free (host);
	g_free (port);

	return ret;
}

static NetworkType
get_manual_network_type (GladeXML *xml)
{
	GtkComboBox *combo = (GtkComboBox *)glade_xml_get_widget (xml, "network_transports");
	int tmp = gtk_combo_box_get_active (combo);
	if (tmp >= 0)
		return (NetworkType)tmp;
	return NETWORK_TYPE_IPP;
}

/* Read the manually selected preferences from the dialog */
static char *
get_manual_network_uri (GladeXML *xml)
{
	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_IPP :	return get_manual_ipp_uri (xml);
	case NETWORK_TYPE_SMB :	return get_manual_smb_uri (xml);
	case NETWORK_TYPE_LPD :	return get_manual_lpd_uri (xml);
	case NETWORK_TYPE_HP :	return get_manual_hp_uri (xml);
	default :
		g_warning ("unsupported type\n");
	}

	return NULL;
}

static char *
get_manual_network_location (GladeXML *xml)
{
	char *ret = NULL;
	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_LPD : {
		char *host  = entry_get_text_stripped (xml, "lpd_host_entry");
		char *queue = entry_get_text_stripped (xml, "lpd_queue_entry");

		if (host != NULL && host[0]) {
			if (queue != NULL && queue[0])	/* queue, host */
				ret = g_strdup_printf (_("UNIX printer %s on %s"), queue, host);
			else				/* host */
				ret = g_strdup_printf (_("UNIX printer on %s"), host);
		}
		g_free (queue);
		g_free (host);
		break;
	}
	case NETWORK_TYPE_SMB : {
		char *host = entry_get_text_stripped (xml, "smb_host_entry");
		char *printer = entry_get_text_stripped (xml, "smb_printer_entry");

		if (host[0] && printer[0])	/* printer, host */
			ret = g_strdup_printf (_("Windows Printer %s on %s"), printer, host);
		break;
	}
	case NETWORK_TYPE_IPP : {
		char *uri = entry_get_text_stripped (xml, "ipp_uri_entry");
		if (uri != NULL && uri[0])	/* uri */
			ret = g_strdup_printf (_("IPP Printer at %s"), uri);
		g_free (uri);
		break;
	}
	default :
		g_warning ("unsupported type\n");
	}

	return ret;
}

static char *
lpd_get_vendor_and_model (char const *host)
{
	int retval;
	char *ret = NULL;
	char *detected = g_strdup (get_snmp_printers ((char*)host, &retval));

	if (NULL != detected) {
		ret = parse_network_detect (detected);
		g_free (detected);
	}
	return ret;
}

static char *
get_manual_network_vendor_and_model (GladeXML *xml)
{
	char *ret;
	char *host;
	host = NULL;

	switch (get_manual_network_type (xml)) {
	case NETWORK_TYPE_LPD :
		host = entry_get_text_stripped (xml, "lpd_host_entry");
		ret = lpd_get_vendor_and_model (host);
		g_free (host);
		return ret;
	case NETWORK_TYPE_SMB :
	case NETWORK_TYPE_IPP :
		return NULL;
	default :
		g_warning ("unsupported type\n");
		return NULL;
	}
}

static void
cb_network_type_changed (GtkComboBox *combo, GladeXML *xml)
{

	GtkWidget *notebook = glade_xml_get_widget (xml, "network_info_notebook");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook),
		gtk_combo_box_get_active (combo));
}

static void
network_location_setup (GladeXML *xml)
{
	GtkWidget *combo = glade_xml_get_widget (xml, "network_transports");
	gtk_combo_box_set_active (GTK_COMBO_BOX (combo), NETWORK_TYPE_IPP);

	druid_watch_sensitivity_widget (xml, "network_transports");
	druid_watch_sensitivity_widget (xml, "lpd_host_entry");
	druid_watch_sensitivity_widget (xml, "smb_username_entry");
	druid_watch_sensitivity_widget (xml, "smb_password_entry");
	druid_watch_sensitivity_widget (xml, "smb_printer_entry");
	druid_watch_sensitivity_widget (xml, "smb_host_entry");
	druid_watch_sensitivity_widget (xml, "ipp_uri_entry");
	druid_watch_sensitivity_widget (xml, "hp_host_entry");
	druid_watch_sensitivity_widget (xml, "hp_port_entry");
	g_signal_connect (glade_xml_get_widget (xml, "network_transports"),
		"changed",
		G_CALLBACK (cb_network_type_changed), xml);
}

static void
network_location_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	char *uri = get_manual_network_uri (xml);
	*back = TRUE;
	*next = (uri != NULL);
	g_free (uri);
}

static GtkWidget *
network_location_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "local_or_remote_page");
}

static GtkWidget *
network_location_next (GladeXML *xml)
{
	char *uri = get_manual_network_uri (xml);

	if (uri != NULL) {
		GCupsDriverSelector *ds = driver_selector (xml);
		char *vendor_and_model = get_manual_network_vendor_and_model (xml);
		char *location = get_manual_network_location (xml);

		set_selected_uri (xml, uri);
		set_selected_location (xml, location);
		g_free (uri);
		g_free (location);

		gcups_driver_selector_set_vendor_and_model (ds, vendor_and_model);
		g_free (vendor_and_model);
		return glade_xml_get_widget (xml, "details_page");
	}
	return NULL;
}

/****************************************************************************/

static void
details_setup (GladeXML *xml)
{
	druid_watch_sensitivity_widget (xml, "driver_selector");
}

static void
details_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	GtkWidget *druid = glade_xml_get_widget (xml, "add_printer_druid");
	*back = TRUE;
	*next = NULL != gcups_driver_selector_get (driver_selector (xml));

	/* see details_prepare for an explanation of the druid hack */
	gnome_druid_set_show_finish (GNOME_DRUID (druid), *next);
}

static GtkWidget *
details_back (GladeXML *xml)
{
	if (toggle_button_is_active (xml, "local_radio"))
		return glade_xml_get_widget (xml, "local_location_page");
	else
		return glade_xml_get_widget (xml, "network_location_page");
}

static ipp_t *
add_printer_request (char const *printer_name,
		     char const *device_uri,
		     GCupsPPD  const *ppd,
		     GError **err)
{
	ipp_t *request = gnome_cups_request_new_for_printer (
		CUPS_ADD_PRINTER, printer_name);
	ippAddString (request,	IPP_TAG_PRINTER, IPP_TAG_TEXT,
		"printer-location", NULL, gnome_cups_strdup (device_uri));
	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_NAME,
		"ppd-name", NULL, gnome_cups_strdup (ppd->filename));
	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_URI,
		"device-uri", NULL, gnome_cups_strdup (device_uri));
	ippAddBoolean (request, IPP_TAG_PRINTER, "printer-is-accepting-jobs", 1);
	ippAddInteger(request, IPP_TAG_PRINTER, IPP_TAG_ENUM, "printer-state",
		      IPP_PRINTER_IDLE);
	return gnome_cups_request_execute (request, NULL, "/admin/", err);
}

static gboolean
add_cups_printer (GladeXML *xml, char const *device_uri, GCupsPPD  const *ppd, char const *printer_name)
{
	GError *err = NULL;
	ipp_t *response = add_printer_request (printer_name, device_uri, ppd, &err);
	ippDelete (response);

	if (err != NULL) {
		gnome_cups_error_dialog ((GtkWindow *)glade_xml_get_widget (xml, "add_printer_window"),
					 _("Couldn't add printer"), err);
		g_error_free (err);
		return FALSE;
	}

	return TRUE;
}

static void
details_prepare (GladeXML *xml)
{
#if 0
	/* Blah.  GnomeDruid is stupid.  finish and next are distinct buttons internally
	 * and there is no way to desensitize finish.  We'll need to cheat */
	GtkWidget *druid = glade_xml_get_widget (xml, "add_printer_druid");
	gnome_druid_set_show_finish (GNOME_DRUID (druid), TRUE);
#endif
}

static GtkWidget *
details_next (GladeXML *xml)
{
	GList *existing;
	char  *name, *uri;
	unsigned i = 0;
	GCupsPPD const  *ppd  = gcups_driver_selector_get (driver_selector (xml));

	if (ppd == NULL)
		return NULL;

	uri  = get_selected_uri (xml);

	name = g_strdup (ppd->model);
	existing = gnome_cups_get_printers ();
	while (NULL != g_list_find_custom (existing, name, (GCompareFunc)strcasecmp )) {
		g_free (name);
		name = g_strdup_printf ("%s-%d", ppd->model, ++i);
	}
	g_list_foreach (existing, (GFunc)g_free, NULL);
	g_list_free (existing);

	if (add_cups_printer (xml, uri, ppd, name)) {
		GError *err = NULL;
		char *argv[] = { "gnome-cups-manager", "-v", NULL, NULL };
		GtkWidget *window = glade_xml_get_widget (xml, "add_printer_window");
		gtk_widget_destroy (window);
		g_object_unref (xml);

		argv[2] = name;
		g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
			NULL, NULL, NULL, &err);

		if (err != NULL) {
			GtkWidget *dialog = gtk_message_dialog_new_with_markup (NULL, 
				 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL, 
				 GTK_MESSAGE_ERROR,
				 GTK_BUTTONS_OK,
				_("Problems launching gnome-cups-manager for the new printer\n\t<b>%s</b>\n%s"), 
				name, err->message);
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog); 
			g_error_free (err);
		}
		gtk_main_quit ();
	}

	g_free (uri);
	g_free (name);

	return NULL;
}

/****************************************************************************/

DruidPageDescription pages[] = {
	{
		"local_or_remote_page",
		NULL,
		NULL,
		local_or_remote_sensitivity,
		NULL,
		local_or_remote_next,
	},
	{
		"local_location_page",
		local_location_setup,
		NULL,
		local_location_sensitivity,
		local_location_back,
		local_location_next,
	},
	{
		"network_location_page",
		network_location_setup,
		NULL,
		network_location_sensitivity,
		network_location_back,
		network_location_next,
	},
	{
		"details_page",
		details_setup,
		details_prepare,
		details_sensitivity,
		details_back,
		details_next
	},
	{ NULL }
};

static int
delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_main_quit ();
	return FALSE;
}

static void
cancel_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

#define SU_APP	"gnomesu"

int
main (int argc, char *argv[])
{
	GladeXML  *xml;
	GtkWidget *window;

	gnome_program_init ("gnome-cups-add",
			    VERSION,
			    LIBGNOMEUI_MODULE, argc, argv,
			    GNOME_PROGRAM_STANDARD_PROPERTIES,
			    GNOME_PARAM_HUMAN_READABLE_NAME, _("Add a Printer"),
			    NULL);
	glade_init ();
	gnome_cups_ui_init ();

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	if (geteuid () != 0) {
		char **args = g_new (char *, argc+2);
		unsigned i;
		GError *err = NULL;

		args[0] = SU_APP;
		for (i = 0 ; i < argc ; i++)
			args[i+1] = argv[i];
		args[i+1] = NULL;
		g_spawn_async (NULL, args, NULL, G_SPAWN_SEARCH_PATH,
			NULL, NULL, NULL, &err);
		if (err != NULL) {
			GtkWidget *dialog = gtk_message_dialog_new_with_markup (NULL, 
				 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL, 
				 GTK_MESSAGE_ERROR,
				 GTK_BUTTONS_OK,
				_("Problems launching %s as root via %s\n%s"),
				argv[0], SU_APP, err->message);
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog); 
			g_error_free (err);
		}
		exit (0);
	}

	xml = glade_xml_new (GNOME_CUPS_MANAGER_DATADIR "/gnome-cups-add.glade",
			     "add_printer_window",
			     GETTEXT_PACKAGE);

	init_smb_combos (xml); /* before the thread */
#ifdef HAVE_LIBSMBCLIENT
	if (smbc_init (smb_auth_fn, 0) >= 0) {
		if (!g_thread_supported ())
			g_thread_init (NULL);
		g_thread_create (cb_smb_thread, NULL, TRUE, NULL);
		g_timeout_add (200, (GtkFunction)cb_smb_req_handler, xml);
	} else {
		g_warning ("smbc_init returned %s (%i)\nDo you have a ~/.smb/smb.conf file?\n",
			   strerror (errno), errno);
	}
#endif

	window = glade_xml_get_widget (xml, "add_printer_window");
	set_window_icon (window, "gnome-dev-printer-new");
	g_signal_connect (window,
		"delete_event",
		G_CALLBACK (delete_event_cb), NULL);
	g_signal_connect (glade_xml_get_widget (xml, "add_printer_druid"),
		"cancel",
		G_CALLBACK (cancel_cb), NULL);

	druid_pages_setup (xml, pages);
	gtk_widget_show (window);

	gtk_main ();

	return 0;
}
