/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "galeon-bookmarks-icon-provider.h"
#include "gul-gobject-misc.h"
#include "galeon-shell.h"
#include "bookmarks-util.h"

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

static GaleonFaviconCache *favicon_cache = NULL;

/**
 * Private data
 */
struct _GbGaleonIconProviderPrivate 
{
	GbIconProvider *default_provider;

	GHashTable *cache_url_to_bookmark;
	GSList *sets;
};

/**
 * Private functions, only availble from this file
 */
static void		gb_galeon_icon_provider_class_init		(GbGaleonIconProviderClass *klass);
static void		gb_galeon_icon_provider_icon_provider_init	(GbIconProviderIface *iface);
static void		gb_galeon_icon_provider_init			(GbGaleonIconProvider *dip);
static void		gb_galeon_icon_provider_finalize_impl		(GObject *o);
static GdkPixbuf *	gb_galeon_icon_provider_icon_provider_get_icon_impl (GbIconProvider *ip,
									      GbBookmark *b);
static void		gb_galeon_icon_provider_favicon_cache_changed_cb (GaleonFaviconCache *cache,
									  const char *url,
									  GbGaleonIconProvider *gip);
static void		gb_galeon_icon_provider_add_cache_url_to_bookmark (GbGaleonIconProvider *gip, 
									   char *curl, GbBookmark *b);
static void		gb_galeon_icon_provider_weak_notify		(gpointer datap,
									 GObject *where_the_object_was);
static void		gb_galeon_icon_provider_bset_weak_notify	(gpointer datap,
									 GObject *where_the_object_was);


static gpointer g_object_class;

MAKE_GET_TYPE_IFACE (gb_galeon_icon_provider, "GaleonIconProvider", GbGaleonIconProvider,
		     gb_galeon_icon_provider_class_init, gb_galeon_icon_provider_init, G_TYPE_OBJECT,
		     gb_galeon_icon_provider_icon_provider_init, GB_TYPE_ICON_PROVIDER);
/**
 * GaleonIconProvider object
 */

static void
gb_galeon_icon_provider_class_init (GbGaleonIconProviderClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_galeon_icon_provider_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	favicon_cache = galeon_shell_get_favicon_cache (galeon_shell);
}

static void
gb_galeon_icon_provider_icon_provider_init (GbIconProviderIface *iface)
{
	iface->get_icon = gb_galeon_icon_provider_icon_provider_get_icon_impl;
}

static void 
gb_galeon_icon_provider_init (GbGaleonIconProvider *dip)
{
	GbGaleonIconProviderPrivate *p = g_new0 (GbGaleonIconProviderPrivate, 1);
	dip->priv = p;

	p->cache_url_to_bookmark = g_hash_table_new_full (g_str_hash, g_str_equal, 
							  (GDestroyNotify) g_free, NULL);

	g_signal_connect (favicon_cache, "changed", 
			  G_CALLBACK (gb_galeon_icon_provider_favicon_cache_changed_cb), dip);
}

static void
gb_galeon_icon_provider_finalize_impl (GObject *o)
{
	GbGaleonIconProvider *dip = GB_GALEON_ICON_PROVIDER (o);
	GbGaleonIconProviderPrivate *p = dip->priv;
	GSList *li;
	
	if (p->default_provider)
	{
		g_object_unref (p->default_provider);
	}

	for (li = p->sets; li; li = li->next)
	{
		g_object_weak_unref (li->data, gb_galeon_icon_provider_bset_weak_notify, dip);
	}
	g_slist_free (p->sets);


	/* note that we don't have (actually, we can't) remove the weak references to the bookmarks
	   if the cache_url_to_bookmark table, because the data refs this object. Once we get here, 
	   the table size must be 0 */
	if (g_hash_table_size (p->cache_url_to_bookmark) > 0)
	{
		g_warning ("Serious memory management problem! Expect a crash soon.");
	}
	g_hash_table_destroy (p->cache_url_to_bookmark);

	g_free (p);
	
	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GbGaleonIconProvider *
gb_galeon_icon_provider_new (void)
{
	GbGaleonIconProvider *ret = g_object_new (GB_TYPE_GALEON_ICON_PROVIDER, NULL);
	return ret;
}

static GdkPixbuf *
gb_galeon_icon_provider_icon_provider_get_icon_impl (GbIconProvider *ip, GbBookmark *b)
{
	GbGaleonIconProviderPrivate *p = GB_GALEON_ICON_PROVIDER (ip)->priv;
	GdkPixbuf *ret = NULL;

	if (GB_IS_SITE (b))
	{
		const gchar *url = GB_SITE (b)->url;
		gchar *curl;

		if ((!url || url[0] == '\0') && GB_IS_SMART_SITE (b))
		{
			url = GB_SMART_SITE (b)->smarturl;
		}

		curl = galeon_favicon_cache_url (url);
		
		if (curl)
		{
			GbBookmark *rb = gb_bookmark_real_bookmark (b);

			DEBUG_MSG (("getting favicon for %s\ncache url %s\n", url, curl));
			ret = galeon_favicon_cache_lookup_direct (favicon_cache, curl);
			DEBUG_MSG (("got %p\n", ret));
			
			if (rb->set && gb_bookmark_set_get_bookmark_by_url (rb->set, curl) != rb)
			{
				gb_galeon_icon_provider_add_cache_url_to_bookmark 
					(GB_GALEON_ICON_PROVIDER (ip), curl, rb);
			}
			else
			{
				g_free (curl);
			}
		}
	}

	if (!ret && p->default_provider)
	{
		return gb_icon_provider_get_icon (p->default_provider, b);
	}
	else
	{
		if (gb_bookmark_is_alias (b))
		{
			ret = gb_util_make_alias_icon (ret);
		}
		return g_object_ref (ret);
	}
}

void
gb_galeon_icon_provider_set_default_provider (GbGaleonIconProvider *gip, 
					      GbIconProvider *ip)
{
	GbGaleonIconProviderPrivate *p = gip->priv;
	if (p->default_provider)
	{
		g_object_unref (p->default_provider);
	}
	p->default_provider = g_object_ref (ip);
}


static void
gb_galeon_icon_provider_favicon_cache_changed_cb (GaleonFaviconCache *cache, const char *url,
						  GbGaleonIconProvider *gip)
{
	GbGaleonIconProviderPrivate *p = gip->priv;
	GSList *li;
	GbBookmark *b = NULL;
	DEBUG_MSG (("favicon url changed %s\n", url));
	

	b = g_hash_table_lookup (p->cache_url_to_bookmark, url);
	if (b)
	{
		DEBUG_MSG (("found the bookmark in the cache_url_to_bookmark table\n"));
		gb_bookmark_emit_changed (b);
	}

	for (li = p->sets; li; li = li->next)
	{
		b = gb_bookmark_set_get_bookmark_by_url (li->data, url);
		if (b)
		{
			DEBUG_MSG (("found the bookmark in a bookmark set\n"));
			gb_bookmark_emit_changed (b);
		}
	}
}

static void
gb_galeon_icon_provider_add_cache_url_to_bookmark (GbGaleonIconProvider *gip, char *curl, GbBookmark *b)
{
	GbGaleonIconProviderPrivate *p = gip->priv;
	gpointer *data;

	if (g_hash_table_lookup (p->cache_url_to_bookmark, curl) != NULL)
	{
		return;
	}

	g_hash_table_insert (p->cache_url_to_bookmark, curl, b);
	
	data = g_new (gpointer, 2);
	data[0] = g_object_ref (gip);
	data[1] = g_strdup (curl);
	g_object_weak_ref (G_OBJECT (b), gb_galeon_icon_provider_weak_notify, data);
	
	DEBUG_MSG (("Added %s (was %s) to the cache_url_to_bookmark table\n", curl, GB_SITE (b)->url));
	DEBUG_MSG (("There are %d items in the cache_url_to_bookmark table\n", 
		    g_hash_table_size (p->cache_url_to_bookmark)));
}

static void
gb_galeon_icon_provider_bset_weak_notify (gpointer datap,
					  GObject *where_the_object_was)
{
	GbGaleonIconProvider *gip = datap;
	GbGaleonIconProviderPrivate *p = gip->priv;
	p->sets = g_slist_remove (p->sets, where_the_object_was);
}

void
gb_galeon_icon_provider_add_bookmark_set (GbGaleonIconProvider *gip,
					  GbBookmarkSet *set)
{
	GbGaleonIconProviderPrivate *p = gip->priv;
	if (!g_slist_find (p->sets, set))
	{
		p->sets = g_slist_prepend (p->sets, set);
		g_object_weak_ref (G_OBJECT (set), gb_galeon_icon_provider_bset_weak_notify, gip);
	}
}

static void
gb_galeon_icon_provider_weak_notify (gpointer datap,
				     GObject *where_the_object_was)
{
	gpointer *data = datap;
	GbGaleonIconProvider *gip = data[0];
	gchar *key = data[1];
	GbGaleonIconProviderPrivate *p = gip->priv;

	DEBUG_MSG (("removing a bookmark from the cache_url_to_bookmark table, key %s\n", key));

	g_hash_table_remove (p->cache_url_to_bookmark, key);
	g_object_unref (gip);
	g_free (key);
	g_free (datap);

	DEBUG_MSG (("%d items left\n", g_hash_table_size (p->cache_url_to_bookmark)));
}

