/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include "gailcanvasitem.h"
#include "gailcanvastext.h"

static void           gail_canvas_text_class_init          (GailCanvasTextClass *klass);
static void           gail_canvas_text_text_interface_init (AtkTextIface        *iface);
static gchar*         gail_canvas_text_get_text            (AtkText             *text,
                                                            gint                start_offset,
                                                            gint                end_offset);
static gchar*         gail_canvas_text_get_text_after_offset 
                                                           (AtkText             *text,
                                                            gint                offset,
                                                            AtkTextBoundary     boundary_type,
                                                            gint                *start_offset,
                                                            gint                *end_offset);
static gchar*         gail_canvas_text_get_text_at_offset  (AtkText             *text,
                                                            gint                offset,
                                                            AtkTextBoundary     boundary_type,
                                                            gint                *start_offset,
                                                            gint                *end_offset);
static gchar*         gail_canvas_text_get_text_before_offset 
                                                           (AtkText             *text,
                                                            gint                offset,
                                                            AtkTextBoundary     boundary_type,
                                                            gint                *start_offset,
                                                            gint                *end_offset);
static gunichar       gail_canvas_text_get_character_at_offset 
                                                            (AtkText            *text,
                                                             gint               offset);
static gint           gail_canvas_text_get_character_count  (AtkText            *text);
static gint           gail_canvas_text_get_caret_offset     (AtkText            *text);
static gboolean       gail_canvas_text_set_caret_offset     (AtkText            *text,
                                                             gint               offset);
static gint           gail_canvas_text_get_offset_at_point  (AtkText            *text,
                                                             gint               x,
                                                             gint               y,
				                             AtkCoordType       coords);
static void           gail_canvas_text_get_character_extents (AtkText           *text,
                                                              gint              offset,
                                                              gint              *x,
                                                              gint              *y,
                                                              gint              *width,
                                                              gint              *height,
                                                              AtkCoordType      coords);
static AtkAttributeSet* 
                      gail_canvas_text_get_run_attributes    (AtkText           *text,
                                                              gint              offset,
                                                              gint              *start_offset,
                                                              gint              *end_offset);
static AtkAttributeSet* 
                      gail_canvas_text_get_default_attributes (AtkText          *text);
static gint           gail_canvas_text_get_n_selections      (AtkText           *text);
static gchar*         gail_canvas_text_get_selection         (AtkText           *text,
                                                              gint              selection_num,
                                                              gint              *start_pos,
                                                              gint              *end_pos);
static gboolean       gail_canvas_text_add_selection         (AtkText           *text,
                                                              gint              start_pos,
                                                              gint              end_pos);
static gboolean       gail_canvas_text_remove_selection      (AtkText           *text,
                                                              gint              selection_num);
static gboolean       gail_canvas_text_set_selection         (AtkText           *text,
                                                              gint              selection_num,
                                                              gint              start_pos,
                                                              gint              end_pos);
static AtkAttributeSet*
                      add_to_attr_set                        (AtkAttributeSet   *attrib_set,
                                                              GtkTextAttributes *attrs,
                                                              AtkTextAttribute  attr);
static gchar*         get_text_near_offset                   (AtkText           *text,
                                                              GailOffsetType    function,
                                                              AtkTextBoundary   boundary_type,
                                                              gint              offset,
                                                              gint              *start_offset,
                                                              gint              *end_offset);

static AtkObjectClass *parent_class = NULL;

GType
gail_canvas_text_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo tinfo =
      {
        sizeof (GailCanvasTextClass),
        (GBaseInitFunc) NULL, /* base init */
        (GBaseFinalizeFunc) NULL, /* base finalize */
        (GClassInitFunc) gail_canvas_text_class_init, /* class init */
        (GClassFinalizeFunc) NULL, /* class finalize */
        NULL, /* class data */
        sizeof (GailCanvasText), /* instance size */
        0, /* nb preallocs */
        (GInstanceInitFunc) NULL, /* instance init */
        NULL /* value table */
      };

      static const GInterfaceInfo atk_text_info =
      {
        (GInterfaceInitFunc) gail_canvas_text_text_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
      };
      type = g_type_register_static (GAIL_TYPE_CANVAS_ITEM,
                                     "GailCanvasText", &tinfo, 0);
      g_type_add_interface_static (type, ATK_TYPE_TEXT,
                                   &atk_text_info);      
    }

  return type;
}

AtkObject*
gail_canvas_text_new (GnomeCanvasItem *item)
{
  GObject *object;
  AtkObject *atk_object;
  GailCanvasItem *gail_item;
  GailCanvasText *gail_text;

  g_return_val_if_fail (GNOME_IS_CANVAS_ITEM (item), NULL);
  object = g_object_new (GAIL_TYPE_CANVAS_TEXT, NULL);
  gail_item = GAIL_CANVAS_ITEM (object);
  gail_text = GAIL_CANVAS_TEXT (object);

  gail_canvas_item_init (gail_item, item);
  gail_text->texthelper = gail_text_helper_new ();
  if (GNOME_IS_CANVAS_RICH_TEXT (item))
    {
      gail_text_helper_buffer_setup (gail_text->texthelper,
				     gnome_canvas_rich_text_get_buffer (GNOME_CANVAS_RICH_TEXT (item)));
    }
  else if (GNOME_IS_CANVAS_TEXT (item))
    {
      gail_text_helper_text_setup (gail_text->texthelper,
				   GNOME_CANVAS_TEXT(item)->text);
    }

  atk_object = ATK_OBJECT (object);
  atk_object->role =  ATK_ROLE_TEXT;
  return atk_object;
}

static void
gail_canvas_text_class_init (GailCanvasTextClass *klass)
{
  parent_class = g_type_class_ref (ATK_TYPE_OBJECT);
}

static void
gail_canvas_text_text_interface_init (AtkTextIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->get_text = gail_canvas_text_get_text;
  iface->get_text_after_offset = gail_canvas_text_get_text_after_offset;
  iface->get_text_at_offset = gail_canvas_text_get_text_at_offset;
  iface->get_text_before_offset = gail_canvas_text_get_text_before_offset;
  iface->get_character_at_offset = gail_canvas_text_get_character_at_offset;
  iface->get_character_count = gail_canvas_text_get_character_count;
  iface->get_caret_offset = gail_canvas_text_get_caret_offset;
  iface->set_caret_offset = gail_canvas_text_set_caret_offset;
  iface->get_offset_at_point = gail_canvas_text_get_offset_at_point;
  iface->get_character_extents = gail_canvas_text_get_character_extents;
  iface->get_n_selections = gail_canvas_text_get_n_selections;
  iface->get_selection = gail_canvas_text_get_selection;
  iface->add_selection = gail_canvas_text_add_selection;
  iface->remove_selection = gail_canvas_text_remove_selection;
  iface->set_selection = gail_canvas_text_set_selection;
  iface->get_run_attributes = gail_canvas_text_get_run_attributes;
  iface->get_default_attributes = gail_canvas_text_get_default_attributes;
}

static gchar*
gail_canvas_text_get_text (AtkText *text,
                           gint    start_offset,
                           gint    end_offset)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter start, end;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), NULL);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, NULL);

  buffer = gail_text->texthelper->buffer;
  gtk_text_buffer_get_iter_at_offset (buffer, &start, start_offset);
  gtk_text_buffer_get_iter_at_offset (buffer, &end, end_offset);

  return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
}

static gchar*
gail_canvas_text_get_text_after_offset (AtkText         *text,
                                        gint            offset,
                                        AtkTextBoundary boundary_type,
                                        gint            *start_offset,
                                        gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_AFTER_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gchar*
gail_canvas_text_get_text_at_offset (AtkText         *text,
                                     gint            offset,
                                     AtkTextBoundary boundary_type,
                                     gint            *start_offset,
                                     gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_AT_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gchar*
gail_canvas_text_get_text_before_offset (AtkText         *text,
                                         gint            offset,
                                         AtkTextBoundary boundary_type,
                                         gint            *start_offset,
                                         gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_BEFORE_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gunichar
gail_canvas_text_get_character_at_offset (AtkText *text,
                                          gint    offset)
{
  GailCanvasText *gail_item;
  GtkTextIter start, end;
  GtkTextBuffer *buffer;
  gchar *string;
  gchar *index;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), 0);
  gail_item = GAIL_CANVAS_TEXT (text);
  buffer = gail_item->texthelper->buffer;
  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
  string = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
  index = g_utf8_offset_to_pointer (string, offset);
  g_free (string);

  return g_utf8_get_char (index);
}

static gint
gail_canvas_text_get_character_count (AtkText *text)
{
  GtkTextBuffer *buffer;
  GailCanvasText *gail_text;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), 0);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, 0);
  buffer = gail_text->texthelper->buffer;
  return gtk_text_buffer_get_char_count (buffer);
}

static gint
gail_canvas_text_get_caret_offset (AtkText *text)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextMark *cursor_mark;
  GtkTextIter cursor_itr;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), 0);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, 0);
  buffer = gail_text->texthelper->buffer;
  cursor_mark = gtk_text_buffer_get_insert (buffer);
  gtk_text_buffer_get_iter_at_mark (buffer, &cursor_itr, cursor_mark);
  return gtk_text_iter_get_offset (&cursor_itr);
}

static gboolean
gail_canvas_text_set_caret_offset (AtkText *text,
                                   gint    offset)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter pos_itr;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), FALSE);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, FALSE);
  buffer = gail_text->texthelper->buffer;
  gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, offset);
  gtk_text_buffer_move_mark_by_name (buffer, "insert", &pos_itr);
  return TRUE;
}

static gint
gail_canvas_text_get_offset_at_point (AtkText      *text,
                                      gint         x,
                                      gint         y,
                                      AtkCoordType coords)
{
  return -1;
}

static void
gail_canvas_text_get_character_extents (AtkText      *text,
                                        gint         offset,
                                        gint         *x,
                                        gint         *y,
                                        gint         *width,
                                        gint         *height,
                                        AtkCoordType coords)
{
  return;
}

static AtkAttributeSet*
gail_canvas_text_get_run_attributes (AtkText *text,
                                     gint    offset,
                                     gint    *start_offset,
                                     gint    *end_offset)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter iter;
  AtkAttributeSet *attrib_set = NULL;
  AtkAttribute *at;
  GSList *tags, *temp_tags;
  gdouble scale = 1;
  gboolean val_set = FALSE;
  PangoFontMask mask;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), NULL);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, NULL);
  buffer = gail_text->texthelper->buffer;
  gtk_text_buffer_get_iter_at_offset (buffer, &iter, offset);

  gtk_text_iter_forward_to_tag_toggle (&iter, NULL);
  *end_offset = gtk_text_iter_get_offset (&iter);

  gtk_text_iter_backward_to_tag_toggle (&iter, NULL);
  *start_offset = gtk_text_iter_get_offset (&iter);

  gtk_text_buffer_get_iter_at_offset (buffer, &iter, offset);

  tags = gtk_text_iter_get_tags (&iter);
  tags = g_slist_reverse (tags);

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG (temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_STYLE;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_STYLE);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_VARIANT;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_VARIANT);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_STRETCH;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_STRETCH);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->justification_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_JUSTIFICATION);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      if (tag->values->direction != GTK_TEXT_DIR_NONE)
        {
          val_set = TRUE;
          attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                        ATK_TEXT_ATTR_DIRECTION);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->wrap_mode_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_WRAP_MODE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->fg_stipple_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_FG_STIPPLE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->bg_stipple_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_BG_STIPPLE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->fg_color_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_FG_COLOR);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
  
      val_set = tag->bg_color_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_BG_COLOR);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_FAMILY;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_FAMILY_NAME);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->language_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_LANGUAGE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_WEIGHT;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_WEIGHT);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;


  /*
   * scale is special as the scale is the product of all scale values
   * specified.
   */
  temp_tags = tags;
  while (temp_tags)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      if (tag->scale_set)
        {
          val_set = TRUE;
          scale *= tag->values->font_scale;
        }
      temp_tags = temp_tags->next;
    }
  if (val_set)
    {
      at = g_malloc(sizeof(AtkAttribute));
      at->name = g_strdup(atk_attribute_get_name (ATK_TEXT_ATTR_SCALE));
      at->value = g_strdup_printf("%g", scale);
      attrib_set = g_slist_prepend(attrib_set, at);
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);
      PangoFontDescription *font;

      font = tag->values->font;

      if (font)
        {
          mask = pango_font_description_get_set_fields (font);
          val_set = mask & PANGO_FONT_MASK_SIZE;
          if (val_set)
            attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                          ATK_TEXT_ATTR_SIZE);
        }
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->strikethrough_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_STRIKETHROUGH);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->underline_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_UNDERLINE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->rise_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_RISE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->bg_full_height_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_BG_FULL_HEIGHT);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->pixels_inside_wrap_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_PIXELS_INSIDE_WRAP);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->pixels_below_lines_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_PIXELS_BELOW_LINES);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->pixels_above_lines_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_PIXELS_ABOVE_LINES);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->editable_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_EDITABLE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->invisible_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_INVISIBLE);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->indent_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_INDENT);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->right_margin_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_RIGHT_MARGIN);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  temp_tags = tags;
  while (temp_tags && !val_set)
    {
      GtkTextTag *tag = GTK_TEXT_TAG(temp_tags->data);

      val_set = tag->left_margin_set;
      if (val_set)
        attrib_set = add_to_attr_set (attrib_set, tag->values, 
                                      ATK_TEXT_ATTR_LEFT_MARGIN);
      temp_tags = temp_tags->next;
    }
  val_set = FALSE;

  g_slist_free (tags);
  return attrib_set;
}

static AtkAttributeSet*
gail_canvas_text_get_default_attributes (AtkText *text)
{
  return NULL;
}

static gint
gail_canvas_text_get_n_selections (AtkText *text)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter start, end;
  gint select_start, select_end;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), -1);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, -1);
  buffer = gail_text->texthelper->buffer;
  
  gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
  select_start = gtk_text_iter_get_offset (&start);
  select_end = gtk_text_iter_get_offset (&end);

  if (select_start != select_end)
     return 1;
  else
     return 0;
}

static gchar*
gail_canvas_text_get_selection (AtkText *text,
                                gint    selection_num,
                                gint    *start_pos,
                                gint    *end_pos)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter start, end;

 /* Only let the user get the selection if one is set, and if the
  * selection_num is 0.
  */
  if (selection_num != 0)
     return NULL;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), NULL);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, NULL);
  buffer = gail_text->texthelper->buffer;

  gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
  *start_pos = gtk_text_iter_get_offset (&start);
  *end_pos = gtk_text_iter_get_offset (&end);

  if (*start_pos != *end_pos)
    return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
  else
    return NULL;
}

static gboolean
gail_canvas_text_add_selection (AtkText *text,
                                gint    start_pos,
                                gint    end_pos)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter pos_itr;
  GtkTextIter start, end;
  gint select_start, select_end;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), FALSE);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, FALSE);
  buffer = gail_text->texthelper->buffer;

  gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
  select_start = gtk_text_iter_get_offset (&start);
  select_end = gtk_text_iter_get_offset (&end);

 /* If there is already a selection, then don't allow another to be added,
  * since GtkTextView only supports one selected region.
  */
  if (select_start == select_end)
    {
      gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, start_pos);
      gtk_text_buffer_move_mark_by_name (buffer, "insert", &pos_itr);
      gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, end_pos);
      gtk_text_buffer_move_mark_by_name (buffer, "selection_bound", &pos_itr);
      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
gail_canvas_text_remove_selection (AtkText *text,
                                 gint    selection_num)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextMark *cursor_mark;
  GtkTextIter cursor_itr;
  GtkTextIter start, end;
  gint select_start, select_end;

  if (selection_num != 0)
     return FALSE;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), FALSE);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, FALSE);
  buffer = gail_text->texthelper->buffer;

  gtk_text_buffer_get_selection_bounds(buffer, &start, &end);
  select_start = gtk_text_iter_get_offset(&start);
  select_end = gtk_text_iter_get_offset(&end);

  if (select_start != select_end)
    {
     /* Setting the start & end of the selected region to the caret position
      * turns off the selection.
      */
      cursor_mark = gtk_text_buffer_get_insert (buffer);
      gtk_text_buffer_get_iter_at_mark (buffer, &cursor_itr, cursor_mark);
      gtk_text_buffer_move_mark_by_name (buffer, "insert", &cursor_itr);
      gtk_text_buffer_move_mark_by_name (buffer, "selection_bound", &cursor_itr);
      return TRUE;
    }
  else
    return FALSE;
}



static gboolean
gail_canvas_text_set_selection (AtkText *text,
                              gint    selection_num,
                              gint    start_pos,
                              gint    end_pos)
{
  GailCanvasText *gail_text;
  GtkTextBuffer *buffer;
  GtkTextIter pos_itr;
  GtkTextIter start, end;
  gint select_start, select_end;

 /* Only let the user move the selection if one is set, and if the
  * selection_num is 0
  */
  if (selection_num != 0)
     return FALSE;

  g_return_val_if_fail (GAIL_IS_CANVAS_TEXT (text), FALSE);
  gail_text = GAIL_CANVAS_TEXT (text);
  g_return_val_if_fail (gail_text->texthelper, FALSE);
  buffer = gail_text->texthelper->buffer;

  gtk_text_buffer_get_selection_bounds(buffer, &start, &end);
  select_start = gtk_text_iter_get_offset(&start);
  select_end = gtk_text_iter_get_offset(&end);

  if (select_start != select_end)
    {
      gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, start_pos);
      gtk_text_buffer_move_mark_by_name (buffer, "insert", &pos_itr);
      gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, end_pos);
      gtk_text_buffer_move_mark_by_name (buffer, "selection_bound", &pos_itr);
      return TRUE;
    }
  else
    return FALSE;
}

static AtkAttributeSet*
add_to_attr_set (AtkAttributeSet   *attrib_set,
                 GtkTextAttributes *attrs,
                 AtkTextAttribute  attr)
{
  gchar *value;

  switch (attr)
    {
    case ATK_TEXT_ATTR_LEFT_MARGIN:
      value = g_strdup_printf ("%i", attrs->left_margin);
      break;
    case ATK_TEXT_ATTR_RIGHT_MARGIN:
      value = g_strdup_printf ("%i", attrs->right_margin);
      break;
    case ATK_TEXT_ATTR_INDENT:
      value = g_strdup_printf ("%i", attrs->indent);
      break;
    case ATK_TEXT_ATTR_INVISIBLE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->invisible));
      break;
    case ATK_TEXT_ATTR_EDITABLE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->editable));
      break;
    case ATK_TEXT_ATTR_PIXELS_ABOVE_LINES:
      value = g_strdup_printf ("%i", attrs->pixels_above_lines);
      break;
    case ATK_TEXT_ATTR_PIXELS_BELOW_LINES:
      value = g_strdup_printf ("%i", attrs->pixels_below_lines);
      break;
    case ATK_TEXT_ATTR_PIXELS_INSIDE_WRAP:
      value = g_strdup_printf ("%i", attrs->pixels_inside_wrap);
      break;
    case ATK_TEXT_ATTR_BG_FULL_HEIGHT:
      value = g_strdup (atk_attribute_get_value (attr, attrs->bg_full_height));
      break;
    case ATK_TEXT_ATTR_RISE:
      value = g_strdup_printf ("%i", attrs->appearance.rise);
      break;
    case ATK_TEXT_ATTR_UNDERLINE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->appearance.underline));
      break;
    case ATK_TEXT_ATTR_STRIKETHROUGH:
      value = g_strdup (atk_attribute_get_value (attr, attrs->appearance.strikethrough));
      break;
    case ATK_TEXT_ATTR_SIZE:
      value = g_strdup_printf ("%i", 
                              pango_font_description_get_size (attrs->font));
      break;
    case ATK_TEXT_ATTR_SCALE:
      value = g_strdup_printf ("%g", attrs->font_scale);
      break;
    case ATK_TEXT_ATTR_WEIGHT:
      value = g_strdup_printf ("%d", 
                              pango_font_description_get_weight (attrs->font));
      break;
    case ATK_TEXT_ATTR_LANGUAGE:
      value = g_strdup ((gchar *)(attrs->language));
      break;
    case ATK_TEXT_ATTR_FAMILY_NAME:
      value = g_strdup (pango_font_description_get_family (attrs->font));
      break;
    case ATK_TEXT_ATTR_BG_COLOR:
      value = g_strdup_printf ("%u,%u,%u",
                               attrs->appearance.bg_color.red,
                               attrs->appearance.bg_color.green,
                               attrs->appearance.bg_color.blue);
      break;
    case ATK_TEXT_ATTR_FG_COLOR:
      value = g_strdup_printf ("%u,%u,%u",
                               attrs->appearance.fg_color.red,
                               attrs->appearance.fg_color.green,
                               attrs->appearance.fg_color.blue);
      break;
    case ATK_TEXT_ATTR_BG_STIPPLE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->appearance.bg_stipple ? 1 : 0));
      break;
    case ATK_TEXT_ATTR_FG_STIPPLE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->appearance.fg_stipple ? 1 : 0));
      break;
    case ATK_TEXT_ATTR_WRAP_MODE:
      value = g_strdup (atk_attribute_get_value (attr, attrs->wrap_mode));
      break;
    case ATK_TEXT_ATTR_DIRECTION:
      value = g_strdup (atk_attribute_get_value (attr, attrs->direction));
      break;
    case ATK_TEXT_ATTR_JUSTIFICATION:
      value = g_strdup (atk_attribute_get_value (attr, attrs->justification));
      break;
    case ATK_TEXT_ATTR_STRETCH:
      value = g_strdup (atk_attribute_get_value (attr, 
                        pango_font_description_get_stretch (attrs->font)));
      break;
    case ATK_TEXT_ATTR_VARIANT:
      value = g_strdup (atk_attribute_get_value (attr, 
                        pango_font_description_get_variant (attrs->font)));
      break;
    case ATK_TEXT_ATTR_STYLE:
      value = g_strdup (atk_attribute_get_value (attr, 
                        pango_font_description_get_style (attrs->font)));
      break;
    default:
      value = NULL;
      break;
    }
  return gail_text_helper_add_attribute (attrib_set, 
                                         attr,
                                         value);
}

static gchar*
get_text_near_offset (AtkText          *text,
                      GailOffsetType   function,
                      AtkTextBoundary  boundary_type,
                      gint             offset,
                      gint             *start_offset,
                      gint             *end_offset)
{
  return gail_text_helper_get_text (GAIL_CANVAS_TEXT (text)->texthelper, NULL,
                                    function, boundary_type, offset, 
                                    start_offset, end_offset);
}
