/*
 * Copyright (C) 2008 Nokia Corporation.
 * Copyright (C) 2008 Zeeshan Ali (Khattak) <zeeshanak@gnome.org>.
 * Copyright (C) 2010 Collabora Ltd.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *          Travis Reitter <travis.reitter@collabora.co.uk>
 *
 * This file was originally part of Rygel.
 */

using Gee;
using GLib;

/**
 * Responsible for backend loading. Probes for shared library files in a
 * specific directory, looking for (and calling) a specific function (by name,
 * signature).
 */
public class Folks.BackendStore : Object {
  [CCode (has_target = false)]
  private delegate void ModuleInitFunc (BackendStore store);
  [CCode (has_target = false)]
  private delegate void ModuleFinalizeFunc (BackendStore store);

  private HashMap<string,Backend> backend_hash;
  private HashMap<string,Backend> _prepared_backends;
  private GLib.List<ModuleFinalizeFunc> finalize_funcs = null;
  private static weak BackendStore instance;
  private static bool _backends_loaded = false;

  /**
   * Emitted when a backend has been added to the BackendStore.
   *
   * This will not be emitted until after {@link BackendStore.load_backends}
   * has been called.
   *
   * {@link Backend}s referenced in this signal are also included in
   * {@link BackendStore.enabled_backends}.
   *
   * @param backend the new {@link Backend}
   */
  public signal void backend_available (Backend backend);

  /**
   * The list of backends visible to this store which have not been explicitly
   * disabled.
   *
   * This list will be empty before {@link BackendStore.load_backends} has been
   * called.
   *
   * The backends in this list have been prepared and are ready to use.
   *
   * @since 0.2.0
   */
  public GLib.List<Backend> enabled_backends
    {
      owned get
        {
          var backends = new GLib.List<Backend> ();
          foreach (var entry in this._prepared_backends)
            backends.prepend (entry.value);

          return backends;
        }

      private set {}
    }

  /**
   * Create a new BackendStore.
   */
  public static BackendStore dup ()
    {
      if (instance == null)
        {
          /* use an intermediate variable to force a strong reference */
          var new_instance = new BackendStore ();
          instance = new_instance;

          return new_instance;
        }

      return instance;
    }

  private BackendStore ()
    {
      /* Treat this as a library init function */
      Debug.set_flags (Environment.get_variable ("FOLKS_DEBUG"));

      this.backend_hash = new HashMap<string,Backend> (str_hash, str_equal);
      this._prepared_backends = new HashMap<string,Backend> (str_hash,
          str_equal);
    }

  ~BackendStore ()
    {
      /* Finalize all the loaded modules */
      foreach (ModuleFinalizeFunc func in this.finalize_funcs)
        func (this);

      /* reset status of backends */
      lock (this._backends_loaded)
        this._backends_loaded = false;

      /* manually clear the singleton instance */
      instance = null;
    }

  /**
   * Find, load, and prepare all backends which are not disabled.
   *
   * Backends will be searched for in the path given by the `FOLKS_BACKEND_DIR`
   * environment variable, if it's set. If it's not set, backends will be
   * searched for in a path set at compilation time.
   */
  public async void load_backends () throws GLib.Error
    {
      lock (this._backends_loaded)
        {
          if (!this._backends_loaded)
            {
              assert (Module.supported());

              var path = Environment.get_variable ("FOLKS_BACKEND_DIR");
              if (path == null)
                {
                  path = BuildConf.BACKEND_DIR;

                  debug ("Using built-in backend dir '%s' (override with " +
                      "environment variable FOLKS_BACKEND_DIR)", path);
                }
              else
                {
                  debug ("Using environment variable FOLKS_BACKEND_DIR = " +
                      "'%s' to look for backends", path);
                }

              File dir = File.new_for_path (path);
              assert (dir != null && yield is_dir (dir));

              yield this.load_modules_from_dir (dir);

              /* all the found modules should call add_backend() for their
               * backends (adding them to the backend hash) as a consequence of
               * load_modules_from_dir(). */

              foreach (var backend in this.backend_hash.values)
                {
                  try
                    {
                      yield backend.prepare ();

                      debug ("New backend '%s' prepared", backend.name);
                      this._prepared_backends.set (backend.name, backend);
                      this.backend_available (backend);
                    }
                  catch (GLib.Error e)
                    {
                      warning ("Error preparing Backend '%s': %s", backend.name,
                          e.message);
                    }
                }

              this._backends_loaded = true;
            }
        }
    }

  /**
   * Add a new {@link Backend} to the BackendStore.
   *
   * @param backend the {@link Backend} to add
   */
  public void add_backend (Backend backend)
    {
      this.backend_hash.set (backend.name, backend);
    }

  /**
   * Get a backend from the store by name.
   *
   * @param name the backend name to retrieve
   * @return the backend, or `null` if none could be found
   */
  public Backend? get_backend_by_name (string name)
    {
      return this.backend_hash.get (name);
    }

  /**
   * List the currently loaded backends.
   *
   * @return a list of the backends currently in the BackendStore
   */
  public Collection<Backend> list_backends ()
    {
      return this.backend_hash.values;
    }

  private async void load_modules_from_dir (File dir)
    {
      debug ("Searching for modules in folder '%s' ..", dir.get_path ());

      string attributes = FILE_ATTRIBUTE_STANDARD_NAME + "," +
                          FILE_ATTRIBUTE_STANDARD_TYPE + "," +
                          FILE_ATTRIBUTE_STANDARD_IS_SYMLINK + "," +
                          FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE;

      GLib.List<FileInfo> infos;
      FileEnumerator enumerator;

      try
        {
          enumerator = yield dir.enumerate_children_async (attributes,
              FileQueryInfoFlags.NONE, Priority.DEFAULT, null);

          infos = yield enumerator.next_files_async (int.MAX,
              Priority.DEFAULT, null);
        }
      catch (Error error)
        {
          critical ("Error listing contents of folder '%s': %s",
              dir.get_path (), error.message);

          return;
        }

      foreach (var info in infos)
        {
          string file_name = info.get_name ();
          string file_path = Path.build_filename (dir.get_path (), file_name);

          File file = File.new_for_path (file_path);
          FileType file_type = info.get_file_type ();
          unowned string content_type = info.get_content_type ();
          /* don't load the library multiple times for its various symlink
           * aliases */
          var is_symlink = info.get_is_symlink ();

          string mime = g_content_type_get_mime_type (content_type);

          if (file_type == FileType.DIRECTORY)
            {
              yield this.load_modules_from_dir (file);
            }
          else if (mime == "application/x-sharedlib" && !is_symlink)
            {
              this.load_module_from_file (file_path);
            }
          else if (mime == null)
            {
              warning ("MIME type could not be determined for file '%s'. " +
                  "Have you installed shared-mime-info?", file_path);
            }
        }

      debug ("Finished searching for modules in folder '%s'",
          dir.get_path ());
    }

  private void load_module_from_file (string file_path)
    {
      Module module = Module.open (file_path, ModuleFlags.BIND_LOCAL);
      if (module == null)
        {
          warning ("Failed to load module from path '%s' : %s",
                    file_path, Module.error ());

          return;
        }

      void* function;

      if (!module.symbol("module_init", out function))
        {
          warning ("Failed to find entry point function '%s' in '%s': %s",
                    "module_init",
                    file_path,
                    Module.error ());

          return;
        }

      ModuleInitFunc module_init = (ModuleInitFunc) function;
      assert (module_init != null);

      /* It's optional for modules to have a finalize function */
      if (module.symbol ("module_finalize", out function))
        {
          ModuleFinalizeFunc module_finalize = (ModuleFinalizeFunc) function;
          this.finalize_funcs.prepend (module_finalize);
        }

      /* We don't want our modules to ever unload */
      module.make_resident ();

      module_init (this);

      debug ("Loaded module source: '%s'", module.name ());
    }

  private async static bool is_dir (File file)
    {
      FileInfo file_info;

      try
        {
          /* Query for the MIME type; if the file doesn't exist, we'll get an
           * appropriate error back, so this also checks for existence. */
          file_info = yield file.query_info_async (FILE_ATTRIBUTE_STANDARD_TYPE,
              FileQueryInfoFlags.NONE, Priority.DEFAULT, null);
        }
      catch (Error error)
        {
          if (error is IOError.NOT_FOUND)
            critical ("File or directory '%s' does not exist",
                      file.get_path ());
          else
            critical ("Failed to get content type for '%s'", file.get_path ());

          return false;
        }

      return file_info.get_file_type () == FileType.DIRECTORY;
    }
}
