/*
 * Copyright (C) 2010 Collabora Ltd.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *       Travis Reitter <travis.reitter@collabora.co.uk>
 */

using BuildConf;
using Folks;

/**
 * A single backend to libfolks, such as Telepathy or evolution-data-server.
 * Each backend provides {@link Persona}s which are aggregated to form
 * {@link Individual}s.
 *
 * After creating a Backend instance, you must connect to the
 * {@link Backend.persona_store_added} and
 * {@link Backend.persona_store_removed} signals, //then// call
 * {@link Backend.prepare}, otherwise a race condition may occur between
 * emission of {@link Backend.persona_store_added} and your code connecting to
 * it.
 */
public abstract class Folks.Backend : Object
{
  /**
   * A unique name for the backend.
   *
   * This will be used to identify the backend, and should also be used as the
   * {@link PersonaStore.type_id} of the {@link PersonaStore}s used by the
   * backend.
   *
   * This is guaranteed to always be available; even before
   * {@link Backend.prepare} is called.
   */
  public abstract string name { get; protected set; }

  /**
   * The {@link PersonaStore}s in use by the backend.
   *
   * A backend may expose {@link Persona}s from multiple servers or accounts
   * (for example), so may have a {@link PersonaStore} for each.
   */
  public abstract HashTable<string, PersonaStore> persona_stores
    {
      get; protected set;
    }

  /**
   * Emitted when a {@link PersonaStore} is added to the backend.
   *
   * This will not be emitted until after {@link Backend.prepare} has been
   * called.
   *
   * @param store the {@link PersonaStore}
   */
  public abstract signal void persona_store_added (PersonaStore store);

  /**
   * Emitted when a {@link PersonaStore} is removed from the backend.
   *
   * This will not be emitted until after {@link Backend.prepare} has been
   * called.
   *
   * @param store the {@link PersonaStore}
   */
  public abstract signal void persona_store_removed (PersonaStore store);

  /**
   * Prepare the Backend for use.
   *
   * This connects the Backend to whichever backend-specific services it
   * requires, and causes it to create its {@link PersonaStore}s. This should be
   * called //after// connecting to the {@link Backend.persona_store_added} and
   * {@link Backend.persona_store_removed} signals, or a race condition could
   * occur, with the signals being emitted before your code has connected to
   * them, and {@link PersonaStore}s getting "lost" as a result.
   *
   * This is normally handled transparently by the {@link IndividualAggregator}.
   *
   * If this function throws an error, the Backend will not be functional.
   */
  public abstract async void prepare () throws GLib.Error;
}
