/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "couchdb-glib.h"
#include "utils.h"

/*
 * CouchDBDatabaseInfo object
 */

GType
couchdb_database_info_get_type (void)
{
	static GType object_type = 0;

	if (G_UNLIKELY (!object_type))
		object_type = g_boxed_type_register_static (g_intern_static_string ("CouchDBDatabaseInfo"),
							    (GBoxedCopyFunc) couchdb_database_info_ref,
							    (GBoxedFreeFunc) couchdb_database_info_unref);

	return object_type;
}

CouchDBDatabaseInfo *
couchdb_database_info_new (const char *dbname,
			   gint doc_count,
			   gint doc_del_count,
			   gint update_seq,
			   gboolean compact_running,
			   gint disk_size)
{
	CouchDBDatabaseInfo *dbinfo;

	dbinfo = g_slice_new (CouchDBDatabaseInfo);
	dbinfo->ref_count = 1;
	dbinfo->dbname = g_strdup (dbname);
	dbinfo->doc_count = doc_count;
	dbinfo->doc_del_count = doc_del_count;
	dbinfo->update_seq = update_seq;
	dbinfo->compact_running = compact_running;
	dbinfo->disk_size = disk_size;

	return dbinfo;
}

CouchDBDatabaseInfo *
couchdb_database_info_ref (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, NULL);
	g_return_val_if_fail (dbinfo->ref_count > 0, NULL);

	g_atomic_int_exchange_and_add (&dbinfo->ref_count, 1);

	return dbinfo;
}

void
couchdb_database_info_unref (CouchDBDatabaseInfo *dbinfo)
{
	gint old_ref;

	g_return_if_fail (dbinfo != NULL);
	g_return_if_fail (dbinfo->ref_count > 0);

	old_ref = g_atomic_int_get (&dbinfo->ref_count);
	if (old_ref > 1)
		g_atomic_int_compare_and_exchange (&dbinfo->ref_count, old_ref, old_ref - 1);
	else {
		g_free (dbinfo->dbname);
		g_slice_free (CouchDBDatabaseInfo, dbinfo);
	}
}

const char *
couchdb_database_info_get_dbname (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, NULL);

	return (const char *) dbinfo->dbname;
}

gint
couchdb_database_info_get_documents_count (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->doc_count;
}

gint
couchdb_database_info_get_deleted_documents_count (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->doc_del_count;
}

gint
couchdb_database_info_get_update_sequence (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->update_seq;
}

gboolean
couchdb_database_info_is_compact_running (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, FALSE);

	return dbinfo->compact_running;
}

gint
couchdb_database_info_get_disk_size (CouchDBDatabaseInfo *dbinfo)
{
	g_return_val_if_fail (dbinfo != NULL, 0);

	return dbinfo->disk_size;
}

/*
 * CouchDBDocumentInfo object
 */

GType
couchdb_document_info_get_type (void)
{
	static GType object_type = 0;

	if (G_UNLIKELY (!object_type))
		object_type = g_boxed_type_register_static (g_intern_static_string ("CouchDBDocumentInfo"),
							    (GBoxedCopyFunc) couchdb_document_info_ref,
							    (GBoxedFreeFunc) couchdb_document_info_unref);

	return object_type;
}

CouchDBDocumentInfo *
couchdb_document_info_new (const char *docid, const char *revision)
{
	CouchDBDocumentInfo *doc_info;

	doc_info = g_slice_new (CouchDBDocumentInfo);
	doc_info->ref_count = 1;
	doc_info->docid = g_strdup (docid);
	doc_info->revision = g_strdup (revision);

	return doc_info;
}

CouchDBDocumentInfo *
couchdb_document_info_ref (CouchDBDocumentInfo *doc_info)
{
	g_return_val_if_fail (doc_info != NULL, NULL);
	g_return_val_if_fail (doc_info->ref_count > 0, NULL);

	g_atomic_int_exchange_and_add (&doc_info->ref_count, 1);

	return doc_info;
}

void
couchdb_document_info_unref (CouchDBDocumentInfo *doc_info)
{
	gint old_ref;

	g_return_if_fail (doc_info != NULL);
	g_return_if_fail (doc_info->ref_count > 0);

	old_ref = g_atomic_int_get (&doc_info->ref_count);
	if (old_ref > 1)
		g_atomic_int_compare_and_exchange (&doc_info->ref_count, old_ref, old_ref - 1);
	else {
		g_free (doc_info->docid);
		g_free (doc_info->revision);
		g_slice_free (CouchDBDocumentInfo, doc_info);
	}
}

const char *
couchdb_document_info_get_docid (CouchDBDocumentInfo *doc_info)
{
	g_return_val_if_fail (doc_info != NULL, NULL);

	return (const char *) doc_info->docid;
}

const char *
couchdb_document_info_get_revision (CouchDBDocumentInfo *doc_info)
{
	g_return_val_if_fail (doc_info != NULL, NULL);

	return (const char *) doc_info->revision;
}

/*
 * CouchDBStructField
 */
GType
couchdb_struct_field_get_type (void)
{
	static GType object_type = 0;

	if (G_UNLIKELY (!object_type))
		object_type = g_boxed_type_register_static (g_intern_static_string ("CouchDBStructField"),
							    (GBoxedCopyFunc) couchdb_struct_field_ref,
							    (GBoxedFreeFunc) couchdb_struct_field_unref);

	return object_type;
}

CouchDBStructField *
couchdb_struct_field_new (void)
{
	CouchDBStructField *sf;

	sf = g_slice_new (CouchDBStructField);
	sf->ref_count = 1;
	sf->json_object = json_object_new ();

	return sf;
}

CouchDBStructField *
couchdb_struct_field_new_from_json_object (JsonObject *json_object)
{
	CouchDBStructField *sf;

	sf = g_slice_new (CouchDBStructField);
	sf->ref_count = 1;
	sf->json_object = json_object_ref (json_object);

	return sf;
}

CouchDBStructField *
couchdb_struct_field_ref (CouchDBStructField *sf)
{
	g_return_val_if_fail (sf != NULL, NULL);
	g_return_val_if_fail (sf->ref_count > 0, NULL);

	g_atomic_int_exchange_and_add (&sf->ref_count, 1);

	return sf;
}

void
couchdb_struct_field_unref (CouchDBStructField *sf)
{
	gint old_ref;

	g_return_if_fail (sf != NULL);
	g_return_if_fail (sf->ref_count > 0);

	old_ref = g_atomic_int_get (&sf->ref_count);
	if (old_ref > 1)
		g_atomic_int_compare_and_exchange (&sf->ref_count, old_ref, old_ref - 1);
	else {
		json_object_unref (sf->json_object);
		g_slice_free (CouchDBStructField, sf);
	}
}

gboolean
couchdb_struct_field_has_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, FALSE);
	g_return_val_if_fail (field != NULL, field);

	return json_object_has_member (sf->json_object, field);
}

void
couchdb_struct_field_remove_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, 0);
	g_return_val_if_fail (field != NULL, 0);

	json_object_remove_member (sf->json_object, field);
}

gboolean
couchdb_struct_field_get_boolean_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, 0);
	g_return_val_if_fail (field != NULL, 0);

	return json_object_get_boolean_member (sf->json_object, field);
}

void
couchdb_struct_field_set_boolean_field (CouchDBStructField *sf, const char *field, gboolean value)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (field != NULL);

	json_object_set_boolean_member (sf->json_object, field, value);
}

gdouble
couchdb_struct_field_get_double_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, 0);
	g_return_val_if_fail (field != NULL, 0);

	return json_object_get_double_member (sf->json_object, field);
}

void
couchdb_struct_field_set_double_field (CouchDBStructField *sf, const char *field, gdouble value)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (field != NULL);

	json_object_set_double_member (sf->json_object, field, value);
}

gint
couchdb_struct_field_get_int_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, 0);
	g_return_val_if_fail (field != NULL, 0);

	return json_object_get_int_member (sf->json_object, field);
}

void
couchdb_struct_field_set_int_field (CouchDBStructField *sf, const char *field, gint value)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (field != NULL);

	json_object_set_int_member (sf->json_object, field, value);
}

const char *
couchdb_struct_field_get_string_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, NULL);
	g_return_val_if_fail (field != NULL, NULL);

	return json_object_get_string_member (sf->json_object, field);
}

void
couchdb_struct_field_set_string_field (CouchDBStructField *sf, const char *field, const char *value)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (field != NULL);
	g_return_if_fail (value != NULL);

	json_object_set_string_member (sf->json_object, field, value);
}

CouchDBStructField *
couchdb_struct_field_get_struct_field (CouchDBStructField *sf, const char *field)
{
	g_return_val_if_fail (sf != NULL, NULL);
	g_return_val_if_fail (field != NULL, NULL);

	return couchdb_struct_field_new_from_json_object (
		json_object_get_object_member (sf->json_object, field));
}

void
couchdb_struct_field_set_struct_field (CouchDBStructField *sf, const char *field, CouchDBStructField *value)
{
	g_return_if_fail (sf != NULL);
	g_return_if_fail (field != NULL);
	g_return_if_fail (value != NULL);

	json_object_set_object_member (sf->json_object, field, json_object_ref (value->json_object));
}
