/* ACME
 * Copyright (C) 2001 Bastien Nocera <hadess@hadess.net>
 *
 * acme.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 */

#include <config.h>

/* system headers */
#include <sys/file.h>
#include <sys/stat.h>
/* X11 headers */
#include <X11/X.h>
#include <X11/XF86keysym.h>
/* Gnome headers */
#include <gdk/gdkx.h>
#include <gtk/gtkinvisible.h>
#include <gnome.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "acme.h"
#include "g-volume.h"
#ifdef USE_FBLEVEL
#include "fb-level.h"
#endif

static gboolean event_cb (GtkWidget * widget, GdkEvent * event,
			  gpointer data);

typedef struct {
	GVolume *volobj;
#ifdef USE_FBLEVEL
	FBLevel *levobj;
#endif
	GladeXML *xml;
	GtkWidget *dialog;
	GtkWidget *invisible;
	int NumLockMask, CapsLockMask, ScrollLockMask;
	GConfClient *conf_client;
	guint dialog_timeout;
	int lock_fd;
} Acme;

enum {
	ICON_MUTED,
	ICON_LOUD,
	ICON_BRIGHT,
	ICON_EJECT,
};

static void
selection_get_func (GtkClipboard *clipboard, GtkSelectionData *selection_data,
		guint info, gpointer user_data_or_owner)
{
}

static void
selection_clear_func (GtkClipboard *clipboard, gpointer user_data_or_owner)
{       
	return;
}

#define SELECTION_NAME "_ACME_SELECTION"

static gboolean
acme_get_lock (Acme *acme)
{
	gboolean result = FALSE;
	GtkClipboard *clipboard;
	Atom clipboard_atom = gdk_x11_get_xatom_by_name (SELECTION_NAME);
	static const GtkTargetEntry targets[] = {
		{ SELECTION_NAME, 0, 0 }
	};

	XGrabServer (GDK_DISPLAY());

	if (XGetSelectionOwner (GDK_DISPLAY(), clipboard_atom) != None)
		goto out;

	clipboard = gtk_clipboard_get (gdk_atom_intern (SELECTION_NAME, FALSE));

	if (!gtk_clipboard_set_with_data  (clipboard, targets,
				G_N_ELEMENTS (targets),
				selection_get_func,
				selection_clear_func, NULL))
		goto out;

	result = TRUE;

out:
	XUngrabServer (GDK_DISPLAY());
	gdk_flush();

	return result;
}

static void
acme_exit (Acme *acme)
{
	if (acme->lock_fd > 0)
	{
		char *path;

		close (acme->lock_fd);
		path = g_strdup_printf ("%s/.acme.lock", g_get_home_dir());
		unlink (path);
		g_free (path);
	}

	exit (0);
}

static void
acme_error (char * msg)
{
	GtkWidget *error_dialog;

	error_dialog =
	    gtk_message_dialog_new (NULL,
			    GTK_DIALOG_MODAL,
			    GTK_MESSAGE_ERROR,
			    GTK_BUTTONS_OK,
			    "%s", msg);
	gtk_dialog_set_default_response (GTK_DIALOG (error_dialog),
			GTK_RESPONSE_OK);
	gtk_widget_show (error_dialog);
	gtk_dialog_run (GTK_DIALOG (error_dialog));
	gtk_widget_destroy (error_dialog);
}

static void
execute (char * cmd)
{
	if (g_spawn_command_line_sync (cmd, NULL, NULL, NULL, NULL) == FALSE)
	{
		char *msg;

		msg = g_strdup_printf
			(_("Couldn't execute command: %s\n"
			   "Verify that this command exists."),
			 cmd);

		acme_error (msg);
		g_free (msg);
	}
}

static void
acme_play_sound (Acme *acme)
{
	char *soundfile, *command;

	soundfile = gconf_client_get_string (acme->conf_client,
			"/apps/acme/soundfile_name", NULL);
	if ((soundfile == NULL) || (strcmp (soundfile, "") == 0)) 
		return;

	if (g_file_test ("/usr/bin/esdplay",
			(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_EXECUTABLE)))
	{
		command = g_strdup_printf ("/usr/bin/esdplay %s",
				soundfile);
	} else if (g_file_test ("/usr/bin/play",
			(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_EXECUTABLE)))
	{
		command = g_strdup_printf ("/usr/bin/play %s",
				soundfile);
	} else {
		return;
	}

	execute (command);
	g_free (command);
}

static char*
permission_problem_string (const char *files)
{
	return g_strdup_printf (_("Permissions on the file %s are broken\n"
				"Please check ACME's documentation, correct "
				"the problem and restart ACME."), files);
}

static gboolean
vol_problem_cb (GVolume *fb, gpointer data)
{
	char *msg;

	msg = permission_problem_string ("/dev/mixer");
	acme_error (msg);
	g_free (msg);

	return TRUE;
}

#ifdef USE_FBLEVEL
static void
fb_problem_cb (void)
{
	char *msg;

	msg = permission_problem_string ("/dev/pmu");
	acme_error (msg);
	g_free (msg);

	return;
}
#endif

static void
update_use_pcm_cb (GConfClient *client, guint id, GConfEntry *entry,
		gpointer data)
{
	Acme *acme = (Acme *)data;
	gboolean use_pcm = FALSE;

	use_pcm = gconf_client_get_bool (acme->conf_client,
			"/apps/acme/use_pcm",
			NULL);
	g_volume_set_use_pcm (acme->volobj, use_pcm);
}

static void
acme_image_set (Acme *acme, int icon)
{
	GtkWidget *image;

	image = glade_xml_get_widget (acme->xml, "image1");
	g_return_if_fail (image != NULL);

	switch (icon) {
	case ICON_LOUD:
		gtk_image_set_from_file (GTK_IMAGE(image),
				ACME_DATA "gnome-speakernotes.png");
		break;
	case ICON_MUTED:
		gtk_image_set_from_file (GTK_IMAGE(image),
				ACME_DATA "gnome-speakernotes-muted.png");
		break;
	case ICON_BRIGHT:
		gtk_image_set_from_file (GTK_IMAGE(image),
				ACME_DATA "acme-brightness.png");
		break;
	case ICON_EJECT:
		gtk_image_set_from_file (GTK_IMAGE(image),
				ACME_DATA "acme-eject.png");
		break;
	default:
		g_assert_not_reached ();
	}
}

static void
ungrab_key (int key_code)
{
	gdk_error_trap_push ();
	XUngrabKey (GDK_DISPLAY (), key_code, AnyModifier,
			GDK_ROOT_WINDOW ());
	gdk_flush ();
	if (gdk_error_trap_pop ()) {
		char *error;

		error = g_strdup_printf
			(_("There was an error removing access to the "
			   "multimedia keys.\nKey %d couldn't be unbound."),
			 key_code);
		acme_error (error);
		g_free (error);
		exit (1);
	}
}

static void
grab_key (int key_code)
{
	gdk_error_trap_push ();
	XGrabKey (GDK_DISPLAY (), key_code, AnyModifier,
		  GDK_ROOT_WINDOW (), True,
		  GrabModeAsync, GrabModeAsync);
	gdk_flush ();
	if (gdk_error_trap_pop ()) {
		char *error;

		error = g_strdup_printf
			(_("It seems that another application already has"
			   " access to the multimedia keys.\n"
			   "Key %d couldn't be bound.\n"
			   "Close this application and restart ACME."),
			 key_code);
		acme_error (error);
		g_free (error);
		exit (1);
	}
}

static void
update_kbd_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	Acme *acme = (Acme *) data;
	int i;

	g_return_if_fail (entry->key != NULL);

	/* Find the key that was modified */
	for (i = 0; i < HANDLED_KEYS; i++) {
		if (strcmp (entry->key, keys[i].key_config) == 0)
		{
			int key_code;

			ungrab_key (keys[i].key_code);
			key_code = gconf_client_get_int (acme->conf_client,
					keys[i].key_config,
					NULL);
			if (key_code > 0) {
				grab_key (key_code);
				keys[i].key_code = key_code;
			}
		}
	}
}

static void
init_kbd (Acme *acme)
{
	GtkWidget *widget = acme->invisible;
	int i;

	gtk_widget_show (widget);

	/* Make the root window send events to the invisible proxy widget */
	gdk_window_set_user_data (gdk_get_default_root_window (), widget);

	gdk_window_set_events (gdk_get_default_root_window (),
			GDK_KEY_PRESS_MASK);

	for (i = 0; i < HANDLED_KEYS; i++) {
		int tmp;

		tmp = gconf_client_get_int (acme->conf_client,
				keys[i].key_config,
				NULL);

		keys[i].key_code = tmp;

		if (tmp != -1)
		{
#ifdef DEBUG
			g_print ("grabbed key %d for gconf key %s\n",
					keys [i].key_code,
					keys[i].key_config);
#endif
			grab_key (keys [i].key_code);
		}
	}

	for (i = 0; i < HANDLED_KEYS; i++) {
		gconf_client_notify_add (acme->conf_client,
				keys[i].key_config,
				update_kbd_cb,
				acme, NULL, NULL);
	}
}

static void
init_sm (Acme *acme)
{
	GnomeClient *master;
	GnomeClientFlags flags;

	master = gnome_master_client ();
	flags = gnome_client_get_flags (master);
	if (flags & GNOME_CLIENT_IS_CONNECTED) {
#ifdef DEBUG
		gnome_client_set_restart_style (master,
				GNOME_RESTART_NEVER);
#else
		gnome_client_set_restart_style (master,
				GNOME_RESTART_ANYWAY);
#endif
		gnome_client_flush (master);
	}

	g_signal_connect (GTK_OBJECT (master), "die",
			(GtkSignalFunc) acme_exit, acme);
}

static gboolean
dialog_hide (Acme *acme)
{
	gtk_widget_hide (acme->dialog);
	acme->dialog_timeout = 0;
	return FALSE;
}

static void
dialog_show (Acme *acme)
{
	int orig_x, orig_y, orig_w, orig_h, orig_d;
	int screen_w, screen_h;
	int x, y;

	gdk_window_get_geometry (GTK_WIDGET (acme->dialog)->window,
				 &orig_x, &orig_y,
				 &orig_w, &orig_h, &orig_d);
	screen_w = gdk_screen_width ();
	screen_h = gdk_screen_height ();

	x = (screen_w - orig_w) / 2;
	y = screen_h / 2 + (screen_h / 2 - orig_h) / 2;

	gdk_window_move (GTK_WIDGET (acme->dialog)->window, x, y);

	gtk_widget_show (acme->dialog);

	/* this makes sure the dialog is actually shown */
	while (gtk_events_pending())
		gtk_main_iteration();

	acme->dialog_timeout = gtk_timeout_add (DIALOG_TIMEOUT,
			(GtkFunction) dialog_hide, acme);
}

static void
do_exit_action (Acme *acme)
{
	GnomeClient *master;

	master = gnome_master_client();
	g_return_if_fail(master != NULL);

	gnome_client_request_save(master,
			GNOME_SAVE_BOTH,
			TRUE,
			GNOME_INTERACT_ANY,
			FALSE,
			TRUE);
}

static void
do_eject_action (Acme *acme)
{
	GtkWidget *progress;
	char *command;

	if (acme->dialog_timeout != 0)
	{
		gtk_timeout_remove (acme->dialog_timeout);
		acme->dialog_timeout = 0;
	}

	progress = glade_xml_get_widget (acme->xml, "progressbar");
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progress),
			(double) 0);
	gtk_widget_set_sensitive (progress, FALSE);

	acme_image_set (acme, ICON_EJECT);
	dialog_show (acme);

	command = gconf_client_get_string (acme->conf_client,
			"/apps/acme/eject_command", NULL);
	if ((command != NULL) && (strcmp (command, "") != 0))
		execute (command);
	else
		execute ("eject");

	gtk_widget_set_sensitive (progress, TRUE);
}

#ifdef USE_FBLEVEL
static void
do_brightness_action (Acme *acme, int type)
{
	GtkWidget *progress;
	int level;

	if (acme->dialog_timeout != 0)
	{
		gtk_timeout_remove (acme->dialog_timeout);
		acme->dialog_timeout = 0;
	}

	level = fb_level_get_level (acme->levobj);
	acme_image_set (acme, ICON_BRIGHT);

	switch (type) {
	case BRIGHT_DOWN_KEY:
		fb_level_set_level (acme->levobj, level - 1);
		break;
	case BRIGHT_UP_KEY:
		fb_level_set_level (acme->levobj, level + 1);
		break;
	}

	level = fb_level_get_level (acme->levobj);
	progress = glade_xml_get_widget (acme->xml, "progressbar");
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progress),
			(double) level / 15);

	dialog_show (acme);
}
#endif

static void
do_sound_action (Acme *acme, int type)
{
	GtkWidget *progress;
	gboolean muted;
	int vol;

	if (acme->dialog_timeout != 0)
	{
		gtk_timeout_remove (acme->dialog_timeout);
		acme->dialog_timeout = 0;
	}

	vol = g_volume_get_volume (acme->volobj);
	muted = g_volume_get_mute (acme->volobj);

	switch (type) {
	case MUTE_KEY:
		g_volume_mute_toggle(acme->volobj);
		break;
	case VOLUME_DOWN_KEY:
		if (muted) {
			g_volume_mute_toggle(acme->volobj);
		} else {
			g_volume_set_volume (acme->volobj, vol - VOLUME_STEP);
		}
		break;
	case VOLUME_UP_KEY:
		if (muted) {
			g_volume_mute_toggle(acme->volobj);
		} else {
			g_volume_set_volume (acme->volobj, vol + VOLUME_STEP);
		}
		break;
	}

	muted = g_volume_get_mute(acme->volobj);
	acme_image_set (acme, muted ? ICON_MUTED : ICON_LOUD);

	vol = g_volume_get_volume (acme->volobj);
	progress = glade_xml_get_widget (acme->xml, "progressbar");
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progress),
			(double) vol / 100);

	dialog_show (acme);

	/* No need to play any sound if we're muted, right ? */
	if (muted == FALSE)
		acme_play_sound (acme);
}

static void
do_action (int type, Acme *acme)
{
#ifdef DEBUG
	g_print ("do_action, type is: %d\n", type);
#endif
	switch (type) {
	case MUTE_KEY:
	case VOLUME_DOWN_KEY:
	case VOLUME_UP_KEY:
		do_sound_action (acme, type);
		break;
	case POWER_KEY:
		do_exit_action (acme);
		break;
	case EJECT_KEY:
		/* Do nothing, handled differently */
		break;
#ifdef USE_FBLEVEL
	case BRIGHT_DOWN_KEY:
	case BRIGHT_UP_KEY:
		if (acme->levobj == NULL)
			break;
		do_brightness_action (acme, type);
		break;
#endif
	default:
		g_assert_not_reached ();
	}
}

static void
check_eject (Acme *acme, GdkEvent *event)
{
	GdkEventKey *key;
	static guint eject_press_time = 0;

	key = (GdkEventKey *) event;

#ifdef DEBUG
	g_print ("type %d (press %d release %d)\n", event->type, GDK_KEY_PRESS, GDK_KEY_RELEASE);
	if (event->type == GDK_KEY_PRESS || event->type == GDK_KEY_RELEASE)
		g_print ("key pressed: %d\n", key->hardware_keycode);
#endif

	if (event->type == GDK_KEY_PRESS
			&& key->hardware_keycode == keys[EJECT_KEY].key_code
			&& eject_press_time == 0)
		eject_press_time = key->time;
	if (event->type == GDK_KEY_RELEASE
			&& key->hardware_keycode == keys[EJECT_KEY].key_code)
	{
		if (key->time - eject_press_time >= EJECT_TIMEOUT)
			do_eject_action (acme);
		eject_press_time = 0;
	}
}

static gboolean
event_cb (GtkWidget * widget, GdkEvent * event, gpointer data)
{
	Acme *acme = (Acme *) data;
	GdkEventKey *key;
	int i;

	/* Eject is handled differently */
	check_eject (acme, event);

	if (event->type != GDK_KEY_PRESS)
		return TRUE;

	key = (GdkEventKey *) event;

	for (i = 0; i < HANDLED_KEYS; i++) {
#ifdef DEBUG
		g_print ("comparing %d and %d\n", keys[i].key_code,
				key->hardware_keycode);
#endif
		if (keys[i].key_code == key->hardware_keycode) {
			do_action (keys[i].key_type, acme);
			break;
		}
	}
	return TRUE;
}

int
main (int argc, char *argv[])
{
	Acme *acme;

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	gnome_program_init ("acme", VERSION,
			LIBGNOMEUI_MODULE,
			argc, argv,
			NULL);

	acme = g_new0 (Acme, 1);
	acme->conf_client = NULL;

	if (acme_get_lock (acme) == FALSE)
	{
		g_print ("Acme is already running, exiting...\n");
		acme_exit (acme);
	}

	glade_gnome_init ();
	acme->xml = glade_xml_new (ACME_DATA "acme.glade", NULL, NULL);

	if (acme->xml == NULL) {
		acme_error (_("Couldn't load the Glade file.\n"
				"Make sure that ACME is properly installed."));
		exit (1);
	}

	acme->dialog = glade_xml_get_widget (acme->xml, "dialog");
	acme_image_set (acme, ICON_LOUD);

	acme->conf_client = gconf_client_get_default ();
	gconf_client_add_dir (acme->conf_client,
			"/apps/acme",
			GCONF_CLIENT_PRELOAD_ONELEVEL,
			NULL);

	acme->invisible = gtk_invisible_new ();
	init_kbd (acme);
	init_sm (acme);
	gtk_widget_realize(acme->dialog);
	acme->dialog_timeout = 0;

	/* initialise Volume handler */
	acme->volobj = g_volume_new();
	if (acme->volobj == NULL)
	{
		acme_error (_("Your system has no sound support, or ACME was"
				" compiled without any sound support\n"
				"ACME will now exit"));
		exit (1);
	}
	g_signal_connect(G_OBJECT(acme->volobj),
			"fd_problem",
			(GCallback) vol_problem_cb, NULL);

	g_volume_set_use_pcm (acme->volobj,
			gconf_client_get_bool (acme->conf_client,
				"/apps/acme/use_pcm", NULL));
	gconf_client_notify_add (acme->conf_client,
			"/apps/acme/use_pcm",
			update_use_pcm_cb,
			acme, NULL, NULL);

#ifdef USE_FBLEVEL
	/* initialise Frame Buffer level handler */
	acme->levobj = fb_level_new();
	if (acme->levobj == NULL)
		fb_problem_cb ();
#endif

	g_signal_connect (GTK_OBJECT (acme->invisible),
			    "event",
			    (GCallback) event_cb, acme);

	gtk_main ();

	return 0;
}

