from desktop.GlassWindow import GlassWindow
from TargetGroup import TargetGroup
from DisplayTarget import DisplayTarget
from utils.Observable import Observable
from DisplayConfigurator import DisplayConfigurator
from main import ICON, DEFAULT_SENSOR
from main import admin
from utils import vfs

import gtk
import os


#
# Class for display windows.
#
class Display(GlassWindow, Observable):

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1

    # mapping: str -> window type
    __WINDOW_FLAGS = {"below": GlassWindow.TYPE_KEEP_BELOW,
                      "above": GlassWindow.TYPE_KEEP_ABOVE,
                      "sticky": GlassWindow.TYPE_STICKY,
                      "managed": GlassWindow.TYPE_MANAGED}



    def __init__(self, id):

        # the path of the .display file
        self.__path = vfs.getcwd()

        # the unique ID of this display
        self.__id = id

        # the sensors of this window
        self.__sensors = {}

        # the last selected targets (used for detecting events)
        self.__last_targets = []

        # the last position of the mouse pointer (used for filtering out
        # unwanted events)
        self.__last_pointer_pos = (-1, -1)

        # timeout flag for mouse buttons
        self.__button_timeout = 1

        # timeout flag for menu detection
        self.__menu_timeout = 0

        # window position for detecting moves
        self.__window_pos = (-1, -1)

        # window size for detecting resizing
        self.__window_size = (0, 0)

        # mapping between sensors and targets; which target watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # temporary data used for dragging windows
        self.__is_dragging = 0
        self.__drag_offset = (0, 0)


        GlassWindow.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.set_size_request(-1, -1)

        self.__ebox = gtk.EventBox()
        self.__ebox.show()
        self.add(self.__ebox)

        # set the icon
        self.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))
        # set up event handlers
        #self.connect("configure-event", self.__on_configure)
        self.__ebox.connect("button-press-event", self.__on_button, 0)
        self.__ebox.connect("button-release-event", self.__on_button, 1)
        self.__ebox.connect("scroll-event", self.__on_scroll)
        self.__ebox.connect("motion-notify-event", self.__on_motion, 0)
        self.__ebox.connect("leave-notify-event", self.__on_motion, 1)
        self.__ebox.connect("delete-event", self.__on_close)
        self.__ebox.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK |
                        gtk.gdk.LEAVE_NOTIFY_MASK |
                        gtk.gdk.POINTER_MOTION_MASK)


        # sawfish needs this
        self.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DOCK)
        self.realize()
        self.set_property("skip-taskbar-hint", 1)
        self.set_property("skip-pager-hint", 1)

        self.set_window_type(self.TYPE_STICKY | self.TYPE_KEEP_BELOW)



    #
    # Returns the path of the .display file.
    #
    def get_path(self):

        return self.__path



    def add_children(self, childrendata):

        # create the root TargetGroup
        self.__group = TargetGroup(None, self)
        self.__group.set_position(-1, -1)
        self.__group.add_children(childrendata)
        self.__ebox.add(self.__group)
        self.__group.add_observer(self.__on_observe_group)




    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):

        configurators = []
        sensors = self.__sensors.values()
        for s in sensors:
            configurators.append(s.get_configurator())

        dconf = DisplayConfigurator(configurators)
        dconf.set_transient_for(self)


    #
    # Removes this display.
    #
    def remove_display(self):

        self.drop_observers()
        for s in self.__sensors.values():
            s.stop()

        del self.__sensors
        del self.__mapping
        del self.__last_targets

        self.remove(self.__group)
        self.__group.delete()
        self.__group.destroy()
        del self.__group
        self.destroy()



    #
    # Reacts on configuring the window.
    #
    def __on_configure(self, src, event):

        width = event.width
        height = event.height

        if ((width, height) != self.__window_size):
            self.__window_size = (width, height)
            self.set_config("width", width)
            self.set_config("height", height)
        #end if



    #
    # Reacts on closing the window.
    #
    def __on_close(self, src, event):

        #self.__remove_display()
        self.update_observer(self.OBS_CLOSE, self.__id)



    #
    # Reacts on moving the mouse.
    #
    def __on_move(self):

        if (self.__is_dragging):
            offx, offy = self.__drag_offset
            winx, winy = self.get_position()
            x, y = self.get_pointer()
            x += winx; y += winy
            self.move(x - offx, y - offy)

            return gtk.TRUE

        else:
            return gtk.FALSE



    def file_drop(self, files, x, y):
        ''' Sends the call with path and files to the sensor '''

        targets = self.__get_target_at(x, y)
        # find target and invoke action handler
        rtargets = targets[:]
        rtargets.reverse()
        for t, path in rtargets:
            if (t.has_action(DisplayTarget.ACTION_FILE_DROP)):
                call = t.get_action_call(DisplayTarget.ACTION_FILE_DROP)
                #index = targets.index(t)
                #targetpath = path[:index + 1]
                print files, t, path
                self.__call_sensor(call, path, files, x, y)
                break
        #end for

        #self.__call_sensor(call, path, files, x, y)



    #
    # Reacts on button events.
    #
    def __on_button(self, src, event, is_release = 0):

        # local timeout function
        def do_timeout(self): self.__button_timeout = 1

        # function for detecting holding down the mouse long enough for menu
        # TODO: this is obsolete, isn't it?
        def do_menu(self, targets, path):
            if (not self.__menu_timeout): return
            targets.reverse()
            for t in targets:
                if (t.has_action(t.ACTION_MENU)):
                    call = t.get_action_call(t.ACTION_MENU)
                    index = targets.index(t)
                    targetpath = path[:index + 1]
                    self.__call_sensor(call, targetpath, button)
                    break
            #end for
        #end def

        self.__menu_timeout = 0
        px, py = self.get_pointer()
        lx, ly = self.__pointer_pos
        button = event.button
        targets = self.__get_target_at(px, py)

        # determine action
        if (not is_release):
            self.__pointer_pos = (px, py)
            if (button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
                action = DisplayTarget.ACTION_DOUBLECLICK
            elif (button == 1):
                action = DisplayTarget.ACTION_PRESS
            elif (button == 2):
                #action = DisplayTarget.ACTION_PRESS
                x, y = self.get_pointer()
                self.__is_dragging = 1
                self.__drag_offset = (x, y)
                self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))
                gtk.timeout_add(20, self.__on_move)
                return
            elif (button == 3):
                return
            else:
                return

        else:
            if (button == 1):
                if (abs(lx - px) < 10 and abs(ly - py) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                #action = DisplayTarget.ACTION_RELEASE
                self.__is_dragging = 0
                self.window.set_cursor(None)
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return

        #end if

        # find target and invoke action handler
        called = 0
        for t, path in targets:
            if (t.has_action(action)):
                call = t.get_action_call(action)
                #index = targets.index(t)
                #targetpath = path[:index + 1]
                self.__call_sensor(call, path, button)
                called = 1
                break
        #end for

        # make sure that there is always a popup menu
        if (action == DisplayTarget.ACTION_MENU and not called):
            self.__call_sensor([(DEFAULT_SENSOR, "menu", [])], [], 3)



    #
    # Reacts on moving the mouse.
    #
    def __on_motion(self, src, event, is_leave):

        px, py = self.get_pointer()
        if ((px, py) == self.__last_pointer_pos): return
        else: self.__last_pointer_pos = (px, py)

        targets = self.__get_target_at(px, py)

        # how to detect enter and leave:
        # get covered targets and compare with previously covered targets;
        # what's new in covered targets is entered;
        # what's only in previously covered targets is left

        # some braindead window managers treat mouse clicks as leave-notify
        # events; work around this by checking if the mouse really has left
        # the window
        nil, nil, width, height = self.__group.get_geometry()
        if (is_leave and 0 <= px < width and 0 <= py < height): is_leave = 0
        if (is_leave): targets = []
        
        # TODO: make this more efficient; don't check for existence in lists
        for t, path in targets + self.__last_targets:
            # enter
            if (not (t, path) in self.__last_targets and
                t.has_action(t.ACTION_ENTER)):
                #index = targets.index(t)
                #targetpath = path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_ENTER), path)

            # leave
            elif (not (t, path) in targets and t.has_action(t.ACTION_LEAVE)):
                #index = self.__last_targets.index(t)
                #targetpath = self.__last_target_path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_LEAVE), path)


            # motion
            elif ((t, path) in targets and t.has_action(t.ACTION_MOTION)):
                tx, ty = t.get_pointer()
                #index = targets.index(t)
                #targetpath = path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_MOTION),
                                   path, tx, ty)

        #end for

        self.__last_targets = targets

        # save the window position if the window has moved
        x, y = self.window.get_position()
        if ((x, y) != self.__window_pos and not is_leave):
            nil, nil, w, h = self.__group.get_geometry()
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)
            dx, dy = x - ax, y - ay
            self.__call_sensor([(DEFAULT_SENSOR, "move", [])], [""],
                               x + dx, y + dy, x, y)

            self.__window_pos = (x, y)
            #print "save window @", x, y
            self.__group.set_position(x + dx, y + dy, update = 0)



    #
    # Reacts on rolling the mouse wheel.
    #
    def __on_scroll(self, src, event):

        px, py = self.get_pointer()
        targets = self.__get_target_at(px, py)


        if (event.direction == gtk.gdk.SCROLL_UP):
            direction = 0
        elif (event.direction == gtk.gdk.SCROLL_DOWN):
            direction = 1
        else:
            direction = -1

        for t, path in targets:
            if (t.has_action(DisplayTarget.ACTION_SCROLL)):
                call = t.get_action_call(DisplayTarget.ACTION_SCROLL)
                #index = targets.index(t)
                #targetpath = path[:index + 1]
                self.__call_sensor(call, path, direction)
                break
        #end for
    


    #
    # Observer for sensors.
    #
    def __on_observe_sensor(self, src, cmd, data):

        # propagate the incoming sensor output
        if (cmd == src.OBS_OUTPUT):
            #if (self.window): self.window.freeze_updates()
            for key, value in data.get_entries():
                if ("[" in key):
                    indexpart = key[key.find("[") + 1:-1]
                    indexes = indexpart.split("][")
                    key = key[:key.find("[")]
                else:
                    indexes = []

                entries = self.__mapping.get((src, key), [])
                for target, property in entries:
                    if (indexes):
                        sensor = self.__get_sensor_id(src)
                        target.distribute_sensor_output(sensor, indexes[:],
                                                        key, value)
                    else:
                        target.set_config(property, value)
                #end for
            #end for
            #if (self.window): self.window.thaw_updates()

            # is this queue_draw() needed? this was the CPU hog
            #self.queue_draw()

        elif (cmd == src.OBS_CMD_CONFIGURE):
            self.__open_configurator()

        elif (cmd == src.OBS_CMD_REMOVE):
            #self.__remove_display(restart = 0)
            self.update_observer(self.OBS_CLOSE, self.__id)

        elif (cmd == src.OBS_CMD_RESTART):
            #self.__remove_display(restart = 1)
            self.update_observer(self.OBS_RESTART, self.__id)

        elif (cmd == src.OBS_CMD_DUPLICATE):
            dsp_uri = admin.get_displays().get(self.__id)
            if (dsp_uri): admin.add_display(dsp_uri)



    #
    # Observer for the root group.
    #
    def __on_observe_group(self, src, cmd, *args):

        if (not self.__is_dragging and cmd == src.OBS_MOVE):
            x, y, w, h = args
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)

            ax = min(max(0, ax), gtk.gdk.screen_width())
            ay = min(max(0, ay), gtk.gdk.screen_height())

            if ((ax, ay) == (0, 0) or (ax, ay) != self.__window_pos):
                self.move(ax, ay)
                self.__window_pos = (ax, ay)
                self.__call_sensor([(DEFAULT_SENSOR, "positioned", [])], [""])
                

            if (w != 0 and h != 0):
                self.resize(w, h)
                # storing the size is useless, but it's added by request;
                # it makes life easy for desklets pagers
                self.__call_sensor([(DEFAULT_SENSOR, "size", [])], [""], w, h)

            # don't show this window before its position has been set
            if (self.__window_pos != (-1, -1)):
                gtk.idle_add(self.show)
                gtk.idle_add(self.move, ax, ay)



    #
    # Calls a function of a Sensor.
    #
    def __call_sensor(self, cmd, path, *args):
        assert(cmd)

        args = list(args)
        for id, callname, userargs in cmd:
            if (not id): id = DEFAULT_SENSOR
            sensor = self.__get_sensor(id) or self.__get_sensor(DEFAULT_SENSOR)
            allargs = args + userargs

            # the sensor is an external module, so we make sure it cannot crash
            # the application

            try:
                sensor.send_action(callname, path, allargs)

            except StandardError, e:
                print "The sensor produced an error:", e
                import traceback; traceback.print_exc()
        #end for



    #
    # Returns the target and its path at the given position.
    #
    def __get_target_at(self, px, py):

        targets = self.__group.get_target_at(px, py, [], 1)
        return targets



    #
    # Sets the configuration.
    #
    def set_config(self, key, value):

        if (key == "window-flags"):
            flags = 0
            value = value.split(",")
            for p in value: flags |= self.__WINDOW_FLAGS[p.strip()]
            self.set_window_type(flags)

        elif (key == "shape"):
            file = vfs.join(self.get_path(), value)
            try:
                loader = gtk.gdk.PixbufLoader()
                fd = vfs.open(file, "r")
                data = vfs.read_all(fd)
                fd.close()
                loader.write(data, len(data))
                loader.close()
                pixbuf = loader.get_pixbuf()
                #pixbuf = gtk.gdk.pixbuf_new_from_file(file)
                pix, mask = pixbuf.render_pixmap_and_mask(1)
                self.shape_combine_mask(mask, 0, 0)
            except:
                print "could not set shape", file

        else:
            self.__group.set_config(key, value)



    #
    # Adds a sensor to this display.
    #
    def add_sensor(self, id, sensor):

        self.__sensors[id] = sensor
        sensor.add_observer(self.__on_observe_sensor)



    #
    # Returns the sensor with the given ID. Returns None if the sensor does not
    # exist.
    #
    def __get_sensor(self, id): return self.__sensors.get(id)



    #
    # Returns the ID of the given sensor.
    #
    def __get_sensor_id(self, sensor):

        for k, v in self.__sensors.items():
            if (v == sensor): return k

        return ""



    #
    # Maps a sensor output to a target.
    #
    def add_mapping(self, sensorplug, target, property):

        id, port = sensorplug.split(":")
        sensor = self.__get_sensor(id)
        if (not self.__mapping.has_key((sensor, port))):
            self.__mapping[(sensor, port)] = []

        if (not (target, property) in self.__mapping[(sensor, port)]):
            self.__mapping[(sensor, port)].append((target, property))
