"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LanguageAgnosticBracketTokens = exports.BracketTokens = void 0;
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
const strings_1 = require("../../../../../base/common/strings");
const ast_1 = require("./ast");
const length_1 = require("./length");
const smallImmutableSet_1 = require("./smallImmutableSet");
const tokenizer_1 = require("./tokenizer");
class BracketTokens {
    constructor(map) {
        this.map = map;
        this.hasRegExp = false;
        this._regExpGlobal = null;
    }
    static createFromLanguage(configuration, denseKeyProvider) {
        function getId(languageId, openingText) {
            return denseKeyProvider.getKey(`${languageId}:::${openingText}`);
        }
        const brackets = configuration.characterPair.getColorizedBrackets();
        const closingBrackets = new Map();
        const openingBrackets = new Set();
        for (const [openingText, closingText] of brackets) {
            openingBrackets.add(openingText);
            let info = closingBrackets.get(closingText);
            const openingTextId = getId(configuration.languageId, openingText);
            if (!info) {
                info = { openingBrackets: smallImmutableSet_1.SmallImmutableSet.getEmpty(), first: openingTextId };
                closingBrackets.set(closingText, info);
            }
            info.openingBrackets = info.openingBrackets.add(openingTextId, smallImmutableSet_1.identityKeyProvider);
        }
        const map = new Map();
        for (const [closingText, info] of closingBrackets) {
            const length = (0, length_1.toLength)(0, closingText.length);
            map.set(closingText, new tokenizer_1.Token(length, 2 /* ClosingBracket */, info.first, info.openingBrackets, ast_1.BracketAstNode.create(length, configuration.languageId, info.openingBrackets)));
        }
        for (const openingText of openingBrackets) {
            const length = (0, length_1.toLength)(0, openingText.length);
            const openingTextId = getId(configuration.languageId, openingText);
            const bracketIds = smallImmutableSet_1.SmallImmutableSet.getEmpty().add(openingTextId, smallImmutableSet_1.identityKeyProvider);
            map.set(openingText, new tokenizer_1.Token(length, 1 /* OpeningBracket */, openingTextId, bracketIds, ast_1.BracketAstNode.create(length, configuration.languageId, bracketIds)));
        }
        return new BracketTokens(map);
    }
    getRegExpStr() {
        if (this.isEmpty) {
            return null;
        }
        else {
            const keys = [...this.map.keys()];
            keys.sort();
            keys.reverse();
            return keys.map(k => prepareBracketForRegExp(k)).join('|');
        }
    }
    /**
     * Returns null if there is no such regexp (because there are no brackets).
    */
    get regExpGlobal() {
        if (!this.hasRegExp) {
            const regExpStr = this.getRegExpStr();
            this._regExpGlobal = regExpStr ? new RegExp(regExpStr, 'g') : null;
            this.hasRegExp = true;
        }
        return this._regExpGlobal;
    }
    getToken(value) {
        return this.map.get(value);
    }
    findClosingTokenText(openingBracketIds) {
        for (const [closingText, info] of this.map) {
            if (info.bracketIds.intersects(openingBracketIds)) {
                return closingText;
            }
        }
        return undefined;
    }
    get isEmpty() {
        return this.map.size === 0;
    }
}
exports.BracketTokens = BracketTokens;
function prepareBracketForRegExp(str) {
    const escaped = (0, strings_1.escapeRegExpCharacters)(str);
    // This bracket pair uses letters like e.g. "begin" - "end" (see https://github.com/microsoft/vscode/issues/132162)
    const needsWordBoundaries = (/^[\w ]+$/.test(str));
    return (needsWordBoundaries ? `\\b${escaped}\\b` : escaped);
}
class LanguageAgnosticBracketTokens {
    constructor(denseKeyProvider, getLanguageConfiguration) {
        this.denseKeyProvider = denseKeyProvider;
        this.getLanguageConfiguration = getLanguageConfiguration;
        this.languageIdToBracketTokens = new Map();
    }
    didLanguageChange(languageId) {
        const existing = this.languageIdToBracketTokens.get(languageId);
        if (!existing) {
            return false;
        }
        const newRegExpStr = BracketTokens.createFromLanguage(this.getLanguageConfiguration(languageId), this.denseKeyProvider).getRegExpStr();
        return existing.getRegExpStr() !== newRegExpStr;
    }
    getSingleLanguageBracketTokens(languageId) {
        let singleLanguageBracketTokens = this.languageIdToBracketTokens.get(languageId);
        if (!singleLanguageBracketTokens) {
            singleLanguageBracketTokens = BracketTokens.createFromLanguage(this.getLanguageConfiguration(languageId), this.denseKeyProvider);
            this.languageIdToBracketTokens.set(languageId, singleLanguageBracketTokens);
        }
        return singleLanguageBracketTokens;
    }
    getToken(value, languageId) {
        const singleLanguageBracketTokens = this.getSingleLanguageBracketTokens(languageId);
        return singleLanguageBracketTokens.getToken(value);
    }
}
exports.LanguageAgnosticBracketTokens = LanguageAgnosticBracketTokens;
//# sourceMappingURL=brackets.js.map