/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
import { Emitter } from '../../../../base/common/event';
import * as strings from '../../../../base/common/strings';
import { Range } from '../../core/range';
import { ApplyEditsResult } from '../../model';
import { PieceTreeBase } from './pieceTreeBase';
import { countEOL } from '../tokensStore';
import { TextChange } from '../textChange';
import { Disposable } from '../../../../base/common/lifecycle';
var PieceTreeTextBuffer = /** @class */ (function (_super) {
    __extends(PieceTreeTextBuffer, _super);
    function PieceTreeTextBuffer(chunks, BOM, eol, containsRTL, containsUnusualLineTerminators, isBasicASCII, eolNormalized) {
        var _this = _super.call(this) || this;
        _this._onDidChangeContent = _this._register(new Emitter());
        _this.onDidChangeContent = _this._onDidChangeContent.event;
        _this._BOM = BOM;
        _this._mightContainNonBasicASCII = !isBasicASCII;
        _this._mightContainRTL = containsRTL;
        _this._mightContainUnusualLineTerminators = containsUnusualLineTerminators;
        _this._pieceTree = new PieceTreeBase(chunks, eol, eolNormalized);
        return _this;
    }
    // #region TextBuffer
    PieceTreeTextBuffer.prototype.equals = function (other) {
        if (!(other instanceof PieceTreeTextBuffer)) {
            return false;
        }
        if (this._BOM !== other._BOM) {
            return false;
        }
        if (this.getEOL() !== other.getEOL()) {
            return false;
        }
        return this._pieceTree.equal(other._pieceTree);
    };
    PieceTreeTextBuffer.prototype.mightContainRTL = function () {
        return this._mightContainRTL;
    };
    PieceTreeTextBuffer.prototype.mightContainUnusualLineTerminators = function () {
        return this._mightContainUnusualLineTerminators;
    };
    PieceTreeTextBuffer.prototype.resetMightContainUnusualLineTerminators = function () {
        this._mightContainUnusualLineTerminators = false;
    };
    PieceTreeTextBuffer.prototype.mightContainNonBasicASCII = function () {
        return this._mightContainNonBasicASCII;
    };
    PieceTreeTextBuffer.prototype.getBOM = function () {
        return this._BOM;
    };
    PieceTreeTextBuffer.prototype.getEOL = function () {
        return this._pieceTree.getEOL();
    };
    PieceTreeTextBuffer.prototype.createSnapshot = function (preserveBOM) {
        return this._pieceTree.createSnapshot(preserveBOM ? this._BOM : '');
    };
    PieceTreeTextBuffer.prototype.getOffsetAt = function (lineNumber, column) {
        return this._pieceTree.getOffsetAt(lineNumber, column);
    };
    PieceTreeTextBuffer.prototype.getPositionAt = function (offset) {
        return this._pieceTree.getPositionAt(offset);
    };
    PieceTreeTextBuffer.prototype.getRangeAt = function (start, length) {
        var end = start + length;
        var startPosition = this.getPositionAt(start);
        var endPosition = this.getPositionAt(end);
        return new Range(startPosition.lineNumber, startPosition.column, endPosition.lineNumber, endPosition.column);
    };
    PieceTreeTextBuffer.prototype.getValueInRange = function (range, eol) {
        if (eol === void 0) { eol = 0 /* TextDefined */; }
        if (range.isEmpty()) {
            return '';
        }
        var lineEnding = this._getEndOfLine(eol);
        return this._pieceTree.getValueInRange(range, lineEnding);
    };
    PieceTreeTextBuffer.prototype.getValueLengthInRange = function (range, eol) {
        if (eol === void 0) { eol = 0 /* TextDefined */; }
        if (range.isEmpty()) {
            return 0;
        }
        if (range.startLineNumber === range.endLineNumber) {
            return (range.endColumn - range.startColumn);
        }
        var startOffset = this.getOffsetAt(range.startLineNumber, range.startColumn);
        var endOffset = this.getOffsetAt(range.endLineNumber, range.endColumn);
        return endOffset - startOffset;
    };
    PieceTreeTextBuffer.prototype.getCharacterCountInRange = function (range, eol) {
        if (eol === void 0) { eol = 0 /* TextDefined */; }
        if (this._mightContainNonBasicASCII) {
            // we must count by iterating
            var result = 0;
            var fromLineNumber = range.startLineNumber;
            var toLineNumber = range.endLineNumber;
            for (var lineNumber = fromLineNumber; lineNumber <= toLineNumber; lineNumber++) {
                var lineContent = this.getLineContent(lineNumber);
                var fromOffset = (lineNumber === fromLineNumber ? range.startColumn - 1 : 0);
                var toOffset = (lineNumber === toLineNumber ? range.endColumn - 1 : lineContent.length);
                for (var offset = fromOffset; offset < toOffset; offset++) {
                    if (strings.isHighSurrogate(lineContent.charCodeAt(offset))) {
                        result = result + 1;
                        offset = offset + 1;
                    }
                    else {
                        result = result + 1;
                    }
                }
            }
            result += this._getEndOfLine(eol).length * (toLineNumber - fromLineNumber);
            return result;
        }
        return this.getValueLengthInRange(range, eol);
    };
    PieceTreeTextBuffer.prototype.getLength = function () {
        return this._pieceTree.getLength();
    };
    PieceTreeTextBuffer.prototype.getLineCount = function () {
        return this._pieceTree.getLineCount();
    };
    PieceTreeTextBuffer.prototype.getLinesContent = function () {
        return this._pieceTree.getLinesContent();
    };
    PieceTreeTextBuffer.prototype.getLineContent = function (lineNumber) {
        return this._pieceTree.getLineContent(lineNumber);
    };
    PieceTreeTextBuffer.prototype.getLineCharCode = function (lineNumber, index) {
        return this._pieceTree.getLineCharCode(lineNumber, index);
    };
    PieceTreeTextBuffer.prototype.getCharCode = function (offset) {
        return this._pieceTree.getCharCode(offset);
    };
    PieceTreeTextBuffer.prototype.getLineLength = function (lineNumber) {
        return this._pieceTree.getLineLength(lineNumber);
    };
    PieceTreeTextBuffer.prototype.getLineMinColumn = function (lineNumber) {
        return 1;
    };
    PieceTreeTextBuffer.prototype.getLineMaxColumn = function (lineNumber) {
        return this.getLineLength(lineNumber) + 1;
    };
    PieceTreeTextBuffer.prototype.getLineFirstNonWhitespaceColumn = function (lineNumber) {
        var result = strings.firstNonWhitespaceIndex(this.getLineContent(lineNumber));
        if (result === -1) {
            return 0;
        }
        return result + 1;
    };
    PieceTreeTextBuffer.prototype.getLineLastNonWhitespaceColumn = function (lineNumber) {
        var result = strings.lastNonWhitespaceIndex(this.getLineContent(lineNumber));
        if (result === -1) {
            return 0;
        }
        return result + 2;
    };
    PieceTreeTextBuffer.prototype._getEndOfLine = function (eol) {
        switch (eol) {
            case 1 /* LF */:
                return '\n';
            case 2 /* CRLF */:
                return '\r\n';
            case 0 /* TextDefined */:
                return this.getEOL();
            default:
                throw new Error('Unknown EOL preference');
        }
    };
    PieceTreeTextBuffer.prototype.setEOL = function (newEOL) {
        this._pieceTree.setEOL(newEOL);
    };
    PieceTreeTextBuffer.prototype.applyEdits = function (rawOperations, recordTrimAutoWhitespace, computeUndoEdits) {
        var _a;
        var mightContainRTL = this._mightContainRTL;
        var mightContainUnusualLineTerminators = this._mightContainUnusualLineTerminators;
        var mightContainNonBasicASCII = this._mightContainNonBasicASCII;
        var canReduceOperations = true;
        var operations = [];
        for (var i = 0; i < rawOperations.length; i++) {
            var op = rawOperations[i];
            if (canReduceOperations && op._isTracked) {
                canReduceOperations = false;
            }
            var validatedRange = op.range;
            if (op.text) {
                var textMightContainNonBasicASCII = true;
                if (!mightContainNonBasicASCII) {
                    textMightContainNonBasicASCII = !strings.isBasicASCII(op.text);
                    mightContainNonBasicASCII = textMightContainNonBasicASCII;
                }
                if (!mightContainRTL && textMightContainNonBasicASCII) {
                    // check if the new inserted text contains RTL
                    mightContainRTL = strings.containsRTL(op.text);
                }
                if (!mightContainUnusualLineTerminators && textMightContainNonBasicASCII) {
                    // check if the new inserted text contains unusual line terminators
                    mightContainUnusualLineTerminators = strings.containsUnusualLineTerminators(op.text);
                }
            }
            var validText = '';
            var eolCount = 0;
            var firstLineLength = 0;
            var lastLineLength = 0;
            if (op.text) {
                var strEOL = void 0;
                _a = __read(countEOL(op.text), 4), eolCount = _a[0], firstLineLength = _a[1], lastLineLength = _a[2], strEOL = _a[3];
                var bufferEOL = this.getEOL();
                var expectedStrEOL = (bufferEOL === '\r\n' ? 2 /* CRLF */ : 1 /* LF */);
                if (strEOL === 0 /* Unknown */ || strEOL === expectedStrEOL) {
                    validText = op.text;
                }
                else {
                    validText = op.text.replace(/\r\n|\r|\n/g, bufferEOL);
                }
            }
            operations[i] = {
                sortIndex: i,
                identifier: op.identifier || null,
                range: validatedRange,
                rangeOffset: this.getOffsetAt(validatedRange.startLineNumber, validatedRange.startColumn),
                rangeLength: this.getValueLengthInRange(validatedRange),
                text: validText,
                eolCount: eolCount,
                firstLineLength: firstLineLength,
                lastLineLength: lastLineLength,
                forceMoveMarkers: Boolean(op.forceMoveMarkers),
                isAutoWhitespaceEdit: op.isAutoWhitespaceEdit || false
            };
        }
        // Sort operations ascending
        operations.sort(PieceTreeTextBuffer._sortOpsAscending);
        var hasTouchingRanges = false;
        for (var i = 0, count = operations.length - 1; i < count; i++) {
            var rangeEnd = operations[i].range.getEndPosition();
            var nextRangeStart = operations[i + 1].range.getStartPosition();
            if (nextRangeStart.isBeforeOrEqual(rangeEnd)) {
                if (nextRangeStart.isBefore(rangeEnd)) {
                    // overlapping ranges
                    throw new Error('Overlapping ranges are not allowed!');
                }
                hasTouchingRanges = true;
            }
        }
        if (canReduceOperations) {
            operations = this._reduceOperations(operations);
        }
        // Delta encode operations
        var reverseRanges = (computeUndoEdits || recordTrimAutoWhitespace ? PieceTreeTextBuffer._getInverseEditRanges(operations) : []);
        var newTrimAutoWhitespaceCandidates = [];
        if (recordTrimAutoWhitespace) {
            for (var i = 0; i < operations.length; i++) {
                var op = operations[i];
                var reverseRange = reverseRanges[i];
                if (op.isAutoWhitespaceEdit && op.range.isEmpty()) {
                    // Record already the future line numbers that might be auto whitespace removal candidates on next edit
                    for (var lineNumber = reverseRange.startLineNumber; lineNumber <= reverseRange.endLineNumber; lineNumber++) {
                        var currentLineContent = '';
                        if (lineNumber === reverseRange.startLineNumber) {
                            currentLineContent = this.getLineContent(op.range.startLineNumber);
                            if (strings.firstNonWhitespaceIndex(currentLineContent) !== -1) {
                                continue;
                            }
                        }
                        newTrimAutoWhitespaceCandidates.push({ lineNumber: lineNumber, oldContent: currentLineContent });
                    }
                }
            }
        }
        var reverseOperations = null;
        if (computeUndoEdits) {
            var reverseRangeDeltaOffset = 0;
            reverseOperations = [];
            for (var i = 0; i < operations.length; i++) {
                var op = operations[i];
                var reverseRange = reverseRanges[i];
                var bufferText = this.getValueInRange(op.range);
                var reverseRangeOffset = op.rangeOffset + reverseRangeDeltaOffset;
                reverseRangeDeltaOffset += (op.text.length - bufferText.length);
                reverseOperations[i] = {
                    sortIndex: op.sortIndex,
                    identifier: op.identifier,
                    range: reverseRange,
                    text: bufferText,
                    textChange: new TextChange(op.rangeOffset, bufferText, reverseRangeOffset, op.text)
                };
            }
            // Can only sort reverse operations when the order is not significant
            if (!hasTouchingRanges) {
                reverseOperations.sort(function (a, b) { return a.sortIndex - b.sortIndex; });
            }
        }
        this._mightContainRTL = mightContainRTL;
        this._mightContainUnusualLineTerminators = mightContainUnusualLineTerminators;
        this._mightContainNonBasicASCII = mightContainNonBasicASCII;
        var contentChanges = this._doApplyEdits(operations);
        var trimAutoWhitespaceLineNumbers = null;
        if (recordTrimAutoWhitespace && newTrimAutoWhitespaceCandidates.length > 0) {
            // sort line numbers auto whitespace removal candidates for next edit descending
            newTrimAutoWhitespaceCandidates.sort(function (a, b) { return b.lineNumber - a.lineNumber; });
            trimAutoWhitespaceLineNumbers = [];
            for (var i = 0, len = newTrimAutoWhitespaceCandidates.length; i < len; i++) {
                var lineNumber = newTrimAutoWhitespaceCandidates[i].lineNumber;
                if (i > 0 && newTrimAutoWhitespaceCandidates[i - 1].lineNumber === lineNumber) {
                    // Do not have the same line number twice
                    continue;
                }
                var prevContent = newTrimAutoWhitespaceCandidates[i].oldContent;
                var lineContent = this.getLineContent(lineNumber);
                if (lineContent.length === 0 || lineContent === prevContent || strings.firstNonWhitespaceIndex(lineContent) !== -1) {
                    continue;
                }
                trimAutoWhitespaceLineNumbers.push(lineNumber);
            }
        }
        this._onDidChangeContent.fire();
        return new ApplyEditsResult(reverseOperations, contentChanges, trimAutoWhitespaceLineNumbers);
    };
    /**
     * Transform operations such that they represent the same logic edit,
     * but that they also do not cause OOM crashes.
     */
    PieceTreeTextBuffer.prototype._reduceOperations = function (operations) {
        if (operations.length < 1000) {
            // We know from empirical testing that a thousand edits work fine regardless of their shape.
            return operations;
        }
        // At one point, due to how events are emitted and how each operation is handled,
        // some operations can trigger a high amount of temporary string allocations,
        // that will immediately get edited again.
        // e.g. a formatter inserting ridiculous ammounts of \n on a model with a single line
        // Therefore, the strategy is to collapse all the operations into a huge single edit operation
        return [this._toSingleEditOperation(operations)];
    };
    PieceTreeTextBuffer.prototype._toSingleEditOperation = function (operations) {
        var forceMoveMarkers = false;
        var firstEditRange = operations[0].range;
        var lastEditRange = operations[operations.length - 1].range;
        var entireEditRange = new Range(firstEditRange.startLineNumber, firstEditRange.startColumn, lastEditRange.endLineNumber, lastEditRange.endColumn);
        var lastEndLineNumber = firstEditRange.startLineNumber;
        var lastEndColumn = firstEditRange.startColumn;
        var result = [];
        for (var i = 0, len = operations.length; i < len; i++) {
            var operation = operations[i];
            var range = operation.range;
            forceMoveMarkers = forceMoveMarkers || operation.forceMoveMarkers;
            // (1) -- Push old text
            result.push(this.getValueInRange(new Range(lastEndLineNumber, lastEndColumn, range.startLineNumber, range.startColumn)));
            // (2) -- Push new text
            if (operation.text.length > 0) {
                result.push(operation.text);
            }
            lastEndLineNumber = range.endLineNumber;
            lastEndColumn = range.endColumn;
        }
        var text = result.join('');
        var _a = __read(countEOL(text), 3), eolCount = _a[0], firstLineLength = _a[1], lastLineLength = _a[2];
        return {
            sortIndex: 0,
            identifier: operations[0].identifier,
            range: entireEditRange,
            rangeOffset: this.getOffsetAt(entireEditRange.startLineNumber, entireEditRange.startColumn),
            rangeLength: this.getValueLengthInRange(entireEditRange, 0 /* TextDefined */),
            text: text,
            eolCount: eolCount,
            firstLineLength: firstLineLength,
            lastLineLength: lastLineLength,
            forceMoveMarkers: forceMoveMarkers,
            isAutoWhitespaceEdit: false
        };
    };
    PieceTreeTextBuffer.prototype._doApplyEdits = function (operations) {
        operations.sort(PieceTreeTextBuffer._sortOpsDescending);
        var contentChanges = [];
        // operations are from bottom to top
        for (var i = 0; i < operations.length; i++) {
            var op = operations[i];
            var startLineNumber = op.range.startLineNumber;
            var startColumn = op.range.startColumn;
            var endLineNumber = op.range.endLineNumber;
            var endColumn = op.range.endColumn;
            if (startLineNumber === endLineNumber && startColumn === endColumn && op.text.length === 0) {
                // no-op
                continue;
            }
            if (op.text) {
                // replacement
                this._pieceTree.delete(op.rangeOffset, op.rangeLength);
                this._pieceTree.insert(op.rangeOffset, op.text, true);
            }
            else {
                // deletion
                this._pieceTree.delete(op.rangeOffset, op.rangeLength);
            }
            var contentChangeRange = new Range(startLineNumber, startColumn, endLineNumber, endColumn);
            contentChanges.push({
                range: contentChangeRange,
                rangeLength: op.rangeLength,
                text: op.text,
                rangeOffset: op.rangeOffset,
                forceMoveMarkers: op.forceMoveMarkers
            });
        }
        return contentChanges;
    };
    PieceTreeTextBuffer.prototype.findMatchesLineByLine = function (searchRange, searchData, captureMatches, limitResultCount) {
        return this._pieceTree.findMatchesLineByLine(searchRange, searchData, captureMatches, limitResultCount);
    };
    // #endregion
    // #region helper
    // testing purpose.
    PieceTreeTextBuffer.prototype.getPieceTree = function () {
        return this._pieceTree;
    };
    PieceTreeTextBuffer._getInverseEditRange = function (range, text) {
        var startLineNumber = range.startLineNumber;
        var startColumn = range.startColumn;
        var _a = __read(countEOL(text), 3), eolCount = _a[0], firstLineLength = _a[1], lastLineLength = _a[2];
        var resultRange;
        if (text.length > 0) {
            // the operation inserts something
            var lineCount = eolCount + 1;
            if (lineCount === 1) {
                // single line insert
                resultRange = new Range(startLineNumber, startColumn, startLineNumber, startColumn + firstLineLength);
            }
            else {
                // multi line insert
                resultRange = new Range(startLineNumber, startColumn, startLineNumber + lineCount - 1, lastLineLength + 1);
            }
        }
        else {
            // There is nothing to insert
            resultRange = new Range(startLineNumber, startColumn, startLineNumber, startColumn);
        }
        return resultRange;
    };
    /**
     * Assumes `operations` are validated and sorted ascending
     */
    PieceTreeTextBuffer._getInverseEditRanges = function (operations) {
        var result = [];
        var prevOpEndLineNumber = 0;
        var prevOpEndColumn = 0;
        var prevOp = null;
        for (var i = 0, len = operations.length; i < len; i++) {
            var op = operations[i];
            var startLineNumber = void 0;
            var startColumn = void 0;
            if (prevOp) {
                if (prevOp.range.endLineNumber === op.range.startLineNumber) {
                    startLineNumber = prevOpEndLineNumber;
                    startColumn = prevOpEndColumn + (op.range.startColumn - prevOp.range.endColumn);
                }
                else {
                    startLineNumber = prevOpEndLineNumber + (op.range.startLineNumber - prevOp.range.endLineNumber);
                    startColumn = op.range.startColumn;
                }
            }
            else {
                startLineNumber = op.range.startLineNumber;
                startColumn = op.range.startColumn;
            }
            var resultRange = void 0;
            if (op.text.length > 0) {
                // the operation inserts something
                var lineCount = op.eolCount + 1;
                if (lineCount === 1) {
                    // single line insert
                    resultRange = new Range(startLineNumber, startColumn, startLineNumber, startColumn + op.firstLineLength);
                }
                else {
                    // multi line insert
                    resultRange = new Range(startLineNumber, startColumn, startLineNumber + lineCount - 1, op.lastLineLength + 1);
                }
            }
            else {
                // There is nothing to insert
                resultRange = new Range(startLineNumber, startColumn, startLineNumber, startColumn);
            }
            prevOpEndLineNumber = resultRange.endLineNumber;
            prevOpEndColumn = resultRange.endColumn;
            result.push(resultRange);
            prevOp = op;
        }
        return result;
    };
    PieceTreeTextBuffer._sortOpsAscending = function (a, b) {
        var r = Range.compareRangesUsingEnds(a.range, b.range);
        if (r === 0) {
            return a.sortIndex - b.sortIndex;
        }
        return r;
    };
    PieceTreeTextBuffer._sortOpsDescending = function (a, b) {
        var r = Range.compareRangesUsingEnds(a.range, b.range);
        if (r === 0) {
            return b.sortIndex - a.sortIndex;
        }
        return -r;
    };
    return PieceTreeTextBuffer;
}(Disposable));
export { PieceTreeTextBuffer };
