/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import * as types from '../../../base/common/types';
import { CursorState, SingleCursorState } from './cursorCommon';
import { MoveOperations } from './cursorMoveOperations';
import { WordOperations } from './cursorWordOperations';
import { Position } from '../core/position';
import { Range } from '../core/range';
var CursorMoveCommands = /** @class */ (function () {
    function CursorMoveCommands() {
    }
    CursorMoveCommands.addCursorDown = function (viewModel, cursors, useLogicalLine) {
        var result = [], resultLen = 0;
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[resultLen++] = new CursorState(cursor.modelState, cursor.viewState);
            if (useLogicalLine) {
                result[resultLen++] = CursorState.fromModelState(MoveOperations.translateDown(viewModel.cursorConfig, viewModel.model, cursor.modelState));
            }
            else {
                result[resultLen++] = CursorState.fromViewState(MoveOperations.translateDown(viewModel.cursorConfig, viewModel, cursor.viewState));
            }
        }
        return result;
    };
    CursorMoveCommands.addCursorUp = function (viewModel, cursors, useLogicalLine) {
        var result = [], resultLen = 0;
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[resultLen++] = new CursorState(cursor.modelState, cursor.viewState);
            if (useLogicalLine) {
                result[resultLen++] = CursorState.fromModelState(MoveOperations.translateUp(viewModel.cursorConfig, viewModel.model, cursor.modelState));
            }
            else {
                result[resultLen++] = CursorState.fromViewState(MoveOperations.translateUp(viewModel.cursorConfig, viewModel, cursor.viewState));
            }
        }
        return result;
    };
    CursorMoveCommands.moveToBeginningOfLine = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = this._moveToLineStart(viewModel, cursor, inSelectionMode);
        }
        return result;
    };
    CursorMoveCommands._moveToLineStart = function (viewModel, cursor, inSelectionMode) {
        var currentViewStateColumn = cursor.viewState.position.column;
        var currentModelStateColumn = cursor.modelState.position.column;
        var isFirstLineOfWrappedLine = currentViewStateColumn === currentModelStateColumn;
        var currentViewStatelineNumber = cursor.viewState.position.lineNumber;
        var firstNonBlankColumn = viewModel.getLineFirstNonWhitespaceColumn(currentViewStatelineNumber);
        var isBeginningOfViewLine = currentViewStateColumn === firstNonBlankColumn;
        if (!isFirstLineOfWrappedLine && !isBeginningOfViewLine) {
            return this._moveToLineStartByView(viewModel, cursor, inSelectionMode);
        }
        else {
            return this._moveToLineStartByModel(viewModel, cursor, inSelectionMode);
        }
    };
    CursorMoveCommands._moveToLineStartByView = function (viewModel, cursor, inSelectionMode) {
        return CursorState.fromViewState(MoveOperations.moveToBeginningOfLine(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode));
    };
    CursorMoveCommands._moveToLineStartByModel = function (viewModel, cursor, inSelectionMode) {
        return CursorState.fromModelState(MoveOperations.moveToBeginningOfLine(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode));
    };
    CursorMoveCommands.moveToEndOfLine = function (viewModel, cursors, inSelectionMode, sticky) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = this._moveToLineEnd(viewModel, cursor, inSelectionMode, sticky);
        }
        return result;
    };
    CursorMoveCommands._moveToLineEnd = function (viewModel, cursor, inSelectionMode, sticky) {
        var viewStatePosition = cursor.viewState.position;
        var viewModelMaxColumn = viewModel.getLineMaxColumn(viewStatePosition.lineNumber);
        var isEndOfViewLine = viewStatePosition.column === viewModelMaxColumn;
        var modelStatePosition = cursor.modelState.position;
        var modelMaxColumn = viewModel.model.getLineMaxColumn(modelStatePosition.lineNumber);
        var isEndLineOfWrappedLine = viewModelMaxColumn - viewStatePosition.column === modelMaxColumn - modelStatePosition.column;
        if (isEndOfViewLine || isEndLineOfWrappedLine) {
            return this._moveToLineEndByModel(viewModel, cursor, inSelectionMode, sticky);
        }
        else {
            return this._moveToLineEndByView(viewModel, cursor, inSelectionMode, sticky);
        }
    };
    CursorMoveCommands._moveToLineEndByView = function (viewModel, cursor, inSelectionMode, sticky) {
        return CursorState.fromViewState(MoveOperations.moveToEndOfLine(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, sticky));
    };
    CursorMoveCommands._moveToLineEndByModel = function (viewModel, cursor, inSelectionMode, sticky) {
        return CursorState.fromModelState(MoveOperations.moveToEndOfLine(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode, sticky));
    };
    CursorMoveCommands.expandLineSelection = function (viewModel, cursors) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var startLineNumber = cursor.modelState.selection.startLineNumber;
            var lineCount = viewModel.model.getLineCount();
            var endLineNumber = cursor.modelState.selection.endLineNumber;
            var endColumn = void 0;
            if (endLineNumber === lineCount) {
                endColumn = viewModel.model.getLineMaxColumn(lineCount);
            }
            else {
                endLineNumber++;
                endColumn = 1;
            }
            result[i] = CursorState.fromModelState(new SingleCursorState(new Range(startLineNumber, 1, startLineNumber, 1), 0, new Position(endLineNumber, endColumn), 0));
        }
        return result;
    };
    CursorMoveCommands.moveToBeginningOfBuffer = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromModelState(MoveOperations.moveToBeginningOfBuffer(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode));
        }
        return result;
    };
    CursorMoveCommands.moveToEndOfBuffer = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromModelState(MoveOperations.moveToEndOfBuffer(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode));
        }
        return result;
    };
    CursorMoveCommands.selectAll = function (viewModel, cursor) {
        var lineCount = viewModel.model.getLineCount();
        var maxColumn = viewModel.model.getLineMaxColumn(lineCount);
        return CursorState.fromModelState(new SingleCursorState(new Range(1, 1, 1, 1), 0, new Position(lineCount, maxColumn), 0));
    };
    CursorMoveCommands.line = function (viewModel, cursor, inSelectionMode, _position, _viewPosition) {
        var position = viewModel.model.validatePosition(_position);
        var viewPosition = (_viewPosition
            ? viewModel.coordinatesConverter.validateViewPosition(new Position(_viewPosition.lineNumber, _viewPosition.column), position)
            : viewModel.coordinatesConverter.convertModelPositionToViewPosition(position));
        if (!inSelectionMode || !cursor.modelState.hasSelection()) {
            // Entering line selection for the first time
            var lineCount = viewModel.model.getLineCount();
            var selectToLineNumber = position.lineNumber + 1;
            var selectToColumn = 1;
            if (selectToLineNumber > lineCount) {
                selectToLineNumber = lineCount;
                selectToColumn = viewModel.model.getLineMaxColumn(selectToLineNumber);
            }
            return CursorState.fromModelState(new SingleCursorState(new Range(position.lineNumber, 1, selectToLineNumber, selectToColumn), 0, new Position(selectToLineNumber, selectToColumn), 0));
        }
        // Continuing line selection
        var enteringLineNumber = cursor.modelState.selectionStart.getStartPosition().lineNumber;
        if (position.lineNumber < enteringLineNumber) {
            return CursorState.fromViewState(cursor.viewState.move(cursor.modelState.hasSelection(), viewPosition.lineNumber, 1, 0));
        }
        else if (position.lineNumber > enteringLineNumber) {
            var lineCount = viewModel.getLineCount();
            var selectToViewLineNumber = viewPosition.lineNumber + 1;
            var selectToViewColumn = 1;
            if (selectToViewLineNumber > lineCount) {
                selectToViewLineNumber = lineCount;
                selectToViewColumn = viewModel.getLineMaxColumn(selectToViewLineNumber);
            }
            return CursorState.fromViewState(cursor.viewState.move(cursor.modelState.hasSelection(), selectToViewLineNumber, selectToViewColumn, 0));
        }
        else {
            var endPositionOfSelectionStart = cursor.modelState.selectionStart.getEndPosition();
            return CursorState.fromModelState(cursor.modelState.move(cursor.modelState.hasSelection(), endPositionOfSelectionStart.lineNumber, endPositionOfSelectionStart.column, 0));
        }
    };
    CursorMoveCommands.word = function (viewModel, cursor, inSelectionMode, _position) {
        var position = viewModel.model.validatePosition(_position);
        return CursorState.fromModelState(WordOperations.word(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode, position));
    };
    CursorMoveCommands.cancelSelection = function (viewModel, cursor) {
        if (!cursor.modelState.hasSelection()) {
            return new CursorState(cursor.modelState, cursor.viewState);
        }
        var lineNumber = cursor.viewState.position.lineNumber;
        var column = cursor.viewState.position.column;
        return CursorState.fromViewState(new SingleCursorState(new Range(lineNumber, column, lineNumber, column), 0, new Position(lineNumber, column), 0));
    };
    CursorMoveCommands.moveTo = function (viewModel, cursor, inSelectionMode, _position, _viewPosition) {
        var position = viewModel.model.validatePosition(_position);
        var viewPosition = (_viewPosition
            ? viewModel.coordinatesConverter.validateViewPosition(new Position(_viewPosition.lineNumber, _viewPosition.column), position)
            : viewModel.coordinatesConverter.convertModelPositionToViewPosition(position));
        return CursorState.fromViewState(cursor.viewState.move(inSelectionMode, viewPosition.lineNumber, viewPosition.column, 0));
    };
    CursorMoveCommands.simpleMove = function (viewModel, cursors, direction, inSelectionMode, value, unit) {
        switch (direction) {
            case 0 /* Left */: {
                if (unit === 4 /* HalfLine */) {
                    // Move left by half the current line length
                    return this._moveHalfLineLeft(viewModel, cursors, inSelectionMode);
                }
                else {
                    // Move left by `moveParams.value` columns
                    return this._moveLeft(viewModel, cursors, inSelectionMode, value);
                }
            }
            case 1 /* Right */: {
                if (unit === 4 /* HalfLine */) {
                    // Move right by half the current line length
                    return this._moveHalfLineRight(viewModel, cursors, inSelectionMode);
                }
                else {
                    // Move right by `moveParams.value` columns
                    return this._moveRight(viewModel, cursors, inSelectionMode, value);
                }
            }
            case 2 /* Up */: {
                if (unit === 2 /* WrappedLine */) {
                    // Move up by view lines
                    return this._moveUpByViewLines(viewModel, cursors, inSelectionMode, value);
                }
                else {
                    // Move up by model lines
                    return this._moveUpByModelLines(viewModel, cursors, inSelectionMode, value);
                }
            }
            case 3 /* Down */: {
                if (unit === 2 /* WrappedLine */) {
                    // Move down by view lines
                    return this._moveDownByViewLines(viewModel, cursors, inSelectionMode, value);
                }
                else {
                    // Move down by model lines
                    return this._moveDownByModelLines(viewModel, cursors, inSelectionMode, value);
                }
            }
            case 4 /* PrevBlankLine */: {
                if (unit === 2 /* WrappedLine */) {
                    return cursors.map(function (cursor) { return CursorState.fromViewState(MoveOperations.moveToPrevBlankLine(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode)); });
                }
                else {
                    return cursors.map(function (cursor) { return CursorState.fromModelState(MoveOperations.moveToPrevBlankLine(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode)); });
                }
            }
            case 5 /* NextBlankLine */: {
                if (unit === 2 /* WrappedLine */) {
                    return cursors.map(function (cursor) { return CursorState.fromViewState(MoveOperations.moveToNextBlankLine(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode)); });
                }
                else {
                    return cursors.map(function (cursor) { return CursorState.fromModelState(MoveOperations.moveToNextBlankLine(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode)); });
                }
            }
            case 6 /* WrappedLineStart */: {
                // Move to the beginning of the current view line
                return this._moveToViewMinColumn(viewModel, cursors, inSelectionMode);
            }
            case 7 /* WrappedLineFirstNonWhitespaceCharacter */: {
                // Move to the first non-whitespace column of the current view line
                return this._moveToViewFirstNonWhitespaceColumn(viewModel, cursors, inSelectionMode);
            }
            case 8 /* WrappedLineColumnCenter */: {
                // Move to the "center" of the current view line
                return this._moveToViewCenterColumn(viewModel, cursors, inSelectionMode);
            }
            case 9 /* WrappedLineEnd */: {
                // Move to the end of the current view line
                return this._moveToViewMaxColumn(viewModel, cursors, inSelectionMode);
            }
            case 10 /* WrappedLineLastNonWhitespaceCharacter */: {
                // Move to the last non-whitespace column of the current view line
                return this._moveToViewLastNonWhitespaceColumn(viewModel, cursors, inSelectionMode);
            }
            default:
                return null;
        }
    };
    CursorMoveCommands.viewportMove = function (viewModel, cursors, direction, inSelectionMode, value) {
        var visibleViewRange = viewModel.getCompletelyVisibleViewRange();
        var visibleModelRange = viewModel.coordinatesConverter.convertViewRangeToModelRange(visibleViewRange);
        switch (direction) {
            case 11 /* ViewPortTop */: {
                // Move to the nth line start in the viewport (from the top)
                var modelLineNumber = this._firstLineNumberInRange(viewModel.model, visibleModelRange, value);
                var modelColumn = viewModel.model.getLineFirstNonWhitespaceColumn(modelLineNumber);
                return [this._moveToModelPosition(viewModel, cursors[0], inSelectionMode, modelLineNumber, modelColumn)];
            }
            case 13 /* ViewPortBottom */: {
                // Move to the nth line start in the viewport (from the bottom)
                var modelLineNumber = this._lastLineNumberInRange(viewModel.model, visibleModelRange, value);
                var modelColumn = viewModel.model.getLineFirstNonWhitespaceColumn(modelLineNumber);
                return [this._moveToModelPosition(viewModel, cursors[0], inSelectionMode, modelLineNumber, modelColumn)];
            }
            case 12 /* ViewPortCenter */: {
                // Move to the line start in the viewport center
                var modelLineNumber = Math.round((visibleModelRange.startLineNumber + visibleModelRange.endLineNumber) / 2);
                var modelColumn = viewModel.model.getLineFirstNonWhitespaceColumn(modelLineNumber);
                return [this._moveToModelPosition(viewModel, cursors[0], inSelectionMode, modelLineNumber, modelColumn)];
            }
            case 14 /* ViewPortIfOutside */: {
                // Move to a position inside the viewport
                var result = [];
                for (var i = 0, len = cursors.length; i < len; i++) {
                    var cursor = cursors[i];
                    result[i] = this.findPositionInViewportIfOutside(viewModel, cursor, visibleViewRange, inSelectionMode);
                }
                return result;
            }
            default:
                return null;
        }
    };
    CursorMoveCommands.findPositionInViewportIfOutside = function (viewModel, cursor, visibleViewRange, inSelectionMode) {
        var viewLineNumber = cursor.viewState.position.lineNumber;
        if (visibleViewRange.startLineNumber <= viewLineNumber && viewLineNumber <= visibleViewRange.endLineNumber - 1) {
            // Nothing to do, cursor is in viewport
            return new CursorState(cursor.modelState, cursor.viewState);
        }
        else {
            if (viewLineNumber > visibleViewRange.endLineNumber - 1) {
                viewLineNumber = visibleViewRange.endLineNumber - 1;
            }
            if (viewLineNumber < visibleViewRange.startLineNumber) {
                viewLineNumber = visibleViewRange.startLineNumber;
            }
            var viewColumn = viewModel.getLineFirstNonWhitespaceColumn(viewLineNumber);
            return this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
    };
    /**
     * Find the nth line start included in the range (from the start).
     */
    CursorMoveCommands._firstLineNumberInRange = function (model, range, count) {
        var startLineNumber = range.startLineNumber;
        if (range.startColumn !== model.getLineMinColumn(startLineNumber)) {
            // Move on to the second line if the first line start is not included in the range
            startLineNumber++;
        }
        return Math.min(range.endLineNumber, startLineNumber + count - 1);
    };
    /**
     * Find the nth line start included in the range (from the end).
     */
    CursorMoveCommands._lastLineNumberInRange = function (model, range, count) {
        var startLineNumber = range.startLineNumber;
        if (range.startColumn !== model.getLineMinColumn(startLineNumber)) {
            // Move on to the second line if the first line start is not included in the range
            startLineNumber++;
        }
        return Math.max(startLineNumber, range.endLineNumber - count + 1);
    };
    CursorMoveCommands._moveLeft = function (viewModel, cursors, inSelectionMode, noOfColumns) {
        var hasMultipleCursors = (cursors.length > 1);
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var skipWrappingPointStop = hasMultipleCursors || !cursor.viewState.hasSelection();
            var newViewState = MoveOperations.moveLeft(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, noOfColumns);
            if (skipWrappingPointStop
                && noOfColumns === 1
                && cursor.viewState.position.column === viewModel.getLineMinColumn(cursor.viewState.position.lineNumber)
                && newViewState.position.lineNumber !== cursor.viewState.position.lineNumber) {
                // moved over to the previous view line
                var newViewModelPosition = viewModel.coordinatesConverter.convertViewPositionToModelPosition(newViewState.position);
                if (newViewModelPosition.lineNumber === cursor.modelState.position.lineNumber) {
                    // stayed on the same model line => pass wrapping point where 2 view positions map to a single model position
                    newViewState = MoveOperations.moveLeft(viewModel.cursorConfig, viewModel, newViewState, inSelectionMode, 1);
                }
            }
            result[i] = CursorState.fromViewState(newViewState);
        }
        return result;
    };
    CursorMoveCommands._moveHalfLineLeft = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var halfLine = Math.round(viewModel.getLineContent(viewLineNumber).length / 2);
            result[i] = CursorState.fromViewState(MoveOperations.moveLeft(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, halfLine));
        }
        return result;
    };
    CursorMoveCommands._moveRight = function (viewModel, cursors, inSelectionMode, noOfColumns) {
        var hasMultipleCursors = (cursors.length > 1);
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var skipWrappingPointStop = hasMultipleCursors || !cursor.viewState.hasSelection();
            var newViewState = MoveOperations.moveRight(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, noOfColumns);
            if (skipWrappingPointStop
                && noOfColumns === 1
                && cursor.viewState.position.column === viewModel.getLineMaxColumn(cursor.viewState.position.lineNumber)
                && newViewState.position.lineNumber !== cursor.viewState.position.lineNumber) {
                // moved over to the next view line
                var newViewModelPosition = viewModel.coordinatesConverter.convertViewPositionToModelPosition(newViewState.position);
                if (newViewModelPosition.lineNumber === cursor.modelState.position.lineNumber) {
                    // stayed on the same model line => pass wrapping point where 2 view positions map to a single model position
                    newViewState = MoveOperations.moveRight(viewModel.cursorConfig, viewModel, newViewState, inSelectionMode, 1);
                }
            }
            result[i] = CursorState.fromViewState(newViewState);
        }
        return result;
    };
    CursorMoveCommands._moveHalfLineRight = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var halfLine = Math.round(viewModel.getLineContent(viewLineNumber).length / 2);
            result[i] = CursorState.fromViewState(MoveOperations.moveRight(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, halfLine));
        }
        return result;
    };
    CursorMoveCommands._moveDownByViewLines = function (viewModel, cursors, inSelectionMode, linesCount) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromViewState(MoveOperations.moveDown(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, linesCount));
        }
        return result;
    };
    CursorMoveCommands._moveDownByModelLines = function (viewModel, cursors, inSelectionMode, linesCount) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromModelState(MoveOperations.moveDown(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode, linesCount));
        }
        return result;
    };
    CursorMoveCommands._moveUpByViewLines = function (viewModel, cursors, inSelectionMode, linesCount) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromViewState(MoveOperations.moveUp(viewModel.cursorConfig, viewModel, cursor.viewState, inSelectionMode, linesCount));
        }
        return result;
    };
    CursorMoveCommands._moveUpByModelLines = function (viewModel, cursors, inSelectionMode, linesCount) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            result[i] = CursorState.fromModelState(MoveOperations.moveUp(viewModel.cursorConfig, viewModel.model, cursor.modelState, inSelectionMode, linesCount));
        }
        return result;
    };
    CursorMoveCommands._moveToViewPosition = function (viewModel, cursor, inSelectionMode, toViewLineNumber, toViewColumn) {
        return CursorState.fromViewState(cursor.viewState.move(inSelectionMode, toViewLineNumber, toViewColumn, 0));
    };
    CursorMoveCommands._moveToModelPosition = function (viewModel, cursor, inSelectionMode, toModelLineNumber, toModelColumn) {
        return CursorState.fromModelState(cursor.modelState.move(inSelectionMode, toModelLineNumber, toModelColumn, 0));
    };
    CursorMoveCommands._moveToViewMinColumn = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var viewColumn = viewModel.getLineMinColumn(viewLineNumber);
            result[i] = this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
        return result;
    };
    CursorMoveCommands._moveToViewFirstNonWhitespaceColumn = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var viewColumn = viewModel.getLineFirstNonWhitespaceColumn(viewLineNumber);
            result[i] = this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
        return result;
    };
    CursorMoveCommands._moveToViewCenterColumn = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var viewColumn = Math.round((viewModel.getLineMaxColumn(viewLineNumber) + viewModel.getLineMinColumn(viewLineNumber)) / 2);
            result[i] = this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
        return result;
    };
    CursorMoveCommands._moveToViewMaxColumn = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var viewColumn = viewModel.getLineMaxColumn(viewLineNumber);
            result[i] = this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
        return result;
    };
    CursorMoveCommands._moveToViewLastNonWhitespaceColumn = function (viewModel, cursors, inSelectionMode) {
        var result = [];
        for (var i = 0, len = cursors.length; i < len; i++) {
            var cursor = cursors[i];
            var viewLineNumber = cursor.viewState.position.lineNumber;
            var viewColumn = viewModel.getLineLastNonWhitespaceColumn(viewLineNumber);
            result[i] = this._moveToViewPosition(viewModel, cursor, inSelectionMode, viewLineNumber, viewColumn);
        }
        return result;
    };
    return CursorMoveCommands;
}());
export { CursorMoveCommands };
export var CursorMove;
(function (CursorMove) {
    var isCursorMoveArgs = function (arg) {
        if (!types.isObject(arg)) {
            return false;
        }
        var cursorMoveArg = arg;
        if (!types.isString(cursorMoveArg.to)) {
            return false;
        }
        if (!types.isUndefined(cursorMoveArg.select) && !types.isBoolean(cursorMoveArg.select)) {
            return false;
        }
        if (!types.isUndefined(cursorMoveArg.by) && !types.isString(cursorMoveArg.by)) {
            return false;
        }
        if (!types.isUndefined(cursorMoveArg.value) && !types.isNumber(cursorMoveArg.value)) {
            return false;
        }
        return true;
    };
    CursorMove.description = {
        description: 'Move cursor to a logical position in the view',
        args: [
            {
                name: 'Cursor move argument object',
                description: "Property-value pairs that can be passed through this argument:\n\t\t\t\t\t* 'to': A mandatory logical position value providing where to move the cursor.\n\t\t\t\t\t\t```\n\t\t\t\t\t\t'left', 'right', 'up', 'down', 'prevBlankLine', 'nextBlankLine',\n\t\t\t\t\t\t'wrappedLineStart', 'wrappedLineEnd', 'wrappedLineColumnCenter'\n\t\t\t\t\t\t'wrappedLineFirstNonWhitespaceCharacter', 'wrappedLineLastNonWhitespaceCharacter'\n\t\t\t\t\t\t'viewPortTop', 'viewPortCenter', 'viewPortBottom', 'viewPortIfOutside'\n\t\t\t\t\t\t```\n\t\t\t\t\t* 'by': Unit to move. Default is computed based on 'to' value.\n\t\t\t\t\t\t```\n\t\t\t\t\t\t'line', 'wrappedLine', 'character', 'halfLine'\n\t\t\t\t\t\t```\n\t\t\t\t\t* 'value': Number of units to move. Default is '1'.\n\t\t\t\t\t* 'select': If 'true' makes the selection. Default is 'false'.\n\t\t\t\t",
                constraint: isCursorMoveArgs,
                schema: {
                    'type': 'object',
                    'required': ['to'],
                    'properties': {
                        'to': {
                            'type': 'string',
                            'enum': ['left', 'right', 'up', 'down', 'prevBlankLine', 'nextBlankLine', 'wrappedLineStart', 'wrappedLineEnd', 'wrappedLineColumnCenter', 'wrappedLineFirstNonWhitespaceCharacter', 'wrappedLineLastNonWhitespaceCharacter', 'viewPortTop', 'viewPortCenter', 'viewPortBottom', 'viewPortIfOutside']
                        },
                        'by': {
                            'type': 'string',
                            'enum': ['line', 'wrappedLine', 'character', 'halfLine']
                        },
                        'value': {
                            'type': 'number',
                            'default': 1
                        },
                        'select': {
                            'type': 'boolean',
                            'default': false
                        }
                    }
                }
            }
        ]
    };
    /**
     * Positions in the view for cursor move command.
     */
    CursorMove.RawDirection = {
        Left: 'left',
        Right: 'right',
        Up: 'up',
        Down: 'down',
        PrevBlankLine: 'prevBlankLine',
        NextBlankLine: 'nextBlankLine',
        WrappedLineStart: 'wrappedLineStart',
        WrappedLineFirstNonWhitespaceCharacter: 'wrappedLineFirstNonWhitespaceCharacter',
        WrappedLineColumnCenter: 'wrappedLineColumnCenter',
        WrappedLineEnd: 'wrappedLineEnd',
        WrappedLineLastNonWhitespaceCharacter: 'wrappedLineLastNonWhitespaceCharacter',
        ViewPortTop: 'viewPortTop',
        ViewPortCenter: 'viewPortCenter',
        ViewPortBottom: 'viewPortBottom',
        ViewPortIfOutside: 'viewPortIfOutside'
    };
    /**
     * Units for Cursor move 'by' argument
     */
    CursorMove.RawUnit = {
        Line: 'line',
        WrappedLine: 'wrappedLine',
        Character: 'character',
        HalfLine: 'halfLine'
    };
    function parse(args) {
        if (!args.to) {
            // illegal arguments
            return null;
        }
        var direction;
        switch (args.to) {
            case CursorMove.RawDirection.Left:
                direction = 0 /* Left */;
                break;
            case CursorMove.RawDirection.Right:
                direction = 1 /* Right */;
                break;
            case CursorMove.RawDirection.Up:
                direction = 2 /* Up */;
                break;
            case CursorMove.RawDirection.Down:
                direction = 3 /* Down */;
                break;
            case CursorMove.RawDirection.PrevBlankLine:
                direction = 4 /* PrevBlankLine */;
                break;
            case CursorMove.RawDirection.NextBlankLine:
                direction = 5 /* NextBlankLine */;
                break;
            case CursorMove.RawDirection.WrappedLineStart:
                direction = 6 /* WrappedLineStart */;
                break;
            case CursorMove.RawDirection.WrappedLineFirstNonWhitespaceCharacter:
                direction = 7 /* WrappedLineFirstNonWhitespaceCharacter */;
                break;
            case CursorMove.RawDirection.WrappedLineColumnCenter:
                direction = 8 /* WrappedLineColumnCenter */;
                break;
            case CursorMove.RawDirection.WrappedLineEnd:
                direction = 9 /* WrappedLineEnd */;
                break;
            case CursorMove.RawDirection.WrappedLineLastNonWhitespaceCharacter:
                direction = 10 /* WrappedLineLastNonWhitespaceCharacter */;
                break;
            case CursorMove.RawDirection.ViewPortTop:
                direction = 11 /* ViewPortTop */;
                break;
            case CursorMove.RawDirection.ViewPortBottom:
                direction = 13 /* ViewPortBottom */;
                break;
            case CursorMove.RawDirection.ViewPortCenter:
                direction = 12 /* ViewPortCenter */;
                break;
            case CursorMove.RawDirection.ViewPortIfOutside:
                direction = 14 /* ViewPortIfOutside */;
                break;
            default:
                // illegal arguments
                return null;
        }
        var unit = 0 /* None */;
        switch (args.by) {
            case CursorMove.RawUnit.Line:
                unit = 1 /* Line */;
                break;
            case CursorMove.RawUnit.WrappedLine:
                unit = 2 /* WrappedLine */;
                break;
            case CursorMove.RawUnit.Character:
                unit = 3 /* Character */;
                break;
            case CursorMove.RawUnit.HalfLine:
                unit = 4 /* HalfLine */;
                break;
        }
        return {
            direction: direction,
            unit: unit,
            select: (!!args.select),
            value: (args.value || 1)
        };
    }
    CursorMove.parse = parse;
})(CursorMove || (CursorMove = {}));
