/**
 * Copyright (c) 2010-2016, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Zoltan Ujhelyi - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.emf.types;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.scoping.IMetamodelProvider;
import org.eclipse.viatra.query.patternlanguage.emf.types.AbstractTypeSystem;
import org.eclipse.viatra.query.patternlanguage.emf.types.BottomTypeKey;
import org.eclipse.viatra.query.patternlanguage.emf.util.IClassLoaderProvider;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ClassType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.JavaType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternBody;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ReferenceType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.RelationType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Type;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.patternlanguage.emf.vql.VariableReference;
import org.eclipse.viatra.query.runtime.emf.EMFQueryMetaContext;
import org.eclipse.viatra.query.runtime.emf.types.EClassTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EClassUnscopedTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EDataTypeInSlotsKey;
import org.eclipse.viatra.query.runtime.emf.types.EStructuralFeatureInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.Primitives;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * @author Zoltan Ujhelyi
 */
@SuppressWarnings("all")
public class EMFTypeSystem extends AbstractTypeSystem {
  private final static String NON_EMF_TYPE_ENCOUNTERED = "EMF Type System only supports EMF Types but %s found.";
  
  /**
   * This function can be used to extract EClassifier instances from IInputKey instances.
   * If the IInputKey instance does not represent an EClassifier, null is returned
   * @since 2.0
   */
  public static EClassifier extractClassifierFromInputKey(final IInputKey key) {
    EClassifier _switchResult = null;
    boolean _matched = false;
    if (key instanceof EClassTransitiveInstancesKey) {
      _matched=true;
      _switchResult = ((EClassTransitiveInstancesKey)key).getEmfKey();
    }
    if (!_matched) {
      if (key instanceof EDataTypeInSlotsKey) {
        _matched=true;
        _switchResult = ((EDataTypeInSlotsKey)key).getEmfKey();
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  /**
   * This function can be used to extract EClassifier instances from Type declaration.
   * If the IInputKey instance does not represent an EClassifier, null is returned
   * @since 2.0
   */
  public static EClassifier extractClassifierFromType(final Type typeDeclaration) {
    EClassifier _switchResult = null;
    boolean _matched = false;
    if (typeDeclaration instanceof ClassType) {
      _matched=true;
      _switchResult = ((ClassType)typeDeclaration).getClassname();
    }
    if (!_matched) {
      if (typeDeclaration instanceof ReferenceType) {
        _matched=true;
        EStructuralFeature _refname = ((ReferenceType)typeDeclaration).getRefname();
        EClassifier _eType = null;
        if (_refname!=null) {
          _eType=_refname.getEType();
        }
        _switchResult = _eType;
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  @Inject
  private IMetamodelProvider metamodelProvider;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  @Inject
  private Primitives primitives;
  
  @Inject
  private TypeReferences typeReferences;
  
  @Inject
  private IClassLoaderProvider classLoaderProvider;
  
  @Inject
  public EMFTypeSystem(final Logger logger) {
    super(EMFQueryMetaContext.DEFAULT);
  }
  
  @Override
  public IInputKey extractTypeDescriptor(final Type type) {
    Preconditions.checkArgument((((type instanceof ClassType) || (type instanceof ReferenceType)) || (type instanceof JavaType)), 
      EMFTypeSystem.NON_EMF_TYPE_ENCOUNTERED, type.getClass());
    if ((type instanceof ClassType)) {
      final EClassifier classifier = ((ClassType)type).getClassname();
      return this.classifierToInputKey(classifier);
    } else {
      if ((type instanceof ReferenceType)) {
        EStructuralFeature _refname = ((ReferenceType)type).getRefname();
        EClassifier _eType = null;
        if (_refname!=null) {
          _eType=_refname.getEType();
        }
        return this.classifierToInputKey(_eType);
      } else {
        if ((type instanceof JavaType)) {
          String _identifier = ((JavaType)type).getClassRef().getIdentifier();
          return new JavaTransitiveInstancesKey(_identifier);
        }
      }
    }
    throw new UnsupportedOperationException();
  }
  
  public IInputKey classifierToInputKey(final EClassifier classifier) {
    IInputKey _switchResult = null;
    boolean _matched = false;
    boolean _eIsProxy = classifier.eIsProxy();
    if (_eIsProxy) {
      _matched=true;
      _switchResult = BottomTypeKey.INSTANCE;
    }
    if (!_matched) {
      if (classifier instanceof EClass) {
        _matched=true;
        _switchResult = new EClassTransitiveInstancesKey(((EClass)classifier));
      }
    }
    if (!_matched) {
      if (classifier instanceof EDataType) {
        _matched=true;
        _switchResult = new EDataTypeInSlotsKey(((EDataType)classifier));
      }
    }
    if (!_matched) {
      _switchResult = BottomTypeKey.INSTANCE;
    }
    return _switchResult;
  }
  
  /**
   * Returns the EClassifier stored in the IInputKey. If no EClassifier is represented in the key,
   * the optional will be empty.
   * @since 2.0
   */
  public Optional<EClassifier> inputKeyToClassifier(final IInputKey key) {
    Optional<EClassifier> _switchResult = null;
    boolean _matched = false;
    if (key instanceof EClassTransitiveInstancesKey) {
      _matched=true;
      _switchResult = Optional.<EClassifier>ofNullable(((EClassTransitiveInstancesKey)key).getEmfKey());
    }
    if (!_matched) {
      if (key instanceof EClassUnscopedTransitiveInstancesKey) {
        _matched=true;
        _switchResult = Optional.<EClassifier>ofNullable(((EClassUnscopedTransitiveInstancesKey)key).getEmfKey());
      }
    }
    if (!_matched) {
      if (key instanceof EDataTypeInSlotsKey) {
        _matched=true;
        _switchResult = Optional.<EClassifier>ofNullable(((EDataTypeInSlotsKey)key).getEmfKey());
      }
    }
    if (!_matched) {
      _switchResult = Optional.<EClassifier>empty();
    }
    return _switchResult;
  }
  
  @Override
  public IInputKey extractColumnDescriptor(final RelationType type, final int columnIndex) {
    Preconditions.checkArgument((type instanceof ReferenceType), EMFTypeSystem.NON_EMF_TYPE_ENCOUNTERED, type.getClass());
    if ((type instanceof ReferenceType)) {
      final EStructuralFeature feature = ((ReferenceType)type).getRefname();
      return this.extractColumnDescriptor(feature, columnIndex);
    }
    throw new UnsupportedOperationException();
  }
  
  private IInputKey extractColumnDescriptor(final EStructuralFeature feature, final int columnIndex) {
    if (((feature == null) || feature.eIsProxy())) {
      return BottomTypeKey.INSTANCE;
    }
    if ((0 == columnIndex)) {
      EClass _eContainingClass = feature.getEContainingClass();
      return new EClassTransitiveInstancesKey(_eContainingClass);
    } else {
      if ((feature instanceof EReference)) {
        EClass _eReferenceType = ((EReference)feature).getEReferenceType();
        return new EClassTransitiveInstancesKey(_eReferenceType);
      } else {
        EDataType _eAttributeType = ((EAttribute) feature).getEAttributeType();
        return new EDataTypeInSlotsKey(_eAttributeType);
      }
    }
  }
  
  @Override
  public boolean isConformant(final IInputKey expectedType, final IInputKey actualType) {
    if ((expectedType instanceof BottomTypeKey)) {
      return false;
    } else {
      if ((actualType instanceof BottomTypeKey)) {
        return false;
      } else {
        if ((expectedType instanceof EClassTransitiveInstancesKey)) {
          if ((actualType instanceof EClassTransitiveInstancesKey)) {
            return this.isConform(((EClassTransitiveInstancesKey)expectedType).getEmfKey(), ((EClassTransitiveInstancesKey)actualType).getEmfKey());
          }
        } else {
          if ((expectedType instanceof EDataTypeInSlotsKey)) {
            if ((actualType instanceof EDataTypeInSlotsKey)) {
              return this.isConform(((EDataTypeInSlotsKey)expectedType).getEmfKey(), ((EDataTypeInSlotsKey)actualType).getEmfKey());
            } else {
              if ((actualType instanceof JavaTransitiveInstancesKey)) {
                final Class<?> expectedJavaClass = this.getJavaClass(((EDataTypeInSlotsKey)expectedType));
                final Class<?> actualJavaClass = this.getJavaClass(((JavaTransitiveInstancesKey)actualType));
                return (((expectedJavaClass != null) && (actualJavaClass != null)) && expectedJavaClass.isAssignableFrom(actualJavaClass));
              }
            }
          } else {
            if ((expectedType instanceof JavaTransitiveInstancesKey)) {
              if ((actualType instanceof JavaTransitiveInstancesKey)) {
                final Class<?> expectedJavaClass_1 = ((JavaTransitiveInstancesKey)expectedType).getInstanceClass();
                final Class<?> actualJavaClass_1 = ((JavaTransitiveInstancesKey)actualType).getInstanceClass();
                return (((expectedJavaClass_1 != null) && (actualJavaClass_1 != null)) && expectedJavaClass_1.isAssignableFrom(actualJavaClass_1));
              } else {
                if ((actualType instanceof EDataTypeInSlotsKey)) {
                  final Class<?> expectedJavaClass_2 = this.getJavaClass(((JavaTransitiveInstancesKey)expectedType));
                  final Class<?> actualJavaClass_2 = this.getJavaClass(((EDataTypeInSlotsKey)actualType));
                  return (((expectedJavaClass_2 != null) && (actualJavaClass_2 != null)) && expectedJavaClass_2.isAssignableFrom(actualJavaClass_2));
                }
              }
            }
          }
        }
      }
    }
    return false;
  }
  
  /**
   * @since 1.5
   */
  public Class<?> getJavaClass(final EDataTypeInSlotsKey key) {
    try {
      Class<?> dataTypeClass = key.getEmfKey().getInstanceClass();
      if ((dataTypeClass == null)) {
        dataTypeClass = this.classLoaderProvider.getClassLoader(key.getEmfKey()).loadClass(key.getEmfKey().getInstanceClassName());
      }
      boolean _isPrimitive = dataTypeClass.isPrimitive();
      if (_isPrimitive) {
        Class<?> _wrapperClassForType = null;
        if (dataTypeClass!=null) {
          _wrapperClassForType=AbstractTypeSystem.getWrapperClassForType(dataTypeClass);
        }
        dataTypeClass = _wrapperClassForType;
      }
      return dataTypeClass;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * @since 2.1
   */
  public String getJavaClassName(final EDataTypeInSlotsKey key) {
    return AbstractTypeSystem.getWrapperClassNameForTypeName(key.getEmfKey().getInstanceClassName());
  }
  
  /**
   * @since 1.5
   */
  public Class<?> getJavaClass(final JavaTransitiveInstancesKey javaKey) {
    Class<?> javaTypeClass = javaKey.getInstanceClass();
    boolean _isPrimitive = javaTypeClass.isPrimitive();
    if (_isPrimitive) {
      javaTypeClass = AbstractTypeSystem.getWrapperClassForType(javaTypeClass);
    }
    return javaTypeClass;
  }
  
  /**
   * @since 2.0
   */
  public boolean isConformant(final ClassType expectedType, final ClassType actualType) {
    final IInputKey expectedClassifier = this.extractTypeDescriptor(expectedType);
    final IInputKey actualClassifier = this.extractTypeDescriptor(actualType);
    return this.isConformant(expectedClassifier, actualClassifier);
  }
  
  private boolean isConform(final EClassifier expectedClassifier, final EClassifier actualClassifier) {
    if ((actualClassifier instanceof EClass)) {
      return EcoreUtil2.getCompatibleTypesOf(((EClass)actualClassifier)).contains(expectedClassifier);
    } else {
      return expectedClassifier.equals(actualClassifier);
    }
  }
  
  @Override
  public boolean isConformToRelationColumn(final IInputKey relationType, final int columnIndex, final IInputKey columnType) {
    if ((relationType instanceof EStructuralFeatureInstancesKey)) {
      final EStructuralFeature feature = ((EStructuralFeatureInstancesKey)relationType).getEmfKey();
      return this.isConformant(this.extractColumnDescriptor(feature, columnIndex), columnType);
    } else {
      return false;
    }
  }
  
  /**
   * @since 2.0
   */
  public boolean isConformToRelationSource(final ReferenceType relationType, final ClassType sourceType) {
    final EStructuralFeature featureType = relationType.getRefname();
    final EClassifier classifier = sourceType.getClassname();
    final EClass sourceClass = featureType.getEContainingClass();
    return this.isConform(sourceClass, classifier);
  }
  
  /**
   * @since 2.0
   */
  public boolean isConformToRelationTarget(final ReferenceType relationType, final ClassType targetType) {
    final EStructuralFeature featureType = relationType.getRefname();
    final EClassifier classifier = targetType.getClassname();
    final EClassifier targetClassifier = featureType.getEType();
    return this.isConform(targetClassifier, classifier);
  }
  
  @Override
  public JvmTypeReference toJvmTypeReference(final IInputKey type, final EObject context) {
    if ((type instanceof EClassTransitiveInstancesKey)) {
      return this.getJvmType(((EClassTransitiveInstancesKey)type).getEmfKey(), context);
    } else {
      if ((type instanceof EDataTypeInSlotsKey)) {
        return this.getJvmType(((EDataTypeInSlotsKey)type).getEmfKey(), context);
      } else {
        if ((type instanceof JavaTransitiveInstancesKey)) {
          return this.typeReferences.getTypeForName(((JavaTransitiveInstancesKey)type).getWrappedKey(), context);
        }
      }
    }
    return this.typeReferences.getTypeForName(Object.class, context);
  }
  
  private JvmTypeReference getJvmType(final EClassifier classifier, final EObject context) {
    JvmTypeReference _xblockexpression = null;
    {
      if ((classifier != null)) {
        final String className = this.metamodelProvider.getQualifiedClassName(classifier, context);
        boolean _isNullOrEmpty = Strings.isNullOrEmpty(className);
        boolean _not = (!_isNullOrEmpty);
        if (_not) {
          return this.getTypeReferenceForTypeName(className, context);
        }
      }
      Class<?> _xifexpression = null;
      if ((classifier instanceof EClass)) {
        _xifexpression = EObject.class;
      } else {
        _xifexpression = Object.class;
      }
      final Class<?> clazz = _xifexpression;
      _xblockexpression = this.typeReferences.getTypeForName(clazz, context);
    }
    return _xblockexpression;
  }
  
  private JvmTypeReference getTypeReferenceForTypeName(final String typeName, final EObject context) {
    final JvmTypeReference typeRef = this.typeReferences.getTypeForName(typeName, context);
    final JvmTypeReference typeReference = this.primitives.asWrapperTypeIfPrimitive(typeRef);
    if ((typeReference == null)) {
      EObject errorContext = context;
      String contextName = context.toString();
      if (((context instanceof Variable) && (((Variable) context).eContainer() instanceof PatternBody))) {
        final Variable variable = ((Variable) context);
        final Optional<VariableReference> reference = PatternLanguageHelper.getReferences(variable).findAny();
        boolean _isPresent = reference.isPresent();
        if (_isPresent) {
          contextName = variable.getName();
          errorContext = reference.get();
        }
      }
      this.errorFeedback.reportError(errorContext, 
        String.format(
          "Cannot resolve corresponding Java type for variable %s. Are the required bundle dependencies set?", contextName), EMFPatternLanguageJvmModelInferrer.INVALID_TYPEREF_CODE, Severity.WARNING, 
        IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
    }
    return typeReference;
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> minimizeTypeInformation(final Set<IInputKey> types, final boolean mergeWithSupertypes) {
    int _size = types.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      return types;
    }
    final Iterable<JavaTransitiveInstancesKey> eJavaTypes = this.minimizeJavaTypeList(IterableExtensions.<JavaTransitiveInstancesKey>filterNull(Iterables.<JavaTransitiveInstancesKey>filter(types, JavaTransitiveInstancesKey.class)));
    final Function1<EDataTypeInSlotsKey, Boolean> _function = (EDataTypeInSlotsKey it) -> {
      boolean _xblockexpression = false;
      {
        Class<?> _wrapperClassForType = AbstractTypeSystem.getWrapperClassForType(it.getEmfKey().getInstanceClass());
        final JavaTransitiveInstancesKey javaType = new JavaTransitiveInstancesKey(_wrapperClassForType);
        final Function1<JavaTransitiveInstancesKey, Boolean> _function_1 = (JavaTransitiveInstancesKey it_1) -> {
          return Boolean.valueOf(Objects.equal(it_1, javaType));
        };
        boolean _exists = IterableExtensions.<JavaTransitiveInstancesKey>exists(eJavaTypes, _function_1);
        _xblockexpression = (!_exists);
      }
      return Boolean.valueOf(_xblockexpression);
    };
    final Iterable<EDataTypeInSlotsKey> eDataTypes = IterableExtensions.<EDataTypeInSlotsKey>filter(this.minimizeEDataTypeList(IterableExtensions.<EDataTypeInSlotsKey>filterNull(Iterables.<EDataTypeInSlotsKey>filter(types, EDataTypeInSlotsKey.class))), _function);
    final Iterable<EClassTransitiveInstancesKey> eClassTypes = this.minimizeEClassKeyList(IterableExtensions.<EClassTransitiveInstancesKey>filterNull(Iterables.<EClassTransitiveInstancesKey>filter(types, EClassTransitiveInstancesKey.class)), mergeWithSupertypes);
    return IterableExtensions.<IInputKey>toSet(Iterables.<IInputKey>concat(eClassTypes, Iterables.<IInputKey>concat(eDataTypes, eJavaTypes)));
  }
  
  private Iterable<EClassTransitiveInstancesKey> minimizeEClassKeyList(final Iterable<EClassTransitiveInstancesKey> types, final boolean mergeWithSupertypes) {
    final Function1<EClassTransitiveInstancesKey, EClass> _function = (EClassTransitiveInstancesKey it) -> {
      return it.getEmfKey();
    };
    final HashSet<EClass> emfTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(IterableExtensions.<EClassTransitiveInstancesKey, EClass>map(types, _function), EClass.class)));
    final Function1<EClass, EClassTransitiveInstancesKey> _function_1 = (EClass it) -> {
      return new EClassTransitiveInstancesKey(it);
    };
    return IterableExtensions.<EClass, EClassTransitiveInstancesKey>map(this.minimizeEClassList(emfTypes, mergeWithSupertypes), _function_1);
  }
  
  private Iterable<EClass> minimizeEClassList(final Iterable<EClass> types, final boolean mergeWithSupertypes) {
    final Function1<EClass, Boolean> _function = (EClass it) -> {
      return Boolean.valueOf(((!Objects.equal("EObject", it.getName())) || (!Objects.equal(it.getEPackage().getNsURI(), EcorePackage.eNS_URI))));
    };
    final Iterable<EClass> nonTopTypes = IterableExtensions.<EClass>filter(types, _function);
    final HashSet<EClass> emfTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(nonTopTypes, EClass.class)));
    final Consumer<EClass> _function_1 = (EClass key) -> {
      emfTypes.removeAll(key.getEAllSuperTypes());
    };
    nonTopTypes.forEach(_function_1);
    if ((mergeWithSupertypes && (emfTypes.size() > 1))) {
      final Function1<EClass, Iterable<EClass>> _function_2 = (EClass key) -> {
        final Function1<EClass, EClass> _function_3 = (EClass current) -> {
          EClass _xblockexpression = null;
          {
            final EClassifier type = EcoreUtil2.getCompatibleType(key, current, null);
            EClass _xifexpression = null;
            if ((type instanceof EClass)) {
              _xifexpression = ((EClass)type);
            } else {
              _xifexpression = current;
            }
            _xblockexpression = _xifexpression;
          }
          return _xblockexpression;
        };
        return IterableExtensions.<EClass, EClass>map(emfTypes, _function_3);
      };
      final Iterable<EClass> compatibleTypes = Iterables.<EClass>concat(IterableExtensions.<EClass, Iterable<EClass>>map(emfTypes, _function_2));
      final Function1<EClass, Boolean> _function_3 = (EClass it) -> {
        final Function1<EClass, Boolean> _function_4 = (EClass supertype) -> {
          final Function1<EClass, Boolean> _function_5 = (EClass it_1) -> {
            return Boolean.valueOf(Objects.equal(it_1, supertype));
          };
          return Boolean.valueOf(IterableExtensions.<EClass>exists(compatibleTypes, _function_5));
        };
        boolean _exists = IterableExtensions.<EClass>exists(EcoreUtil2.getAllSuperTypes(it), _function_4);
        return Boolean.valueOf((!_exists));
      };
      final Set<EClass> filteredTypes = IterableExtensions.<EClass>toSet(IterableExtensions.<EClass>filter(compatibleTypes, _function_3));
      int _size = filteredTypes.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        return CollectionLiterals.<EClass>newHashSet(EcorePackage.Literals.EOBJECT);
      } else {
        return filteredTypes;
      }
    }
    return emfTypes;
  }
  
  private Iterable<EDataTypeInSlotsKey> minimizeEDataTypeList(final Iterable<EDataTypeInSlotsKey> types) {
    final Function1<EDataTypeInSlotsKey, EDataType> _function = (EDataTypeInSlotsKey it) -> {
      return it.getEmfKey();
    };
    final Function1<EDataType, Boolean> _function_1 = (EDataType it) -> {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(it.getInstanceClassName());
      return Boolean.valueOf((!_isNullOrEmpty));
    };
    final HashSet<EDataType> emfTypes = CollectionLiterals.<EDataType>newHashSet(((EDataType[])Conversions.unwrapArray(IterableExtensions.<EDataType>filter(IterableExtensions.<EDataTypeInSlotsKey, EDataType>map(types, _function), _function_1), EDataType.class)));
    Object _clone = emfTypes.clone();
    final HashSet<EDataType> result = ((HashSet<EDataType>) _clone);
    final Iterator<EDataType> it = emfTypes.iterator();
    while (it.hasNext()) {
      {
        final EDataType dataType = it.next();
        final Function1<EDataType, Boolean> _function_2 = (EDataType it_1) -> {
          String _instanceClassName = it_1.getInstanceClassName();
          String _instanceClassName_1 = dataType.getInstanceClassName();
          return Boolean.valueOf(Objects.equal(_instanceClassName, _instanceClassName_1));
        };
        CollectionExtensions.<EDataType>removeAll(result, IterableExtensions.<EDataType>drop(IterableExtensions.<EDataType>filter(emfTypes, _function_2), 1));
      }
    }
    final Function1<EDataType, EDataTypeInSlotsKey> _function_2 = (EDataType it_1) -> {
      return new EDataTypeInSlotsKey(it_1);
    };
    return IterableExtensions.<EDataType, EDataTypeInSlotsKey>map(result, _function_2);
  }
  
  private Iterable<JavaTransitiveInstancesKey> minimizeJavaTypeList(final Iterable<JavaTransitiveInstancesKey> types) {
    final Function1<JavaTransitiveInstancesKey, Boolean> _function = (JavaTransitiveInstancesKey it) -> {
      return Boolean.valueOf(((it.getInstanceClass() != null) && (!Objects.equal(it.getInstanceClass(), Object.class))));
    };
    final Function1<JavaTransitiveInstancesKey, Class<?>> _function_1 = (JavaTransitiveInstancesKey it) -> {
      return it.getInstanceClass();
    };
    final Iterable<Class<?>> nonTopTypes = IterableExtensions.<Class<?>>filterNull(IterableExtensions.<JavaTransitiveInstancesKey, Class<?>>map(IterableExtensions.<JavaTransitiveInstancesKey>filter(types, _function), _function_1));
    final HashSet<Class<?>> javaTypes = CollectionLiterals.<Class<?>>newHashSet(((Class<?>[])Conversions.unwrapArray(nonTopTypes, Class.class)));
    final Consumer<Class<?>> _function_2 = (Class<?> key) -> {
      final Function1<Class<?>, Boolean> _function_3 = (Class<?> it) -> {
        return Boolean.valueOf(((!Objects.equal(it, key)) && key.isAssignableFrom(it)));
      };
      CollectionExtensions.<Class<?>>removeAll(javaTypes, IterableExtensions.<Class<?>>filter(javaTypes, _function_3));
    };
    nonTopTypes.forEach(_function_2);
    final Function1<Class<?>, JavaTransitiveInstancesKey> _function_3 = (Class<?> it) -> {
      return new JavaTransitiveInstancesKey(it);
    };
    return IterableExtensions.<Class<?>, JavaTransitiveInstancesKey>map(javaTypes, _function_3);
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> addTypeInformation(final Set<IInputKey> types, final IInputKey newType) {
    return this.minimizeTypeInformation(ImmutableSet.<IInputKey>builder().addAll(types).add(newType).build(), false);
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> addTypeInformation(final Set<IInputKey> types, final Set<IInputKey> newTypes) {
    return this.minimizeTypeInformation(ImmutableSet.<IInputKey>builder().addAll(types).addAll(newTypes).build(), 
      false);
  }
  
  /**
   * @since 1.3
   * @return True if the given classifiers has a common subtype in the selected EPackages.
   */
  public boolean hasCommonSubtype(final Set<IInputKey> typeKeys, final Iterable<EPackage> ePackages) {
    boolean _xblockexpression = false;
    {
      final Function1<EPackage, Set<EClass>> _function = (EPackage it) -> {
        return EMFTypeSystem.getAllEClassifiers(it);
      };
      final Iterable<EClass> knownTypes = Iterables.<EClass>concat(IterableExtensions.<EPackage, Set<EClass>>map(ePackages, _function));
      boolean _xifexpression = false;
      final Function1<IInputKey, Boolean> _function_1 = (IInputKey it) -> {
        return Boolean.valueOf((it instanceof EClassTransitiveInstancesKey));
      };
      boolean _forall = IterableExtensions.<IInputKey>forall(typeKeys, _function_1);
      if (_forall) {
        boolean _xblockexpression_1 = false;
        {
          final Function1<IInputKey, EClass> _function_2 = (IInputKey it) -> {
            return ((EClassTransitiveInstancesKey) it).getEmfKey();
          };
          final Set<EClass> classifiers = IterableExtensions.<EClass>toSet(IterableExtensions.<IInputKey, EClass>map(typeKeys, _function_2));
          final Function1<EClass, Boolean> _function_3 = (EClass it) -> {
            return Boolean.valueOf(it.getEAllSuperTypes().containsAll(classifiers));
          };
          _xblockexpression_1 = IterableExtensions.<EClass>exists(knownTypes, _function_3);
        }
        _xifexpression = _xblockexpression_1;
      } else {
        return false;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private static Set<EClass> getAllEClassifiers(final EPackage ePackage) {
    return CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(Iterables.<EClass>filter(ePackage.getEClassifiers(), EClass.class), EClass.class)));
  }
  
  @Override
  public String typeString(final IInputKey type) {
    String _switchResult = null;
    boolean _matched = false;
    if ((type == null)) {
      _matched=true;
      _switchResult = "«null»";
    }
    if (!_matched) {
      if (((type instanceof EClassTransitiveInstancesKey) && (!((EClassTransitiveInstancesKey) type).getEmfKey().eIsProxy()))) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        String _nsURI = ((EClassTransitiveInstancesKey) type).getEmfKey().getEPackage().getNsURI();
        _builder.append(_nsURI);
        _builder.append("::");
        String _name = ((EClassTransitiveInstancesKey) type).getEmfKey().getName();
        _builder.append(_name);
        _switchResult = _builder.toString();
      }
    }
    if (!_matched) {
      if (type instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        StringConcatenation _builder_1 = new StringConcatenation();
        String _string = ((EClassTransitiveInstancesKey)type).getEmfKey().toString();
        _builder_1.append(_string);
        _switchResult = _builder_1.toString();
      }
    }
    if (!_matched) {
      if (type instanceof EDataTypeInSlotsKey) {
        _matched=true;
        StringConcatenation _builder_1 = new StringConcatenation();
        String _nsURI_1 = ((EDataTypeInSlotsKey)type).getEmfKey().getEPackage().getNsURI();
        _builder_1.append(_nsURI_1);
        _builder_1.append("::");
        String _name_1 = ((EDataTypeInSlotsKey)type).getEmfKey().getName();
        _builder_1.append(_name_1);
        _switchResult = _builder_1.toString();
      }
    }
    if (!_matched) {
      _switchResult = super.typeString(type);
    }
    return _switchResult;
  }
  
  /**
   * @since 1.3
   */
  @Override
  public Set<IInputKey> getCompatibleSupertypes(final Set<IInputKey> types) {
    Set<IInputKey> _xifexpression = null;
    final Function1<IInputKey, Boolean> _function = (IInputKey it) -> {
      return Boolean.valueOf((it instanceof EClassTransitiveInstancesKey));
    };
    boolean _forall = IterableExtensions.<IInputKey>forall(types, _function);
    if (_forall) {
      _xifexpression = this.getCompatibleEClasses(Iterables.<EClassTransitiveInstancesKey>filter(types, EClassTransitiveInstancesKey.class));
    } else {
      final Function1<IInputKey, Boolean> _function_1 = (IInputKey it) -> {
        return Boolean.valueOf((it instanceof JavaTransitiveInstancesKey));
      };
      boolean _forall_1 = IterableExtensions.<IInputKey>forall(types, _function_1);
      if (_forall_1) {
        return types;
      } else {
        return types;
      }
    }
    return _xifexpression;
  }
  
  private Set<IInputKey> getCompatibleEClasses(final Iterable<EClassTransitiveInstancesKey> types) {
    final Function1<EClassTransitiveInstancesKey, List<EClass>> _function = (EClassTransitiveInstancesKey it) -> {
      return EcoreUtil2.getCompatibleTypesOf(it.getEmfKey());
    };
    final Iterable<List<EClass>> candidates = IterableExtensions.<EClassTransitiveInstancesKey, List<EClass>>map(types, _function);
    final Iterator<List<EClass>> iterator = candidates.iterator();
    boolean _hasNext = iterator.hasNext();
    if (_hasNext) {
      final Set<EClass> compatibleTypes = CollectionLiterals.<EClass>newHashSet(((EClass[])Conversions.unwrapArray(iterator.next(), EClass.class)));
      while (iterator.hasNext()) {
        compatibleTypes.retainAll(iterator.next());
      }
      final Function1<EClass, IInputKey> _function_1 = (EClass it) -> {
        EClassTransitiveInstancesKey _eClassTransitiveInstancesKey = new EClassTransitiveInstancesKey(it);
        return ((IInputKey) _eClassTransitiveInstancesKey);
      };
      return IterableExtensions.<IInputKey>toSet(IterableExtensions.<EClass, IInputKey>map(compatibleTypes, _function_1));
    }
    return Collections.<IInputKey>unmodifiableSet(CollectionLiterals.<IInputKey>newHashSet());
  }
  
  @Override
  public boolean isValidType(final Type type) {
    boolean _xblockexpression = false;
    {
      if ((type instanceof ClassType)) {
        final EClassifier classifier = ((ClassType)type).getClassname();
        return ((classifier != null) && (!classifier.eIsProxy()));
      } else {
        if ((type instanceof ReferenceType)) {
          final EStructuralFeature feature = ((ReferenceType)type).getRefname();
          return ((feature != null) && (!feature.eIsProxy()));
        }
      }
      _xblockexpression = super.isValidType(type);
    }
    return _xblockexpression;
  }
}
